      SUBROUTINE wrt_his (ng, tile)
!
!svn $Id$
!================================================== Hernan G. Arango ===
!  Copyright (c) 2002-2020 The ROMS/TOMS Group                         !
!    Licensed under a MIT/X style license                              !
!    See License_ROMS.txt                                              !
!=======================================================================
!                                                                      !
!  This routine writes requested model fields at requested levels      !
!  into history NetCDF file.                                           !
!                                                                      !
!  Notice that only momentum is affected by the full time-averaged     !
!  masks.  If applicable, these mask contains information about        !
!  river runoff and time-dependent wetting and drying variations.      !
!                                                                      !
!=======================================================================
!
      USE mod_param
      USE mod_parallel
      USE mod_coupling
      USE mod_forces
      USE mod_grid
      USE mod_ice
      USE mod_iounits
      USE mod_mixing
      USE mod_ncparam
      USE mod_netcdf
      USE mod_ocean
      USE mod_scalars
      USE mod_stepping
!
      USE nf_fwrite2d_mod,     ONLY : nf_fwrite2d
      USE nf_fwrite3d_mod,     ONLY : nf_fwrite3d
      USE omega_mod,           ONLY : scale_omega
      USE uv_rotate_mod,       ONLY : uv_rotate2d
      USE uv_rotate_mod,       ONLY : uv_rotate3d
      USE strings_mod,         ONLY : FoundError
!
      implicit none
!
!  Imported variable declarations.
!
      integer, intent(in) :: ng, tile
!
!  Local variable declarations.
!
      integer :: LBi, UBi, LBj, UBj
      integer :: Fcount, gfactor, gtype, status
      integer :: i, itrc, j, k
      real(r8) :: scale
      real(r8), allocatable :: Ur2d(:,:)
      real(r8), allocatable :: Vr2d(:,:)
      real(r8), allocatable :: Ur3d(:,:,:)
      real(r8), allocatable :: Vr3d(:,:,:)
      real(r8), allocatable :: Wr3d(:,:,:)
      real(r8) :: cff
      real(r8), allocatable :: strain(:,:,:)
      real(r8), allocatable :: dUde(:,:), dVdx(:,:)
      real(r8), allocatable :: dUdx(:,:), dVde(:,:)
      real(r8), allocatable :: pvor(:,:,:), pvork(:,:,:)
      real(r8), allocatable :: pvori(:,:,:), pvorj(:,:,:)
      real(r8), allocatable :: dUde2d(:,:), dVdx2d(:,:)
!     real(r8), allocatable :: dUdx2d(:,:), dVde2d(:,:)
      real(r8), allocatable :: rdVdx(:,:), rdUde(:,:)
      real(r8), allocatable :: dRdx(:,:), dRde(:,:)
      integer :: k1,k2
      real(r8) :: orho0, fomn_p
      real(r8) :: dRde_pr, dRdx_pr, dRdz_pr, dUdz_pr, dVdz_pr
      real(r8), allocatable :: dUdz(:,:), dVdz(:,:), dRdz(:,:)
      real(r8), allocatable :: dz_r(:,:,:),dz_u(:,:,:),dz_v(:,:,:)
!
!-----------------------------------------------------------------------
!  Set lower and upper tile bounds and staggered variables bounds for
!  this horizontal domain partition.  Notice that if tile=-1, it will
!  set the values for the global grid.
!-----------------------------------------------------------------------
!
      integer :: Istr, IstrB, IstrP, IstrR, IstrT, IstrM, IstrU
      integer :: Iend, IendB, IendP, IendR, IendT
      integer :: Jstr, JstrB, JstrP, JstrR, JstrT, JstrM, JstrV
      integer :: Jend, JendB, JendP, JendR, JendT
      integer :: Istrm3, Istrm2, Istrm1, IstrUm2, IstrUm1
      integer :: Iendp1, Iendp2, Iendp2i, Iendp3
      integer :: Jstrm3, Jstrm2, Jstrm1, JstrVm2, JstrVm1
      integer :: Jendp1, Jendp2, Jendp2i, Jendp3
!
      Istr   =BOUNDS(ng) % Istr   (tile)
      IstrB  =BOUNDS(ng) % IstrB  (tile)
      IstrM  =BOUNDS(ng) % IstrM  (tile)
      IstrP  =BOUNDS(ng) % IstrP  (tile)
      IstrR  =BOUNDS(ng) % IstrR  (tile)
      IstrT  =BOUNDS(ng) % IstrT  (tile)
      IstrU  =BOUNDS(ng) % IstrU  (tile)
      Iend   =BOUNDS(ng) % Iend   (tile)
      IendB  =BOUNDS(ng) % IendB  (tile)
      IendP  =BOUNDS(ng) % IendP  (tile)
      IendR  =BOUNDS(ng) % IendR  (tile)
      IendT  =BOUNDS(ng) % IendT  (tile)
      Jstr   =BOUNDS(ng) % Jstr   (tile)
      JstrB  =BOUNDS(ng) % JstrB  (tile)
      JstrM  =BOUNDS(ng) % JstrM  (tile)
      JstrP  =BOUNDS(ng) % JstrP  (tile)
      JstrR  =BOUNDS(ng) % JstrR  (tile)
      JstrT  =BOUNDS(ng) % JstrT  (tile)
      JstrV  =BOUNDS(ng) % JstrV  (tile)
      Jend   =BOUNDS(ng) % Jend   (tile)
      JendB  =BOUNDS(ng) % JendB  (tile)
      JendP  =BOUNDS(ng) % JendP  (tile)
      JendR  =BOUNDS(ng) % JendR  (tile)
      JendT  =BOUNDS(ng) % JendT  (tile)
!
      Istrm3 =BOUNDS(ng) % Istrm3 (tile)            ! Istr-3
      Istrm2 =BOUNDS(ng) % Istrm2 (tile)            ! Istr-2
      Istrm1 =BOUNDS(ng) % Istrm1 (tile)            ! Istr-1
      IstrUm2=BOUNDS(ng) % IstrUm2(tile)            ! IstrU-2
      IstrUm1=BOUNDS(ng) % IstrUm1(tile)            ! IstrU-1
      Iendp1 =BOUNDS(ng) % Iendp1 (tile)            ! Iend+1
      Iendp2 =BOUNDS(ng) % Iendp2 (tile)            ! Iend+2
      Iendp2i=BOUNDS(ng) % Iendp2i(tile)            ! Iend+2 interior
      Iendp3 =BOUNDS(ng) % Iendp3 (tile)            ! Iend+3
      Jstrm3 =BOUNDS(ng) % Jstrm3 (tile)            ! Jstr-3
      Jstrm2 =BOUNDS(ng) % Jstrm2 (tile)            ! Jstr-2
      Jstrm1 =BOUNDS(ng) % Jstrm1 (tile)            ! Jstr-1
      JstrVm2=BOUNDS(ng) % JstrVm2(tile)            ! JstrV-2
      JstrVm1=BOUNDS(ng) % JstrVm1(tile)            ! JstrV-1
      Jendp1 =BOUNDS(ng) % Jendp1 (tile)            ! Jend+1
      Jendp2 =BOUNDS(ng) % Jendp2 (tile)            ! Jend+2
      Jendp2i=BOUNDS(ng) % Jendp2i(tile)            ! Jend+2 interior
      Jendp3 =BOUNDS(ng) % Jendp3 (tile)            ! Jend+3
!
      SourceFile="ROMS/Utility/wrt_his.F"
!
      LBi=LBOUND(GRID(ng)%h,DIM=1)
      UBi=UBOUND(GRID(ng)%h,DIM=1)
      LBj=LBOUND(GRID(ng)%h,DIM=2)
      UBj=UBOUND(GRID(ng)%h,DIM=2)
!
!-----------------------------------------------------------------------
!  Write out history fields.
!-----------------------------------------------------------------------
!
      IF (FoundError(exit_flag, NoError, 133,                      &
     &               "ROMS/Utility/wrt_his.F")) RETURN
!
!  Set grid type factor to write full (gfactor=1) fields or water
!  points (gfactor=-1) fields only.
!
      gfactor=1
!
!  Set time record index.
!
      HIS(ng)%Rindex=HIS(ng)%Rindex+1
      Fcount=HIS(ng)%load
      HIS(ng)%Nrec(Fcount)=HIS(ng)%Nrec(Fcount)+1
!
!  Write out model time (s).
!
      CALL netcdf_put_fvar (ng, iNLM, HIS(ng)%name,                     &
     &                      TRIM(Vname(idtime,ng)), time(ng:),          &
     &                      (/HIS(ng)%Rindex/), (/1/),                  &
     &                      ncid = HIS(ng)%ncid,                        &
     &                      varid = HIS(ng)%Vid(idtime))
      IF (FoundError(exit_flag, NoError, 158,                      &
     &               "ROMS/Utility/wrt_his.F")) RETURN
!
!  Write time-varying depths of RHO-points.
!
      IF (Hout(idpthR,ng)) THEN
        scale=1.0_r8
        gtype=gfactor*r3dvar
        status=nf_fwrite3d(ng, iNLM, HIS(ng)%ncid, HIS(ng)%Vid(idpthR), &
     &                     HIS(ng)%Rindex, gtype,                       &
     &                     LBi, UBi, LBj, UBj, 1, N(ng), scale,         &
     &                     GRID(ng) % rmask,                            &
     &                     GRID(ng) % z_r)
        IF (FoundError(status, nf90_noerr, 291,                    &
     &                 "ROMS/Utility/wrt_his.F")) THEN
          IF (Master) THEN
            WRITE (stdout,10) TRIM(Vname(1,idpthR)), HIS(ng)%Rindex
          END IF
          exit_flag=3
          ioerror=status
          RETURN
        END IF
      END IF
!-----------------------------------------------------------------------
!  Write strain.
!-----------------------------------------------------------------------
!psl20170702: idStr is absent from varinfo.
!     IF (Hout(idStr,ng)) THEN
!       IF (.not.allocated(strain)) THEN
!         allocate (strain(LBi:UBi,LBj:UBj,N(ng)))
!                   strain(LBi:UBi,LBj:UBj,1:N(ng))=0.0_r8
!         allocate (dUde(LBi:UBi,LBj:UBj))
!                   dUde(LBi:UBi,LBj:UBj)=0.0_r8
!         allocate (dVdx(LBi:UBi,LBj:UBj))
!                   dVdx(LBi:UBi,LBj:UBj)=0.0_r8
!         allocate (dUdx(LBi:UBi,LBj:UBj))
!                   dUdx(LBi:UBi,LBj:UBj)=0.0_r8
!         allocate (dVde(LBi:UBi,LBj:UBj))
!                   dVde(LBi:UBi,LBj:UBj)=0.0_r8
!       END IF
!       DO k=1,N(ng)
!         DO i=LBi+1,UBi-1
!           DO j=LBj+1,UBj-1
!           dVdx(i,j)= ( OCEAN(ng) % v(i  ,j, k, nrhs(ng)) -                &
!    &                   OCEAN(ng) % v(i-1,j, k, nrhs(ng))    )             &
!    &               / GRID(ng) % om_v(i  ,j)
!           dUde(i,j)= ( OCEAN(ng) % u(i  ,j, k, nrhs(ng)) -                &
!    &                   OCEAN(ng) % u(i,j-1, k, nrhs(ng))    )             &
!    &               / GRID(ng) % on_u(i  ,j)
!           dUdx(i,j)= ( OCEAN(ng) % u(i  ,j, k, nrhs(ng)) -                &
!    &                   OCEAN(ng) % u(i-1,j, k, nrhs(ng))    )             &
!    &               / GRID(ng) % om_v(i  ,j)
!           dVde(i,j)= ( OCEAN(ng) % v(i  ,j, k, nrhs(ng)) -                &
!    &                   OCEAN(ng) % v(i,j-1, k, nrhs(ng))    )             &
!    &               / GRID(ng) % on_u(i  ,j)
!           strain(i,j,k)=sqrt(  (dUdx(i,j) - dVde(i,j) )**2 +          &
!    &                       (  dVdx(i,j) + dUde(i,j) )**2  )
!           END DO
!         END DO
!       END DO
!       scale=1.0_r8
!       gtype=gfactor*p3dvar
!       status=nf_fwrite3d(ng, iNLM, HIS(ng)%ncid, HIS(ng)%Vid(idStr),  &
!    &                     HIS(ng)%Rindex, gtype,                       &
!    &                     LBi, UBi, LBj, UBj, 1, N(ng), scale,         &
!    &                     GRID(ng) % pmask,                            &
!    &                     strain)
!         IF (FoundError(status, nf90_noerr,                            &
!    &        356,   "ROMS/Utility/wrt_his.F")) THEN
!         IF (Master) THEN
!           WRITE (stdout,10) TRIM(Vname(1,idStr)), HIS(ng)%Rindex
!         END IF
!         exit_flag=3
!         ioerror=status
!         RETURN
!       END IF
!     END IF
!-----------------------------------------------------------------------
!  Write potential vorticity.
!-----------------------------------------------------------------------
!
      IF (Hout(id3dPV,ng)) THEN !psl20170702: id3dPVI,J,K absent from varinfo.
!     IF (Hout(id3dPVI,ng).or.Hout(id3dPVJ,ng).or.                      &
!    &      Hout(id3dPVK,ng).or.Hout(id3dPV,ng)) THEN
        IF (.not.allocated(pvor)) THEN
          allocate (pvor(LBi:UBi,LBj:UBj,N(ng)))
                    pvor(LBi:UBi,LBj:UBj,1:N(ng))=0.0_r8
        END IF
        IF (.not.allocated(pvori)) THEN
          allocate (pvori(LBi:UBi,LBj:UBj,N(ng)))
                    pvori(LBi:UBi,LBj:UBj,1:N(ng))=0.0_r8
        END IF
        IF (.not.allocated(pvorj)) THEN
          allocate (pvorj(LBi:UBi,LBj:UBj,N(ng)))
                    pvorj(LBi:UBi,LBj:UBj,1:N(ng))=0.0_r8
        END IF
        IF (.not.allocated(pvork)) THEN
          allocate (pvork(LBi:UBi,LBj:UBj,N(ng)))
                    pvork(LBi:UBi,LBj:UBj,1:N(ng))=0.0_r8
        END IF
        IF (.not.allocated(dz_r)) THEN
          allocate (dz_r(LBi:UBi,LBj:UBj,N(ng)))
                    dz_r(LBi:UBi,LBj:UBj,1:N(ng))=0.0_r8
        END IF
        IF (.not.allocated(dz_u)) THEN
          allocate (dz_u(LBi:UBi,LBj:UBj,N(ng)))
                    dz_u(LBi:UBi,LBj:UBj,1:N(ng))=0.0_r8
        END IF
        IF (.not.allocated(dz_v)) THEN
          allocate (dz_v(LBi:UBi,LBj:UBj,N(ng)))
                    dz_v(LBi:UBi,LBj:UBj,1:N(ng))=0.0_r8
        END IF
        IF (.not.allocated(dRdx)) THEN
          allocate (dRdx(LBi:UBi,LBj:UBj))
            dRdx(LBi:UBi,LBj:UBj)=0.0_r8
        END IF
        IF (.not.allocated(dRde)) THEN
          allocate (dRde(LBi:UBi,LBj:UBj))
            dRde(LBi:UBi,LBj:UBj)=0.0_r8
        END IF
        IF (.not.allocated(dRdz)) THEN
          allocate (dRdz(LBi:UBi,LBj:UBj))
            dRdz(LBi:UBi,LBj:UBj)=0.0_r8
        END IF
        IF (.not.allocated(dUdz)) THEN
          allocate (dUdz(LBi:UBi,LBj:UBj))
            dUdz(LBi:UBi,LBj:UBj)=0.0_r8
        END IF
        IF (.not.allocated(dVdz)) THEN
          allocate (dVdz(LBi:UBi,LBj:UBj))
            dVdz(LBi:UBi,LBj:UBj)=0.0_r8
        END IF
        IF (.not.allocated(dUde)) THEN
          allocate (dUde(LBi:UBi,LBj:UBj))
                    dUde(LBi:UBi,LBj:UBj)=0.0_r8
        END IF
        IF (.not.allocated(dVdx)) THEN
          allocate (dVdx(LBi:UBi,LBj:UBj))
                    dVdx(LBi:UBi,LBj:UBj)=0.0_r8
        END IF
!
! Find dz on the various grids
!
      DO j=Jstr,JendR
        DO i=Istr,IendR
          DO k=1,N(ng)
            dz_r(i,j,k)= GRID(ng) % z_w(i,j,k) - GRID(ng) % z_w(i,j,k-1)
          END DO
        END DO
      END DO
      DO k=1,N(ng)
        DO j=Jstr,JendR
          DO i=Istr,IendR
            dz_u(i,j,k) = ( dz_r(i,j,k) + dz_r(i+1,j  ,k) ) / 2
            dz_v(i,j,k) = ( dz_r(i,j,k) + dz_r(i  ,j+1,k) ) / 2
          END DO
        END DO
      END DO
      DO k=1,N(ng)
        DO j=Jstr,JendR
          dz_r(Istr-1,j,k) = dz_r(Istr,j,k)
          dz_u(Istr-1,j,k) = dz_u(Istr,j,k)
          dz_v(Istr-1,j,k) = dz_v(Istr,j,k)
        END DO
        DO i=Istr,IendR
          dz_r(i,Jstr-1,k) = dz_r(i,Jstr,k)
          dz_u(i,Jstr-1,k) = dz_u(i,Jstr,k)
          dz_v(i,Jstr-1,k) = dz_v(i,Jstr,k)
        END DO
        dz_r(Istr-1,Jstr-1,k) = dz_r(Istr,Jstr,k)
        dz_u(Istr-1,Jstr-1,k) = dz_u(Istr,Jstr,k)
        dz_v(Istr-1,Jstr-1,k) = dz_v(Istr,Jstr,k)
      END DO
!
! Start of main loop
!
      DO k=2,N(ng)
!
! find the horizontal derivatives on the psi grid
!
          DO j=Jstr,JendR
            DO i=Istr,IendR
              cff = ( OCEAN(ng) % rho(i  ,j  , k) -                     &
     &                OCEAN(ng) % rho(i-1,j  , k)    )                  &
     &               / GRID(ng) % om_r(i  ,j)  +                        &
     &              ( OCEAN(ng) % rho(i  ,j-1, k) -                     &
     &                OCEAN(ng) % rho(i-1,j-1, k)    )                  &
     &               / GRID(ng) % om_r(i  ,j-1)
              dRdx(i,j) = cff / 2
              cff = ( OCEAN(ng) % rho(i  ,j  , k) -                     &
     &                OCEAN(ng) % rho(i,  j-1, k)    )                  &
     &               / GRID(ng) % on_r(i  ,j)  +                        &
     &              ( OCEAN(ng) % rho(i-1,j  , k) -                     &
     &                OCEAN(ng) % rho(i-1,j-1, k)    )                  &
     &               / GRID(ng) % om_r(i-1,j  )
              dRde(i,j) = cff / 2
              dVdx(i,j)= ( OCEAN(ng) % v(i  ,j, k, nrhs(ng)) -              &
     &                   OCEAN(ng) % v(i-1,j, k, nrhs(ng))    )             &
     &               / GRID(ng) % om_v(i  ,j)
              dUde(i,j)= ( OCEAN(ng) % u(i  ,j, k, nrhs(ng)) -              &
     &                   OCEAN(ng) % u(i,j-1, k, nrhs(ng))    )             &
     &               / GRID(ng) % on_u(i  ,j)
              dUde(i,j)=dUde(i,j)*GRID(ng)%pmask(i,j)
              dVdx(i,j)=dVdx(i,j)*GRID(ng)%pmask(i,j)
              dRde(i,j)=dRde(i,j)*GRID(ng)%pmask(i,j)
              drdx(i,j)=drdx(i,j)*GRID(ng)%pmask(i,j)
            END DO
          END DO
!
! vertical derivatives
!
          DO j=Jstr,JendR
            DO i=Istr,IendR
              cff = ( OCEAN(ng) % rho(i  ,j  ,k  ) -                    &
     &                OCEAN(ng) % rho(i  ,j  ,k-1)   )  /               &
     &              (   GRID(ng) % z_r(i  ,j  ,k  ) -                   &
     &                  GRID(ng) % z_r(i  ,j  ,k-1)   )
              cff = cff +                                               &
     &              ( OCEAN(ng) % rho(i-1,j  ,k  ) -                    &
     &                OCEAN(ng) % rho(i-1,j  ,k-1)   )   /              &
     &              (   GRID(ng) % z_r(i-1,j  ,k  ) -                   &
     &                  GRID(ng) % z_r(i-1,j  ,k-1)   )
              cff = cff +                                               &
     &              ( OCEAN(ng) % rho(i  ,j-1,k  ) -                    &
     &                OCEAN(ng) % rho(i  ,j-1,k-1)   )   /              &
     &              (   GRID(ng) % z_r(i  ,j-1,k  ) -                   &
     &                  GRID(ng) % z_r(i  ,j-1,k-1)   )
              cff = cff +                                               &
     &              ( OCEAN(ng) % rho(i-1,j-1,k  ) -                    &
     &                OCEAN(ng) % rho(i-1,j-1,k-1)   )   /              &
     &              (   GRID(ng) % z_r(i-1,j-1,k  ) -                   &
     &                  GRID(ng) % z_r(i-1,j-1,k-1)   )
              dRdz(i,j) = cff / 4
              cff = ( OCEAN(ng) % v(i  ,j  ,k  ,nrhs(ng)) -                 &
     &                OCEAN(ng) % v(i  ,j  ,k-1,nrhs(ng))   ) / dz_v(i,j,k)
              cff = cff +                                               &
     &           ( OCEAN(ng) % v(i-1,j  ,k  ,nrhs(ng)) -                    &
     &             OCEAN(ng) % v(i-1,j  ,k-1,nrhs(ng))   ) / dz_v(i-1,j,k)
              dVdz(i,j) = cff / 2
              cff = ( OCEAN(ng) % u(i  ,j  ,k  ,nrhs(ng)) -                 &
     &                OCEAN(ng) % u(i  ,j  ,k-1,nrhs(ng))   ) / dz_u(i,j,k)
              cff = cff +                                               &
     &           ( OCEAN(ng) % u(i  ,j-1,k  ,nrhs(ng)) -                    &
     &             OCEAN(ng) % u(i  ,j-1,k-1,nrhs(ng))   ) / dz_u(i,j-1,k)
              dUdz(i,j) = cff / 2
            END DO
          END DO
!
!  Compute potential vorticity (meter-1 second-1)
!  at horizontal PSI-points and vertical RHO-points.
!
          DO j=Jstr,JendR
            DO i=Istr,IendR
              fomn_p=0.25_r8*(GRID(ng)%f(i-1,j-1)+GRID(ng)%f(i-1,j)+    &
     &                  GRID(ng)%f(i,j-1)+GRID(ng)%f(i,j))
              pvori(i,j,k) = -dVdz(i,j) * dRdx(i,j) / rho0
              pvorj(i,j,k) =  dUdz(i,j) * dRde(i,j) / rho0
              pvork(i,j,k) =  dRdz(i,j) *                               &
     &                       (fomn_p + dVdx(i,j) - dUde(i,j) ) / rho0
              pvor(i,j,k) = pvori(i,j,k) + pvorj(i,j,k) + pvork(i,j,k)
            END DO
          END DO
        END DO
!
        IF (Hout(id3dPV,ng)) THEN
        scale=1.0_r8
        gtype=gfactor*p3dvar
        status=nf_fwrite3d(ng, iNLM, HIS(ng)%ncid, HIS(ng)%Vid(id3DPV), &
     &                     HIS(ng)%Rindex, gtype,                       &
     &                     LBi, UBi, LBj, UBj, 1, N(ng), scale,         &
     &                     GRID(ng) % pmask,                            &
     &                     pvor)
          IF (FoundError(status, nf90_noerr,                            &
     &          589, "ROMS/Utility/wrt_his.F")) THEN
            IF (Master) THEN
                WRITE (stdout,10) TRIM(Vname(1,id3DPV)), HIS(ng)%Rindex
            END IF
            exit_flag=3
            ioerror=status
            RETURN
          END IF
        END IF
!psl20170702: id3dPVI,J,K are absent from varinfo.
!       IF (Hout(id3dPVI,ng)) THEN
!         scale=1.0_r8
!         gtype=gfactor*p3dvar
!         status=nf_fwrite3d(ng, iNLM, HIS(ng)%ncid,                    &
!    &                       HIS(ng)%Vid(id3dPVI),HIS(ng)%Rindex, gtype,&
!    &                       LBi, UBi, LBj, UBj, 1, N(ng), scale,       &
!    &                       GRID(ng) % pmask,                          &
!    &                       pvori)
!         IF (FoundError(status, nf90_noerr,                            &
!    &        610, "ROMS/Utility/wrt_his.F")) THEN
!           IF (Master) THEN
!             WRITE (stdout,10) TRIM(Vname(1,id3dPVI)), HIS(ng)%Rindex
!           END IF
!           exit_flag=3
!           ioerror=status
!           RETURN
!         END IF
!       END IF
!
!       IF (Hout(id3dPVJ,ng)) THEN
!         scale=1.0_r8
!         gtype=gfactor*p3dvar
!         status=nf_fwrite3d(ng, iNLM, HIS(ng)%ncid,                    &
!    &                       HIS(ng)%Vid(id3dPVJ),HIS(ng)%Rindex, gtype,&
!    &                       LBi, UBi, LBj, UBj, 1, N(ng), scale,       &
!    &                     GRID(ng) % pmask,                            &
!    &                     pvorj)
!         IF (FoundError(status, nf90_noerr,                            &
!    &          631, "ROMS/Utility/wrt_his.F")) THEN
!           IF (Master) THEN
!               WRITE (stdout,10) TRIM(Vname(1,id3dPVJ)), HIS(ng)%Rindex
!           END IF
!           exit_flag=3
!           ioerror=status
!           RETURN
!         END IF
!       END IF
!
!       IF (Hout(id3dPVK,ng)) THEN
!         scale=1.0_r8
!         gtype=gfactor*p3dvar
!         status=nf_fwrite3d(ng, iNLM, HIS(ng)%ncid,                    &
!    &                       HIS(ng)%Vid(id3dPVK),HIS(ng)%Rindex, gtype,&
!    &                       LBi, UBi, LBj, UBj, 1, N(ng), scale,       &
!    &                       GRID(ng) % pmask,                          &
!    &                       pvork)
!         IF (FoundError(status, nf90_noerr,                            &
!    &          652, "ROMS/Utility/wrt_his.F")) THEN
!           IF (Master) THEN
!             WRITE (stdout,10) TRIM(Vname(1,id3dPVK)), HIS(ng)%Rindex
!           END IF
!           exit_flag=3
!           ioerror=status
!           RETURN
!         END IF
!       END IF
      END IF
!
!  Write time-varying depths of U-points.
!
      IF (Hout(idpthU,ng)) THEN
        scale=1.0_r8
        gtype=gfactor*u3dvar
        DO k=1,N(ng)
          DO j=Jstr-1,Jend+1
            DO i=IstrU-1,Iend+1
              GRID(ng)%z_v(i,j,k)=0.5_r8*(GRID(ng)%z_r(i-1,j,k)+        &
     &                                    GRID(ng)%z_r(i  ,j,k))
            END DO
          END DO
        END DO
        status=nf_fwrite3d(ng, iNLM, HIS(ng)%ncid, HIS(ng)%Vid(idpthU), &
     &                     HIS(ng)%Rindex, gtype,                       &
     &                     LBi, UBi, LBj, UBj, 1, N(ng), scale,         &
     &                     GRID(ng) % umask,                            &
     &                     GRID(ng) % z_v)
        IF (FoundError(status, nf90_noerr, 683,                    &
     &                 "ROMS/Utility/wrt_his.F")) THEN
          IF (Master) THEN
            WRITE (stdout,10) TRIM(Vname(1,idpthU)), HIS(ng)%Rindex
          END IF
          exit_flag=3
          ioerror=status
          RETURN
        END IF
      END IF
!
!  Write time-varying depths of V-points.
!
      IF (Hout(idpthV,ng)) THEN
        scale=1.0_r8
        gtype=gfactor*v3dvar
        DO k=1,N(ng)
          DO j=JstrV-1,Jend+1
            DO i=Istr-1,Iend+1
              GRID(ng)%z_v(i,j,k)=0.5_r8*(GRID(ng)%z_r(i,j-1,k)+       &
     &                                    GRID(ng)%z_r(i,j  ,k))
            END DO
          END DO
        END DO
        status=nf_fwrite3d(ng, iNLM, HIS(ng)%ncid, HIS(ng)%Vid(idpthV), &
     &                     HIS(ng)%Rindex, gtype,                       &
     &                     LBi, UBi, LBj, UBj, 1, N(ng), scale,         &
     &                     GRID(ng) % vmask,                            &
     &                     GRID(ng) % z_v)
        IF (FoundError(status, nf90_noerr, 714,                    &
     &                 "ROMS/Utility/wrt_his.F")) THEN
          IF (Master) THEN
            WRITE (stdout,10) TRIM(Vname(1,idpthV)), HIS(ng)%Rindex
          END IF
          exit_flag=3
          ioerror=status
          RETURN
        END IF
      END IF
!
!  Write time-varying depths of W-points.
!
      IF (Hout(idpthW,ng)) THEN
        scale=1.0_r8
        gtype=gfactor*w3dvar
        status=nf_fwrite3d(ng, iNLM, HIS(ng)%ncid, HIS(ng)%Vid(idpthW), &
     &                     HIS(ng)%Rindex, gtype,                       &
     &                     LBi, UBi, LBj, UBj, 0, N(ng), scale,         &
     &                     GRID(ng) % rmask,                            &
     &                     GRID(ng) % z_w)
        IF (FoundError(status, nf90_noerr, 737,                    &
     &                 "ROMS/Utility/wrt_his.F")) THEN
          IF (Master) THEN
            WRITE (stdout,10) TRIM(Vname(1,idpthW)), HIS(ng)%Rindex
          END IF
          exit_flag=3
          ioerror=status
          RETURN
        END IF
      END IF
!
!  Write out free-surface (m)
!
      IF (Hout(idFsur,ng)) THEN
        scale=1.0_r8
        gtype=gfactor*r2dvar
        status=nf_fwrite2d(ng, iNLM, HIS(ng)%ncid, HIS(ng)%Vid(idFsur), &
     &                     HIS(ng)%Rindex, gtype,                       &
     &                     LBi, UBi, LBj, UBj, scale,                   &
     &                     GRID(ng) % rmask,                            &
     &                     OCEAN(ng) % zeta(:,:,kstp(ng)))
        IF (FoundError(status, nf90_noerr, 766,                    &
     &                 "ROMS/Utility/wrt_his.F")) THEN
          IF (Master) THEN
            WRITE (stdout,10) TRIM(Vname(1,idFsur)), HIS(ng)%Rindex
          END IF
          exit_flag=3
          ioerror=status
          RETURN
        END IF
      END IF
!
!  Write out 2D U-momentum component (m/s).
!
      IF (Hout(idUbar,ng)) THEN
        scale=1.0_r8
        gtype=gfactor*u2dvar
        status=nf_fwrite2d(ng, iNLM, HIS(ng)%ncid, HIS(ng)%Vid(idUbar), &
     &                     HIS(ng)%Rindex, gtype,                       &
     &                     LBi, UBi, LBj, UBj, scale,                   &
     &                     GRID(ng) % umask_full,                       &
     &                     OCEAN(ng) % ubar(:,:,kstp(ng)))
        IF (FoundError(status, nf90_noerr, 832,                    &
     &                 "ROMS/Utility/wrt_his.F")) THEN
          IF (Master) THEN
            WRITE (stdout,10) TRIM(Vname(1,idUbar)), HIS(ng)%Rindex
          END IF
          exit_flag=3
          ioerror=status
          RETURN
        END IF
      END IF
!
!  Write out 2D V-momentum component (m/s).
!
      IF (Hout(idVbar,ng)) THEN
        scale=1.0_r8
        gtype=gfactor*v2dvar
        status=nf_fwrite2d(ng, iNLM, HIS(ng)%ncid, HIS(ng)%Vid(idVbar), &
     &                     HIS(ng)%Rindex, gtype,                       &
     &                     LBi, UBi, LBj, UBj, scale,                   &
     &                     GRID(ng) % vmask_full,                       &
     &                     OCEAN(ng) % vbar(:,:,kstp(ng)))
        IF (FoundError(status, nf90_noerr, 952,                    &
     &                 "ROMS/Utility/wrt_his.F")) THEN
          IF (Master) THEN
            WRITE (stdout,10) TRIM(Vname(1,idVbar)), HIS(ng)%Rindex
          END IF
          exit_flag=3
          ioerror=status
          RETURN
        END IF
      END IF
!
!  Write out 2D Eastward and Northward momentum components (m/s) at
!  RHO-points.
!
      IF (Hout(idu2dE,ng).and.Hout(idv2dN,ng)) THEN
        IF (.not.allocated(Ur2d)) THEN
          allocate (Ur2d(LBi:UBi,LBj:UBj))
            Ur2d(LBi:UBi,LBj:UBj)=0.0_r8
        END IF
        IF (.not.allocated(Vr2d)) THEN
          allocate (Vr2d(LBi:UBi,LBj:UBj))
            Vr2d(LBi:UBi,LBj:UBj)=0.0_r8
        END IF
        CALL uv_rotate2d (ng, tile, .FALSE., .TRUE.,                    &
     &                    LBi, UBi, LBj, UBj,                           &
     &                    GRID(ng) % CosAngler,                         &
     &                    GRID(ng) % SinAngler,                         &
     &                    GRID(ng) % rmask_full,                        &
     &                    OCEAN(ng) % ubar(:,:,kstp(ng)),                   &
     &                    OCEAN(ng) % vbar(:,:,kstp(ng)),                   &
     &                    Ur2d, Vr2d)
        scale=1.0_r8
        gtype=gfactor*r2dvar
        status=nf_fwrite2d(ng, iNLM, HIS(ng)%ncid, HIS(ng)%Vid(idu2dE), &
     &                     HIS(ng)%Rindex, gtype,                       &
     &                     LBi, UBi, LBj, UBj, scale,                   &
     &                     GRID(ng) % rmask_full,                       &
     &                     Ur2d)
        IF (FoundError(status, nf90_noerr, 1092,                    &
     &                 "ROMS/Utility/wrt_his.F")) THEN
          IF (Master) THEN
            WRITE (stdout,10) TRIM(Vname(1,idu2dE)), HIS(ng)%Rindex
          END IF
          exit_flag=3
          ioerror=status
          RETURN
        END IF
        status=nf_fwrite2d(ng, iNLM, HIS(ng)%ncid, HIS(ng)%Vid(idv2dN), &
     &                     HIS(ng)%Rindex, gtype,                       &
     &                     LBi, UBi, LBj, UBj, scale,                   &
     &                     GRID(ng) % rmask_full,                       &
     &                     Vr2d)
        IF (FoundError(status, nf90_noerr, 1109,                    &
     &                 "ROMS/Utility/wrt_his.F")) THEN
          IF (Master) THEN
            WRITE (stdout,10) TRIM(Vname(1,idv2dN)), HIS(ng)%Rindex
          END IF
          exit_flag=3
          ioerror=status
          RETURN
        END IF
        deallocate (Ur2d)
        deallocate (Vr2d)
      END IF
!
!  Write out 3D U-momentum component (m/s).
!
      IF (Hout(idUvel,ng)) THEN
        scale=1.0_r8
        gtype=gfactor*u3dvar
        status=nf_fwrite3d(ng, iNLM, HIS(ng)%ncid, HIS(ng)%Vid(idUvel), &
     &                     HIS(ng)%Rindex, gtype,                       &
     &                     LBi, UBi, LBj, UBj, 1, N(ng), scale,         &
     &                     GRID(ng) % umask_full,                       &
     &                     OCEAN(ng) % u(:,:,:,nrhs(ng)))
        IF (FoundError(status, nf90_noerr, 1136,                    &
     &                 "ROMS/Utility/wrt_his.F")) THEN
          IF (Master) THEN
            WRITE (stdout,10) TRIM(Vname(1,idUvel)), HIS(ng)%Rindex
          END IF
          exit_flag=3
          ioerror=status
          RETURN
        END IF
      END IF
!
!  Write out 3D V-momentum component (m/s).
!
      IF (Hout(idVvel,ng)) THEN
        scale=1.0_r8
        gtype=gfactor*v3dvar
        status=nf_fwrite3d(ng, iNLM, HIS(ng)%ncid, HIS(ng)%Vid(idVvel), &
     &                     HIS(ng)%Rindex, gtype,                       &
     &                     LBi, UBi, LBj, UBj, 1, N(ng), scale,         &
     &                     GRID(ng) % vmask_full,                       &
     &                     OCEAN(ng) % v(:,:,:,nrhs(ng)))
        IF (FoundError(status, nf90_noerr, 1202,                    &
     &                 "ROMS/Utility/wrt_his.F")) THEN
          IF (Master) THEN
            WRITE (stdout,10) TRIM(Vname(1,idVvel)), HIS(ng)%Rindex
          END IF
          exit_flag=3
          ioerror=status
          RETURN
        END IF
      END IF
!
!  Write out 3D Eastward and Northward momentum components (m/s) at
!  RHO-points.
!
      IF (Hout(idu3dE,ng).and.Hout(idv3dN,ng)) THEN
        IF (.not.allocated(Ur3d)) THEN
          allocate (Ur3d(LBi:UBi,LBj:UBj,N(ng)))
          Ur3d(LBi:UBi,LBj:UBj,1:N(ng))=0.0_r8
        END IF
        IF (.not.allocated(Vr3d)) THEN
          allocate (Vr3d(LBi:UBi,LBj:UBj,N(ng)))
          Vr3d(LBi:UBi,LBj:UBj,1:N(ng))=0.0_r8
        END IF
        CALL uv_rotate3d (ng, tile, .FALSE., .TRUE.,                    &
     &                    LBi, UBi, LBj, UBj, 1, N(ng),                 &
     &                    GRID(ng) % CosAngler,                         &
     &                    GRID(ng) % SinAngler,                         &
     &                    GRID(ng) % rmask_full,                        &
     &                    OCEAN(ng) % u(:,:,:,nrhs(ng)),                    &
     &                    OCEAN(ng) % v(:,:,:,nrhs(ng)),                    &
     &                    Ur3d, Vr3d)
        scale=1.0_r8
        gtype=gfactor*r3dvar
        status=nf_fwrite3d(ng, iNLM, HIS(ng)%ncid, HIS(ng)%Vid(idu3dE), &
     &                     HIS(ng)%Rindex, gtype,                       &
     &                     LBi, UBi, LBj, UBj, 1, N(ng), scale,         &
     &                     GRID(ng) % rmask_full,                       &
     &                     Ur3d)
        IF (FoundError(status, nf90_noerr, 1288,                    &
     &                 "ROMS/Utility/wrt_his.F")) THEN
          IF (Master) THEN
            WRITE (stdout,10) TRIM(Vname(1,idu3dE)), HIS(ng)%Rindex
          END IF
          exit_flag=3
          ioerror=status
          RETURN
        END IF
        status=nf_fwrite3d(ng, iNLM, HIS(ng)%ncid, HIS(ng)%Vid(idv3dN), &
     &                     HIS(ng)%Rindex, gtype,                       &
     &                     LBi, UBi, LBj, UBj, 1, N(ng), scale,         &
     &                     GRID(ng) % rmask_full,                       &
     &                     Vr3d)
        IF (FoundError(status, nf90_noerr, 1305,                    &
     &                 "ROMS/Utility/wrt_his.F")) THEN
          IF (Master) THEN
            WRITE (stdout,10) TRIM(Vname(1,idv3dN)), HIS(ng)%Rindex
          END IF
          exit_flag=3
          ioerror=status
          RETURN
        END IF
        deallocate (Ur3d)
        deallocate (Vr3d)
      END IF
!
!  Write out S-coordinate omega vertical velocity (m/s).
!
      IF (Hout(idOvel,ng)) THEN
        IF (.not.allocated(Wr3d)) THEN
          allocate (Wr3d(LBi:UBi,LBj:UBj,0:N(ng)))
          Wr3d(LBi:UBi,LBj:UBj,0:N(ng))=0.0_r8
        END IF
        scale=1.0_r8
        gtype=gfactor*w3dvar
        CALL scale_omega (ng, tile, LBi, UBi, LBj, UBj, 0, N(ng),       &
     &                    GRID(ng) % pm,                                &
     &                    GRID(ng) % pn,                                &
     &                    OCEAN(ng) % W,                                &
     &                    Wr3d)
        status=nf_fwrite3d(ng, iNLM, HIS(ng)%ncid, HIS(ng)%Vid(idOvel), &
     &                     HIS(ng)%Rindex, gtype,                       &
     &                     LBi, UBi, LBj, UBj, 0, N(ng), scale,         &
     &                     GRID(ng) % rmask,                            &
     &                     Wr3d)
        IF (FoundError(status, nf90_noerr, 1339,                    &
     &                 "ROMS/Utility/wrt_his.F")) THEN
          IF (Master) THEN
            WRITE (stdout,10) TRIM(Vname(1,idOvel)), HIS(ng)%Rindex
          END IF
          exit_flag=3
          ioerror=status
          RETURN
        END IF
        deallocate (Wr3d)
      END IF
!
!  Write out vertical velocity (m/s).
!
      IF (Hout(idWvel,ng)) THEN
        scale=1.0_r8
        gtype=gfactor*w3dvar
        status=nf_fwrite3d(ng, iNLM, HIS(ng)%ncid, HIS(ng)%Vid(idWvel), &
     &                     HIS(ng)%Rindex, gtype,                       &
     &                     LBi, UBi, LBj, UBj, 0, N(ng), scale,         &
     &                     GRID(ng) % rmask,                            &
     &                     OCEAN(ng) % wvel)
        IF (FoundError(status, nf90_noerr, 1363,                    &
     &                 "ROMS/Utility/wrt_his.F")) THEN
          IF (Master) THEN
            WRITE (stdout,10) TRIM(Vname(1,idWvel)), HIS(ng)%Rindex
          END IF
          exit_flag=3
          ioerror=status
          RETURN
        END IF
      END IF
!
!  Write out tracer type variables.
!
      DO itrc=1,NT(ng)
        IF (Hout(idTvar(itrc),ng)) THEN
          scale=1.0_r8
          gtype=gfactor*r3dvar
          status=nf_fwrite3d(ng, iNLM, HIS(ng)%ncid, HIS(ng)%Tid(itrc), &
     &                       HIS(ng)%Rindex, gtype,                     &
     &                       LBi, UBi, LBj, UBj, 1, N(ng), scale,       &
     &                       GRID(ng) % rmask,                          &
     &                       OCEAN(ng) % t(:,:,:,nrhs(ng),itrc))
          IF (FoundError(status, nf90_noerr, 1387,                  &
     &                   "ROMS/Utility/wrt_his.F")) THEN
            IF (Master) THEN
              WRITE (stdout,10) TRIM(Vname(1,idTvar(itrc))),            &
     &                          HIS(ng)%Rindex
            END IF
            exit_flag=3
            ioerror=status
            RETURN
          END IF
        END IF
      END DO
!----------------------------
!  Write out density anomaly.
!----------------------------
      IF (Hout(idDano,ng)) THEN
        scale=1.0_r8
        gtype=gfactor*r3dvar
        status=nf_fwrite3d(ng, iNLM, HIS(ng)%ncid, HIS(ng)%Vid(idDano), &
     &                     HIS(ng)%Rindex, gtype,                       &
     &                     LBi, UBi, LBj, UBj, 1, N(ng), scale,         &
     &                     GRID(ng) % rmask_full,                       &
     &                     OCEAN(ng) % rho)
        IF (FoundError(status, nf90_noerr, 1806,                    &
     &                 "ROMS/Utility/wrt_his.F")) THEN
          IF (Master) THEN
            WRITE (stdout,10) TRIM(Vname(1,idDano)), HIS(ng)%Rindex
          END IF
          exit_flag=3
          ioerror=status
          RETURN
        END IF
      END IF
!
!  Write out depth surface boundary layer.
!
      IF (Hout(idHsbl,ng)) THEN
        scale=1.0_r8
        gtype=gfactor*r2dvar
        status=nf_fwrite2d(ng, iNLM, HIS(ng)%ncid, HIS(ng)%Vid(idHsbl), &
     &                     HIS(ng)%Rindex, gtype,                       &
     &                     LBi, UBi, LBj, UBj, scale,                   &
     &                     GRID(ng) % rmask_full,                       &
     &                     MIXING(ng) % hsbl)
        IF (FoundError(status, nf90_noerr, 1978,                    &
     &                 "ROMS/Utility/wrt_his.F")) THEN
          IF (Master) THEN
            WRITE (stdout,10) TRIM(Vname(1,idHsbl)), HIS(ng)%Rindex
          END IF
          exit_flag=3
          ioerror=status
          RETURN
        END IF
      END IF
!
!  Write out depth surface boundary layer.
!
      IF (Hout(idHbbl,ng)) THEN
        scale=1.0_r8
        gtype=gfactor*r2dvar
        status=nf_fwrite2d(ng, iNLM, HIS(ng)%ncid, HIS(ng)%Vid(idHbbl), &
     &                     HIS(ng)%Rindex, gtype,                       &
     &                     LBi, UBi, LBj, UBj, scale,                   &
     &                     GRID(ng) % rmask_full,                       &
     &                     MIXING(ng) % hbbl)
        IF (FoundError(status, nf90_noerr, 2003,                    &
     &                 "ROMS/Utility/wrt_his.F")) THEN
          IF (Master) THEN
            WRITE (stdout,10) TRIM(Vname(1,idHbbl)), HIS(ng)%Rindex
          END IF
          exit_flag=3
          ioerror=status
          RETURN
        END IF
      END IF
!
!  Write out vertical viscosity coefficient.
!
      IF (Hout(idVvis,ng)) THEN
        scale=1.0_r8
        gtype=gfactor*w3dvar
        status=nf_fwrite3d(ng, iNLM, HIS(ng)%ncid, HIS(ng)%Vid(idVvis), &
     &                     HIS(ng)%Rindex, gtype,                       &
     &                     LBi, UBi, LBj, UBj, 0, N(ng), scale,         &
     &                     GRID(ng) % rmask_full,                       &
     &                     MIXING(ng) % Akv,                            &
     &                     SetFillVal = .FALSE.)
        IF (FoundError(status, nf90_noerr, 2082,                    &
     &                 "ROMS/Utility/wrt_his.F")) THEN
          IF (Master) THEN
            WRITE (stdout,10) TRIM(Vname(1,idVvis)), HIS(ng)%Rindex
          END IF
          exit_flag=3
          ioerror=status
          RETURN
        END IF
      END IF
!
!  Write out vertical diffusion coefficient for potential temperature.
!
      IF (Hout(idTdif,ng)) THEN
        scale=1.0_r8
        gtype=gfactor*w3dvar
        status=nf_fwrite3d(ng, iNLM, HIS(ng)%ncid, HIS(ng)%Vid(idTdif), &
     &                     HIS(ng)%Rindex, gtype,                       &
     &                     LBi, UBi, LBj, UBj, 0, N(ng), scale,         &
     &                     GRID(ng) % rmask_full,                       &
     &                     MIXING(ng) % Akt(:,:,:,itemp),               &
     &                     SetFillVal = .FALSE.)
        IF (FoundError(status, nf90_noerr, 2106,                    &
     &                 "ROMS/Utility/wrt_his.F")) THEN
          IF (Master) THEN
            WRITE (stdout,10) TRIM(Vname(1,idTdif)), HIS(ng)%Rindex
          END IF
          exit_flag=3
          ioerror=status
          RETURN
        END IF
      END IF
!
!  Write out vertical diffusion coefficient for salinity.
!
      IF (Hout(idSdif,ng)) THEN
        scale=1.0_r8
        gtype=gfactor*w3dvar
        status=nf_fwrite3d(ng, iNLM, HIS(ng)%ncid, HIS(ng)%Vid(idSdif), &
     &                     HIS(ng)%Rindex, gtype,                       &
     &                     LBi, UBi, LBj, UBj, 0, N(ng), scale,         &
     &                     GRID(ng) % rmask_full,                       &
     &                     MIXING(ng) % Akt(:,:,:,isalt),               &
     &                     SetFillVal = .FALSE.)
        IF (FoundError(status, nf90_noerr, 2131,                    &
     &                 "ROMS/Utility/wrt_his.F")) THEN
          IF (Master) THEN
            WRITE (stdout,10) TRIM(Vname(1,idSdif)), HIS(ng)%Rindex
          END IF
          exit_flag=3
          ioerror=status
          RETURN
        END IF
      END IF
!
!  Write out ice 2D momentum component (m/s) in the XI-direction.
!
      IF (Hout(idUice,ng)) THEN
        scale=1.0_r8
        gtype=gfactor*u2dvar
        status=nf_fwrite2d(ng, iNLM, HIS(ng)%ncid,                      &
     &                     HIS(ng)%Vid(idUice),                         &
     &                     HIS(ng)%Rindex, gtype,                       &
     &                     LBi, UBi, LBj, UBj, scale,                   &
     &                     GRID(ng) % umask_full,                       &
     &                     ICE(ng) % ui(:,:,liunw(ng)))
        IF (FoundError(status, nf90_noerr, 2267,                    &
     &                 "ROMS/Utility/wrt_his.F")) THEN
          IF (Master) THEN
            WRITE (stdout,10) TRIM(Vname(1,idUice)),                    &
     &                        HIS(ng)%Rindex
          END IF
          exit_flag=3
          ioerror=status
          RETURN
        END IF
      END IF
!
!  Write out ice 2D momentum component (m/s) in the ETA-direction.
!
      IF (Hout(idVice,ng)) THEN
        scale=1.0_r8
        gtype=gfactor*v2dvar
        status=nf_fwrite2d(ng, iNLM, HIS(ng)%ncid,                      &
     &                     HIS(ng)%Vid(idVice),                         &
     &                     HIS(ng)%Rindex, gtype,                       &
     &                     LBi, UBi, LBj, UBj, scale,                   &
     &                     GRID(ng) % vmask_full,                       &
     &                     ICE(ng) % vi(:,:,liunw(ng)))
        IF (FoundError(status, nf90_noerr, 2292,                    &
     &                 "ROMS/Utility/wrt_his.F")) THEN
          IF (Master) THEN
            WRITE (stdout,10) TRIM(Vname(1,idVice)),                    &
     &                        HIS(ng)%Rindex
          END IF
          exit_flag=3
          ioerror=status
          RETURN
        END IF
      END IF
!
!  Write out 2D Eastward and Northward ice momentum components (m/s) at
!  RHO-points.
!
      IF (Hout(idUiceE,ng).and.Hout(idViceN,ng)) THEN
        IF (.not.allocated(Ur2d)) THEN
          allocate (Ur2d(LBi:UBi,LBj:UBj))
            Ur2d(LBi:UBi,LBj:UBj)=0.0_r8
        END IF
        IF (.not.allocated(Vr2d)) THEN
          allocate (Vr2d(LBi:UBi,LBj:UBj))
            Vr2d(LBi:UBi,LBj:UBj)=0.0_r8
        END IF
        CALL uv_rotate2d (ng, tile, .FALSE., .TRUE.,                    &
     &                    LBi, UBi, LBj, UBj,                           &
     &                    GRID(ng) % CosAngler,                         &
     &                    GRID(ng) % SinAngler,                         &
     &                    GRID(ng) % rmask_full,                        &
     &                    ICE(ng) % ui(:,:,liunw(ng)),                      &
     &                    ICE(ng) % vi(:,:,liunw(ng)),                      &
     &                    Ur2d, Vr2d)
        scale=1.0_r8
        gtype=gfactor*r2dvar
        status=nf_fwrite2d(ng, iNLM, HIS(ng)%ncid, HIS(ng)%Vid(idUiceE),&
     &                     HIS(ng)%Rindex, gtype,                       &
     &                     LBi, UBi, LBj, UBj, scale,                   &
     &                     GRID(ng) % rmask_full,                       &
     &                     Ur2d)
        IF (FoundError(status, nf90_noerr, 2336,                    &
     &                 "ROMS/Utility/wrt_his.F")) THEN
          IF (Master) THEN
            WRITE (stdout,10) TRIM(Vname(1,idUiceE)), HIS(ng)%Rindex
          END IF
          exit_flag=3
          ioerror=status
          RETURN
        END IF
        status=nf_fwrite2d(ng, iNLM, HIS(ng)%ncid, HIS(ng)%Vid(idViceN),&
     &                     HIS(ng)%Rindex, gtype,                       &
     &                     LBi, UBi, LBj, UBj, scale,                   &
     &                     GRID(ng) % rmask_full,                       &
     &                     Vr2d)
        IF (FoundError(status, nf90_noerr, 2353,                    &
     &                 "ROMS/Utility/wrt_his.F")) THEN
          IF (Master) THEN
            WRITE (stdout,10) TRIM(Vname(1,idViceN)), HIS(ng)%Rindex
          END IF
          exit_flag=3
          ioerror=status
          RETURN
        END IF
        deallocate (Ur2d)
        deallocate (Vr2d)
      END IF
!
!  Write out ice concentration
!
      IF (Hout(idAice,ng)) THEN
        scale=1.0_r8
        gtype=gfactor*r2dvar
        status=nf_fwrite2d(ng, iNLM, HIS(ng)%ncid,                      &
     &                     HIS(ng)%Vid(idAice),                         &
     &                     HIS(ng)%Rindex, gtype,                       &
     &                     LBi, UBi, LBj, UBj, scale,                   &
     &                     GRID(ng) % rmask_full,                       &
     &                     ICE(ng) % ai(:,:,liunw(ng)))
        IF (FoundError(status, nf90_noerr, 2379,                    &
     &                 "ROMS/Utility/wrt_his.F")) THEN
          IF (Master) THEN
            WRITE (stdout,10) TRIM(Vname(1,idAice)),                    &
     &                        HIS(ng)%Rindex
          END IF
          exit_flag=3
          ioerror=status
          RETURN
        END IF
      END IF
!
!  Write out ice average thickness
!
      IF (Hout(idHice,ng)) THEN
        scale=1.0_r8
        gtype=gfactor*r2dvar
        status=nf_fwrite2d(ng, iNLM, HIS(ng)%ncid,                      &
     &                     HIS(ng)%Vid(idHice),                         &
     &                     HIS(ng)%Rindex, gtype,                       &
     &                     LBi, UBi, LBj, UBj, scale,                   &
     &                     GRID(ng) % rmask_full,                       &
     &                     ICE(ng) % hi(:,:,liunw(ng)))
        IF (FoundError(status, nf90_noerr, 2404,                    &
     &                 "ROMS/Utility/wrt_his.F")) THEN
          IF (Master) THEN
            WRITE (stdout,10) TRIM(Vname(1,idHice)),                    &
     &                        HIS(ng)%Rindex
          END IF
          exit_flag=3
          ioerror=status
          RETURN
        END IF
      END IF
!
!  Write out snow average thickness
!
      IF (Hout(idHsno,ng)) THEN
        scale=1.0_r8
        gtype=gfactor*r2dvar
        status=nf_fwrite2d(ng, iNLM, HIS(ng)%ncid,                      &
     &                     HIS(ng)%Vid(idHsno),                         &
     &                     HIS(ng)%Rindex, gtype,                       &
     &                     LBi, UBi, LBj, UBj, scale,                   &
     &                     GRID(ng) % rmask_full,                       &
     &                     ICE(ng) % hsn(:,:,liunw(ng)))
        IF (FoundError(status, nf90_noerr, 2429,                    &
     &                 "ROMS/Utility/wrt_his.F")) THEN
          IF (Master) THEN
            WRITE (stdout,10) TRIM(Vname(1,idHsno)),                    &
     &                        HIS(ng)%Rindex
          END IF
          exit_flag=3
          ioerror=status
          RETURN
        END IF
      END IF
!
!  Write out ice age.
!
      IF (Hout(idAgeice,ng)) THEN
        scale=1.0_r8
        gtype=gfactor*r2dvar
        status=nf_fwrite2d(ng, iNLM, HIS(ng)%ncid,                      &
     &                     HIS(ng)%Vid(idAgeice),                       &
     &                     HIS(ng)%Rindex, gtype,                       &
     &                     LBi, UBi, LBj, UBj, scale,                   &
     &                     GRID(ng) % rmask_full,                       &
     &                     ICE(ng) % ageice(:,:,liunw(ng)))
        IF (FoundError(status, nf90_noerr, 2454,                    &
     &                 "ROMS/Utility/wrt_his.F")) THEN
          IF (Master) THEN
            WRITE (stdout,10) TRIM(Vname(1,idAgeice)),                  &
     &                        HIS(ng)%Rindex
          END IF
          exit_flag=3
          ioerror=status
          RETURN
        END IF
      END IF
!
!  Write out ice-ocean mass flux
!
      IF (Hout(idIomflx,ng)) THEN
        scale=1.0_r8
        gtype=gfactor*r2dvar
        status=nf_fwrite2d(ng, iNLM, HIS(ng)%ncid,                      &
     &                     HIS(ng)%Vid(idIomflx),                       &
     &                     HIS(ng)%Rindex, gtype,                       &
     &                     LBi, UBi, LBj, UBj, scale,                   &
     &                     GRID(ng) % rmask_full,                       &
     &                     ICE(ng) % io_mflux)
        IF (FoundError(status, nf90_noerr, 2479,                    &
     &                 "ROMS/Utility/wrt_his.F")) THEN
          IF (Master) THEN
            WRITE (stdout,10) TRIM(Vname(1,idIomflx)),                  &
     &                        HIS(ng)%Rindex
          END IF
          exit_flag=3
          ioerror=status
          RETURN
        END IF
      END IF
!
!  Write out ice/snow surface temperature
!
      IF (Hout(idTice,ng)) THEN
        scale=1.0_r8
        gtype=gfactor*r2dvar
        status=nf_fwrite2d(ng, iNLM, HIS(ng)%ncid,                      &
     &                     HIS(ng)%Vid(idTice),                         &
     &                     HIS(ng)%Rindex, gtype,                       &
     &                     LBi, UBi, LBj, UBj, scale,                   &
     &                     GRID(ng) % rmask_full,                       &
     &                     ICE(ng) % tis)
        IF (FoundError(status, nf90_noerr, 2504,                    &
     &                 "ROMS/Utility/wrt_his.F")) THEN
          IF (Master) THEN
            WRITE (stdout,10) TRIM(Vname(1,idTice)),                    &
     &                        HIS(ng)%Rindex
          END IF
          exit_flag=3
          ioerror=status
          RETURN
        END IF
      END IF
!
!  Write out ice interior temperature
!
      IF (Hout(idTimid,ng)) THEN
        scale=1.0_r8
        gtype=gfactor*r2dvar
        status=nf_fwrite2d(ng, iNLM, HIS(ng)%ncid,                      &
     &                     HIS(ng)%Vid(idTimid),                        &
     &                     HIS(ng)%Rindex, gtype,                       &
     &                     LBi, UBi, LBj, UBj, scale,                   &
     &                     GRID(ng) % rmask_full,                       &
     &                     ICE(ng) % ti(:,:,liunw(ng)))
        IF (FoundError(status, nf90_noerr, 2529,                    &
     &                 "ROMS/Utility/wrt_his.F")) THEN
          IF (Master) THEN
            WRITE (stdout,10) TRIM(Vname(1,idTimid)),                   &
     &                        HIS(ng)%Rindex
          END IF
          exit_flag=3
          ioerror=status
          RETURN
        END IF
      END IF
!
!  Write out ice top temperature
!
      IF (Hout(idT2ice,ng)) THEN
        scale=1.0_r8
        gtype=gfactor*r2dvar
        status=nf_fwrite2d(ng, iNLM, HIS(ng)%ncid,                      &
     &                     HIS(ng)%Vid(idT2ice),                        &
     &                     HIS(ng)%Rindex, gtype,                       &
     &                     LBi, UBi, LBj, UBj, scale,                   &
     &                     GRID(ng) % rmask_full,                       &
     &                     ICE(ng) % t2)
        IF (FoundError(status, nf90_noerr, 2554,                    &
     &                 "ROMS/Utility/wrt_his.F")) THEN
          IF (Master) THEN
            WRITE (stdout,10) TRIM(Vname(1,idT2ice)),                   &
     &                        HIS(ng)%Rindex
          END IF
          exit_flag=3
          ioerror=status
          RETURN
        END IF
      END IF
!
!  Write out internal ice stress component 11
!
      IF (Hout(idSig11,ng)) THEN
        scale=1.0_r8
        gtype=gfactor*r2dvar
        status=nf_fwrite2d(ng, iNLM, HIS(ng)%ncid,                      &
     &                     HIS(ng)%Vid(idSig11),                        &
     &                     HIS(ng)%Rindex, gtype,                       &
     &                     LBi, UBi, LBj, UBj, scale,                   &
     &                     GRID(ng) % rmask,                            &
     &                     ICE(ng) % sig11(:,:,liunw(ng)))
        IF (FoundError(status, nf90_noerr, 2579,                    &
     &                 "ROMS/Utility/wrt_his.F")) THEN
          IF (Master) THEN
            WRITE (stdout,10) TRIM(Vname(1,idSig11)),                   &
     &                        HIS(ng)%Rindex
          END IF
          exit_flag=3
          ioerror=status
          RETURN
        END IF
      END IF
!
!  Write out internal ice stress component 12
!
      IF (Hout(idSig12,ng)) THEN
        scale=1.0_r8
        gtype=gfactor*r2dvar
        status=nf_fwrite2d(ng, iNLM, HIS(ng)%ncid,                      &
     &                     HIS(ng)%Vid(idSig12),                        &
     &                     HIS(ng)%Rindex, gtype,                       &
     &                     LBi, UBi, LBj, UBj, scale,                   &
     &                     GRID(ng) % rmask,                            &
     &                     ICE(ng) % sig12(:,:,liunw(ng)))
        IF (FoundError(status, nf90_noerr, 2604,                    &
     &                 "ROMS/Utility/wrt_his.F")) THEN
          IF (Master) THEN
            WRITE (stdout,10) TRIM(Vname(1,idSig12)),                   &
     &                        HIS(ng)%Rindex
          END IF
          exit_flag=3
          ioerror=status
          RETURN
        END IF
      END IF
!
!  Write out internal ice stress component 22
!
      IF (Hout(idSig22,ng)) THEN
        scale=1.0_r8
        gtype=gfactor*r2dvar
        status=nf_fwrite2d(ng, iNLM, HIS(ng)%ncid,                      &
     &                     HIS(ng)%Vid(idSig22),                        &
     &                     HIS(ng)%Rindex, gtype,                       &
     &                     LBi, UBi, LBj, UBj, scale,                   &
     &                     GRID(ng) % rmask_full,                       &
     &                     ICE(ng) % sig22(:,:,liunw(ng)))
        IF (FoundError(status, nf90_noerr, 2629,                    &
     &                 "ROMS/Utility/wrt_his.F")) THEN
          IF (Master) THEN
            WRITE (stdout,10) TRIM(Vname(1,idSig22)),                   &
     &                        HIS(ng)%Rindex
          END IF
          exit_flag=3
          ioerror=status
          RETURN
        END IF
      END IF
!
!  Write out ice-ocean friction velocity
!
      IF (Hout(idTauiw,ng)) THEN
        scale=1.0_r8
        gtype=gfactor*r2dvar
        status=nf_fwrite2d(ng, iNLM, HIS(ng)%ncid,                      &
     &                     HIS(ng)%Vid(idTauiw),                        &
     &                     HIS(ng)%Rindex, gtype,                       &
     &                     LBi, UBi, LBj, UBj, scale,                   &
     &                     GRID(ng) % rmask_full,                       &
     &                     ICE(ng) % utau_iw)
        IF (FoundError(status, nf90_noerr, 2654,                    &
     &                 "ROMS/Utility/wrt_his.F")) THEN
          IF (Master) THEN
            WRITE (stdout,10) TRIM(Vname(1,idTauiw)),                   &
     &                        HIS(ng)%Rindex
          END IF
          exit_flag=3
          ioerror=status
          RETURN
        END IF
      END IF
!
!  Write out ice-ocean momentum transfer coefficient
!
      IF (Hout(idChuiw,ng)) THEN
        scale=1.0_r8
        gtype=gfactor*r2dvar
        status=nf_fwrite2d(ng, iNLM, HIS(ng)%ncid,                      &
     &                     HIS(ng)%Vid(idChuiw),                        &
     &                     HIS(ng)%Rindex, gtype,                       &
     &                     LBi, UBi, LBj, UBj, scale,                   &
     &                     GRID(ng) % rmask_full,                       &
     &                     ICE(ng) % chu_iw)
        IF (FoundError(status, nf90_noerr, 2679,                    &
     &                 "ROMS/Utility/wrt_his.F")) THEN
          IF (Master) THEN
            WRITE (stdout,10) TRIM(Vname(1,idChuiw)),                   &
     &                        HIS(ng)%Rindex
          END IF
          exit_flag=3
          ioerror=status
          RETURN
        END IF
      END IF
!
!  Write out temperature of molecular sublayer under ice
!
      IF (Hout(idT0mk,ng)) THEN
        scale=1.0_r8
        gtype=gfactor*r2dvar
        status=nf_fwrite2d(ng, iNLM, HIS(ng)%ncid,                      &
     &                     HIS(ng)%Vid(idT0mk),                         &
     &                     HIS(ng)%Rindex, gtype,                       &
     &                     LBi, UBi, LBj, UBj, scale,                   &
     &                     GRID(ng) % rmask,                            &
     &                     ICE(ng) % t0mk)
        IF (FoundError(status, nf90_noerr, 2704,                    &
     &                 "ROMS/Utility/wrt_his.F")) THEN
          IF (Master) THEN
            WRITE (stdout,10) TRIM(Vname(1,idT0mk)),                    &
     &                        HIS(ng)%Rindex
          END IF
          exit_flag=3
          ioerror=status
          RETURN
        END IF
      END IF
!
!  Write out salinity of molecular sublayer under ice
!
      IF (Hout(idS0mk,ng)) THEN
        scale=1.0_r8
        gtype=gfactor*r2dvar
        status=nf_fwrite2d(ng, iNLM, HIS(ng)%ncid,                      &
     &                     HIS(ng)%Vid(idS0mk),                         &
     &                     HIS(ng)%Rindex, gtype,                       &
     &                     LBi, UBi, LBj, UBj, scale,                   &
     &                     GRID(ng) % rmask,                            &
     &                     ICE(ng) % s0mk)
        IF (FoundError(status, nf90_noerr, 2729,                    &
     &                 "ROMS/Utility/wrt_his.F")) THEN
          IF (Master) THEN
            WRITE (stdout,10) TRIM(Vname(1,idS0mk)),                    &
     &                        HIS(ng)%Rindex
          END IF
          exit_flag=3
          ioerror=status
          RETURN
        END IF
      END IF
!
!  Write out ice freeze Wfr
!
      IF (Hout(idWfr,ng)) THEN
        scale=1.0_r8
        gtype=gfactor*r2dvar
        status=nf_fwrite2d(ng, iNLM, HIS(ng)%ncid,                      &
     &                     HIS(ng)%Vid(idWfr),                          &
     &                     HIS(ng)%Rindex, gtype,                       &
     &                     LBi, UBi, LBj, UBj, scale,                   &
     &                     GRID(ng) % rmask,                            &
     &                     ICE(ng) % wfr)
        IF (FoundError(status, nf90_noerr, 2754,                    &
     &                 "ROMS/Utility/wrt_his.F")) THEN
          IF (Master) THEN
            WRITE (stdout,10) TRIM(Vname(1,idWfr)),                     &
     &                        HIS(ng)%Rindex
          END IF
          exit_flag=3
          ioerror=status
          RETURN
        END IF
      END IF
!
!  Write out ice melt/freeze wai
!
      IF (Hout(idWai,ng)) THEN
        scale=1.0_r8
        gtype=gfactor*r2dvar
        status=nf_fwrite2d(ng, iNLM, HIS(ng)%ncid,                      &
     &                     HIS(ng)%Vid(idWai),                          &
     &                     HIS(ng)%Rindex, gtype,                       &
     &                     LBi, UBi, LBj, UBj, scale,                   &
     &                     GRID(ng) % rmask,                            &
     &                     ICE(ng) % wai)
        IF (FoundError(status, nf90_noerr, 2779,                    &
     &                 "ROMS/Utility/wrt_his.F")) THEN
          IF (Master) THEN
            WRITE (stdout,10) TRIM(Vname(1,idWai)),                     &
     &                        HIS(ng)%Rindex
          END IF
          exit_flag=3
          ioerror=status
          RETURN
        END IF
      END IF
!
!  Write out ice melt/freeze Wao
!
      IF (Hout(idWao,ng)) THEN
        scale=1.0_r8
        gtype=gfactor*r2dvar
        status=nf_fwrite2d(ng, iNLM, HIS(ng)%ncid,                      &
     &                     HIS(ng)%Vid(idWao),                          &
     &                     HIS(ng)%Rindex, gtype,                       &
     &                     LBi, UBi, LBj, UBj, scale,                   &
     &                     GRID(ng) % rmask,                            &
     &                     ICE(ng) % wao)
        IF (FoundError(status, nf90_noerr, 2804,                    &
     &                 "ROMS/Utility/wrt_his.F")) THEN
          IF (Master) THEN
            WRITE (stdout,10) TRIM(Vname(1,idWao)),                     &
     &                        HIS(ng)%Rindex
          END IF
          exit_flag=3
          ioerror=status
          RETURN
        END IF
      END IF
!
!  Write out ice melt/freeze wio
!
      IF (Hout(idWio,ng)) THEN
        scale=1.0_r8
        gtype=gfactor*r2dvar
        status=nf_fwrite2d(ng, iNLM, HIS(ng)%ncid,                      &
     &                     HIS(ng)%Vid(idWio),                          &
     &                     HIS(ng)%Rindex, gtype,                       &
     &                     LBi, UBi, LBj, UBj, scale,                   &
     &                     GRID(ng) % rmask,                            &
     &                     ICE(ng) % wio)
        IF (FoundError(status, nf90_noerr, 2829,                    &
     &                 "ROMS/Utility/wrt_his.F")) THEN
          IF (Master) THEN
            WRITE (stdout,10) TRIM(Vname(1,idWio)),                     &
     &                        HIS(ng)%Rindex
          END IF
          exit_flag=3
          ioerror=status
          RETURN
        END IF
      END IF
!
!  Write out ice melt/freeze wro
!
      IF (Hout(idWro,ng)) THEN
        scale=1.0_r8
        gtype=gfactor*r2dvar
        status=nf_fwrite2d(ng, iNLM, HIS(ng)%ncid,                      &
     &                     HIS(ng)%Vid(idWro),                          &
     &                     HIS(ng)%Rindex, gtype,                       &
     &                     LBi, UBi, LBj, UBj, scale,                   &
     &                     GRID(ng) % rmask,                            &
     &                     ICE(ng) % wro)
        IF (FoundError(status, nf90_noerr, 2854,                    &
     &                 "ROMS/Utility/wrt_his.F")) THEN
          IF (Master) THEN
            WRITE (stdout,10) TRIM(Vname(1,idWro)),                     &
     &                        HIS(ng)%Rindex
          END IF
          exit_flag=3
          ioerror=status
          RETURN
        END IF
      END IF
!
!  Write out ice divergence rate
!
      IF (Hout(idWdiv,ng)) THEN
        scale=1.0_r8
        gtype=gfactor*r2dvar
        status=nf_fwrite2d(ng, iNLM, HIS(ng)%ncid,                      &
     &                     HIS(ng)%Vid(idWdiv),                         &
     &                     HIS(ng)%Rindex, gtype,                       &
     &                     LBi, UBi, LBj, UBj, scale,                   &
     &                     GRID(ng) % rmask,                            &
     &                     ICE(ng) % wdiv)
        IF (FoundError(status, nf90_noerr, 2879,                    &
     &                 "ROMS/Utility/wrt_his.F")) THEN
          IF (Master) THEN
            WRITE (stdout,10) TRIM(Vname(1,idWdiv)),                    &
     &                        HIS(ng)%Rindex
          END IF
          exit_flag=3
          ioerror=status
          RETURN
        END IF
      END IF
!
!  Write out surface air pressure.
!
      IF (Hout(idPair,ng)) THEN
        scale=1.0_r8
        gtype=gfactor*r2dvar
        status=nf_fwrite2d(ng, iNLM, HIS(ng)%ncid, HIS(ng)%Vid(idPair), &
     &                     HIS(ng)%Rindex, gtype,                       &
     &                     LBi, UBi, LBj, UBj, scale,                   &
     &                     GRID(ng) % rmask,                            &
     &                     FORCES(ng) % Pair)
        IF (FoundError(status, nf90_noerr, 3408,                    &
     &                 "ROMS/Utility/wrt_his.F")) THEN
          IF (Master) THEN
            WRITE (stdout,10) TRIM(Vname(1,idPair)), HIS(ng)%Rindex
          END IF
          exit_flag=3
          ioerror=status
          RETURN
        END IF
      END IF
!
!  Write out surface air temperature.
!
      IF (Hout(idTair,ng)) THEN
        scale=1.0_r8
        gtype=gfactor*r2dvar
        status=nf_fwrite2d(ng, iNLM, HIS(ng)%ncid, HIS(ng)%Vid(idTair), &
     &                     HIS(ng)%Rindex, gtype,                       &
     &                     LBi, UBi, LBj, UBj, scale,                   &
     &                     GRID(ng) % rmask,                            &
     &                     FORCES(ng) % Tair)
        IF (FoundError(status, nf90_noerr, 3433,                    &
     &                 "ROMS/Utility/wrt_his.F")) THEN
          IF (Master) THEN
            WRITE (stdout,10) TRIM(Vname(1,idTair)), HIS(ng)%Rindex
          END IF
          exit_flag=3
          ioerror=status
          RETURN
        END IF
      END IF
!
!  Write out surface winds.
!
      IF (Hout(idUair,ng)) THEN
        scale=1.0_r8
        gtype=gfactor*r2dvar
        status=nf_fwrite2d(ng, iNLM, HIS(ng)%ncid, HIS(ng)%Vid(idUair), &
     &                     HIS(ng)%Rindex, gtype,                       &
     &                     LBi, UBi, LBj, UBj, scale,                   &
     &                     GRID(ng) % rmask,                            &
     &                     FORCES(ng) % Uwind)
        IF (FoundError(status, nf90_noerr, 3458,                    &
     &                 "ROMS/Utility/wrt_his.F")) THEN
          IF (Master) THEN
            WRITE (stdout,10) TRIM(Vname(1,idUair)), HIS(ng)%Rindex
          END IF
          exit_flag=3
          ioerror=status
          RETURN
        END IF
      END IF
      IF (Hout(idVair,ng)) THEN
        scale=1.0_r8
        gtype=gfactor*r2dvar
        status=nf_fwrite2d(ng, iNLM, HIS(ng)%ncid, HIS(ng)%Vid(idVair), &
     &                     HIS(ng)%Rindex, gtype,                       &
     &                     LBi, UBi, LBj, UBj, scale,                   &
     &                     GRID(ng) % rmask,                            &
     &                     FORCES(ng) % Vwind)
        IF (FoundError(status, nf90_noerr, 3479,                    &
     &                 "ROMS/Utility/wrt_his.F")) THEN
          IF (Master) THEN
            WRITE (stdout,10) TRIM(Vname(1,idVair)), HIS(ng)%Rindex
          END IF
          exit_flag=3
          ioerror=status
          RETURN
        END IF
      END IF
!
!  Write out 2D Eastward and Northward surface winds (m/s) at
!  RHO-points.
!
      IF (Hout(idUairE,ng).and.Hout(idVairN,ng)) THEN
        IF (.not.allocated(Ur2d)) THEN
          allocate (Ur2d(LBi:UBi,LBj:UBj))
            Ur2d(LBi:UBi,LBj:UBj)=0.0_r8
        END IF
        IF (.not.allocated(Vr2d)) THEN
          allocate (Vr2d(LBi:UBi,LBj:UBj))
            Vr2d(LBi:UBi,LBj:UBj)=0.0_r8
        END IF
        CALL uv_rotate2d (ng, tile, .FALSE., .TRUE.,                    &
     &                    LBi, UBi, LBj, UBj,                           &
     &                    GRID(ng) % CosAngler,                         &
     &                    GRID(ng) % SinAngler,                         &
     &                    GRID(ng) % rmask_full,                        &
     &                    FORCES(ng) % Uwind,                           &
     &                    FORCES(ng) % Vwind,                           &
     &                    Ur2d, Vr2d)
        scale=1.0_r8
        gtype=gfactor*r2dvar
        status=nf_fwrite2d(ng, iNLM, HIS(ng)%ncid, HIS(ng)%Vid(idUairE),&
     &                     HIS(ng)%Rindex, gtype,                       &
     &                     LBi, UBi, LBj, UBj, scale,                   &
     &                     GRID(ng) % rmask_full,                       &
     &                     Ur2d)
        IF (FoundError(status, nf90_noerr, 3522,                    &
     &                 "ROMS/Utility/wrt_his.F")) THEN
          IF (Master) THEN
            WRITE (stdout,10) TRIM(Vname(1,idUairE)), HIS(ng)%Rindex
          END IF
          exit_flag=3
          ioerror=status
          RETURN
        END IF
        status=nf_fwrite2d(ng, iNLM, HIS(ng)%ncid, HIS(ng)%Vid(idVairN),&
     &                     HIS(ng)%Rindex, gtype,                       &
     &                     LBi, UBi, LBj, UBj, scale,                   &
     &                     GRID(ng) % rmask_full,                       &
     &                     Vr2d)
        IF (FoundError(status, nf90_noerr, 3539,                    &
     &                 "ROMS/Utility/wrt_his.F")) THEN
          IF (Master) THEN
            WRITE (stdout,10) TRIM(Vname(1,idVairN)), HIS(ng)%Rindex
          END IF
          exit_flag=3
          ioerror=status
          RETURN
        END IF
        deallocate (Ur2d)
        deallocate (Vr2d)
      END IF
!
!  Write out surface active traces fluxes.
!
      DO itrc=1,NAT
        IF (Hout(idTsur(itrc),ng)) THEN
          IF (itrc.eq.itemp) THEN
            scale=rho0*Cp                   ! Celsius m/s to W/m2
          ELSE IF (itrc.eq.isalt) THEN
            scale=1.0_r8
          END IF
          gtype=gfactor*r2dvar
          status=nf_fwrite2d(ng, iNLM, HIS(ng)%ncid,                    &
     &                       HIS(ng)%Vid(idTsur(itrc)),                 &
     &                       HIS(ng)%Rindex, gtype,                     &
     &                       LBi, UBi, LBj, UBj, scale,                 &
     &                       GRID(ng) % rmask,                          &
     &                       FORCES(ng) % stflx(:,:,itrc))
          IF (FoundError(status, nf90_noerr, 3576,                  &
     &                   "ROMS/Utility/wrt_his.F")) THEN
            IF (Master) THEN
              WRITE (stdout,10) TRIM(Vname(1,idTsur(itrc))),            &
     &                          HIS(ng)%Rindex
            END IF
            exit_flag=3
            ioerror=status
            RETURN
          END IF
        END IF
      END DO
!
!  Write out latent heat flux.
!
      IF (Hout(idLhea,ng)) THEN
        scale=rho0*Cp
        gtype=gfactor*r2dvar
        status=nf_fwrite2d(ng, iNLM, HIS(ng)%ncid, HIS(ng)%Vid(idLhea), &
     &                     HIS(ng)%Rindex, gtype,                       &
     &                     LBi, UBi, LBj, UBj, scale,                   &
     &                     GRID(ng) % rmask,                            &
     &                     FORCES(ng) % lhflx)
        IF (FoundError(status, nf90_noerr, 3603,                    &
     &                 "ROMS/Utility/wrt_his.F")) THEN
          IF (Master) THEN
            WRITE (stdout,10) TRIM(Vname(1,idLhea)), HIS(ng)%Rindex
          END IF
          exit_flag=3
          ioerror=status
          RETURN
        END IF
      END IF
!
!  Write out sensible heat flux.
!
      IF (Hout(idShea,ng)) THEN
        scale=rho0*Cp
        gtype=gfactor*r2dvar
        status=nf_fwrite2d(ng, iNLM, HIS(ng)%ncid, HIS(ng)%Vid(idShea), &
     &                     HIS(ng)%Rindex, gtype,                       &
     &                     LBi, UBi, LBj, UBj, scale,                   &
     &                     GRID(ng) % rmask,                            &
     &                     FORCES(ng) % shflx)
        IF (FoundError(status, nf90_noerr, 3626,                    &
     &                 "ROMS/Utility/wrt_his.F")) THEN
          IF (Master) THEN
            WRITE (stdout,10) TRIM(Vname(1,idShea)), HIS(ng)%Rindex
          END IF
          exit_flag=3
          ioerror=status
          RETURN
        END IF
      END IF
!
!  Write out longwave radiation flux.
!
      IF (Hout(idLrad,ng)) THEN
        scale=rho0*Cp
        gtype=gfactor*r2dvar
        status=nf_fwrite2d(ng, iNLM, HIS(ng)%ncid, HIS(ng)%Vid(idLrad), &
     &                     HIS(ng)%Rindex, gtype,                       &
     &                     LBi, UBi, LBj, UBj, scale,                   &
     &                     GRID(ng) % rmask,                            &
     &                     FORCES(ng) % lrflx)
        IF (FoundError(status, nf90_noerr, 3649,                    &
     &                 "ROMS/Utility/wrt_his.F")) THEN
          IF (Master) THEN
            WRITE (stdout,10) TRIM(Vname(1,idLrad)), HIS(ng)%Rindex
          END IF
          exit_flag=3
          ioerror=status
          RETURN
        END IF
      END IF
!
!  Write out evaporation rate (kg/m2/s).
!
      IF (Hout(idevap,ng)) THEN
        scale=1.0_r8
        gtype=gfactor*r2dvar
        status=nf_fwrite2d(ng, iNLM, HIS(ng)%ncid, HIS(ng)%Vid(idevap), &
     &                     HIS(ng)%Rindex, gtype,                       &
     &                     LBi, UBi, LBj, UBj, scale,                   &
     &                     GRID(ng) % rmask,                            &
     &                     FORCES(ng) % evap)
        IF (FoundError(status, nf90_noerr, 3673,                    &
     &                 "ROMS/Utility/wrt_his.F")) THEN
          IF (Master) THEN
            WRITE (stdout,10) TRIM(Vname(1,idevap)), HIS(ng)%Rindex
          END IF
          exit_flag=3
          ioerror=status
          RETURN
        END IF
      END IF
!
!  Write out precipitation rate (kg/m2/s).
!
      IF (Hout(idrain,ng)) THEN
        scale=1.0_r8
        gtype=gfactor*r2dvar
        status=nf_fwrite2d(ng, iNLM, HIS(ng)%ncid, HIS(ng)%Vid(idrain), &
     &                     HIS(ng)%Rindex, gtype,                       &
     &                     LBi, UBi, LBj, UBj, scale,                   &
     &                     GRID(ng) % rmask,                            &
     &                     FORCES(ng) % rain)
        IF (FoundError(status, nf90_noerr, 3696,                    &
     &                 "ROMS/Utility/wrt_his.F")) THEN
          IF (Master) THEN
            WRITE (stdout,10) TRIM(Vname(1,idrain)), HIS(ng)%Rindex
          END IF
          exit_flag=3
          ioerror=status
          RETURN
        END IF
      END IF
!
!  Write out E-P (m/s).
!
      IF (Hout(idEmPf,ng)) THEN
        scale=1.0_r8
        gtype=gfactor*r2dvar
        status=nf_fwrite2d(ng, iNLM, HIS(ng)%ncid, HIS(ng)%Vid(idEmPf), &
     &                     HIS(ng)%Rindex, gtype,                       &
     &                     LBi, UBi, LBj, UBj, scale,                   &
     &                     GRID(ng) % rmask,                            &
     &                     FORCES(ng) % EminusP)
        IF (FoundError(status, nf90_noerr, 3722,                    &
     &                 "ROMS/Utility/wrt_his.F")) THEN
          IF (Master) THEN
            WRITE (stdout,10) TRIM(Vname(1,idEmPf)), HIS(ng)%Rindex
          END IF
          exit_flag=3
          ioerror=status
          RETURN
        END IF
      END IF
!
!  Write out shortwave radiation flux.
!
      IF (Hout(idSrad,ng)) THEN
        scale=rho0*Cp
        gtype=gfactor*r2dvar
        status=nf_fwrite2d(ng, iNLM, HIS(ng)%ncid, HIS(ng)%Vid(idSrad), &
     &                     HIS(ng)%Rindex, gtype,                       &
     &                     LBi, UBi, LBj, UBj, scale,                   &
     &                     GRID(ng) % rmask,                            &
     &                     FORCES(ng) % srflx)
        IF (FoundError(status, nf90_noerr, 3747,                    &
     &                 "ROMS/Utility/wrt_his.F")) THEN
          IF (Master) THEN
            WRITE (stdout,10) TRIM(Vname(1,idSrad)), HIS(ng)%Rindex
          END IF
          exit_flag=3
          ioerror=status
          RETURN
        END IF
      END IF
!
!  Write out surface U-momentum stress.
!
      IF (Hout(idUsms,ng)) THEN
        scale=rho0                          ! m2/s2 to Pa
        gtype=gfactor*u2dvar
        status=nf_fwrite2d(ng, iNLM, HIS(ng)%ncid, HIS(ng)%Vid(idUsms), &
     &                     HIS(ng)%Rindex, gtype,                       &
     &                     LBi, UBi, LBj, UBj, scale,                   &
     &                     GRID(ng) % umask,                            &
     &                     FORCES(ng) % sustr)
        IF (FoundError(status, nf90_noerr, 3776,                    &
     &                 "ROMS/Utility/wrt_his.F")) THEN
          IF (Master) THEN
            WRITE (stdout,10) TRIM(Vname(1,idUsms)), HIS(ng)%Rindex
          END IF
          exit_flag=3
          ioerror=status
          RETURN
        END IF
      END IF
!
!  Write out surface V-momentum stress.
!
      IF (Hout(idVsms,ng)) THEN
        scale=rho0
        gtype=gfactor*v2dvar
        status=nf_fwrite2d(ng, iNLM, HIS(ng)%ncid, HIS(ng)%Vid(idVsms), &
     &                     HIS(ng)%Rindex, gtype,                       &
     &                     LBi, UBi, LBj, UBj, scale,                   &
     &                     GRID(ng) % vmask,                            &
     &                     FORCES(ng) % svstr)
        IF (FoundError(status, nf90_noerr, 3803,                    &
     &                 "ROMS/Utility/wrt_his.F")) THEN
          IF (Master) THEN
            WRITE (stdout,10) TRIM(Vname(1,idVsms)), HIS(ng)%Rindex
          END IF
          exit_flag=3
          ioerror=status
          RETURN
        END IF
      END IF
!
!  Write out bottom U-momentum stress.
!
      IF (Hout(idUbms,ng)) THEN
        scale=-rho0
        gtype=gfactor*u2dvar
        status=nf_fwrite2d(ng, iNLM, HIS(ng)%ncid, HIS(ng)%Vid(idUbms), &
     &                     HIS(ng)%Rindex, gtype,                       &
     &                     LBi, UBi, LBj, UBj, scale,                   &
     &                     GRID(ng) % umask,                            &
     &                     FORCES(ng) % bustr)
        IF (FoundError(status, nf90_noerr, 3826,                    &
     &                 "ROMS/Utility/wrt_his.F")) THEN
          IF (Master) THEN
            WRITE (stdout,10) TRIM(Vname(1,idUbms)), HIS(ng)%Rindex
          END IF
          exit_flag=3
          ioerror=status
          RETURN
        END IF
      END IF
!
!  Write out bottom V-momentum stress.
!
      IF (Hout(idVbms,ng)) THEN
        scale=-rho0
        gtype=gfactor*v2dvar
        status=nf_fwrite2d(ng, iNLM, HIS(ng)%ncid, HIS(ng)%Vid(idVbms), &
     &                     HIS(ng)%Rindex, gtype,                       &
     &                     LBi, UBi, LBj, UBj, scale,                   &
     &                     GRID(ng) % vmask,                            &
     &                     FORCES(ng) % bvstr)
        IF (FoundError(status, nf90_noerr, 3849,                    &
     &                 "ROMS/Utility/wrt_his.F")) THEN
          IF (Master) THEN
            WRITE (stdout,10) TRIM(Vname(1,idVbms)), HIS(ng)%Rindex
          END IF
          exit_flag=3
          ioerror=status
          RETURN
        END IF
      END IF
!
!-----------------------------------------------------------------------
!  Synchronize history NetCDF file to disk to allow other processes
!  to access data immediately after it is written.
!-----------------------------------------------------------------------
!
      CALL netcdf_sync (ng, iNLM, HIS(ng)%name, HIS(ng)%ncid)
      IF (FoundError(exit_flag, NoError, 4814,                      &
     &               "ROMS/Utility/wrt_his.F")) RETURN
      IF (Master) WRITE (stdout,20) kstp(ng), nrhs(ng), HIS(ng)%Rindex
!
  10  FORMAT (/,' WRT_HIS - error while writing variable: ',a,/,11x,    &
     &        'into history NetCDF file for time record: ',i4)
  20  FORMAT (6x,'WRT_HIS     - wrote history', t39,                    &
     &        'fields (Index=',i1,',',i1,') in record = ',i7.7)
      RETURN
      END SUBROUTINE wrt_his
