      SUBROUTINE set_data (ng, tile)
!
!svn $Id$
!================================================== Hernan G. Arango ===
!  Copyright (c) 2002-2020 The ROMS/TOMS Group                         !
!    Licensed under a MIT/X style license                              !
!    See License_ROMS.txt                                              !
!=======================================================================
!                                                                      !
!  This subroutine processes forcing, boundary, climatology, and       !
!  other input data. It time-interpolates between snapshots.           !
!                                                                      !
!=======================================================================
!
      USE mod_param
!
!  Imported variable declarations.
!
      integer, intent(in) :: ng, tile
!
!  Local variable declarations.
!
      integer :: IminS, ImaxS, JminS, JmaxS
      integer :: LBi, UBi, LBj, UBj, LBij, UBij
!
!  Set horizontal starting and ending indices for automatic private
!  storage arrays.
!
      IminS=BOUNDS(ng)%Istr(tile)-3
      ImaxS=BOUNDS(ng)%Iend(tile)+3
      JminS=BOUNDS(ng)%Jstr(tile)-3
      JmaxS=BOUNDS(ng)%Jend(tile)+3
!
!  Determine array lower and upper bounds in the I- and J-directions.
!
      LBi=BOUNDS(ng)%LBi(tile)
      UBi=BOUNDS(ng)%UBi(tile)
      LBj=BOUNDS(ng)%LBj(tile)
      UBj=BOUNDS(ng)%UBj(tile)
!
!  Set array lower and upper bounds for MIN(I,J) directions and
!  MAX(I,J) directions.
!
      LBij=BOUNDS(ng)%LBij
      UBij=BOUNDS(ng)%UBij
!
      CALL wclock_on (ng, iNLM, 4, 28, "ROMS/Nonlinear/set_data.F")
      CALL set_data_tile (ng, tile,                                     &
     &                    LBi, UBi, LBj, UBj,                           &
     &                    IminS, ImaxS, JminS, JmaxS)
      CALL wclock_off (ng, iNLM, 4, 34, "ROMS/Nonlinear/set_data.F")
      RETURN
      END SUBROUTINE set_data
!
!***********************************************************************
      SUBROUTINE set_data_tile (ng, tile,                               &
     &                          LBi, UBi, LBj, UBj,                     &
     &                          IminS, ImaxS, JminS, JmaxS)
!***********************************************************************
!
      USE mod_param
      USE mod_boundary
      USE mod_clima
      USE mod_forces
      USE mod_grid
      USE mod_mixing
      USE mod_ncparam
      USE mod_ocean
      USE mod_stepping
      USE mod_scalars
      USE mod_sources
!
      USE analytical_mod
      USE exchange_2d_mod
      USE set_2dfld_mod
      USE set_ngfld_mod
      USE set_3dfld_mod
      USE mp_exchange_mod, ONLY : mp_exchange2d
      USE mp_exchange_mod, ONLY : mp_exchange3d
      USE strings_mod,     ONLY : FoundError
!
      implicit none
!
!  Imported variable declarations.
!
      integer, intent(in) :: ng, tile
      integer, intent(in) :: LBi, UBi, LBj, UBj
      integer, intent(in) :: IminS, ImaxS, JminS, JmaxS
!
!  Local variable declarations.
!
      logical :: SetBC
      logical :: update = .FALSE.
      integer :: ILB, IUB, JLB, JUB
      integer :: i, ic, itrc, j, k, my_tile
      real(r8) :: cff, cff1, cff2
!
!-----------------------------------------------------------------------
!  Set lower and upper tile bounds and staggered variables bounds for
!  this horizontal domain partition.  Notice that if tile=-1, it will
!  set the values for the global grid.
!-----------------------------------------------------------------------
!
      integer :: Istr, IstrB, IstrP, IstrR, IstrT, IstrM, IstrU
      integer :: Iend, IendB, IendP, IendR, IendT
      integer :: Jstr, JstrB, JstrP, JstrR, JstrT, JstrM, JstrV
      integer :: Jend, JendB, JendP, JendR, JendT
      integer :: Istrm3, Istrm2, Istrm1, IstrUm2, IstrUm1
      integer :: Iendp1, Iendp2, Iendp2i, Iendp3
      integer :: Jstrm3, Jstrm2, Jstrm1, JstrVm2, JstrVm1
      integer :: Jendp1, Jendp2, Jendp2i, Jendp3
!
      Istr   =BOUNDS(ng) % Istr   (tile)
      IstrB  =BOUNDS(ng) % IstrB  (tile)
      IstrM  =BOUNDS(ng) % IstrM  (tile)
      IstrP  =BOUNDS(ng) % IstrP  (tile)
      IstrR  =BOUNDS(ng) % IstrR  (tile)
      IstrT  =BOUNDS(ng) % IstrT  (tile)
      IstrU  =BOUNDS(ng) % IstrU  (tile)
      Iend   =BOUNDS(ng) % Iend   (tile)
      IendB  =BOUNDS(ng) % IendB  (tile)
      IendP  =BOUNDS(ng) % IendP  (tile)
      IendR  =BOUNDS(ng) % IendR  (tile)
      IendT  =BOUNDS(ng) % IendT  (tile)
      Jstr   =BOUNDS(ng) % Jstr   (tile)
      JstrB  =BOUNDS(ng) % JstrB  (tile)
      JstrM  =BOUNDS(ng) % JstrM  (tile)
      JstrP  =BOUNDS(ng) % JstrP  (tile)
      JstrR  =BOUNDS(ng) % JstrR  (tile)
      JstrT  =BOUNDS(ng) % JstrT  (tile)
      JstrV  =BOUNDS(ng) % JstrV  (tile)
      Jend   =BOUNDS(ng) % Jend   (tile)
      JendB  =BOUNDS(ng) % JendB  (tile)
      JendP  =BOUNDS(ng) % JendP  (tile)
      JendR  =BOUNDS(ng) % JendR  (tile)
      JendT  =BOUNDS(ng) % JendT  (tile)
!
      Istrm3 =BOUNDS(ng) % Istrm3 (tile)            ! Istr-3
      Istrm2 =BOUNDS(ng) % Istrm2 (tile)            ! Istr-2
      Istrm1 =BOUNDS(ng) % Istrm1 (tile)            ! Istr-1
      IstrUm2=BOUNDS(ng) % IstrUm2(tile)            ! IstrU-2
      IstrUm1=BOUNDS(ng) % IstrUm1(tile)            ! IstrU-1
      Iendp1 =BOUNDS(ng) % Iendp1 (tile)            ! Iend+1
      Iendp2 =BOUNDS(ng) % Iendp2 (tile)            ! Iend+2
      Iendp2i=BOUNDS(ng) % Iendp2i(tile)            ! Iend+2 interior
      Iendp3 =BOUNDS(ng) % Iendp3 (tile)            ! Iend+3
      Jstrm3 =BOUNDS(ng) % Jstrm3 (tile)            ! Jstr-3
      Jstrm2 =BOUNDS(ng) % Jstrm2 (tile)            ! Jstr-2
      Jstrm1 =BOUNDS(ng) % Jstrm1 (tile)            ! Jstr-1
      JstrVm2=BOUNDS(ng) % JstrVm2(tile)            ! JstrV-2
      JstrVm1=BOUNDS(ng) % JstrVm1(tile)            ! JstrV-1
      Jendp1 =BOUNDS(ng) % Jendp1 (tile)            ! Jend+1
      Jendp2 =BOUNDS(ng) % Jendp2 (tile)            ! Jend+2
      Jendp2i=BOUNDS(ng) % Jendp2i(tile)            ! Jend+2 interior
      Jendp3 =BOUNDS(ng) % Jendp3 (tile)            ! Jend+3
!
!  Lower and upper bounds for nontiled (global values) boundary arrays.
!
      my_tile=-1                           ! for global values
      ILB=BOUNDS(ng)%LBi(my_tile)
      IUB=BOUNDS(ng)%UBi(my_tile)
      JLB=BOUNDS(ng)%LBj(my_tile)
      JUB=BOUNDS(ng)%UBj(my_tile)
!
!-----------------------------------------------------------------------
!  Set surface air temperature (degC).
!-----------------------------------------------------------------------
!
      CALL set_2dfld_tile (ng, tile, iNLM, idTair,                      &
     &                     LBi, UBi, LBj, UBj,                          &
     &                     FORCES(ng)%TairG,                            &
     &                     FORCES(ng)%Tair,                             &
     &                     update)
      IF (FoundError(exit_flag, NoError, 338,                      &
     &               "ROMS/Nonlinear/set_data.F")) RETURN
!
!-----------------------------------------------------------------------
!  Set surface air relative or specific humidity.
!-----------------------------------------------------------------------
!
      CALL set_2dfld_tile (ng, tile, iNLM, idQair,                      &
     &                     LBi, UBi, LBj, UBj,                          &
     &                     FORCES(ng)%HairG,                            &
     &                     FORCES(ng)%Hair,                             &
     &                     update)
      IF (FoundError(exit_flag, NoError, 359,                      &
     &               "ROMS/Nonlinear/set_data.F")) RETURN
!
!
!-----------------------------------------------------------------------
!  Set kinematic surface solar shortwave radiation flux (degC m/s).
!-----------------------------------------------------------------------
!
      CALL set_2dfld_tile (ng, tile, iNLM, idSrad,                      &
     &                     LBi, UBi, LBj, UBj,                          &
     &                     FORCES(ng)%srflxG,                           &
     &                     FORCES(ng)%srflx,                            &
     &                     update)
      IF (FoundError(exit_flag, NoError, 415,                      &
     &               "ROMS/Nonlinear/set_data.F")) RETURN
!psl20171017
      CALL set_2dfld_tile (ng, tile, iNLM, idSpar,                      &
     &                     LBi, UBi, LBj, UBj,                          &
     &                     FORCES(ng)%spflxG,                           &
     &                     FORCES(ng)%spflx,                            &
     &                     update)
      IF (FoundError(exit_flag, NoError, 434,                      &
     &               "ROMS/Nonlinear/set_data.F")) RETURN
!
!-----------------------------------------------------------------------
!  Surface downwelling longwave radiation (degC m/s).
!-----------------------------------------------------------------------
!
      CALL set_2dfld_tile (ng, tile, iNLM, idLdwn,                      &
     &                     LBi, UBi, LBj, UBj,                          &
     &                     FORCES(ng)%lrflxG,                           &
     &                     FORCES(ng)%lrflx,                            &
     &                     update)
      IF (FoundError(exit_flag, NoError, 483,                      &
     &               "ROMS/Nonlinear/set_data.F")) RETURN
!
!-----------------------------------------------------------------------
!  Set surface winds (m/s).
!-----------------------------------------------------------------------
!
      CALL set_2dfld_tile (ng, tile, iNLM, idUair,                      &
     &                     LBi, UBi, LBj, UBj,                          &
     &                     FORCES(ng)%UwindG,                           &
     &                     FORCES(ng)%Uwind,                            &
     &                     update)
      IF (FoundError(exit_flag, NoError, 510,                      &
     &               "ROMS/Nonlinear/set_data.F")) RETURN
      CALL set_2dfld_tile (ng, tile, iNLM, idVair,                      &
     &                     LBi, UBi, LBj, UBj,                          &
     &                     FORCES(ng)%VwindG,                           &
     &                     FORCES(ng)%Vwind,                            &
     &                     update)
      IF (FoundError(exit_flag, NoError, 518,                      &
     &               "ROMS/Nonlinear/set_data.F")) RETURN
!
!  If input point surface winds or interpolated from coarse data, rotate
!  to curvilinear grid.
!
      IF (.not.Linfo(1,idUair,ng).or.                                   &
     &    (Iinfo(5,idUair,ng).ne.Lm(ng)+2).or.                          &
     &    (Iinfo(6,idUair,ng).ne.Mm(ng)+2)) THEN
        DO j=JstrR,JendR
          DO i=IstrR,IendR
            cff1=FORCES(ng)%Uwind(i,j)*GRID(ng)%CosAngler(i,j)+         &
     &           FORCES(ng)%Vwind(i,j)*GRID(ng)%SinAngler(i,j)
            cff2=FORCES(ng)%Vwind(i,j)*GRID(ng)%CosAngler(i,j)-         &
     &           FORCES(ng)%Uwind(i,j)*GRID(ng)%SinAngler(i,j)
            FORCES(ng)%Uwind(i,j)=cff1
            FORCES(ng)%Vwind(i,j)=cff2
          END DO
        END DO
        IF (EWperiodic(ng).or.NSperiodic(ng)) THEN
          CALL exchange_r2d_tile (ng, tile,                             &
     &                            LBi, UBi, LBj, UBj,                   &
     &                            FORCES(ng)%UWind)
          CALL exchange_r2d_tile (ng, tile,                             &
     &                            LBi, UBi, LBj, UBj,                   &
     &                            FORCES(ng)%VWind)
        END IF
        CALL mp_exchange2d (ng, tile, iNLM, 2,                          &
     &                      LBi, UBi, LBj, UBj,                         &
     &                      NghostPoints,                               &
     &                      EWperiodic(ng), NSperiodic(ng),             &
     &                      FORCES(ng)%UWind,                           &
     &                      FORCES(ng)%VWind)
      END IF
!
!-----------------------------------------------------------------------
!  Set rain fall rate (kg/m2/s).
!-----------------------------------------------------------------------
!
      CALL set_2dfld_tile (ng, tile, iNLM, idrain,                      &
     &                     LBi, UBi, LBj, UBj,                          &
     &                     FORCES(ng)%rainG,                            &
     &                     FORCES(ng)%rain,                             &
     &                     update)
      IF (FoundError(exit_flag, NoError, 578,                      &
     &               "ROMS/Nonlinear/set_data.F")) RETURN
!
!-----------------------------------------------------------------------
! Fresh water runoff (kg/m2/s)
!-----------------------------------------------------------------------
!
!
!-----------------------------------------------------------------------
!  Set kinematic bottom net heat flux (degC m/s).
!-----------------------------------------------------------------------
!
      CALL ana_btflux (ng, tile, iNLM, itemp)
!
!-----------------------------------------------------------------------
!  Set kinematic surface freshwater (E-P) flux (m/s).
!-----------------------------------------------------------------------
!
!
!-----------------------------------------------------------------------
!  Set kinematic bottom salt flux (m/s).
!-----------------------------------------------------------------------
!
      CALL ana_btflux (ng, tile, iNLM, isalt)
!
!-----------------------------------------------------------------------
!  Set kinematic surface and bottom passive tracer fluxes (T m/s).
!-----------------------------------------------------------------------
!
      DO itrc=NAT+1,NT(ng)
        CALL ana_stflux (ng, tile, iNLM, itrc)
        CALL ana_btflux (ng, tile, iNLM, itrc)
      END DO
!
!-----------------------------------------------------------------------
!  Set surface air pressure (mb).
!-----------------------------------------------------------------------
!
      SetBC=.TRUE.
!     SetBC=.FALSE.
      CALL set_2dfld_tile (ng, tile, iNLM, idPair,                      &
     &                     LBi, UBi, LBj, UBj,                          &
     &                     FORCES(ng)%PairG,                            &
     &                     FORCES(ng)%Pair,                             &
     &                     update, SetBC)
      IF (FoundError(exit_flag, NoError, 933,                      &
     &               "ROMS/Nonlinear/set_data.F")) RETURN
!
!-----------------------------------------------------------------------
!  Set point Sources/Sinks (river runoff).
!-----------------------------------------------------------------------
!
      IF (DOMAIN(ng)%SouthWest_Test(tile)) THEN
        IF (LuvSrc(ng).or.LwSrc(ng)) THEN
          CALL set_ngfld (ng, iNLM, idRtra, 1, Nsrc(ng), 1,             &
     &                    1, Nsrc(ng), 1,                               &
     &                    SOURCES(ng) % QbarG,                          &
     &                    SOURCES(ng) % Qbar,                           &
     &                    update)
          IF (FoundError(exit_flag, NoError, 1136,                  &
     &                   "ROMS/Nonlinear/set_data.F")) RETURN
          DO k=1,N(ng)
            DO i=1,Nsrc(ng)
              SOURCES(ng)%Qsrc(i,k)=SOURCES(ng)%Qbar(i)*                &
     &                              SOURCES(ng)%Qshape(i,k)
            END DO
          END DO
        END IF
        DO itrc=1,NT(ng)
          IF (LtracerSrc(itrc,ng)) THEN
            CALL set_ngfld (ng, iNLM, idRtrc(itrc), 1, Nsrc(ng), N(ng), &
     &                      1, Nsrc(ng), N(ng),                         &
     &                      SOURCES(ng) % TsrcG(:,:,:,itrc),            &
     &                      SOURCES(ng) % Tsrc(:,:,itrc),               &
     &                      update)
            IF (FoundError(exit_flag, NoError, 1171,                &
     &                     "ROMS/Nonlinear/set_data.F")) RETURN
          END IF
        END DO
      END IF
!
!-----------------------------------------------------------------------
!  Set open boundary conditions fields.
!-----------------------------------------------------------------------
!
!  Free-surface
!
      IF (LprocessOBC(ng)) THEN
        IF (DOMAIN(ng)%SouthWest_Test(tile)) THEN
          IF (LBC(iwest,isFsur,ng)%acquire) THEN
            CALL set_ngfld (ng, iNLM, idZbry(iwest), JLB, JUB, 1,       &
     &                      0, Mm(ng)+1, 1,                             &
     &                      BOUNDARY(ng) % zetaG_west,                  &
     &                      BOUNDARY(ng) % zeta_west,                   &
     &                      update)
            IF (FoundError(exit_flag, NoError, 1230,                &
     &                     "ROMS/Nonlinear/set_data.F")) RETURN
          END IF
          IF (LBC(ieast,isFsur,ng)%acquire) THEN
            CALL set_ngfld (ng, iNLM, idZbry(ieast), JLB, JUB, 1,       &
     &                      0, Mm(ng)+1, 1,                             &
     &                      BOUNDARY(ng) % zetaG_east,                  &
     &                      BOUNDARY(ng) % zeta_east,                   &
     &                      update)
            IF (FoundError(exit_flag, NoError, 1240,                &
     &                     "ROMS/Nonlinear/set_data.F")) RETURN
          END IF
          IF (LBC(isouth,isFsur,ng)%acquire) THEN
            CALL set_ngfld (ng, iNLM, idZbry(isouth), ILB, IUB, 1,      &
     &                      0, Lm(ng)+1 ,1,                             &
     &                      BOUNDARY(ng) % zetaG_south,                 &
     &                      BOUNDARY(ng) % zeta_south,                  &
     &                      update)
            IF (FoundError(exit_flag, NoError, 1250,                &
     &                     "ROMS/Nonlinear/set_data.F")) RETURN
          END IF
          IF (LBC(inorth,isFsur,ng)%acquire) THEN
            CALL set_ngfld (ng, iNLM, idZbry(inorth), ILB, IUB, 1,      &
     &                      0, Lm(ng)+1, 1,                             &
     &                      BOUNDARY(ng) % zetaG_north,                 &
     &                      BOUNDARY(ng) % zeta_north,                  &
     &                      update)
            IF (FoundError(exit_flag, NoError, 1260,                &
     &                     "ROMS/Nonlinear/set_data.F")) RETURN
          END IF
        END IF
      END IF
!
!  2D momentum.
!
      IF (LprocessOBC(ng)) THEN
        IF (DOMAIN(ng)%SouthWest_Test(tile)) THEN
          IF (LBC(iwest,isUbar,ng)%acquire) THEN
            CALL set_ngfld (ng, iNLM, idU2bc(iwest), JLB, JUB, 1,       &
     &                      0, Mm(ng)+1, 1,                             &
     &                      BOUNDARY(ng) % ubarG_west,                  &
     &                      BOUNDARY(ng) % ubar_west,                   &
     &                      update)
            IF (FoundError(exit_flag, NoError, 1357,                &
     &                     "ROMS/Nonlinear/set_data.F")) RETURN
          END IF
          IF (LBC(iwest,isVbar,ng)%acquire) THEN
            CALL set_ngfld (ng, iNLM, idV2bc(iwest), JLB, JUB, 1,       &
     &                      1, Mm(ng)+1, 1,                             &
     &                      BOUNDARY(ng) % vbarG_west,                  &
     &                      BOUNDARY(ng) % vbar_west,                   &
     &                      update)
            IF (FoundError(exit_flag, NoError, 1367,                &
     &                     "ROMS/Nonlinear/set_data.F")) RETURN
          END IF
          IF (LBC(ieast,isUbar,ng)%acquire) THEN
            CALL set_ngfld (ng, iNLM, idU2bc(ieast), JLB, JUB, 1,       &
     &                      0, Mm(ng)+1, 1,                             &
     &                      BOUNDARY(ng) % ubarG_east,                  &
     &                      BOUNDARY(ng) % ubar_east,                   &
     &                      update)
            IF (FoundError(exit_flag, NoError, 1377,                &
     &                     "ROMS/Nonlinear/set_data.F")) RETURN
          END IF
          IF (LBC(ieast,isVbar,ng)%acquire) THEN
            CALL set_ngfld (ng, iNLM, idV2bc(ieast), JLB, JUB, 1,       &
     &                      1, Mm(ng)+1, 1,                             &
     &                      BOUNDARY(ng) % vbarG_east,                  &
     &                      BOUNDARY(ng) % vbar_east,                   &
     &                      update)
            IF (FoundError(exit_flag, NoError, 1387,                &
     &                     "ROMS/Nonlinear/set_data.F")) RETURN
          END IF
          IF (LBC(isouth,isUbar,ng)%acquire) THEN
            CALL set_ngfld (ng, iNLM, idU2bc(isouth), ILB, IUB, 1,      &
     &                      1, Lm(ng)+1, 1,                             &
     &                      BOUNDARY(ng) % ubarG_south,                 &
     &                      BOUNDARY(ng) % ubar_south,                  &
     &                      update)
            IF (FoundError(exit_flag, NoError, 1397,                &
     &                     "ROMS/Nonlinear/set_data.F")) RETURN
          END IF
          IF (LBC(isouth,isVbar,ng)%acquire) THEN
            CALL set_ngfld (ng, iNLM, idV2bc(isouth), ILB, IUB, 1,      &
     &                      0, Lm(ng)+1, 1,                             &
     &                      BOUNDARY(ng) % vbarG_south,                 &
     &                      BOUNDARY(ng) % vbar_south,                  &
     &                      update)
            IF (FoundError(exit_flag, NoError, 1407,                &
     &                     "ROMS/Nonlinear/set_data.F")) RETURN
          END IF
          IF (LBC(inorth,isUbar,ng)%acquire) THEN
            CALL set_ngfld (ng, iNLM, idU2bc(inorth), ILB, IUB, 1,      &
     &                      1, Lm(ng)+1, 1,                             &
     &                      BOUNDARY(ng) % ubarG_north,                 &
     &                      BOUNDARY(ng) % ubar_north,                  &
     &                      update)
            IF (FoundError(exit_flag, NoError, 1417,                &
     &                     "ROMS/Nonlinear/set_data.F")) RETURN
          END IF
          IF (LBC(inorth,isVbar,ng)%acquire) THEN
            CALL set_ngfld (ng, iNLM, idV2bc(inorth), ILB, IUB, 1,      &
     &                      0, Lm(ng)+1, 1,                             &
     &                      BOUNDARY(ng) % vbarG_north,                 &
     &                      BOUNDARY(ng) % vbar_north,                  &
     &                      update)
            IF (FoundError(exit_flag, NoError, 1427,                &
     &                     "ROMS/Nonlinear/set_data.F")) RETURN
          END IF
        END IF
      END IF
!
!  3D momentum.
!
      IF (LprocessOBC(ng)) THEN
        IF (DOMAIN(ng)%SouthWest_Test(tile)) THEN
          IF (LBC(iwest,isUvel,ng)%acquire) THEN
            CALL set_ngfld (ng, iNLM, idU3bc(iwest), JLB, JUB, N(ng),   &
     &                      0, Mm(ng)+1, N(ng),                         &
     &                      BOUNDARY(ng) % uG_west,                     &
     &                      BOUNDARY(ng) % u_west,                      &
     &                      update)
            IF (FoundError(exit_flag, NoError, 1449,                &
     &                     "ROMS/Nonlinear/set_data.F")) RETURN
          END IF
          IF (LBC(iwest,isVvel,ng)%acquire) THEN
            CALL set_ngfld (ng, iNLM, idV3bc(iwest), JLB, JUB, N(ng),   &
     &                      1, Mm(ng)+1, N(ng),                         &
     &                      BOUNDARY(ng) % vG_west,                     &
     &                      BOUNDARY(ng) % v_west,                      &
     &                      update)
            IF (FoundError(exit_flag, NoError, 1459,                &
     &                     "ROMS/Nonlinear/set_data.F")) RETURN
          END IF
          IF (LBC(ieast,isUvel,ng)%acquire) THEN
            CALL set_ngfld (ng, iNLM, idU3bc(ieast), JLB, JUB, N(ng),   &
     &                      0, Mm(ng)+1, N(ng),                         &
     &                      BOUNDARY(ng) % uG_east,                     &
     &                      BOUNDARY(ng) % u_east,                      &
     &                      update)
            IF (FoundError(exit_flag, NoError, 1469,                &
     &                     "ROMS/Nonlinear/set_data.F")) RETURN
          END IF
          IF (LBC(ieast,isVvel,ng)%acquire) THEN
            CALL set_ngfld (ng, iNLM, idV3bc(ieast), JLB, JUB, N(ng),   &
     &                      1, Mm(ng)+1, N(ng),                         &
     &                      BOUNDARY(ng) % vG_east,                     &
     &                      BOUNDARY(ng) % v_east,                      &
     &                      update)
            IF (FoundError(exit_flag, NoError, 1479,                &
     &                     "ROMS/Nonlinear/set_data.F")) RETURN
          END IF
          IF (LBC(isouth,isUvel,ng)%acquire) THEN
            CALL set_ngfld (ng, iNLM, idU3bc(isouth), ILB, IUB, N(ng),  &
     &                      1, Lm(ng)+1, N(ng),                         &
     &                      BOUNDARY(ng) % uG_south,                    &
     &                      BOUNDARY(ng) % u_south,                     &
     &                      update)
            IF (FoundError(exit_flag, NoError, 1489,                &
     &                     "ROMS/Nonlinear/set_data.F")) RETURN
          END IF
          IF (LBC(isouth,isVvel,ng)%acquire) THEN
            CALL set_ngfld (ng, iNLM, idV3bc(isouth), ILB, IUB, N(ng),  &
     &                      0, Lm(ng)+1, N(ng),                         &
     &                      BOUNDARY(ng) % vG_south,                    &
     &                      BOUNDARY(ng) % v_south,                     &
     &                      update)
            IF (FoundError(exit_flag, NoError, 1499,                &
     &                     "ROMS/Nonlinear/set_data.F")) RETURN
          END IF
          IF (LBC(inorth,isUvel,ng)%acquire) THEN
            CALL set_ngfld (ng, iNLM, idU3bc(inorth), ILB, IUB, N(ng),  &
     &                      1, Lm(ng)+1, N(ng),                         &
     &                      BOUNDARY(ng) % uG_north,                    &
     &                      BOUNDARY(ng) % u_north,                     &
     &                      update)
            IF (FoundError(exit_flag, NoError, 1509,                &
     &                     "ROMS/Nonlinear/set_data.F")) RETURN
          END IF
          IF (LBC(inorth,isVvel,ng)%acquire) THEN
            CALL set_ngfld (ng, iNLM, idV3bc(inorth), ILB, IUB, N(ng),  &
     &                      0, Lm(ng)+1, N(ng),                         &
     &                      BOUNDARY(ng) % vG_north,                    &
     &                      BOUNDARY(ng) % v_north,                     &
     &                      update)
            IF (FoundError(exit_flag, NoError, 1519,                &
     &                     "ROMS/Nonlinear/set_data.F")) RETURN
          END IF
        END IF
      END IF
!
!  Tracers.
!
      IF (LprocessOBC(ng)) THEN
        IF (DOMAIN(ng)%SouthWest_Test(tile)) THEN
           DO itrc=1,NT(ng)
            IF (LBC(iwest,isTvar(itrc),ng)%acquire) THEN
              CALL set_ngfld (ng, iNLM, idTbry(iwest,itrc),             &
     &                        JLB, JUB, N(ng), 0, Mm(ng)+1, N(ng),      &
     &                        BOUNDARY(ng) % tG_west(:,:,:,itrc),       &
     &                        BOUNDARY(ng) % t_west(:,:,itrc),          &
     &                        update)
              IF (FoundError(exit_flag, NoError, 1540,              &
     &                       "ROMS/Nonlinear/set_data.F")) RETURN
            END IF
            IF (LBC(ieast,isTvar(itrc),ng)%acquire) THEN
              CALL set_ngfld (ng, iNLM, idTbry(ieast,itrc),             &
     &                        JLB, JUB, N(ng), 0, Mm(ng)+1, N(ng),      &
     &                        BOUNDARY(ng) % tG_east(:,:,:,itrc),       &
     &                        BOUNDARY(ng) % t_east(:,:,itrc),          &
     &                        update)
              IF (FoundError(exit_flag, NoError, 1550,              &
     &                       "ROMS/Nonlinear/set_data.F")) RETURN
            END IF
            IF (LBC(isouth,isTvar(itrc),ng)%acquire) THEN
              CALL set_ngfld (ng, iNLM, idTbry(isouth,itrc),            &
     &                        ILB, IUB, N(ng), 0, Lm(ng)+1, N(ng),      &
     &                        BOUNDARY(ng) % tG_south(:,:,:,itrc),      &
     &                        BOUNDARY(ng) % t_south(:,:,itrc),         &
     &                        update)
              IF (FoundError(exit_flag, NoError, 1560,              &
     &                       "ROMS/Nonlinear/set_data.F")) RETURN
            END IF
            IF (LBC(inorth,isTvar(itrc),ng)%acquire) THEN
              CALL set_ngfld (ng, iNLM, idTbry(inorth,itrc),            &
     &                        ILB, IUB, N(ng), 0, Lm(ng)+1, N(ng),      &
     &                        BOUNDARY(ng) % tG_north(:,:,:,itrc),      &
     &                        BOUNDARY(ng) % t_north(:,:,itrc),         &
     &                        update)
              IF (FoundError(exit_flag, NoError, 1570,              &
     &                       "ROMS/Nonlinear/set_data.F")) RETURN
            END IF
          END DO
        END IF
      END IF
      IF (.not.(RefinedGrid(ng).and.RefineScale(ng).gt.0)) THEN
        IF (DOMAIN(ng)%SouthWest_Test(tile)) THEN
          IF (LBC(iwest,isAice,ng)%acquire) THEN
            CALL set_ngfld (ng, iNLM, idAibc(iwest), JLB, JUB, 1,       &
     &                      0, Mm(ng)+1, 1,                             &
     &                      BOUNDARY(ng) % aiG_west,                    &
     &                      BOUNDARY(ng) % ai_west,                     &
     &                      update)
            IF (FoundError(exit_flag, NoError, 1591,                &
     &                     "ROMS/Nonlinear/set_data.F")) RETURN
          END IF
          IF (LBC(ieast,isAice,ng)%acquire) THEN
            CALL set_ngfld (ng, iNLM, idAibc(ieast), JLB, JUB, 1,       &
     &                      0, Mm(ng)+1, 1,                             &
     &                      BOUNDARY(ng) % aiG_east,                    &
     &                      BOUNDARY(ng) % ai_east,                     &
     &                      update)
            IF (FoundError(exit_flag, NoError, 1600,                &
     &                     "ROMS/Nonlinear/set_data.F")) RETURN
          END IF
          IF (LBC(isouth,isAice,ng)%acquire) THEN
            CALL set_ngfld (ng, iNLM, idAibc(isouth), ILB, IUB, 1,      &
     &                      0, Lm(ng)+1 ,1,                             &
     &                      BOUNDARY(ng) % aiG_south,                   &
     &                      BOUNDARY(ng) % ai_south,                    &
     &                      update)
            IF (FoundError(exit_flag, NoError, 1609,                &
     &                     "ROMS/Nonlinear/set_data.F")) RETURN
          END IF
          IF (LBC(inorth,isAice,ng)%acquire) THEN
            CALL set_ngfld (ng, iNLM, idAibc(inorth), ILB, IUB, 1,      &
     &                      0, Lm(ng)+1, 1,                             &
     &                      BOUNDARY(ng) % aiG_north,                   &
     &                      BOUNDARY(ng) % ai_north,                    &
     &                      update)
            IF (FoundError(exit_flag, NoError, 1618,                &
     &                     "ROMS/Nonlinear/set_data.F")) RETURN
          END IF
        END IF
      END IF
      IF (.not.(RefinedGrid(ng).and.RefineScale(ng).gt.0)) THEN
        IF (DOMAIN(ng)%SouthWest_Test(tile)) THEN
          IF (LBC(iwest,isHice,ng)%acquire) THEN
            CALL set_ngfld (ng, iNLM, idHibc(iwest), JLB, JUB, 1,       &
     &                      0, Mm(ng)+1, 1,                             &
     &                      BOUNDARY(ng) % hiG_west,                    &
     &                      BOUNDARY(ng) % hi_west,                     &
     &                      update)
            IF (FoundError(exit_flag, NoError, 1636,                &
     &                     "ROMS/Nonlinear/set_data.F")) RETURN
          END IF
          IF (LBC(ieast,isHice,ng)%acquire) THEN
            CALL set_ngfld (ng, iNLM, idHibc(ieast), JLB, JUB, 1,       &
     &                      0, Mm(ng)+1, 1,                             &
     &                      BOUNDARY(ng) % hiG_east,                    &
     &                      BOUNDARY(ng) % hi_east,                     &
     &                      update)
            IF (FoundError(exit_flag, NoError, 1645,                &
     &                     "ROMS/Nonlinear/set_data.F")) RETURN
          END IF
          IF (LBC(isouth,isHice,ng)%acquire) THEN
            CALL set_ngfld (ng, iNLM, idHibc(isouth), ILB, IUB, 1,      &
     &                      0, Lm(ng)+1 ,1,                             &
     &                      BOUNDARY(ng) % hiG_south,                   &
     &                      BOUNDARY(ng) % hi_south,                    &
     &                      update)
            IF (FoundError(exit_flag, NoError, 1654,                &
     &                     "ROMS/Nonlinear/set_data.F")) RETURN
          END IF
          IF (LBC(inorth,isHice,ng)%acquire) THEN
            CALL set_ngfld (ng, iNLM, idHibc(inorth), ILB, IUB, 1,      &
     &                      0, Lm(ng)+1, 1,                             &
     &                      BOUNDARY(ng) % hiG_north,                   &
     &                      BOUNDARY(ng) % hi_north,                    &
     &                      update)
            IF (FoundError(exit_flag, NoError, 1663,                &
     &                     "ROMS/Nonlinear/set_data.F")) RETURN
          END IF
        END IF
      END IF
      IF (.not.(RefinedGrid(ng).and.RefineScale(ng).gt.0)) THEN
      END IF
      IF (.not.(RefinedGrid(ng).and.RefineScale(ng).gt.0)) THEN
        IF (DOMAIN(ng)%SouthWest_Test(tile)) THEN
          IF (LBC(iwest,isHsno,ng)%acquire) THEN
            CALL set_ngfld (ng, iNLM, idHsnbc(iwest), JLB, JUB, 1,      &
     &                      0, Mm(ng)+1, 1,                             &
     &                      BOUNDARY(ng) % hsnG_west,                   &
     &                      BOUNDARY(ng) % hsn_west,                    &
     &                      update)
            IF (FoundError(exit_flag, NoError, 1691,                &
     &                     "ROMS/Nonlinear/set_data.F")) RETURN
          END IF
          IF (LBC(ieast,isHsno,ng)%acquire) THEN
            CALL set_ngfld (ng, iNLM, idHsnbc(ieast), JLB, JUB, 1,      &
     &                      0, Mm(ng)+1, 1,                             &
     &                      BOUNDARY(ng) % hsnG_east,                   &
     &                      BOUNDARY(ng) % hsn_east,                    &
     &                      update)
            IF (FoundError(exit_flag, NoError, 1700,                &
     &                     "ROMS/Nonlinear/set_data.F")) RETURN
          END IF
          IF (LBC(isouth,isHsno,ng)%acquire) THEN
            CALL set_ngfld (ng, iNLM, idHsnbc(isouth), ILB, IUB, 1,     &
     &                      0, Lm(ng)+1 ,1,                             &
     &                      BOUNDARY(ng) % hsnG_south,                  &
     &                      BOUNDARY(ng) % hsn_south,                   &
     &                      update)
            IF (FoundError(exit_flag, NoError, 1709,                &
     &                     "ROMS/Nonlinear/set_data.F")) RETURN
          END IF
          IF (LBC(inorth,isHsno,ng)%acquire) THEN
            CALL set_ngfld (ng, iNLM, idHsnbc(inorth), ILB, IUB, 1,     &
     &                      0, Lm(ng)+1, 1,                             &
     &                      BOUNDARY(ng) % hsnG_north,                  &
     &                      BOUNDARY(ng) % hsn_north,                   &
     &                      update)
            IF (FoundError(exit_flag, NoError, 1718,                &
     &                     "ROMS/Nonlinear/set_data.F")) RETURN
          END IF
        END IF
      END IF
      IF (.not.(RefinedGrid(ng).and.RefineScale(ng).gt.0)) THEN
        IF (DOMAIN(ng)%SouthWest_Test(tile)) THEN
          IF (LBC(iwest,isTice,ng)%acquire) THEN
            CALL set_ngfld (ng, iNLM, idTibc(iwest), JLB, JUB, 1,       &
     &                      0, Mm(ng)+1, 1,                             &
     &                      BOUNDARY(ng) % tiG_west,                    &
     &                      BOUNDARY(ng) % ti_west,                     &
     &                      update)
            IF (FoundError(exit_flag, NoError, 1736,                &
     &                     "ROMS/Nonlinear/set_data.F")) RETURN
          END IF
          IF (LBC(ieast,isTice,ng)%acquire) THEN
            CALL set_ngfld (ng, iNLM, idTibc(ieast), JLB, JUB, 1,       &
     &                      0, Mm(ng)+1, 1,                             &
     &                      BOUNDARY(ng) % tiG_east,                    &
     &                      BOUNDARY(ng) % ti_east,                     &
     &                      update)
            IF (FoundError(exit_flag, NoError, 1745,                &
     &                     "ROMS/Nonlinear/set_data.F")) RETURN
          END IF
          IF (LBC(isouth,isTice,ng)%acquire) THEN
            CALL set_ngfld (ng, iNLM, idTibc(isouth), ILB, IUB, 1,      &
     &                      0, Lm(ng)+1 ,1,                             &
     &                      BOUNDARY(ng) % tiG_south,                   &
     &                      BOUNDARY(ng) % ti_south,                    &
     &                      update)
            IF (FoundError(exit_flag, NoError, 1754,                &
     &                     "ROMS/Nonlinear/set_data.F")) RETURN
          END IF
          IF (LBC(inorth,isTice,ng)%acquire) THEN
            CALL set_ngfld (ng, iNLM, idTibc(inorth), ILB, IUB, 1,      &
     &                      0, Lm(ng)+1, 1,                             &
     &                      BOUNDARY(ng) % tiG_north,                   &
     &                      BOUNDARY(ng) % ti_north,                    &
     &                      update)
            IF (FoundError(exit_flag, NoError, 1763,                &
     &                     "ROMS/Nonlinear/set_data.F")) RETURN
          END IF
        END IF
      END IF
      IF (.not.(RefinedGrid(ng).and.RefineScale(ng).gt.0)) THEN
        IF (DOMAIN(ng)%SouthWest_Test(tile)) THEN
          IF (LBC(iwest,isSig11,ng)%acquire) THEN
            CALL set_ngfld (ng, iNLM, idS11bc(iwest), JLB, JUB, 1,      &
     &                      0, Mm(ng)+1, 1,                             &
     &                      BOUNDARY(ng) % sig11G_west,                 &
     &                      BOUNDARY(ng) % sig11_west,                  &
     &                      update)
            IF (FoundError(exit_flag, NoError, 1781,                &
     &                     "ROMS/Nonlinear/set_data.F")) RETURN
          END IF
          IF (LBC(ieast,isSig11,ng)%acquire) THEN
            CALL set_ngfld (ng, iNLM, idS11bc(ieast), JLB, JUB, 1,      &
     &                      0, Mm(ng)+1, 1,                             &
     &                      BOUNDARY(ng) % sig11G_east,                 &
     &                      BOUNDARY(ng) % sig11_east,                  &
     &                      update)
            IF (FoundError(exit_flag, NoError, 1790,                &
     &                     "ROMS/Nonlinear/set_data.F")) RETURN
          END IF
          IF (LBC(isouth,isSig11,ng)%acquire) THEN
            CALL set_ngfld (ng, iNLM, idS11bc(isouth), ILB, IUB, 1,     &
     &                      0, Lm(ng)+1 ,1,                             &
     &                      BOUNDARY(ng) % sig11G_south,                &
     &                      BOUNDARY(ng) % sig11_south,                 &
     &                      update)
            IF (FoundError(exit_flag, NoError, 1799,                &
     &                     "ROMS/Nonlinear/set_data.F")) RETURN
          END IF
          IF (LBC(inorth,isSig11,ng)%acquire) THEN
            CALL set_ngfld (ng, iNLM, idS11bc(inorth), ILB, IUB, 1,     &
     &                      0, Lm(ng)+1, 1,                             &
     &                      BOUNDARY(ng) % sig11G_north,                &
     &                      BOUNDARY(ng) % sig11_north,                 &
     &                      update)
            IF (FoundError(exit_flag, NoError, 1808,                &
     &                     "ROMS/Nonlinear/set_data.F")) RETURN
          END IF
        END IF
      END IF
      IF (.not.(RefinedGrid(ng).and.RefineScale(ng).gt.0)) THEN
        IF (DOMAIN(ng)%SouthWest_Test(tile)) THEN
          IF (LBC(iwest,isSig22,ng)%acquire) THEN
            CALL set_ngfld (ng, iNLM, idS22bc(iwest), JLB, JUB, 1,      &
     &                      0, Mm(ng)+1, 1,                             &
     &                      BOUNDARY(ng) % sig22G_west,                 &
     &                      BOUNDARY(ng) % sig22_west,                  &
     &                      update)
            IF (FoundError(exit_flag, NoError, 1826,                &
     &                     "ROMS/Nonlinear/set_data.F")) RETURN
          END IF
          IF (LBC(ieast,isSig22,ng)%acquire) THEN
            CALL set_ngfld (ng, iNLM, idS22bc(ieast), JLB, JUB, 1,      &
     &                      0, Mm(ng)+1, 1,                             &
     &                      BOUNDARY(ng) % sig22G_east,                 &
     &                      BOUNDARY(ng) % sig22_east,                  &
     &                      update)
            IF (FoundError(exit_flag, NoError, 1835,                &
     &                     "ROMS/Nonlinear/set_data.F")) RETURN
          END IF
          IF (LBC(isouth,isSig22,ng)%acquire) THEN
            CALL set_ngfld (ng, iNLM, idS22bc(isouth), ILB, IUB, 1,     &
     &                      0, Lm(ng)+1 ,1,                             &
     &                      BOUNDARY(ng) % sig22G_south,                &
     &                      BOUNDARY(ng) % sig22_south,                 &
     &                      update)
            IF (FoundError(exit_flag, NoError, 1844,                &
     &                     "ROMS/Nonlinear/set_data.F")) RETURN
          END IF
          IF (LBC(inorth,isSig22,ng)%acquire) THEN
            CALL set_ngfld (ng, iNLM, idS22bc(inorth), ILB, IUB, 1,     &
     &                      0, Lm(ng)+1, 1,                             &
     &                      BOUNDARY(ng) % sig22G_north,                &
     &                      BOUNDARY(ng) % sig22_north,                 &
     &                      update)
            IF (FoundError(exit_flag, NoError, 1853,                &
     &                     "ROMS/Nonlinear/set_data.F")) RETURN
          END IF
        END IF
      END IF
      IF (.not.(RefinedGrid(ng).and.RefineScale(ng).gt.0)) THEN
        IF (DOMAIN(ng)%SouthWest_Test(tile)) THEN
          IF (LBC(iwest,isSig12,ng)%acquire) THEN
            CALL set_ngfld (ng, iNLM, idS12bc(iwest), JLB, JUB, 1,      &
     &                      0, Mm(ng)+1, 1,                             &
     &                      BOUNDARY(ng) % sig12G_west,                 &
     &                      BOUNDARY(ng) % sig12_west,                  &
     &                      update)
            IF (FoundError(exit_flag, NoError, 1871,                &
     &                     "ROMS/Nonlinear/set_data.F")) RETURN
          END IF
          IF (LBC(ieast,isSig12,ng)%acquire) THEN
            CALL set_ngfld (ng, iNLM, idS12bc(ieast), JLB, JUB, 1,      &
     &                      0, Mm(ng)+1, 1,                             &
     &                      BOUNDARY(ng) % sig12G_east,                 &
     &                      BOUNDARY(ng) % sig12_east,                  &
     &                      update)
            IF (FoundError(exit_flag, NoError, 1880,                &
     &                     "ROMS/Nonlinear/set_data.F")) RETURN
          END IF
          IF (LBC(isouth,isSig12,ng)%acquire) THEN
            CALL set_ngfld (ng, iNLM, idS12bc(isouth), ILB, IUB, 1,     &
     &                      0, Lm(ng)+1 ,1,                             &
     &                      BOUNDARY(ng) % sig12G_south,                &
     &                      BOUNDARY(ng) % sig12_south,                 &
     &                      update)
            IF (FoundError(exit_flag, NoError, 1889,                &
     &                     "ROMS/Nonlinear/set_data.F")) RETURN
          END IF
          IF (LBC(inorth,isSig12,ng)%acquire) THEN
            CALL set_ngfld (ng, iNLM, idS12bc(inorth), ILB, IUB, 1,     &
     &                      0, Lm(ng)+1, 1,                             &
     &                      BOUNDARY(ng) % sig12G_north,                &
     &                      BOUNDARY(ng) % sig12_north,                 &
     &                      update)
            IF (FoundError(exit_flag, NoError, 1898,                &
     &                     "ROMS/Nonlinear/set_data.F")) RETURN
          END IF
        END IF
      END IF
      IF (.not.(RefinedGrid(ng).and.RefineScale(ng).gt.0)) THEN
        IF (DOMAIN(ng)%SouthWest_Test(tile)) THEN
          IF (LBC(iwest,isUice,ng)%acquire) THEN
            CALL set_ngfld (ng, iNLM, idUibc(iwest), JLB, JUB, 1,       &
     &                      0, Mm(ng)+1, 1,                             &
     &                      BOUNDARY(ng) % uiG_west,                    &
     &                      BOUNDARY(ng) % ui_west,                     &
     &                      update)
            IF (FoundError(exit_flag, NoError, 1916,                &
     &                     "ROMS/Nonlinear/set_data.F")) RETURN
          END IF
          IF (LBC(iwest,isVice,ng)%acquire) THEN
            CALL set_ngfld (ng, iNLM, idVibc(iwest), JLB, JUB, 1,       &
     &                      1, Mm(ng)+1, 1,                             &
     &                      BOUNDARY(ng) % viG_west,                    &
     &                      BOUNDARY(ng) % vi_west,                     &
     &                      update)
            IF (FoundError(exit_flag, NoError, 1925,                &
     &                     "ROMS/Nonlinear/set_data.F")) RETURN
          END IF
          IF (LBC(ieast,isUice,ng)%acquire) THEN
            CALL set_ngfld (ng, iNLM, idUibc(ieast), JLB, JUB, 1,       &
     &                      0, Mm(ng)+1, 1,                             &
     &                      BOUNDARY(ng) % uiG_east,                    &
     &                      BOUNDARY(ng) % ui_east,                     &
     &                      update)
            IF (FoundError(exit_flag, NoError, 1934,                &
     &                     "ROMS/Nonlinear/set_data.F")) RETURN
          END IF
          IF (LBC(ieast,isVice,ng)%acquire) THEN
            CALL set_ngfld (ng, iNLM, idVibc(ieast), JLB, JUB, 1,       &
     &                      1, Mm(ng)+1, 1,                             &
     &                      BOUNDARY(ng) % viG_east,                    &
     &                      BOUNDARY(ng) % vi_east,                     &
     &                      update)
            IF (FoundError(exit_flag, NoError, 1943,                &
     &                     "ROMS/Nonlinear/set_data.F")) RETURN
          END IF
          IF (LBC(isouth,isUice,ng)%acquire) THEN
            CALL set_ngfld (ng, iNLM, idUibc(isouth), ILB, IUB, 1,      &
     &                      1, Lm(ng)+1 ,1,                             &
     &                      BOUNDARY(ng) % uiG_south,                   &
     &                      BOUNDARY(ng) % ui_south,                    &
     &                      update)
            IF (FoundError(exit_flag, NoError, 1952,                &
     &                     "ROMS/Nonlinear/set_data.F")) RETURN
          END IF
          IF (LBC(isouth,isVice,ng)%acquire) THEN
            CALL set_ngfld (ng, iNLM, idVibc(isouth), ILB, IUB, 1,      &
     &                      0, Lm(ng)+1 ,1,                             &
     &                      BOUNDARY(ng) % viG_south,                   &
     &                      BOUNDARY(ng) % vi_south,                    &
     &                      update)
            IF (FoundError(exit_flag, NoError, 1961,                &
     &                     "ROMS/Nonlinear/set_data.F")) RETURN
          END IF
          IF (LBC(inorth,isUice,ng)%acquire) THEN
            CALL set_ngfld (ng, iNLM, idUibc(inorth), ILB, IUB, 1,      &
     &                      1, Lm(ng)+1, 1,                             &
     &                      BOUNDARY(ng) % uiG_north,                   &
     &                      BOUNDARY(ng) % ui_north,                    &
     &                      update)
            IF (FoundError(exit_flag, NoError, 1970,                &
     &                     "ROMS/Nonlinear/set_data.F")) RETURN
          END IF
          IF (LBC(inorth,isVice,ng)%acquire) THEN
            CALL set_ngfld (ng, iNLM, idVibc(inorth), ILB, IUB, 1,      &
     &                      0, Lm(ng)+1, 1,                             &
     &                      BOUNDARY(ng) % viG_north,                   &
     &                      BOUNDARY(ng) % vi_north,                    &
     &                      update)
            IF (FoundError(exit_flag, NoError, 1979,                &
     &                     "ROMS/Nonlinear/set_data.F")) RETURN
          END IF
        END IF
      END IF
!
!-----------------------------------------------------------------------
!  Set sea surface height climatology (m).
!-----------------------------------------------------------------------
!
      IF (LsshCLM(ng)) THEN
        CALL set_2dfld_tile (ng, tile, iNLM, idSSHc,                    &
     &                       LBi, UBi, LBj, UBj,                        &
     &                       CLIMA(ng)%sshG,                            &
     &                       CLIMA(ng)%ssh,                             &
     &                       update)
        IF (FoundError(exit_flag, NoError, 2000,                    &
     &                 "ROMS/Nonlinear/set_data.F")) RETURN
      END IF
!
!-----------------------------------------------------------------------
!  Set 2D momentum climatology (m/s).
!-----------------------------------------------------------------------
!
      IF (Lm2CLM(ng)) THEN
        CALL set_2dfld_tile (ng, tile, iNLM, idUbcl,                    &
     &                       LBi, UBi, LBj, UBj,                        &
     &                       CLIMA(ng)%ubarclmG,                        &
     &                       CLIMA(ng)%ubarclm,                         &
     &                       update)
        IF (FoundError(exit_flag, NoError, 2038,                    &
     &                 "ROMS/Nonlinear/set_data.F")) RETURN
!
        CALL set_2dfld_tile (ng, tile, iNLM, idVbcl,                    &
     &                       LBi, UBi, LBj, UBj,                        &
     &                       CLIMA(ng)%vbarclmG,                        &
     &                       CLIMA(ng)%vbarclm,                         &
     &                       update)
        IF (FoundError(exit_flag, NoError, 2046,                    &
     &                 "ROMS/Nonlinear/set_data.F")) RETURN
      END IF
!
!-----------------------------------------------------------------------
!  Set 3D momentum climatology (m/s).
!-----------------------------------------------------------------------
!
      IF (Lm3CLM(ng)) THEN
        CALL set_3dfld_tile (ng, tile, iNLM, idUclm,                    &
     &                       LBi, UBi, LBj, UBj, 1, N(ng),              &
     &                       CLIMA(ng)%uclmG,                           &
     &                       CLIMA(ng)%uclm,                            &
     &                       update)
        IF (FoundError(exit_flag, NoError, 2091,                    &
     &                 "ROMS/Nonlinear/set_data.F")) RETURN
!
        CALL set_3dfld_tile (ng, tile, iNLM, idVclm,                    &
     &                       LBi, UBi, LBj, UBj, 1, N(ng),              &
     &                       CLIMA(ng)%vclmG,                           &
     &                       CLIMA(ng)%vclm,                            &
     &                       update)
        IF (FoundError(exit_flag, NoError, 2099,                    &
     &                 "ROMS/Nonlinear/set_data.F")) RETURN
      END IF
!
!-----------------------------------------------------------------------
!  Set tracers climatology.
!-----------------------------------------------------------------------
!
      ic=0
      DO itrc=1,NT(ng)
        IF (LtracerCLM(itrc,ng)) THEN
          ic=ic+1
          CALL set_3dfld_tile (ng, tile, iNLM, idTclm(itrc),            &
     &                         LBi, UBi, LBj, UBj, 1, N(ng),            &
     &                         CLIMA(ng)%tclmG(:,:,:,:,ic),             &
     &                         CLIMA(ng)%tclm (:,:,:,ic),               &
     &                         update)
          IF (FoundError(exit_flag, NoError, 2153,                  &
     &                   "ROMS/Nonlinear/set_data.F")) RETURN
        END IF
      END DO
      IF (LaiCLM(ng)) THEN
        CALL set_2dfld_tile (ng, tile, iNLM, idAIclm,                   &
     &                       LBi, UBi, LBj, UBj,                        &
     &                       CLIMA(ng)%aiclmG,                          &
     &                       CLIMA(ng)%aiclm,                           &
     &                       update)
        IF (FoundError(exit_flag, NoError, 2242,                    &
     &               "ROMS/Nonlinear/set_data.F")) RETURN
        CALL set_2dfld_tile (ng, tile, iNLM, idHIclm,                   &
     &                       LBi, UBi, LBj, UBj,                        &
     &                       CLIMA(ng)%hiclmG,                          &
     &                       CLIMA(ng)%hiclm,                           &
     &                       update)
        IF (FoundError(exit_flag, NoError, 2249,                    &
     &               "ROMS/Nonlinear/set_data.F")) RETURN
        CALL set_2dfld_tile (ng, tile, iNLM, idHSNclm,                  &
     &                       LBi, UBi, LBj, UBj,                        &
     &                       CLIMA(ng)%hsnclmG,                         &
     &                       CLIMA(ng)%hsnclm,                          &
     &                       update)
        IF (FoundError(exit_flag, NoError, 2256,                    &
     &               "ROMS/Nonlinear/set_data.F")) RETURN
      END IF
      IF (LsiCLM(ng)) THEN
        CALL set_2dfld_tile (ng, tile, iNLM, idS11clm,                  &
     &                       LBi, UBi, LBj, UBj,                        &
     &                       CLIMA(ng)%sig11clmG,                       &
     &                       CLIMA(ng)%sig11clm,                        &
     &                       update)
        IF (FoundError(exit_flag, NoError, 2270,                    &
     &               "ROMS/Nonlinear/set_data.F")) RETURN
        CALL set_2dfld_tile (ng, tile, iNLM, idS12clm,                  &
     &                       LBi, UBi, LBj, UBj,                        &
     &                       CLIMA(ng)%sig12clmG,                       &
     &                       CLIMA(ng)%sig12clm,                        &
     &                       update)
        IF (FoundError(exit_flag, NoError, 2277,                    &
     &               "ROMS/Nonlinear/set_data.F")) RETURN
        CALL set_2dfld_tile (ng, tile, iNLM, idS22clm,                  &
     &                       LBi, UBi, LBj, UBj,                        &
     &                       CLIMA(ng)%sig22clmG,                       &
     &                       CLIMA(ng)%sig22clm,                        &
     &                       update)
        IF (FoundError(exit_flag, NoError, 2284,                    &
     &               "ROMS/Nonlinear/set_data.F")) RETURN
      END IF
      IF (LmiCLM(ng)) THEN
        CALL set_2dfld_tile (ng, tile, iNLM, idUIclm,                   &
     &                       LBi, UBi, LBj, UBj,                        &
     &                       CLIMA(ng)%uiclmG,                          &
     &                       CLIMA(ng)%uiclm,                           &
     &                       update)
        IF (FoundError(exit_flag, NoError, 2298,                    &
     &               "ROMS/Nonlinear/set_data.F")) RETURN
        CALL set_2dfld_tile (ng, tile, iNLM, idVIclm,                   &
     &                       LBi, UBi, LBj, UBj,                        &
     &                       CLIMA(ng)%viclmG,                          &
     &                       CLIMA(ng)%viclm,                           &
     &                       update)
        IF (FoundError(exit_flag, NoError, 2305,                    &
     &               "ROMS/Nonlinear/set_data.F")) RETURN
      END IF
      RETURN
      END SUBROUTINE set_data_tile
