      SUBROUTINE get_data (ng)
!
!svn $Id$
!================================================== Hernan G. Arango ===
!  Copyright (c) 2002-2020 The ROMS/TOMS Group                         !
!    Licensed under a MIT/X style license                              !
!    See License_ROMS.txt                                              !
!=======================================================================
!                                                                      !
!  This routine reads in forcing, climatology and other data from      !
!  NetCDF files.  If there is more than one time-record,  data is      !
!  loaded into global  two-time  record arrays. The interpolation      !
!  is carried elsewhere.                                               !
!                                                                      !
!  Currently, this routine is only executed in serial mode by the      !
!  main thread.                                                        !
!                                                                      !
!=======================================================================
!
      USE mod_param
      USE mod_boundary
      USE mod_clima
      USE mod_forces
      USE mod_grid
      USE mod_iounits
      USE mod_ncparam
      USE mod_scalars
      USE mod_sources
      USE mod_stepping
!
      USE strings_mod, ONLY : FoundError
!
      implicit none
!
!  Imported variable declarations.
!
      integer, intent(in) :: ng
!
!  Local variable declarations.
!
      logical, dimension(3) :: update =                                 &
     &         (/ .FALSE., .FALSE., .FALSE. /)
      integer :: ILB, IUB, JLB, JUB
      integer :: LBi, UBi, LBj, UBj
      integer :: i, ic, my_tile
!
!  Lower and upper bounds for nontiled (global values) boundary arrays.
!
      my_tile=-1                           ! for global values
      ILB=BOUNDS(ng)%LBi(my_tile)
      IUB=BOUNDS(ng)%UBi(my_tile)
      JLB=BOUNDS(ng)%LBj(my_tile)
      JUB=BOUNDS(ng)%UBj(my_tile)
!
!  Lower and upper bounds for tiled arrays.
!
      LBi=LBOUND(GRID(ng)%h,DIM=1)
      UBi=UBOUND(GRID(ng)%h,DIM=1)
      LBj=LBOUND(GRID(ng)%h,DIM=2)
      UBj=UBOUND(GRID(ng)%h,DIM=2)
!
!-----------------------------------------------------------------------
!  Turn on input data time wall clock.
!-----------------------------------------------------------------------
!
      CALL wclock_on (ng, iNLM, 3, 80, "ROMS/Nonlinear/get_data.F")
!
!=======================================================================
!  Read in forcing data from FORCING NetCDF file.
!=======================================================================
!
!-----------------------------------------------------------------------
!  Point Sources/Sinks time dependent data.
!-----------------------------------------------------------------------
!
!  Point Source/Sink vertically integrated mass transport.
!
      IF (LuvSrc(ng).or.LwSrc(ng)) THEN
        CALL get_ngfld (ng, iNLM, idRtra, ncSSFid(idRtra,ng),           &
     &                  nSSFfiles(ng), SSF(1,ng), update(1),            &
     &                  1, Nsrc(ng), 1, 2, 1, Nsrc(ng), 1,              &
     &                  SOURCES(ng) % QbarG)
        IF (FoundError(exit_flag, NoError, 100,                    &
     &                 "ROMS/Nonlinear/get_data.F")) RETURN
      END IF
!
!  Tracer Sources/Sinks.
!
      DO i=1,NT(ng)
        IF (LtracerSrc(i,ng)) THEN
          CALL get_ngfld (ng, iNLM, idRtrc(i), ncSSFid(idRtrc(i),ng),   &
     &                    nSSFfiles(ng), SSF(1,ng), update(1),          &
     &                    1, Nsrc(ng), N(ng), 2, 1, Nsrc(ng), N(ng),    &
     &                    SOURCES(ng) % TsrcG(:,:,:,i))
          IF (FoundError(exit_flag, NoError, 122,                  &
     &                   "ROMS/Nonlinear/get_data.F")) RETURN
        END IF
      END DO
!
!-----------------------------------------------------------------------
!  Surface wind components.
!-----------------------------------------------------------------------
!
      CALL get_2dfld (ng, iNLM, idUair, ncFRCid(idUair,ng),             &
     &                nFfiles(ng), FRC(1,ng), update(1),                &
     &                LBi, UBi, LBj, UBj, 2, 1,                         &
     &                GRID(ng) % rmask,                                 &
     &                FORCES(ng) % UwindG)
      IF (FoundError(exit_flag, NoError, 307,                      &
     &               "ROMS/Nonlinear/get_data.F")) RETURN
      CALL get_2dfld (ng , iNLM, idVair, ncFRCid(idVair,ng),            &
     &                nFfiles(ng), FRC(1,ng), update(1),                &
     &                LBi, UBi, LBj, UBj, 2, 1,                         &
     &                GRID(ng) % rmask,                                 &
     &                FORCES(ng) % VwindG)
      IF (FoundError(exit_flag, NoError, 317,                      &
     &               "ROMS/Nonlinear/get_data.F")) RETURN
!
!-----------------------------------------------------------------------
!  Surface air pressure.
!-----------------------------------------------------------------------
!
      CALL get_2dfld (ng, iNLM, idPair, ncFRCid(idPair,ng),             &
     &                nFfiles(ng), FRC(1,ng), update(1),                &
     &                LBi, UBi, LBj, UBj, 2, 1,                         &
     &                GRID(ng) % rmask,                                 &
     &                FORCES(ng) % PairG)
      IF (FoundError(exit_flag, NoError, 421,                      &
     &               "ROMS/Nonlinear/get_data.F")) RETURN
!
!-----------------------------------------------------------------------
!  Surface air temperature.
!-----------------------------------------------------------------------
!
      CALL get_2dfld (ng, iNLM, idTair, ncFRCid(idTair,ng),             &
     &                nFfiles(ng), FRC(1,ng), update(1),                &
     &                LBi, UBi, LBj, UBj, 2, 1,                         &
     &                GRID(ng) % rmask,                                 &
     &                FORCES(ng) % TairG)
      IF (FoundError(exit_flag, NoError, 543,                      &
     &               "ROMS/Nonlinear/get_data.F")) RETURN
!
!-----------------------------------------------------------------------
!  Surface air humidity.
!-----------------------------------------------------------------------
!
      CALL get_2dfld (ng, iNLM, idQair, ncFRCid(idQair,ng),             &
     &                nFfiles(ng), FRC(1,ng), update(1),                &
     &                LBi, UBi, LBj, UBj, 2, 1,                         &
     &                GRID(ng) % rmask,                                 &
     &                FORCES(ng) % HairG)
      IF (FoundError(exit_flag, NoError, 563,                      &
     &               "ROMS/Nonlinear/get_data.F")) RETURN
!
!-----------------------------------------------------------------------
!  Surface solar shortwave radiation.
!-----------------------------------------------------------------------
!
      CALL get_2dfld (ng, iNLM, idSrad, ncFRCid(idSrad,ng),             &
     &                nFfiles(ng), FRC(1,ng), update(1),                &
     &                LBi, UBi, LBj, UBj, 2, 1,                         &
     &                GRID(ng) % rmask,                                 &
     &                FORCES(ng) % srflxG)
      IF (FoundError(exit_flag, NoError, 617,                      &
     &               "ROMS/Nonlinear/get_data.F")) RETURN
!
!-----------------------------------------------------------------------
!  Surface PAR (psl20171017).
!-----------------------------------------------------------------------
!
      CALL get_2dfld (ng, iNLM, idSpar, ncFRCid(idSpar,ng),             &
     &                nFfiles(ng), FRC(1,ng), update(1),                &
     &                LBi, UBi, LBj, UBj, 2, 1,                         &
     &                GRID(ng) % rmask,                                 &
     &                FORCES(ng) % spflxG)
      IF (FoundError(exit_flag, NoError, 632,                      &
     &               "ROMS/Nonlinear/get_data.F")) RETURN
!
!-----------------------------------------------------------------------
!  Surface downwelling longwave radiation.
!-----------------------------------------------------------------------
!
      CALL get_2dfld (ng, iNLM, idLdwn, ncFRCid(idLdwn,ng),             &
     &                nFfiles(ng), FRC(1,ng), update(1),                &
     &                LBi, UBi, LBj, UBj, 2, 1,                         &
     &                GRID(ng) % rmask,                                 &
     &                FORCES(ng) % lrflxG)
      IF (FoundError(exit_flag, NoError, 687,                      &
     &               "ROMS/Nonlinear/get_data.F")) RETURN
!
!-----------------------------------------------------------------------
!  Rain fall rate.  Snow fall rate.
!-----------------------------------------------------------------------
!
      CALL get_2dfld (ng, iNLM, idrain, ncFRCid(idrain,ng),             &
     &                nFfiles(ng), FRC(1,ng), update(1),                &
     &                LBi, UBi, LBj, UBj, 2, 1,                         &
     &                GRID(ng) % rmask,                                 &
     &                FORCES(ng) % rainG)
      IF (FoundError(exit_flag, NoError, 707,                      &
     &               "ROMS/Nonlinear/get_data.F")) RETURN
!
!=======================================================================
!  Read in open boundary conditions from BOUNDARY NetCDF file.
!=======================================================================
!
      IF (LprocessOBC(ng)) THEN
        IF (LBC(iwest,isFsur,ng)%acquire) THEN
          CALL get_ngfld (ng, iNLM, idZbry(iwest),                      &
     &                    ncBRYid(idZbry(iwest),ng),                    &
     &                    nBCfiles(ng), BRY(1,ng), update(1),           &
     &                    JLB, JUB, 1, 2, 0, Mm(ng)+1, 1,               &
     &                    BOUNDARY(ng) % zetaG_west)
          IF (FoundError(exit_flag, NoError, 1012,                  &
     &                   "ROMS/Nonlinear/get_data.F")) RETURN
        END IF
        IF (LBC(ieast,isFsur,ng)%acquire) THEN
          CALL get_ngfld (ng, iNLM, idZbry(ieast),                      &
     &                    ncBRYid(idZbry(ieast),ng),                    &
     &                    nBCfiles(ng), BRY(1,ng), update(1),           &
     &                    JLB, JUB, 1, 2, 0, Mm(ng)+1, 1,               &
     &                    BOUNDARY(ng) % zetaG_east)
          IF (FoundError(exit_flag, NoError, 1022,                  &
     &                   "ROMS/Nonlinear/get_data.F")) RETURN
        END IF
        IF (LBC(isouth,isFsur,ng)%acquire) THEN
          CALL get_ngfld (ng, iNLM, idZbry(isouth),                     &
     &                    ncBRYid(idZbry(isouth),ng),                   &
     &                    nBCfiles(ng), BRY(1,ng), update(1),           &
     &                    ILB, IUB, 1, 2, 0, Lm(ng)+1, 1,               &
     &                    BOUNDARY(ng) % zetaG_south)
          IF (FoundError(exit_flag, NoError, 1032,                  &
     &                   "ROMS/Nonlinear/get_data.F")) RETURN
        END IF
        IF (LBC(inorth,isFsur,ng)%acquire) THEN
          CALL get_ngfld (ng, iNLM, idZbry(inorth),                     &
     &                    ncBRYid(idZbry(inorth),ng),                   &
     &                    nBCfiles(ng), BRY(1,ng), update(1),           &
     &                    ILB, IUB, 1, 2, 0, Lm(ng)+1, 1,               &
     &                    BOUNDARY(ng) % zetaG_north)
          IF (FoundError(exit_flag, NoError, 1042,                  &
     &                   "ROMS/Nonlinear/get_data.F")) RETURN
        END IF
      END IF
!
      IF (LprocessOBC(ng)) THEN
        IF (LBC(iwest,isUbar,ng)%acquire) THEN
          CALL get_ngfld (ng, iNLM, idU2bc(iwest),                      &
     &                    ncBRYid(idU2bc(iwest),ng),                    &
     &                    nBCfiles(ng), BRY(1,ng), update(1),           &
     &                    JLB, JUB, 1, 2, 0, Mm(ng)+1, 1,               &
     &                    BOUNDARY(ng) % ubarG_west)
          IF (FoundError(exit_flag, NoError, 1057,                  &
     &                   "ROMS/Nonlinear/get_data.F")) RETURN
        END IF
        IF (LBC(iwest,isVbar,ng)%acquire) THEN
          CALL get_ngfld (ng, iNLM, idV2bc(iwest),                      &
     &                    ncBRYid(idV2bc(iwest),ng),                    &
     &                    nBCfiles(ng), BRY(1,ng), update(1),           &
     &                    JLB, JUB, 1, 2, 1, Mm(ng)+1, 1,               &
     &                    BOUNDARY(ng) % vbarG_west)
          IF (FoundError(exit_flag, NoError, 1067,                  &
     &                   "ROMS/Nonlinear/get_data.F")) RETURN
        END IF
        IF (LBC(ieast,isUbar,ng)%acquire) THEN
          CALL get_ngfld (ng, iNLM, idU2bc(ieast),                      &
     &                    ncBRYid(idU2bc(ieast),ng),                    &
     &                    nBCfiles(ng), BRY(1,ng), update(1),           &
     &                    JLB, JUB, 1, 2, 0, Mm(ng)+1, 1,               &
     &                    BOUNDARY(ng) % ubarG_east)
          IF (FoundError(exit_flag, NoError, 1077,                  &
     &                   "ROMS/Nonlinear/get_data.F")) RETURN
        END IF
        IF (LBC(ieast,isVbar,ng)%acquire) THEN
          CALL get_ngfld (ng, iNLM, idV2bc(ieast),                      &
     &                    ncBRYid(idV2bc(ieast),ng),                    &
     &                    nBCfiles(ng), BRY(1,ng), update(1),           &
     &                    JLB, JUB, 1, 2, 1, Mm(ng)+1, 1,               &
     &                    BOUNDARY(ng) % vbarG_east)
          IF (FoundError(exit_flag, NoError, 1087,                  &
     &                   "ROMS/Nonlinear/get_data.F")) RETURN
        END IF
        IF (LBC(isouth,isUbar,ng)%acquire) THEN
          CALL get_ngfld (ng, iNLM, idU2bc(isouth),                     &
     &                    ncBRYid(idU2bc(isouth),ng),                   &
     &                    nBCfiles(ng), BRY(1,ng), update(1),           &
     &                    ILB, IUB, 1, 2, 1, Lm(ng)+1, 1,               &
     &                    BOUNDARY(ng) % ubarG_south)
          IF (FoundError(exit_flag, NoError, 1097,                  &
     &                   "ROMS/Nonlinear/get_data.F")) RETURN
        END IF
        IF (LBC(isouth,isVbar,ng)%acquire) THEN
          CALL get_ngfld (ng, iNLM, idV2bc(isouth),                     &
     &                    ncBRYid(idV2bc(isouth),ng),                   &
     &                    nBCfiles(ng), BRY(1,ng), update(1),           &
     &                    ILB, IUB, 1, 2, 0, Lm(ng)+1, 1,               &
     &                    BOUNDARY(ng) % vbarG_south)
          IF (FoundError(exit_flag, NoError, 1107,                  &
     &                   "ROMS/Nonlinear/get_data.F")) RETURN
        END IF
        IF (LBC(inorth,isUbar,ng)%acquire) THEN
          CALL get_ngfld (ng, iNLM, idU2bc(inorth),                     &
     &                    ncBRYid(idU2bc(inorth),ng),                   &
     &                    nBCfiles(ng), BRY(1,ng), update(1),           &
     &                    ILB, IUB, 1, 2, 1, Lm(ng)+1, 1,               &
     &                    BOUNDARY(ng) % ubarG_north)
          IF (FoundError(exit_flag, NoError, 1117,                  &
     &                   "ROMS/Nonlinear/get_data.F")) RETURN
        END IF
        IF (LBC(inorth,isVbar,ng)%acquire) THEN
          CALL get_ngfld (ng, iNLM, idV2bc(inorth),                     &
     &                    ncBRYid(idV2bc(inorth),ng),                   &
     &                    nBCfiles(ng), BRY(1,ng), update(1),           &
     &                    ILB, IUB, 1, 2, 0, Lm(ng)+1, 1,               &
     &                    BOUNDARY(ng) % vbarG_north)
          IF (FoundError(exit_flag, NoError, 1127,                  &
     &                   "ROMS/Nonlinear/get_data.F")) RETURN
        END IF
      END IF
!
      IF (LprocessOBC(ng)) THEN
        IF (LBC(iwest,isUvel,ng)%acquire) THEN
          CALL get_ngfld (ng, iNLM, idU3bc(iwest),                      &
     &                    ncBRYid(idU3bc(iwest),ng),                    &
     &                    nBCfiles(ng), BRY(1,ng), update(1),           &
     &                    JLB, JUB, N(ng), 2, 0, Mm(ng)+1, N(ng),       &
     &                    BOUNDARY(ng) % uG_west)
          IF (FoundError(exit_flag, NoError, 1143,                  &
     &                   "ROMS/Nonlinear/get_data.F")) RETURN
        END IF
        IF (LBC(iwest,isVvel,ng)%acquire) THEN
          CALL get_ngfld (ng, iNLM, idV3bc(iwest),                      &
     &                    ncBRYid(idV3bc(iwest),ng),                    &
     &                    nBCfiles(ng), BRY(1,ng), update(1),           &
     &                    JLB, JUB, N(ng), 2, 1, Mm(ng)+1, N(ng),       &
     &                    BOUNDARY(ng) % vG_west)
          IF (FoundError(exit_flag, NoError, 1153,                  &
     &                   "ROMS/Nonlinear/get_data.F")) RETURN
        END IF
        IF (LBC(ieast,isUvel,ng)%acquire) THEN
          CALL get_ngfld (ng, iNLM, idU3bc(ieast),                      &
     &                    ncBRYid(idU3bc(ieast),ng),                    &
     &                    nBCfiles(ng), BRY(1,ng), update(1),           &
     &                    JLB, JUB, N(ng), 2, 0, Mm(ng)+1, N(ng),       &
     &                    BOUNDARY(ng) % uG_east)
          IF (FoundError(exit_flag, NoError, 1163,                  &
     &                   "ROMS/Nonlinear/get_data.F")) RETURN
        END IF
        IF (LBC(ieast,isVvel,ng)%acquire) THEN
          CALL get_ngfld (ng, iNLM, idV3bc(ieast),                      &
     &                    ncBRYid(idV3bc(ieast),ng),                    &
     &                    nBCfiles(ng), BRY(1,ng), update(1),           &
     &                    JLB, JUB, N(ng), 2, 1, Mm(ng)+1, N(ng),       &
     &                    BOUNDARY(ng) % vG_east)
          IF (FoundError(exit_flag, NoError, 1173,                  &
     &                   "ROMS/Nonlinear/get_data.F")) RETURN
        END IF
        IF (LBC(isouth,isUvel,ng)%acquire) THEN
          CALL get_ngfld (ng, iNLM, idU3bc(isouth),                     &
     &                    ncBRYid(idU3bc(isouth),ng),                   &
     &                    nBCfiles(ng), BRY(1,ng), update(1),           &
     &                    ILB, IUB, N(ng), 2, 1, Lm(ng)+1, N(ng),       &
     &                    BOUNDARY(ng) % uG_south)
          IF (FoundError(exit_flag, NoError, 1183,                  &
     &                   "ROMS/Nonlinear/get_data.F")) RETURN
        END IF
        IF (LBC(isouth,isVvel,ng)%acquire) THEN
          CALL get_ngfld (ng, iNLM, idV3bc(isouth),                     &
     &                    ncBRYid(idV3bc(isouth),ng),                   &
     &                    nBCfiles(ng), BRY(1,ng), update(1),           &
     &                    ILB, IUB, N(ng), 2, 0, Lm(ng)+1, N(ng),       &
     &                    BOUNDARY(ng) % vG_south)
          IF (FoundError(exit_flag, NoError, 1193,                  &
     &                   "ROMS/Nonlinear/get_data.F")) RETURN
        END IF
        IF (LBC(inorth,isUvel,ng)%acquire) THEN
          CALL get_ngfld (ng, iNLM, idU3bc(inorth),                     &
     &                    ncBRYid(idU3bc(inorth),ng),                   &
     &                    nBCfiles(ng), BRY(1,ng), update(1),           &
     &                    ILB, IUB, N(ng), 2, 1, Lm(ng)+1, N(ng),       &
     &                    BOUNDARY(ng) % uG_north)
          IF (FoundError(exit_flag, NoError, 1203,                  &
     &                   "ROMS/Nonlinear/get_data.F")) RETURN
        END IF
        IF (LBC(inorth,isVvel,ng)%acquire) THEN
          CALL get_ngfld (ng, iNLM, idV3bc(inorth),                     &
     &                    ncBRYid(idV3bc(inorth),ng),                   &
     &                    nBCfiles(ng), BRY(1,ng), update(1),           &
     &                    ILB, IUB, N(ng), 2, 0, Lm(ng)+1, N(ng),       &
     &                    BOUNDARY(ng) % vG_north)
          IF (FoundError(exit_flag, NoError, 1213,                  &
     &                   "ROMS/Nonlinear/get_data.F")) RETURN
        END IF
      END IF
!
      IF (LprocessOBC(ng)) THEN
        DO i=1,NT(ng)
          IF (LBC(iwest,isTvar(i),ng)%acquire) THEN
            CALL get_ngfld (ng, iNLM, idTbry(iwest,i),                  &
     &                      ncBRYid(idTbry(iwest,i),ng),                &
     &                      nBCfiles(ng), BRY(1,ng), update(1),         &
     &                      JLB, JUB, N(ng), 2, 0, Mm(ng)+1, N(ng),     &
     &                      BOUNDARY(ng) % tG_west(:,:,:,i))
            IF (FoundError(exit_flag, NoError, 1229,                &
     &                     "ROMS/Nonlinear/get_data.F")) RETURN
          END IF
        END DO
        DO i=1,NT(ng)
          IF (LBC(ieast,isTvar(i),ng)%acquire) THEN
            CALL get_ngfld (ng, iNLM, idTbry(ieast,i),                  &
     &                      ncBRYid(idTbry(ieast,i),ng),                &
     &                      nBCfiles(ng), BRY(1,ng), update(1),         &
     &                      JLB, JUB, N(ng), 2, 0, Mm(ng)+1, N(ng),     &
     &                      BOUNDARY(ng) % tG_east(:,:,:,i))
            IF (FoundError(exit_flag, NoError, 1241,                &
     &                     "ROMS/Nonlinear/get_data.F")) RETURN
          END IF
        END DO
        DO i=1,NT(ng)
          IF (LBC(isouth,isTvar(i),ng)%acquire) THEN
            CALL get_ngfld (ng, iNLM, idTbry(isouth,i),                 &
     &                      ncBRYid(idTbry(isouth,i),ng),               &
     &                      nBCfiles(ng), BRY(1,ng), update(1),         &
     &                      ILB, IUB, N(ng), 2, 0, Lm(ng)+1, N(ng),     &
     &                      BOUNDARY(ng) % tG_south(:,:,:,i))
            IF (FoundError(exit_flag, NoError, 1253,                &
     &                     "ROMS/Nonlinear/get_data.F")) RETURN
          END IF
        END DO
        DO i=1,NT(ng)
          IF (LBC(inorth,isTvar(i),ng)%acquire) THEN
            CALL get_ngfld (ng, iNLM, idTbry(inorth,i),                 &
     &                      ncBRYid(idTbry(inorth,i),ng),               &
     &                      nBCfiles(ng), BRY(1,ng), update(1),         &
     &                      ILB, IUB, N(ng), 2, 0, Lm(ng)+1, N(ng),     &
     &                      BOUNDARY(ng) % tG_north(:,:,:,i))
            IF (FoundError(exit_flag, NoError, 1265,                &
     &                     "ROMS/Nonlinear/get_data.F")) RETURN
          END IF
        END DO
      END IF
      IF (.not.(RefinedGrid(ng).and.RefineScale(ng).gt.0)) THEN
        IF (LBC(iwest,isUice,ng)%acquire) THEN
          CALL get_ngfld (ng, iNLM, idUibc(iwest),                      &
     &                    ncBRYid(idUibc(iwest),ng),                    &
     &                    nBCfiles(ng), BRY(1,ng), update(1),           &
     &                    JLB, JUB, 1, 2, 0, Mm(ng)+1, 1,               &
     &                    BOUNDARY(ng) % uiG_west)
          IF (FoundError(exit_flag, NoError, 1281,                  &
     &               "ROMS/Nonlinear/get_data.F")) RETURN
        END IF
        IF (LBC(iwest,isVice,ng)%acquire) THEN
          CALL get_ngfld (ng, iNLM, idVibc(iwest),                      &
     &                    ncBRYid(idVibc(iwest),ng),                    &
     &                    nBCfiles(ng), BRY(1,ng), update(1),           &
     &                    JLB, JUB, 1, 2, 1, Mm(ng)+1, 1,               &
     &                    BOUNDARY(ng) % viG_west)
          IF (FoundError(exit_flag, NoError, 1290,                  &
     &               "ROMS/Nonlinear/get_data.F")) RETURN
        END IF
        IF (LBC(ieast,isUice,ng)%acquire) THEN
          CALL get_ngfld (ng, iNLM, idUibc(ieast),                      &
     &                    ncBRYid(idUibc(ieast),ng),                    &
     &                    nBCfiles(ng), BRY(1,ng), update(1),           &
     &                    JLB, JUB, 1, 2, 0, Mm(ng)+1, 1,               &
     &                    BOUNDARY(ng) % uiG_east)
          IF (FoundError(exit_flag, NoError, 1299,                  &
     &               "ROMS/Nonlinear/get_data.F")) RETURN
        END IF
        IF (LBC(ieast,isVice,ng)%acquire) THEN
          CALL get_ngfld (ng, iNLM, idVibc(ieast),                      &
     &                    ncBRYid(idVibc(ieast),ng),                    &
     &                    nBCfiles(ng), BRY(1,ng), update(1),           &
     &                    JLB, JUB, 1, 2, 1, Mm(ng)+1, 1,               &
     &                    BOUNDARY(ng) % viG_east)
          IF (FoundError(exit_flag, NoError, 1308,                  &
     &               "ROMS/Nonlinear/get_data.F")) RETURN
        END IF
        IF (LBC(isouth,isUice,ng)%acquire) THEN
          CALL get_ngfld (ng, iNLM, idUibc(isouth),                     &
     &                    ncBRYid(idUibc(isouth),ng),                   &
     &                    nBCfiles(ng), BRY(1,ng), update(1),           &
     &                    ILB, IUB, 1, 2, 1, Lm(ng)+1, 1,               &
     &                    BOUNDARY(ng) % uiG_south)
          IF (FoundError(exit_flag, NoError, 1317,                  &
     &               "ROMS/Nonlinear/get_data.F")) RETURN
        END IF
        IF (LBC(isouth,isVice,ng)%acquire) THEN
          CALL get_ngfld (ng, iNLM, idVibc(isouth),                     &
     &                    ncBRYid(idVibc(isouth),ng),                   &
     &                    nBCfiles(ng), BRY(1,ng), update(1),           &
     &                    ILB, IUB, 1, 2, 0, Lm(ng)+1, 1,               &
     &                    BOUNDARY(ng) % viG_south)
          IF (FoundError(exit_flag, NoError, 1326,                  &
     &               "ROMS/Nonlinear/get_data.F")) RETURN
        END IF
        IF (LBC(inorth,isUice,ng)%acquire) THEN
          CALL get_ngfld (ng, iNLM, idUibc(inorth),                     &
     &                    ncBRYid(idUibc(inorth),ng),                   &
     &                    nBCfiles(ng), BRY(1,ng), update(1),           &
     &                    ILB, IUB, 1, 2, 1, Lm(ng)+1, 1,               &
     &                    BOUNDARY(ng) % uiG_north)
          IF (FoundError(exit_flag, NoError, 1335,                  &
     &               "ROMS/Nonlinear/get_data.F")) RETURN
        END IF
        IF (LBC(inorth,isVice,ng)%acquire) THEN
          CALL get_ngfld (ng, iNLM, idVibc(inorth),                     &
     &                    ncBRYid(idVibc(inorth),ng),                   &
     &                    nBCfiles(ng), BRY(1,ng), update(1),           &
     &                    ILB, IUB, 1, 2, 0, Lm(ng)+1, 1,               &
     &                    BOUNDARY(ng) % viG_north)
          IF (FoundError(exit_flag, NoError, 1344,                  &
     &               "ROMS/Nonlinear/get_data.F")) RETURN
        END IF
      END IF
      IF (.not.(RefinedGrid(ng).and.RefineScale(ng).gt.0)) THEN
        IF (LBC(iwest,isAice,ng)%acquire) THEN
          CALL get_ngfld (ng, iNLM, idAibc(iwest),                      &
     &                    ncBRYid(idAibc(iwest),ng),                    &
     &                    nBCfiles(ng), BRY(1,ng), update(1),           &
     &                    JLB, JUB, 1, 2, 0, Mm(ng)+1, 1,               &
     &                    BOUNDARY(ng) % aiG_west)
          IF (FoundError(exit_flag, NoError, 1357,                  &
     &               "ROMS/Nonlinear/get_data.F")) RETURN
        END IF
        IF (LBC(ieast,isAice,ng)%acquire) THEN
          CALL get_ngfld (ng, iNLM, idAibc(ieast),                      &
     &                    ncBRYid(idAibc(ieast),ng),                    &
     &                    nBCfiles(ng), BRY(1,ng), update(1),           &
     &                    JLB, JUB, 1, 2, 0, Mm(ng)+1, 1,               &
     &                    BOUNDARY(ng) % aiG_east)
          IF (FoundError(exit_flag, NoError, 1366,                  &
     &               "ROMS/Nonlinear/get_data.F")) RETURN
        END IF
        IF (LBC(isouth,isAice,ng)%acquire) THEN
          CALL get_ngfld (ng, iNLM, idAibc(isouth),                     &
     &                    ncBRYid(idAibc(isouth),ng),                   &
     &                    nBCfiles(ng), BRY(1,ng), update(1),           &
     &                    ILB, IUB, 1, 2, 0, Lm(ng)+1, 1,               &
     &                    BOUNDARY(ng) % aiG_south)
          IF (FoundError(exit_flag, NoError, 1375,                  &
     &               "ROMS/Nonlinear/get_data.F")) RETURN
        END IF
        IF (LBC(inorth,isAice,ng)%acquire) THEN
          CALL get_ngfld (ng, iNLM, idAibc(inorth),                     &
     &                    ncBRYid(idAibc(inorth),ng),                   &
     &                    nBCfiles(ng), BRY(1,ng), update(1),           &
     &                    ILB, IUB, 1, 2, 0, Lm(ng)+1, 1,               &
     &                    BOUNDARY(ng) % aiG_north)
          IF (FoundError(exit_flag, NoError, 1384,                  &
     &               "ROMS/Nonlinear/get_data.F")) RETURN
        END IF
      END IF
      IF (.not.(RefinedGrid(ng).and.RefineScale(ng).gt.0)) THEN
        IF (LBC(iwest,isHice,ng)%acquire) THEN
          CALL get_ngfld (ng, iNLM, idHibc(iwest),                      &
     &                    ncBRYid(idHibc(iwest),ng),                    &
     &                    nBCfiles(ng), BRY(1,ng), update(1),           &
     &                    JLB, JUB, 1, 2, 0, Mm(ng)+1, 1,               &
     &                    BOUNDARY(ng) % hiG_west)
          IF (FoundError(exit_flag, NoError, 1398,                  &
     &               "ROMS/Nonlinear/get_data.F")) RETURN
        END IF
        IF (LBC(ieast,isHice,ng)%acquire) THEN
          CALL get_ngfld (ng, iNLM, idHibc(ieast),                      &
     &                    ncBRYid(idHibc(ieast),ng),                    &
     &                    nBCfiles(ng), BRY(1,ng), update(1),           &
     &                    JLB, JUB, 1, 2, 0, Mm(ng)+1, 1,               &
     &                    BOUNDARY(ng) % hiG_east)
          IF (FoundError(exit_flag, NoError, 1407,                  &
     &               "ROMS/Nonlinear/get_data.F")) RETURN
        END IF
        IF (LBC(isouth,isHice,ng)%acquire) THEN
          CALL get_ngfld (ng, iNLM, idHibc(isouth),                     &
     &                    ncBRYid(idHibc(isouth),ng),                   &
     &                    nBCfiles(ng), BRY(1,ng), update(1),           &
     &                    ILB, IUB, 1, 2, 0, Lm(ng)+1, 1,               &
     &                    BOUNDARY(ng) % hiG_south)
          IF (FoundError(exit_flag, NoError, 1416,                  &
     &               "ROMS/Nonlinear/get_data.F")) RETURN
        END IF
        IF (LBC(inorth,isHice,ng)%acquire) THEN
          CALL get_ngfld (ng, iNLM, idHibc(inorth),                     &
     &                    ncBRYid(idHibc(inorth),ng),                   &
     &                    nBCfiles(ng), BRY(1,ng), update(1),           &
     &                    ILB, IUB, 1, 2, 0, Lm(ng)+1, 1,               &
     &                    BOUNDARY(ng) % hiG_north)
          IF (FoundError(exit_flag, NoError, 1425,                  &
     &               "ROMS/Nonlinear/get_data.F")) RETURN
        END IF
      END IF
      IF (.not.(RefinedGrid(ng).and.RefineScale(ng).gt.0)) THEN
        IF (LBC(iwest,isHsno,ng)%acquire) THEN
          CALL get_ngfld (ng, iNLM, idHsnbc(iwest),                     &
     &                    ncBRYid(idHsnbc(iwest),ng),                   &
     &                    nBCfiles(ng), BRY(1,ng), update(1),           &
     &                    JLB, JUB, 1, 2, 0, Mm(ng)+1, 1,               &
     &                    BOUNDARY(ng) % hsnG_west)
          IF (FoundError(exit_flag, NoError, 1438,                  &
     &               "ROMS/Nonlinear/get_data.F")) RETURN
        END IF
        IF (LBC(ieast,isHsno,ng)%acquire) THEN
          CALL get_ngfld (ng, iNLM, idHsnbc(ieast),                     &
     &                    ncBRYid(idHsnbc(ieast),ng),                   &
     &                    nBCfiles(ng), BRY(1,ng), update(1),           &
     &                    JLB, JUB, 1, 2, 0, Mm(ng)+1, 1,               &
     &                    BOUNDARY(ng) % hsnG_east)
          IF (FoundError(exit_flag, NoError, 1447,                  &
     &               "ROMS/Nonlinear/get_data.F")) RETURN
        END IF
        IF (LBC(isouth,isHsno,ng)%acquire) THEN
          CALL get_ngfld (ng, iNLM, idHsnbc(isouth),                    &
     &                    ncBRYid(idHsnbc(isouth),ng),                  &
     &                    nBCfiles(ng), BRY(1,ng), update(1),           &
     &                    ILB, IUB, 1, 2, 0, Lm(ng)+1, 1,               &
     &                    BOUNDARY(ng) % hsnG_south)
          IF (FoundError(exit_flag, NoError, 1456,                  &
     &               "ROMS/Nonlinear/get_data.F")) RETURN
        END IF
        IF (LBC(inorth,isHsno,ng)%acquire) THEN
          CALL get_ngfld (ng, iNLM, idHsnbc(inorth),                    &
     &                    ncBRYid(idHsnbc(inorth),ng),                  &
     &                    nBCfiles(ng), BRY(1,ng), update(1),           &
     &                    ILB, IUB, 1, 2, 0, Lm(ng)+1, 1,               &
     &                    BOUNDARY(ng) % hsnG_north)
          IF (FoundError(exit_flag, NoError, 1465,                  &
     &               "ROMS/Nonlinear/get_data.F")) RETURN
        END IF
      END IF
      IF (.not.(RefinedGrid(ng).and.RefineScale(ng).gt.0)) THEN
        IF (LBC(iwest,isTice,ng)%acquire) THEN
          CALL get_ngfld (ng, iNLM, idTibc(iwest),                      &
     &                    ncBRYid(idTibc(iwest),ng),                    &
     &                    nBCfiles(ng), BRY(1,ng), update(1),           &
     &                    JLB, JUB, 1, 2, 0, Mm(ng)+1, 1,               &
     &                    BOUNDARY(ng) % tiG_west)
          IF (FoundError(exit_flag, NoError, 1478,                  &
     &               "ROMS/Nonlinear/get_data.F")) RETURN
        END IF
        IF (LBC(ieast,isTice,ng)%acquire) THEN
          CALL get_ngfld (ng, iNLM, idTibc(ieast),                      &
     &                    ncBRYid(idTibc(ieast),ng),                    &
     &                    nBCfiles(ng), BRY(1,ng), update(1),           &
     &                    JLB, JUB, 1, 2, 0, Mm(ng)+1, 1,               &
     &                    BOUNDARY(ng) % tiG_east)
          IF (FoundError(exit_flag, NoError, 1487,                  &
     &               "ROMS/Nonlinear/get_data.F")) RETURN
        END IF
        IF (LBC(isouth,isTice,ng)%acquire) THEN
          CALL get_ngfld (ng, iNLM, idTibc(isouth),                     &
     &                    ncBRYid(idTibc(isouth),ng),                   &
     &                    nBCfiles(ng), BRY(1,ng), update(1),           &
     &                    ILB, IUB, 1, 2, 0, Lm(ng)+1, 1,               &
     &                    BOUNDARY(ng) % tiG_south)
          IF (FoundError(exit_flag, NoError, 1496,                  &
     &               "ROMS/Nonlinear/get_data.F")) RETURN
        END IF
        IF (LBC(inorth,isTice,ng)%acquire) THEN
          CALL get_ngfld (ng, iNLM, idTibc(inorth),                     &
     &                    ncBRYid(idTibc(inorth),ng),                   &
     &                    nBCfiles(ng), BRY(1,ng), update(1),           &
     &                    ILB, IUB, 1, 2, 0, Lm(ng)+1, 1,               &
     &                    BOUNDARY(ng) % tiG_north)
          IF (FoundError(exit_flag, NoError, 1505,                  &
     &               "ROMS/Nonlinear/get_data.F")) RETURN
        END IF
      END IF
      IF (.not.(RefinedGrid(ng).and.RefineScale(ng).gt.0)) THEN
        IF (LBC(iwest,isSig11,ng)%acquire) THEN
          CALL get_ngfld (ng, iNLM, idS11bc(iwest),                     &
     &                    ncBRYid(idS11bc(iwest),ng),                   &
     &                    nBCfiles(ng), BRY(1,ng), update(1),           &
     &                    JLB, JUB, 1, 2, 0, Mm(ng)+1, 1,               &
     &                    BOUNDARY(ng) % sig11G_west)
          IF (FoundError(exit_flag, NoError, 1518,                  &
     &               "ROMS/Nonlinear/get_data.F")) RETURN
        END IF
        IF (LBC(ieast,isSig11,ng)%acquire) THEN
          CALL get_ngfld (ng, iNLM, idS11bc(ieast),                     &
     &                    ncBRYid(idS11bc(ieast),ng),                   &
     &                    nBCfiles(ng), BRY(1,ng), update(1),           &
     &                    JLB, JUB, 1, 2, 0, Mm(ng)+1, 1,               &
     &                    BOUNDARY(ng) % sig11G_east)
          IF (FoundError(exit_flag, NoError, 1527,                  &
     &               "ROMS/Nonlinear/get_data.F")) RETURN
        END IF
        IF (LBC(isouth,isSig11,ng)%acquire) THEN
          CALL get_ngfld (ng, iNLM, idS11bc(isouth),                    &
     &                    ncBRYid(idS11bc(isouth),ng),                  &
     &                    nBCfiles(ng), BRY(1,ng), update(1),           &
     &                    ILB, IUB, 1, 2, 0, Lm(ng)+1, 1,               &
     &                    BOUNDARY(ng) % sig11G_south)
          IF (FoundError(exit_flag, NoError, 1536,                  &
     &               "ROMS/Nonlinear/get_data.F")) RETURN
        END IF
        IF (LBC(inorth,isSig11,ng)%acquire) THEN
          CALL get_ngfld (ng, iNLM, idS11bc(inorth),                    &
     &                    ncBRYid(idS11bc(inorth),ng),                  &
     &                    nBCfiles(ng), BRY(1,ng), update(1),           &
     &                    ILB, IUB, 1, 2, 0, Lm(ng)+1, 1,               &
     &                    BOUNDARY(ng) % sig11G_north)
          IF (FoundError(exit_flag, NoError, 1545,                  &
     &               "ROMS/Nonlinear/get_data.F")) RETURN
        END IF
      END IF
      IF (.not.(RefinedGrid(ng).and.RefineScale(ng).gt.0)) THEN
        IF (LBC(iwest,isSig22,ng)%acquire) THEN
          CALL get_ngfld (ng, iNLM, idS22bc(iwest),                     &
     &                    ncBRYid(idS22bc(iwest),ng),                   &
     &                    nBCfiles(ng), BRY(1,ng), update(1),           &
     &                    JLB, JUB, 1, 2, 0, Mm(ng)+1, 1,               &
     &                    BOUNDARY(ng) % sig22G_west)
          IF (FoundError(exit_flag, NoError, 1558,                  &
     &               "ROMS/Nonlinear/get_data.F")) RETURN
        END IF
        IF (LBC(ieast,isSig22,ng)%acquire) THEN
          CALL get_ngfld (ng, iNLM, idS22bc(ieast),                     &
     &                    ncBRYid(idS22bc(ieast),ng),                   &
     &                    nBCfiles(ng), BRY(1,ng), update(1),           &
     &                    JLB, JUB, 1, 2, 0, Mm(ng)+1, 1,               &
     &                    BOUNDARY(ng) % sig22G_east)
          IF (FoundError(exit_flag, NoError, 1567,                  &
     &               "ROMS/Nonlinear/get_data.F")) RETURN
        END IF
        IF (LBC(isouth,isSig22,ng)%acquire) THEN
          CALL get_ngfld (ng, iNLM, idS22bc(isouth),                    &
     &                    ncBRYid(idS22bc(isouth),ng),                  &
     &                    nBCfiles(ng), BRY(1,ng), update(1),           &
     &                    ILB, IUB, 1, 2, 0, Lm(ng)+1, 1,               &
     &                    BOUNDARY(ng) % sig22G_south)
          IF (FoundError(exit_flag, NoError, 1576,                  &
     &               "ROMS/Nonlinear/get_data.F")) RETURN
        END IF
        IF (LBC(inorth,isSig22,ng)%acquire) THEN
          CALL get_ngfld (ng, iNLM, idS22bc(inorth),                    &
     &                    ncBRYid(idS22bc(inorth),ng),                  &
     &                    nBCfiles(ng), BRY(1,ng), update(1),           &
     &                    ILB, IUB, 1, 2, 0, Lm(ng)+1, 1,               &
     &                    BOUNDARY(ng) % sig22G_north)
          IF (FoundError(exit_flag, NoError, 1585,                  &
     &               "ROMS/Nonlinear/get_data.F")) RETURN
        END IF
      END IF
      IF (.not.(RefinedGrid(ng).and.RefineScale(ng).gt.0)) THEN
        IF (LBC(iwest,isSig12,ng)%acquire) THEN
          CALL get_ngfld (ng, iNLM, idS12bc(iwest),                     &
     &                    ncBRYid(idS12bc(iwest),ng),                   &
     &                    nBCfiles(ng), BRY(1,ng), update(1),           &
     &                    JLB, JUB, 1, 2, 0, Mm(ng)+1, 1,               &
     &                    BOUNDARY(ng) % sig12G_west)
          IF (FoundError(exit_flag, NoError, 1598,                  &
     &               "ROMS/Nonlinear/get_data.F")) RETURN
        END IF
        IF (LBC(ieast,isSig12,ng)%acquire) THEN
          CALL get_ngfld (ng, iNLM, idS12bc(ieast),                     &
     &                    ncBRYid(idS12bc(ieast),ng),                   &
     &                    nBCfiles(ng), BRY(1,ng), update(1),           &
     &                    JLB, JUB, 1, 2, 0, Mm(ng)+1, 1,               &
     &                    BOUNDARY(ng) % sig12G_east)
          IF (FoundError(exit_flag, NoError, 1607,                  &
     &               "ROMS/Nonlinear/get_data.F")) RETURN
        END IF
        IF (LBC(isouth,isSig12,ng)%acquire) THEN
          CALL get_ngfld (ng, iNLM, idS12bc(isouth),                    &
     &                    ncBRYid(idS12bc(isouth),ng),                  &
     &                    nBCfiles(ng), BRY(1,ng), update(1),           &
     &                    ILB, IUB, 1, 2, 0, Lm(ng)+1, 1,               &
     &                    BOUNDARY(ng) % sig12G_south)
          IF (FoundError(exit_flag, NoError, 1616,                  &
     &               "ROMS/Nonlinear/get_data.F")) RETURN
        END IF
        IF (LBC(inorth,isSig12,ng)%acquire) THEN
          CALL get_ngfld (ng, iNLM, idS12bc(inorth),                    &
     &                    ncBRYid(idS12bc(inorth),ng),                  &
     &                    nBCfiles(ng), BRY(1,ng), update(1),           &
     &                    ILB, IUB, 1, 2, 0, Lm(ng)+1, 1,               &
     &                    BOUNDARY(ng) % sig12G_north)
          IF (FoundError(exit_flag, NoError, 1625,                  &
     &               "ROMS/Nonlinear/get_data.F")) RETURN
        END IF
      END IF
!
!=======================================================================
!  Read in data from Climatology NetCDF file.
!=======================================================================
!
!  Free-surface.
!
      IF (LsshCLM(ng)) THEN
        CALL get_2dfld (ng, iNLM, idSSHc, ncCLMid(idSSHc,ng),           &
     &                  nCLMfiles(ng), CLM(1,ng), update(1),            &
     &                  LBi, UBi, LBj, UBj, 2, 1,                       &
     &                  GRID(ng) % rmask,                               &
     &                  CLIMA(ng) % sshG)
        IF (FoundError(exit_flag, NoError, 1649,                    &
     &                 "ROMS/Nonlinear/get_data.F")) RETURN
      END IF
!
!  2D momentum.
!
      IF (Lm2CLM(ng)) THEN
        CALL get_2dfld (ng, iNLM, idUbcl, ncCLMid(idUbcl,ng),           &
     &                  nCLMfiles(ng), CLM(1,ng), update(1),            &
     &                  LBi, UBi, LBj, UBj, 2, 1,                       &
     &                  GRID(ng) % umask,                               &
     &                  CLIMA(ng) % ubarclmG)
        IF (FoundError(exit_flag, NoError, 1665,                    &
     &                 "ROMS/Nonlinear/get_data.F")) RETURN
!
        CALL get_2dfld (ng, iNLM, idVbcl, ncCLMid(idVbcl,ng),           &
     &                  nCLMfiles(ng), CLM(1,ng), update(1),            &
     &                  LBi, UBi, LBj, UBj, 2, 1,                       &
     &                  GRID(ng) % vmask,                               &
     &                  CLIMA(ng) % vbarclmG)
        IF (FoundError(exit_flag, NoError, 1675,                    &
     &                 "ROMS/Nonlinear/get_data.F")) RETURN
      END IF
!
!  3D momentum.
!
      IF (Lm3CLM(ng)) THEN
        CALL get_3dfld (ng, iNLM, idUclm, ncCLMid(idUclm,ng),           &
     &                  nCLMfiles(ng), CLM(1,ng), update(1),            &
     &                  LBi, UBi, LBj, UBj, 1, N(ng), 2, 1,             &
     &                  GRID(ng) % umask,                               &
     &                  CLIMA(ng) % uclmG)
        IF (FoundError(exit_flag, NoError, 1692,                    &
     &                 "ROMS/Nonlinear/get_data.F")) RETURN
!
        CALL get_3dfld (ng, iNLM, idVclm, ncCLMid(idVclm,ng),           &
     &                  nCLMfiles(ng), CLM(1,ng), update(1),            &
     &                  LBi, UBi, LBj, UBj, 1, N(ng), 2, 1,             &
     &                  GRID(ng) % vmask,                               &
     &                  CLIMA(ng) % vclmG)
        IF (FoundError(exit_flag, NoError, 1702,                    &
     &                 "ROMS/Nonlinear/get_data.F")) RETURN
      END IF
!
!  Tracers.
!
      ic=0
      DO i=1,NT(ng)
        IF (LtracerCLM(i,ng)) THEN
          ic=ic+1
          CALL get_3dfld (ng, iNLM, idTclm(i),                          &
     &                    ncCLMid(idTclm(i),ng),                        &
     &                    nCLMfiles(ng), CLM(1,ng), update(1),          &
     &                    LBi, UBi, LBj, UBj, 1, N(ng), 2, 1,           &
     &                    GRID(ng) % rmask,                             &
     &                    CLIMA(ng) % tclmG(:,:,:,:,ic))
          IF (FoundError(exit_flag, NoError, 1722,                  &
     &                   "ROMS/Nonlinear/get_data.F")) RETURN
        END IF
      END DO
      IF (LaiCLM(ng)) THEN
        CALL get_2dfld (ng, iNLM, idAIclm, ncCLMid(idAIclm,ng),         &
     &                  nCLMfiles(ng), CLM(1,ng), update(1),            &
     &                  LBi, UBi, LBj, UBj, 2, 1,                       &
     &                  GRID(ng) % rmask,                               &
     &                  CLIMA(ng) % aiclmG)
        IF (FoundError(exit_flag, NoError, 1784,                    &
     &               "ROMS/Nonlinear/get_data.F")) RETURN
        CALL get_2dfld (ng, iNLM, idHIclm, ncCLMid(idHIclm,ng),         &
     &                  nCLMfiles(ng), CLM(1,ng), update(1),            &
     &                  LBi, UBi, LBj, UBj, 2, 1,                       &
     &                  GRID(ng) % rmask,                               &
     &                  CLIMA(ng) % hiclmG)
        IF (FoundError(exit_flag, NoError, 1793,                    &
     &               "ROMS/Nonlinear/get_data.F")) RETURN
        CALL get_2dfld (ng, iNLM, idHSNclm, ncCLMid(idHSNclm,ng),       &
     &                  nCLMfiles(ng), CLM(1,ng), update(1),            &
     &                  LBi, UBi, LBj, UBj, 2, 1,                       &
     &                  GRID(ng) % rmask,                               &
     &                  CLIMA(ng) % hsnclmG)
        IF (FoundError(exit_flag, NoError, 1802,                    &
     &               "ROMS/Nonlinear/get_data.F")) RETURN
      END IF
      IF (LsiCLM(ng)) THEN
        CALL get_2dfld (ng, iNLM, idS11clm, ncCLMid(idS11clm,ng),       &
     &                  nCLMfiles(ng), CLM(1,ng), update(1),            &
     &                  LBi, UBi, LBj, UBj, 2, 1,                       &
     &                  GRID(ng) % rmask,                               &
     &                  CLIMA(ng) % sig11clmG)
        IF (FoundError(exit_flag, NoError, 1816,                    &
     &               "ROMS/Nonlinear/get_data.F")) RETURN
        CALL get_2dfld (ng, iNLM, idS12clm, ncCLMid(idS12clm,ng),       &
     &                  nCLMfiles(ng), CLM(1,ng), update(1),            &
     &                  LBi, UBi, LBj, UBj, 2, 1,                       &
     &                  GRID(ng) % rmask,                               &
     &                  CLIMA(ng) % sig12clmG)
        IF (FoundError(exit_flag, NoError, 1825,                    &
     &               "ROMS/Nonlinear/get_data.F")) RETURN
        CALL get_2dfld (ng, iNLM, idS22clm, ncCLMid(idS22clm,ng),       &
     &                  nCLMfiles(ng), CLM(1,ng), update(1),            &
     &                  LBi, UBi, LBj, UBj, 2, 1,                       &
     &                  GRID(ng) % rmask,                               &
     &                  CLIMA(ng) % sig22clmG)
        IF (FoundError(exit_flag, NoError, 1834,                    &
     &               "ROMS/Nonlinear/get_data.F")) RETURN
      END IF
      IF (LmiCLM(ng)) THEN
        CALL get_2dfld (ng, iNLM, idUIclm, ncCLMid(idUIclm,ng),         &
     &                  nCLMfiles(ng), CLM(1,ng), update(1),            &
     &                  LBi, UBi, LBj, UBj, 2, 1,                       &
     &                  GRID(ng) % rmask,                               &
     &                  CLIMA(ng) % uiclmG)
        IF (FoundError(exit_flag, NoError, 1848,                    &
     &               "ROMS/Nonlinear/get_data.F")) RETURN
        CALL get_2dfld (ng, iNLM, idVIclm, ncCLMid(idVIclm,ng),         &
     &                  nCLMfiles(ng), CLM(1,ng), update(1),            &
     &                  LBi, UBi, LBj, UBj, 2, 1,                       &
     &                  GRID(ng) % rmask,                               &
     &                  CLIMA(ng) % viclmG)
        IF (FoundError(exit_flag, NoError, 1857,                    &
     &               "ROMS/Nonlinear/get_data.F")) RETURN
      END IF
!
!-----------------------------------------------------------------------
!  Turn off input data time wall clock.
!-----------------------------------------------------------------------
!
      CALL wclock_off (ng, iNLM, 3, 2042, "ROMS/Nonlinear/get_data.F")
      RETURN
      END SUBROUTINE get_data
