      MODULE diag_mod
!
!svn $Id$
!================================================== Hernan G. Arango ===
!  Copyright (c) 2002-2020 The ROMS/TOMS Group                         !
!    Licensed under a MIT/X style license                              !
!    See License_ROMS.txt                                              !
!=======================================================================
!                                                                      !
!  This routine computes various diagnostic fields.                    !
!                                                                      !
!=======================================================================
!
      implicit none
      PRIVATE
      PUBLIC  :: diag
      CONTAINS
!
!***********************************************************************
      SUBROUTINE diag (ng, tile)
!***********************************************************************
!
      USE mod_param
      USE mod_grid
      USE mod_ocean
      USE mod_stepping
!
!
!  Imported variable declarations.
!
      integer, intent(in) :: ng, tile
!
!  Local variable declarations.
!
      integer :: IminS, ImaxS, JminS, JmaxS
      integer :: LBi, UBi, LBj, UBj, LBij, UBij
!
!  Set horizontal starting and ending indices for automatic private
!  storage arrays.
!
      IminS=BOUNDS(ng)%Istr(tile)-3
      ImaxS=BOUNDS(ng)%Iend(tile)+3
      JminS=BOUNDS(ng)%Jstr(tile)-3
      JmaxS=BOUNDS(ng)%Jend(tile)+3
!
!  Determine array lower and upper bounds in the I- and J-directions.
!
      LBi=BOUNDS(ng)%LBi(tile)
      UBi=BOUNDS(ng)%UBi(tile)
      LBj=BOUNDS(ng)%LBj(tile)
      UBj=BOUNDS(ng)%UBj(tile)
!
!  Set array lower and upper bounds for MIN(I,J) directions and
!  MAX(I,J) directions.
!
      LBij=BOUNDS(ng)%LBij
      UBij=BOUNDS(ng)%UBij
!
      CALL wclock_on (ng, iNLM, 7, 44, "ROMS/Nonlinear/diag.F")
      CALL diag_tile (ng, tile,                                         &
     &                LBi, UBi, LBj, UBj,                               &
     &                IminS, ImaxS, JminS, JmaxS,                       &
     &                nstp(ng), krhs(ng),                               &
     &                GRID(ng) % h,                                     &
     &                GRID(ng) % pm,                                    &
     &                GRID(ng) % pn,                                    &
     &                GRID(ng) % omn,                                   &
     &                GRID(ng) % Hz,                                    &
     &                GRID(ng) % z_r,                                   &
     &                GRID(ng) % z_w,                                   &
     &                OCEAN(ng) % rho,                                  &
     &                OCEAN(ng) % u,                                    &
     &                OCEAN(ng) % v,                                    &
     &                OCEAN(ng) % t,                                    &
     &                OCEAN(ng) % wvel,                                 &
     &                GRID(ng) % zice,                                  &
     &                OCEAN(ng) % ubar,                                 &
     &                OCEAN(ng) % vbar,                                 &
     &                OCEAN(ng) % zeta)
      CALL wclock_off (ng, iNLM, 7, 74, "ROMS/Nonlinear/diag.F")
      RETURN
      END SUBROUTINE diag
!
!***********************************************************************
      SUBROUTINE diag_tile (ng, tile,                                   &
     &                      LBi, UBi, LBj, UBj,                         &
     &                      IminS, ImaxS, JminS, JmaxS,                 &
     &                      nstp, krhs,                                 &
     &                      h, pm, pn, omn,                             &
     &                      Hz, z_r, z_w,                               &
     &                      rho, u, v, t, wvel,                         &
     &                      zice,                                       &
     &                      ubar, vbar, zeta)
!***********************************************************************
!
      USE mod_param
      USE mod_parallel
      USE mod_iounits
      USE mod_scalars
      USE mod_biology, ONLY : idbio
      USE distribute_mod, ONLY : mp_reduce
      USE distribute_mod, ONLY : mp_reduce2
!
      implicit none
!
!  Imported variable declarations.
!
      integer, intent(in) :: ng, tile
      integer, intent(in) :: LBi, UBi, LBj, UBj
      integer, intent(in) :: IminS, ImaxS, JminS, JmaxS
      integer, intent(in) :: nstp, krhs
      real(r8), intent(in) :: h(LBi:,LBj:)
      real(r8), intent(in) :: pm(LBi:,LBj:)
      real(r8), intent(in) :: pn(LBi:,LBj:)
      real(r8), intent(in) :: omn(LBi:,LBj:)
      real(r8), intent(in) :: Hz(LBi:,LBj:,:)
      real(r8), intent(in) :: z_r(LBi:,LBj:,:)
      real(r8), intent(in) :: z_w(LBi:,LBj:,0:)
      real(r8), intent(in) :: rho(LBi:,LBj:,:)
      real(r8), intent(in) :: u(LBi:,LBj:,:,:)
      real(r8), intent(in) :: v(LBi:,LBj:,:,:)
      real(r8), intent(in) :: t(LBi:,LBj:,:,:,:)
      real(r8), intent(in) :: wvel(LBi:,LBj:,0:)
      real(r8), intent(in) :: zice(LBi:,LBj:)
      real(r8), intent(in) :: ubar(LBi:,LBj:,:)
      real(r8), intent(in) :: vbar(LBi:,LBj:,:)
      real(r8), intent(in) :: zeta(LBi:,LBj:,:)
!
!  Local variable declarations.
!
      integer :: ibt, itrc
      integer :: NSUB, i, ispace, j, k, trd
      integer :: my_max_Ci, my_max_Cj, my_max_Ck
      integer :: Nreduce
      integer, parameter :: Ncourant = 7
      real(r8), allocatable           :: buffer(:)
      character (len=3), allocatable  :: op_handle(:)
      character (len=6), allocatable  :: C_handle(:)
      real(r8), dimension(Ncourant) :: Courant
      real(r8) :: cff, my_avgke, my_avgpe, my_volume
      real(r8) :: my_C , my_max_C
      real(r8) :: my_Cu, my_max_Cu
      real(r8) :: my_Cv, my_max_Cv
      real(r8) :: my_Cw, my_max_Cw
      real(r8) :: my_maxspeed, u2v2
      real(r8) :: my_maxrho, my_mintemp
      real(r8) :: my_maxbio(NBT)
      real(r8), dimension(IminS:ImaxS,JminS:JmaxS) :: ke2d
      real(r8), dimension(IminS:ImaxS,JminS:JmaxS) :: pe2d
      integer :: ii_max, jj_max, kk_max
      real(r8) :: uv_max, ts_min
      character (len=8 ) :: kechar, pechar, maxvalchar
      character (len=60) :: frmt
!
!-----------------------------------------------------------------------
!  Set lower and upper tile bounds and staggered variables bounds for
!  this horizontal domain partition.  Notice that if tile=-1, it will
!  set the values for the global grid.
!-----------------------------------------------------------------------
!
      integer :: Istr, IstrB, IstrP, IstrR, IstrT, IstrM, IstrU
      integer :: Iend, IendB, IendP, IendR, IendT
      integer :: Jstr, JstrB, JstrP, JstrR, JstrT, JstrM, JstrV
      integer :: Jend, JendB, JendP, JendR, JendT
      integer :: Istrm3, Istrm2, Istrm1, IstrUm2, IstrUm1
      integer :: Iendp1, Iendp2, Iendp2i, Iendp3
      integer :: Jstrm3, Jstrm2, Jstrm1, JstrVm2, JstrVm1
      integer :: Jendp1, Jendp2, Jendp2i, Jendp3
!
      Istr   =BOUNDS(ng) % Istr   (tile)
      IstrB  =BOUNDS(ng) % IstrB  (tile)
      IstrM  =BOUNDS(ng) % IstrM  (tile)
      IstrP  =BOUNDS(ng) % IstrP  (tile)
      IstrR  =BOUNDS(ng) % IstrR  (tile)
      IstrT  =BOUNDS(ng) % IstrT  (tile)
      IstrU  =BOUNDS(ng) % IstrU  (tile)
      Iend   =BOUNDS(ng) % Iend   (tile)
      IendB  =BOUNDS(ng) % IendB  (tile)
      IendP  =BOUNDS(ng) % IendP  (tile)
      IendR  =BOUNDS(ng) % IendR  (tile)
      IendT  =BOUNDS(ng) % IendT  (tile)
      Jstr   =BOUNDS(ng) % Jstr   (tile)
      JstrB  =BOUNDS(ng) % JstrB  (tile)
      JstrM  =BOUNDS(ng) % JstrM  (tile)
      JstrP  =BOUNDS(ng) % JstrP  (tile)
      JstrR  =BOUNDS(ng) % JstrR  (tile)
      JstrT  =BOUNDS(ng) % JstrT  (tile)
      JstrV  =BOUNDS(ng) % JstrV  (tile)
      Jend   =BOUNDS(ng) % Jend   (tile)
      JendB  =BOUNDS(ng) % JendB  (tile)
      JendP  =BOUNDS(ng) % JendP  (tile)
      JendR  =BOUNDS(ng) % JendR  (tile)
      JendT  =BOUNDS(ng) % JendT  (tile)
!
      Istrm3 =BOUNDS(ng) % Istrm3 (tile)            ! Istr-3
      Istrm2 =BOUNDS(ng) % Istrm2 (tile)            ! Istr-2
      Istrm1 =BOUNDS(ng) % Istrm1 (tile)            ! Istr-1
      IstrUm2=BOUNDS(ng) % IstrUm2(tile)            ! IstrU-2
      IstrUm1=BOUNDS(ng) % IstrUm1(tile)            ! IstrU-1
      Iendp1 =BOUNDS(ng) % Iendp1 (tile)            ! Iend+1
      Iendp2 =BOUNDS(ng) % Iendp2 (tile)            ! Iend+2
      Iendp2i=BOUNDS(ng) % Iendp2i(tile)            ! Iend+2 interior
      Iendp3 =BOUNDS(ng) % Iendp3 (tile)            ! Iend+3
      Jstrm3 =BOUNDS(ng) % Jstrm3 (tile)            ! Jstr-3
      Jstrm2 =BOUNDS(ng) % Jstrm2 (tile)            ! Jstr-2
      Jstrm1 =BOUNDS(ng) % Jstrm1 (tile)            ! Jstr-1
      JstrVm2=BOUNDS(ng) % JstrVm2(tile)            ! JstrV-2
      JstrVm1=BOUNDS(ng) % JstrVm1(tile)            ! JstrV-1
      Jendp1 =BOUNDS(ng) % Jendp1 (tile)            ! Jend+1
      Jendp2 =BOUNDS(ng) % Jendp2 (tile)            ! Jend+2
      Jendp2i=BOUNDS(ng) % Jendp2i(tile)            ! Jend+2 interior
      Jendp3 =BOUNDS(ng) % Jendp3 (tile)            ! Jend+3
!
!-----------------------------------------------------------------------
!  Compute and report out volume averaged kinetic, potential
!  total energy, volume, Courant numbers.
!-----------------------------------------------------------------------
!
      Nreduce = 6 + NBT
      IF (.not. allocated(buffer))    allocate (buffer(Nreduce))
      IF (.not. allocated(op_handle)) allocate (op_handle(Nreduce))
      IF (.not. allocated(C_handle))  allocate (C_handle(Nreduce))
      IF (MOD(iic(ng)-1,ninfo(ng)).eq.0) THEN
        my_max_C =0.0_r8
        my_max_Cu=0.0_r8
        my_max_Cv=0.0_r8
        my_max_Cw=0.0_r8
        my_max_Ci=0
        my_max_Cj=0
        my_max_Ck=0
        my_maxspeed=0.0_r8
        my_maxrho=-spval
        my_mintemp=1.0E+20_r8
        DO ibt=1,NBT
          my_maxbio(ibt)=-1.0E+20_r8
        END DO
        DO j=Jstr,Jend
          DO i=Istr,Iend
            ke2d(i,j)=0.0_r8
            pe2d(i,j)=0.5_r8*g*z_w(i,j,N(ng))*z_w(i,j,N(ng))
          END DO
          cff=g/rho0
          DO k=N(ng),1,-1
            DO i=Istr,Iend
              u2v2=u(i  ,j,k,nstp)*u(i  ,j,k,nstp)+                     &
     &             u(i+1,j,k,nstp)*u(i+1,j,k,nstp)+                     &
     &             v(i,j  ,k,nstp)*v(i,j  ,k,nstp)+                     &
     &             v(i,j+1,k,nstp)*v(i,j+1,k,nstp)
              ke2d(i,j)=ke2d(i,j)+                                      &
     &                  Hz(i,j,k)*0.25_r8*u2v2
              pe2d(i,j)=pe2d(i,j)+                                      &
     &                  cff*Hz(i,j,k)*(rho(i,j,k)+1000.0_r8)*           &
     &                  (z_r(i,j,k)-z_w(i,j,0))
              my_Cu=0.5_r8*ABS(u(i,j,k,nstp)+u(i+1,j,k,nstp))*          &
     &              dt(ng)*pm(i,j)
              my_Cv=0.5_r8*ABS(v(i,j,k,nstp)+v(i,j+1,k,nstp))*          &
     &              dt(ng)*pn(i,j)
              my_Cw=0.5_r8*ABS(wvel(i,j,k-1)+wvel(i,j,k))*              &
     &              dt(ng)/Hz(i,j,k)
              my_C=my_Cu+my_Cv+my_Cw
              IF (my_C.gt.my_max_C) THEN
                my_max_C =my_C
                my_max_Cu=my_Cu
                my_max_Cv=my_Cv
                my_max_Cw=my_Cw
                my_max_Ci=i
                my_max_Cj=j
                my_max_Ck=k
              END IF
              my_maxspeed=MAX(my_maxspeed,SQRT(0.5_r8*u2v2))
              my_maxrho=MAX(my_maxrho,rho(i,j,k))
              my_mintemp=MIN(my_mintemp,t(i,j,k,nstp,itemp))
            END DO
          END DO
          DO k=N(ng),1,-1
            DO ibt=1,NBT
              itrc=idbio(ibt)
              DO i=Istr,Iend
                my_maxbio(ibt)=MAX(my_maxbio(ibt),t(i,j,k,nstp,itrc))
              END DO
            END DO
          END DO
        END DO
!
!  Integrate horizontally within one tile. In order to reduce the
!  round-off errors, the summation is performed in two stages. First,
!  the index j is collapsed and then the accumulation is carried out
!  along index i. In this order, the partial sums consist on much
!  fewer number of terms than in a straightforward two-dimensional
!  summation. Thus, adding numbers which are orders of magnitude
!  apart is avoided.
!
        DO i=Istr,Iend
          pe2d(i,Jend+1)=0.0_r8
          pe2d(i,Jstr-1)=0.0_r8
          ke2d(i,Jstr-1)=0.0_r8
        END DO
        DO j=Jstr,Jend
          DO i=Istr,Iend
            pe2d(i,Jend+1)=pe2d(i,Jend+1)+                              &
     &                     omn(i,j)*(z_w(i,j,N(ng))-z_w(i,j,0))
            pe2d(i,Jstr-1)=pe2d(i,Jstr-1)+omn(i,j)*pe2d(i,j)
            ke2d(i,Jstr-1)=ke2d(i,Jstr-1)+omn(i,j)*ke2d(i,j)
          END DO
        END DO
        my_volume=0.0_r8
        my_avgpe=0.0_r8
        my_avgke=0.0_r8
        DO i=Istr,Iend
          my_volume=my_volume+pe2d(i,Jend+1)
          my_avgpe =my_avgpe +pe2d(i,Jstr-1)
          my_avgke =my_avgke +ke2d(i,Jstr-1)
        END DO
!
!  Perform global summation: whoever gets first to the critical region
!  resets global sums before global summation starts; after the global
!  summation is completed, thread, which is the last one to enter the
!  critical region, finalizes the computation of diagnostics and prints
!  them out.
!
        NSUB=1                           ! distributed-memory
!$OMP CRITICAL (NL_DIAGNOSTICS)
          maxrho(ng)=my_maxrho
          mintemp(ng)=my_mintemp
          DO ibt=1,NBT
            maxbio(ibt,ng)=0.0_r8
          END DO
        volume=volume+my_volume
        avgke=avgke+my_avgke
        avgpe=avgpe+my_avgpe
        maxspeed(ng)=MAX(maxspeed(ng),my_maxspeed)
        maxrho(ng)=MAX(maxrho(ng),my_maxrho)
        mintemp(ng)=MIN(mintemp(ng),my_mintemp)
        DO ibt=1,NBT
          maxbio(ibt,ng)=MAX(maxbio(ibt,ng),my_maxbio(ibt))
        END DO
        IF (my_max_C.eq.max_C) THEN
          max_Ci=MIN(max_Ci,my_max_Ci)
          max_Cj=MIN(max_Cj,my_max_Cj)
          max_Ck=MIN(max_Ck,my_max_Ck)
        ELSE IF (my_max_C.gt.max_C) THEN
          max_C =my_max_C
          max_Cu=my_max_Cu
          max_Cv=my_max_Cv
          max_Cw=my_max_Cw
          max_Ci=my_max_Ci
          max_Cj=my_max_Cj
          max_Ck=my_max_Ck
        END IF
        tile_count=tile_count+1
        IF (tile_count.eq.NSUB) THEN
          tile_count=0
          buffer(1)=volume
          buffer(2)=avgke
          buffer(3)=avgpe
          buffer(4)=maxspeed(ng)
          buffer(5)=maxrho(ng)
          buffer(6)=mintemp(ng)
          DO ibt=1,NBT
            buffer(6+ibt)=maxbio(ibt,ng)
          END DO
          op_handle(1)='SUM'
          op_handle(2)='SUM'
          op_handle(3)='SUM'
          op_handle(4)='MAX'
          op_handle(5)='MAX'
          op_handle(6)='MIN'
          DO ibt=1,NBT
            op_handle(6+ibt)='MAX'
          END DO
          CALL mp_reduce (ng, iNLM, Nreduce, buffer, op_handle)
          volume=buffer(1)
          avgke=buffer(2)
          avgpe=buffer(3)
          maxspeed(ng)=buffer(4)
          maxrho(ng)=buffer(5)
          mintemp(ng)=buffer(6)
          DO ibt=1,NBT
            maxbio(ibt,ng)=buffer(6+ibt)
          END DO
!
          Courant(1)=max_C
          Courant(2)=max_Cu
          Courant(3)=max_Cv
          Courant(4)=REAL(max_Ci,r8)
          Courant(5)=REAL(max_Cj,r8)
          Courant(6)=max_Cw
          Courant(7)=REAL(max_Ck,r8)
          C_handle(1)='MAXLOC'
          CALL mp_reduce2 (ng, iNLM, Ncourant, 1, Courant, C_handle)
          max_C =Courant(1)
          max_Cu=Courant(2)
          max_Cv=Courant(3)
          max_Ci=INT(Courant(4))
          max_Cj=INT(Courant(5))
          max_Cw=Courant(6)
          max_Ck=INT(Courant(7))
!
          trd=MyMaster
          avgke=avgke/volume
          avgpe=avgpe/volume
          avgkp=avgke+avgpe
          IF (first_time(ng).eq.0) THEN
            first_time(ng)=1
            IF (Master.and.(ng.eq.1)) THEN
              WRITE (stdout,10) 'TIME-STEP', 'YYYY-MM-DD hh:mm:ss.ss',  &
     &                          'KINETIC_ENRG', 'POTEN_ENRG',           &
     &                          'TOTAL_ENRG', 'NET_VOLUME'
              WRITE (stdout,20) 'C => (i,j,k)', 'Cu', 'Cv',             &
     &                          '  Cw  ', 'Max Speed'
 10           FORMAT (/,1x,a,1x,a,2x,a,3x,a,4x,a,4x,a)
 20           FORMAT (21x,a,7x,a,12x,a,10x,a,7x,a,/)
            END IF
          END IF
          IF (Master) THEN    ! restart counter after 10 billion steps
            WRITE (stdout,30) INT(MOD(REAL(iic(ng)-1,r8),1.0E+10_r8)),  &
                              time_code(ng),                            &
     &                        avgke, avgpe, avgkp, volume
            ispace=35-(6+Idigits(ng)+Jdigits(ng)+Kdigits(ng))
            WRITE (frmt,40) ispace,                                     &
     &                      '"("', Idigits(ng), Idigits(ng),            &
     &                      '","', Jdigits(ng), Jdigits(ng),            &
     &                      '","', Kdigits(ng), Kdigits(ng), '")"'
            WRITE (stdout,frmt) max_Ci, max_Cj,                         &
     &                          max_Ck, max_Cu, max_Cv, max_Cw,         &
     &                          maxspeed(ng)
            CALL my_flush (stdout)
 30         FORMAT (i10,1x,a,4(1pe14.6))
 40         FORMAT ('(',i2.2,'x,',a,',i',i1,'.',i1,',',                 &
     &                            a,',i',i1,'.',i1,',',                 &
     &                            a,',i',i1,'.',i1,',',                 &
     &                            a,',t35,4(1pe13.6,1x))')
          END IF
!
!  If blowing-up, set exit_flag to stop computations.
!
          WRITE (kechar,'(1pe8.1)') avgke
          WRITE (pechar,'(1pe8.1)') avgpe
          DO i=1,8
            IF ((kechar(i:i).eq.'N').or.(pechar(i:i).eq.'N').or.        &
     &          (kechar(i:i).eq.'n').or.(pechar(i:i).eq.'n').or.        &
     &          (kechar(i:i).eq.'*').or.(pechar(i:i).eq.'*')) THEN
              exit_flag=1
              blowup_string='KEchar = '//kechar//', PEchar = '//pechar
            END IF
          END DO
!
!  Stop computations if exceeding maximum speed allowed.  This will be
!  useful during debugging to avoid NaNs in output NetCDF files.
!
          IF (maxspeed(ng).gt.max_speed) THEN
            exit_flag=1
            IF (Master) print *, 'DIAG speed trouble ', maxspeed(ng)
            ii_max = Istr
            jj_max = Jstr
            kk_max = 1
            uv_max = max(u(Istr,Jstr,1,nstp),v(Istr,Jstr,1,nstp))
            DO j=Jstr,Jend
              DO k=1,N(ng)
                DO i=Istr,Iend
                  IF (max(u(i,j,k,nstp),v(i,j,k,nstp)) > uv_max) THEN
                    ii_max = i
                    jj_max = j
                    kk_max = k
                    uv_max = max(u(i,j,k,nstp),v(i,j,k,nstp))
                  END IF
                END DO
              END DO
            END DO
            IF (uv_max .gt. max_speed) THEN
              print *, 'DIAG speed ijk ', uv_max, ii_max, jj_max, kk_max, ng
            END IF
            WRITE (maxvalchar,'(1pe8.1)') maxspeed(ng)
            blowup_string='MaxSpeed = '//maxvalchar
          END IF
!
!  Stop computation if exceeding maximum density anomaly allowed.
!  Recall that  density is computed from potential temperature and
!  salinity.  This is a good way to screen for very bad values which
!  indicates that the model is blowing-up.
!
          IF (maxrho(ng) .gt. max_rho) THEN
            exit_flag=1
            IF (Master) print *, 'DIAG rho trouble ', maxrho(ng), ng
          END IF
          IF (mintemp(ng) .lt. min_temp) THEN
            exit_flag=1
            IF (Master) print *, 'DIAG temperature trouble ', mintemp(ng), ng
            ii_max = Istr
            jj_max = Jstr
            kk_max = 1
            ts_min = t(Istr,Jstr,1,nstp,itemp)
            DO j=Jstr,Jend
              DO k=1,N(ng)
                DO i=Istr,Iend
                  IF (t(i,j,k,itemp,nstp) < ts_min) THEN
                    ii_max = i
                    jj_max = j
                    kk_max = k
                    ts_min = t(i,j,k,nstp,itemp)
                  END IF
                END DO
              END DO
            END DO
            IF (ts_min .lt. min_temp) THEN
              print *, 'DIAG temp ijk ', ts_min, ii_max, jj_max, kk_max, ng
            END IF
            WRITE (maxvalchar,'(1pe8.1)') maxrho(ng)
            blowup_string='MaxDensity = '//maxvalchar
          END IF
          DO ibt=1,NBT
            IF (maxbio(ibt,ng).gt.max_bio(ibt)) THEN
              IF (Master) print *, 'BIO trouble in DIAG ', maxbio(:,ng)
              exit_flag=1
            END IF
            WRITE (pechar,'(1pe8.1)') maxbio(ibt,ng)
            DO i=1,8
              IF ((pechar(i:i).eq.'N').or.(pechar(i:i).eq.'n').or.      &
     &            (pechar(i:i).eq.'*')) THEN
                IF (Master) print *, 'BIO trouble in DIAG ', pechar, ibt
                exit_flag=1
              END IF
            END DO
          END DO
!
!  Reset global reduction variables for the next call.
!
          volume=0.0_r8
          avgke=0.0_r8
          avgpe=0.0_r8
          maxspeed(ng)=-Large
          maxrho(ng)=-Large
          max_C =0.0_r8
          max_Cu=0.0_r8
          max_Cv=0.0_r8
          max_Cw=0.0_r8
          max_Ci=0
          max_Cj=0
          max_Ck=0
        END IF
!$OMP END CRITICAL (NL_DIAGNOSTICS)
      END IF
      RETURN
      END SUBROUTINE diag_tile
      END MODULE diag_mod
