#include "cppdefs.h"

      MODULE sedtr_reactions_mod

#if defined NONLINEAR && defined SEDIMENT && defined SEDBIO_COUP
!
!svn $Id: sed_bed_mod.F 2011 2009-12-20 17:34:23Z arango $
!==================================================== C. R. Sherwood ===
!  Copyright (c) 2002-2010 The ROMS/TOMS Group      Hernan G. Arango   !
!    Licensed under a MIT/X style license                              !
!    See License_ROMS.txt                                              !
!=======================================================================
!                                                                      !
!     This routine computes reaction terms for seabed tracers          !
!     (particulate and dissolved) following Soetaert, et al.'s model as!
!     described in Moriarty et al. (2017;Biogeosciences). The diffusive!
!     piece is calculated in sedtr_biodiff, and the advective piece is !
!     covered in the sediment bed calculations (sed_bed).              !
!                                                                      !
!=======================================================================
!
      implicit none

      PRIVATE
      PUBLIC  :: sedtr_reactions

      CONTAINS
!
!***********************************************************************
      SUBROUTINE sedtr_reactions (ng, tile)
!***********************************************************************
!
      USE mod_param
      USE mod_forces
      USE mod_grid
      USE mod_ocean
      USE mod_sedbed
      USE mod_biology
      USE mod_stepping
!
!  Imported variable declarations.
!
      integer, intent(in) :: ng, tile
!
!  Local variable declarations.
!
# include "tile.h"
!
# ifdef PROFILE
      CALL wclock_on (ng, iNLM, 16)
# endif
      CALL sedtr_reactions_tile (ng, tile,                                &
     &                   LBi, UBi, LBj, UBj,                            &
     &                   IminS, ImaxS, JminS, JmaxS,                    &
     &                   nstp(ng), nnew(ng),                            &
# ifdef WET_DRY
     &                   GRID(ng) % rmask_wet,                          &
# endif
     &                   OCEAN(ng) % t,                                 &
# if defined SED_MORPH
     &                   SEDBED(ng) % bed_thick,                        &
# endif
# if defined SEDBIO_COUP && defined SED_BIODIFF
     &                   SEDBED(ng) % o2uptake,                         &
     &                   SEDBED(ng) % boxcon,                           &
     &                   SEDBED(ng) % carrem,                           &
     &                   SEDBED(ng) % bed,                              &
     &                   SEDBED(ng) % bed_tracer)
# else
     &                   SEDBED(ng) % bed)
# endif
# ifdef PROFILE
      CALL wclock_off (ng, iNLM, 16)
# endif
      RETURN
      END SUBROUTINE sedtr_reactions
!
!***********************************************************************
      SUBROUTINE sedtr_reactions_tile (ng, tile,                          &
     &                         LBi, UBi, LBj, UBj,                      &
     &                         IminS, ImaxS, JminS, JmaxS,              &
     &                         nstp, nnew,                              &
# ifdef WET_DRY
     &                         rmask_wet,                               &
# endif
     &                         t,                                       &
# if defined SED_MORPH
     &                         bed_thick,                               &
# endif
# if defined SEDBIO_COUP && defined SED_BIODIFF
     &                         o2uptake,                                &
     &                         boxcon,                                  &
     &                         carrem,                                  &
     &                         bed,                                     &
     &                         bed_tracer)
# else
     &                         bed)
# endif
!***********************************************************************
!
      USE mod_param
      USE mod_scalars
      USE mod_sediment
      USE mod_biology
      USE bc_3d_mod, ONLY : bc_r3d_tile
      USE bc_2d_mod, ONLY : bc_r2d_tile
# if defined EW_PERIODIC || defined NS_PERIODIC
      USE exchange_2d_mod, ONLY : exchange_r2d_tile
# endif
# ifdef DISTRIBUTE
      USE mp_exchange_mod, ONLY : mp_exchange3d, mp_exchange4d
# endif
!
!  Imported variable declarations.
!
      integer, intent(in) :: ng, tile
      integer, intent(in) :: LBi, UBi, LBj, UBj
      integer, intent(in) :: IminS, ImaxS, JminS, JmaxS
      integer, intent(in) :: nstp, nnew
# ifdef ASSUMED_SHAPE
#  ifdef WET_DRY
      real(r8), intent(in) :: rmask_wet(LBi:,LBj:)
#  endif
#  if defined SED_MORPH
      real(r8), intent(inout):: bed_thick(LBi:,LBj:,:)
#  endif
      real(r8), intent(inout) :: t(LBi:,LBj:,:,:,:)
      real(r8), intent(inout) :: bed(LBi:,LBj:,:,:)
# if defined SEDBIO_COUP && defined SED_BIODIFF
      real(r8), intent(inout) :: bed_tracer(LBi:,LBj:,:,:)
# endif
# if defined SEDBIO_COUP
      real(r8), intent(inout) :: o2uptake(LBi:,LBj:)
      real(r8), intent(inout) :: boxcon(LBi:,LBj:,:,:)
      real(r8), intent(inout) :: carrem(LBi:,LBj:,:,:)
# endif
# else
#  ifdef WET_DRY
      real(r8), intent(in) :: rmask_wet(LBi:UBi,LBj:UBj)
#  endif
#  if defined SED_MORPH
      real(r8), intent(inout):: bed_thick(LBi:UBi,LBj:UBj,2)
#  endif
      real(r8), intent(inout) :: t(LBi:UBi,LBj:UBj,N(ng),3,NT(ng))
      real(r8), intent(inout) :: bed(LBi:UBi,LBj:UBj,Nbed,MBEDP)
# if defined SEDBIO_COUP && defined SED_BIODIFF
      real(r8), intent(inout) :: bed_tracer(LBi:UBi,LBj:UBj,Nbed,NSBT)
# endif
# if defined SEDBIO_COUP
      real(r8), intent(inout) :: o2uptake(LBi:UBi,LBj:UBj)
      real(r8), intent(inout) :: boxcon(LBi:UBi,LBj:UBj,Nbed,NSBT)
      real(r8), intent(inout) :: carrem(LBi:UBi,LBj:UBj,Nbed,NSBT)
# endif
# endif
!
!  Local variable declarations.
!
# ifdef DISTRIBUTE
#  ifdef EW_PERIODIC
      logical :: EWperiodic=.TRUE.
#  else
      logical :: EWperiodic=.FALSE.
#  endif
#  ifdef NS_PERIODIC
      logical :: NSperiodic=.TRUE.
#  else
      logical :: NSperiodic=.FALSE.
#  endif
# endif
      logical, parameter :: OC_decay_t_var = .FALSE. !.TRUE.
      integer :: i, j, k, itrc
      integer :: iiTemp, r1Tvar, r2Tvar, DpwTvar, DsTvar
      real(r8), parameter :: eps = 1.0E-14_r8
!
!     Coefficients for oxidation and remineralization reactions
!     Rates are from Pastor et al. (2011; BGS)
!
!     Remineralization rate for particulate organic matter (POM) [1/d]
!     Convention: rem2 is for slowly decaying POM, MaOM 
!                 rem1 is for quickly decaying POM, MiOM
      real(r8) :: rem2_2 = 0.31_r8/365.0_r8
      real(r8) :: rem1_2 = 11.0_r8/365.0_r8
!     Ratio of mol N: mol C stoichiometry 
      real(r8), parameter :: r_om2 = 6.993e-02 
      real(r8), parameter :: r_om1 = 1.408e-01 
!     Half-sat for O2 lim. in aerobic rem, [umolO2/cm3]
      real(r8), parameter :: k_o2 = 1.000e+00*1.0e-3  
!     Half-sat for O2 inhib. of anaerobic rem, [umolO2/cm3]
      real(r8), parameter :: kin_odu = 1.000e+00*1.0e-3 
!     Max oxidation rate of oxygen demand units (ODUs) [1/day]
      real(r8) :: oxodu2 = 20.0_r8
!     Half-sat for O2 in ODU oxidation, [umolO2/cm3]
      real(r8), parameter :: k_odu = 1.0e+00*1.0e-3 
!     Max nit. rate, [1/day]
      real(r8) :: nit2 = 100.0_r8 
!     Half-sat for O2 inh on nit., [umolO2/cm3]
      real(r8), parameter :: k_nit = 1.0e+01*1.0e-3 
!     Half-sat for NO3 lim of DNF, [umolNO3/cm3]
      real(r8), parameter :: k_dnf = 2.0e+01*1.0e-3 
!     O2 inhib. of DNF, [umolO2/cm3]
      real(r8), parameter :: kin_dnf = 1.000e+00*1.0e-3 
!     Half-sat for NO3 inhib. of anaerobic rem (creating ODUs), [umolNO3/cm3]
      real(r8), parameter :: kin_anox_NO3 = 1.000e+01*1.0e-3 
!     Permanent burial (a fraction; non-dimensional) of ODU
      real(r8), parameter :: PB = 0.995_r8
!
!End of Section with coefficients for oxidation and remineralization reactions
!
      real(r8) :: o2(Nbed), oc1(Nbed), oc2(Nbed)
      real(r8) :: no3(Nbed), nh4(Nbed), odu(Nbed)
      real(r8) :: doc1dt(Nbed), doc2dt(Nbed), do2dt(Nbed), dodudt(Nbed)
      real(r8) :: dnh4dt(Nbed), dno3dt(Nbed)
      real(r8) :: sConv(Nbed), pwConv(Nbed)
      real(r8) :: rxnOC1(Nbed), rxnOC2(Nbed) 
      real(r8) :: rxnOC1_o2(Nbed), rxnOC2_o2(Nbed), rxnOC_o2(Nbed)
      real(r8) :: rxnNH4_o2(Nbed), rxnODU_o2(Nbed), rxnOC_odu(Nbed)
      real(r8) :: rxnOC_no3(Nbed), rxnNH4_amx(Nbed)
      real(r8) :: o2lim, totlim, odulim, dnflim
      real(r8) :: phic(Nbed)
      real(r8) :: oxodu,nit,rem1,rem2

# include "set_bounds.h"
# if defined SEDBIO_COUP && defined SED_BIODIFF
!
!     Module built based on version of Soetaert model as coded by 
!     Dalhousie researchers, K. Fennel and Robin West, and VIMS 
!     researchers C. Harris and J. Moriarty.  At present, we
!     include two organic carbon fractions, oxygen, nitrate, ammonium, 
!     and oxygen demand units (ODUs).  We account for oxic 
!     and anerobic mineralization, and denitrification, as well as 
!     nitrification and oxidation of ODUs.  Annamox and phosphorus code
!     are neglected, and chould be added later.  Changes in porosity,
!     annamox, and inert organic matter are 
!     also neglected, and could be added later. 
!
!----------------------------------------------------------------------
!Adjust variables based on user parameters, i.e. for sensitivity tests
!----------------------------------------------------------------------
!
     rem1=rem1_2*user(1)
     rem2=rem2_2*user(2)
     nit=nit2*user(8)
     oxodu=oxodu2*user(9)
!-----------------------------------------------------------------------
! Loop through I- J- space.
!-----------------------------------------------------------------------
!
      J_LOOP : DO j=Jstr,Jend
        I_LOOP : DO i=Istr,Iend
!
!-----------------------------------------------------------------------
! Initialize rates of change to zero at the beginning of each timestep.
! Also, convert units from ROMS to Soetaert units, and calculate 
! temperature-dependant coefficients.
!-----------------------------------------------------------------------
!
          DO k=1,Nbed

!
!           -Calculate conversion factors to account for porosity and 
!            bed thickness. Units: sConv, pwConv: meters
!           -Re-set doc1dt, doc2dt to zero for each layer in the seabed
!
             sConv(k)  = (1.0_r8-bed(i,j,k,iporo))*bed(i,j,k,ithck)
             pwConv(k) = bed(i,j,k,iporo)*bed(i,j,k,ithck)
             phic(k)=bed(i,j,k,iporo)
             doc1dt(k) = 0.0_r8
             doc2dt(k) = 0.0_r8
!
! Map ROMS variables into the ones used in this routine.
! Initially, bed_tracer had units of mmol tracer /m2 
! (e.g. mmol O2/m2 of seabed, mmol N of POM/m2 of seabed, 
! mmol NO3/m2 seabed, mmol NH4/m2 seabed, or mmol ODU /m2
! Here, convert units to micromoles tracer /cm3 porewater or
! to micromoles tracer/cm3 solids
!
             DO itrc=1,NSBT
                IF (idsedbio(itrc,2).eq.iOxyg) THEN
                   o2(k) =bed_tracer(i,j,k,itrc) 
                ELSEIF (idsedbio(itrc,2).eq.iMaOM) THEN
                   oc2(k)=bed_tracer(i,j,k,itrc) 
                   oc2(k)=oc2(k)/r_om2 ! convert units to mmol Carbon OC / m2 
                ELSEIF (idsedbio(itrc,2).eq.iMiOM) THEN
                   oc1(k)=bed_tracer(i,j,k,itrc) 
                   oc1(k)=oc1(k)/r_om1 ! convert units to mmol Carbon OC / m2
                ELSEIF (idsedbio(itrc,2).eq.iNO3_) THEN
                   no3(k)=bed_tracer(i,j,k,itrc) 
                ELSEIF (idsedbio(itrc,2).eq.iNH4_) THEN
                   nh4(k)=bed_tracer(i,j,k,itrc) 
                ELSEIF (idsedbio(itrc,2).eq.iODU) THEN
                   odu(k)=bed_tracer(i,j,k,itrc) 
                ELSE
!                     Error, tracer not associated with a biological tracer
                ENDIF
             END DO
!
!            Porewater tracers:
!
             o2(k)=o2(k)/pwConv(k)   
             no3(k)=no3(k)/pwConv(k)
             nh4(k)=nh4(k)/pwConv(k)
             odu(k)=odu(k)/pwConv(k)
             o2(k) =o2(k) *1000.0_r8/(100.0_r8*100.0_r8*100.0_r8)
             no3(k) =no3(k) *1000.0_r8/(100.0_r8*100.0_r8*100.0_r8)
             nh4(k) =nh4(k) *1000.0_r8/(100.0_r8*100.0_r8*100.0_r8)
             odu(k) =odu(k) *1000.0_r8/(100.0_r8*100.0_r8*100.0_r8)
!
!            Solids:
!
             oc1(k)=oc1(k)/sConv(k)  
             oc2(k)=oc2(k)/sConv(k)  
             oc1(k)=oc1(k)*1000.0_r8/(100.0_r8*100.0_r8*100.0_r8)
             oc2(k)=oc2(k)*1000.0_r8/(100.0_r8*100.0_r8*100.0_r8)
!
!            Units are now umol / cm3 solids or porewater
!
!
!          Initialize d_tracer/dt to zero
!
             dnh4dt(k) = 0.0_r8
             dno3dt(k) = 0.0_r8
             do2dt(k)  = 0.0_r8
             dodudt(k) = 0.0_r8
!
! Prevent overshooting 
!
             o2(k)  = max( o2(k),0.0_r8)
             no3(k) = max(no3(k),0.0_r8)
             nh4(k) = max(nh4(k),0.0_r8)
             odu(k) = max(odu(k),0.0_r8)
          ENDDO

          IF(OC_decay_t_var) THEN
! This sets the coefficients to have remineralization rates 
! that vary with temperature, but for right now, neglect this
! and add it in later.
!
! T has to be between 0 and 35'C
!simplify             iiTemp = max(1,min(351,int((temp)*10+1)))
!simplify             r1Tvar  =  R1T20(iiTemp)
!simplify             r2Tvar  =  R2T20(iiTemp)
!simplify             DpwTvar =  DpwT20(iiTemp)
!simplify             DsTvar  =  DsT20(iiTemp)
             r1Tvar  =  1.0_r8
             r2Tvar  =  1.0_r8
             DpwTvar =  1.0_r8
             DsTvar  =  1.0_r8
          ELSE
             r1Tvar  =  1.0_r8
             r2Tvar  =  1.0_r8
             DpwTvar =  1.0_r8
             DsTvar  =  1.0_r8
          ENDIF
!
!-----------------------------------------------------------------------
! Compute reaction terms for biological tracers in the sediment bed
!-----------------------------------------------------------------------
!
          DO k=1,Nbed
!
! Calculate total amount of POC removed due to remineralization
! rxnOC* has units umol C/cm3 solids / day
!
             rxnOC1(k) = oc1(k)*rem1*r1Tvar
             rxnOC2(k) = oc2(k)*rem2*r2Tvar
             doc1dt(k)=doc1dt(k)-rxnOC1(k)
             doc2dt(k)=doc2dt(k)-rxnOC2(k)
             carrem(i,j,k,1)=rxnOC1(k)
             carrem(i,j,k,2)=rxnOC2(k)
!
! Calculate limitation terms that are used to partition OM remineralization
! between aerobic, nitrate and anerobic remineralization
!
             o2lim = o2(k)/(o2(k)+k_o2)
             odulim = kin_odu/(o2(k)+kin_odu)                  &
     &            *kin_anox_NO3/(no3(k)+kin_anox_NO3)
             dnflim = no3(k)/(no3(k)+k_dnf)*kin_dnf/(kin_dnf+o2(k))
             totlim = o2lim+odulim+dnflim+eps
!
! Calculate O2 consumed by aerobic respiration (O2:C = 1:1) 
! (rxnOC*_o2) in umol O2/cm3/d porewater
! and POM lost due to aerobic respiration (carrem_03; umol C/cm3 solids/d)
!
             rxnOC1_o2(k)  = rxnOC1(k)*o2lim/totlim                     &
     &            *(1.0_r8-phic(k))/phic(k)
             rxnOC2_o2(k)  = rxnOC2(k)*o2lim/totlim                     &
     &            *(1.0_r8-phic(k))/phic(k)  
             rxnOC_o2(k)  = (rxnOC1(k) + rxnOC2(k))*o2lim/totlim        &
     &            *(1.0_r8-phic(k))/phic(k)  
             carrem(i,j,k,3)=(rxnOC1(k) + rxnOC2(k))*o2lim/totlim 
!
! Calculate NO3 consumed by denitrification (NO3:C = 0.8:1)
! (rxnOC_no3; umol NO3/cm3 porewater /day
! and OM lost due to denitrification in umol C/cm3 solids/day (carrem_04)
! and in umol N/cm3 porewater /day (carrem_05)
!
             rxnOC_no3(k) = 0.8_r8*(rxnOC1(k) + rxnOC2(k))*dnflim/totlim&
     &            *(1.0_r8-phic(k))/phic(k)
             carrem(i,j,k,4)=(rxnOC1(k) + rxnOC2(k))*dnflim/totlim 
             carrem(i,j,k,5)=0.8_r8*(rxnOC1(k) + rxnOC2(k))             &
     &            *dnflim/totlim                                        &
     &            *(1.0_r8-phic(k))/phic(k)
!
! Calculate ODUs produced by anaerobic metabolism (ODU:C = 1:1).
! (rxnOC_odu; umol ODU/cm3 porewater/day) 
! Note that a fraction (=PB) of the ODUs are considered to be lost
! because they are assumed to be non-reactive. 
! Also calculate loss of OM due to anerobic metabolism 
! (carrem_06; umol C/cm3 solids/day)
!
             rxnOC_odu(k) = (rxnOC1(k) + rxnOC2(k))*odulim/totlim       &
     &            *(1.0_r8-phic(k))/phic(k)*(1.0_r8-PB)
             carrem(i,j,k,6)=(rxnOC1(k) + rxnOC2(k))*odulim/totlim 
!
! Calculate O2 consumed by nitrification (O2:NH3 = 2:1) (rxnNH4_o2)
! and by ODU oxidation (O2:ODU = 1:1) (rxnODU_o2)
! in umol O2/cm3 porewater /day
!
             rxnNH4_o2(k) = 2.0*nit*nh4(k)*o2(k)/(o2(k)+k_nit)
             rxnODU_o2(k) = oxodu*o2(k)/(o2(k)+k_odu)*odu(k)
!
! In Future, could calculate NH4 consumed by anammox. 
! Ratios are: 5NH3 + 3HNO3 -> 4N2 + 9H2O
!
!simplify             IF(anammox)THEN
!simplify             rxnNH4_amx(k) = amx*nh4(k)                              &
!simplify     &               *no3(k)/(no3(k)+k_amx)*kin_amx/(o2(k)+kin_amx)
!simplify             ELSE
                rxnNH4_amx(k) = 0.0_r8
!simplify             ENDIF

!
!-----------------------------------------------------------------------
! Update derivatives of dissolved tracers
!-----------------------------------------------------------------------
! All units are umol tracer /cm3 porewater/day
!
             do2dt(k) = do2dt(k)   - rxnOC_o2(k) - rxnNH4_o2(k) -       &
     &            rxnODU_o2(k)
             dodudt(k) = dodudt(k) + rxnOC_odu(k) - rxnODU_o2(k)
             dno3dt(k) = dno3dt(k) - rxnOC_no3(k) + rxnNH4_o2(k)/2      &
     &            - rxnNH4_amx(k)*3.0/5.0
             dnh4dt(k) = dnh4dt(k) - rxnNH4_o2(k)/2 - rxnNH4_amx(k) +   &
     &            (rxnOC1(k)*r_om1 + rxnOC2(k)*r_om2 )*(1.0_r8-phic(k)) &
     &            /phic(k)
!
! Account for ammonium adsorption by dividing dnh4dt by (1+1.3) 
! (Pastor et al., 2011 Biogeosciences; Soetaert et al., 1996)
!
             dnh4dt(k)=dnh4dt(k)/(1.0+1.3)

!
! Save oxygen consumption rates
! and convert units from umol O2 /cm3 porewater/day to
! millimoles/m2 of bed /sec
! 
 boxcon(i,j,k,1)=rxnOC1_o2(k)    !remineralization of OC1
 boxcon(i,j,k,2)=rxnOC2_o2(k)    !remineralization of OC2
 boxcon(i,j,k,3)=do2dt(k)        !total O2 consumption rate
 boxcon(i,j,k,4)=0.0_r8          !Nitrate does not consume O2
 boxcon(i,j,k,5)=rxnNH4_o2(k)    !nitrification
 boxcon(i,j,k,6)=rxnODU_o2(k)    !oxidation of ODUs
 boxcon(i,j,k,1)=boxcon(i,j,k,1)*1000.0_r8*pwConv(k)/86400.0_r8
 boxcon(i,j,k,2)=boxcon(i,j,k,2)*1000.0_r8*pwConv(k)/86400.0_r8
 boxcon(i,j,k,3)=boxcon(i,j,k,3)*1000.0_r8*pwConv(k)/86400.0_r8
 boxcon(i,j,k,4)=boxcon(i,j,k,4)*1000.0_r8*pwConv(k)/86400.0_r8
 boxcon(i,j,k,5)=boxcon(i,j,k,5)*1000.0_r8*pwConv(k)/86400.0_r8
 boxcon(i,j,k,6)=boxcon(i,j,k,6)*1000.0_r8*pwConv(k)/86400.0_r8
!
! Convert carbon remineralization rates 
! from umol /cm3 solids (or porewater)/day 
! to mmol / m2 seabed (or porewater) /s
!
 carrem(i,j,k,1)=carrem(i,j,k,1)*1000.0_r8*sConv(k)/86400.0_r8
 carrem(i,j,k,2)=carrem(i,j,k,2)*1000.0_r8*sConv(k)/86400.0_r8
 carrem(i,j,k,3)=carrem(i,j,k,3)*1000.0_r8*sConv(k)/86400.0_r8
 carrem(i,j,k,4)=carrem(i,j,k,4)*1000.0_r8*sConv(k)/86400.0_r8
 carrem(i,j,k,5)=carrem(i,j,k,5)*1000.0_r8*pwConv(k)/86400.0_r8
 carrem(i,j,k,6)=carrem(i,j,k,6)*1000.0_r8*sConv(k)/86400.0_r8
!
! Use updated derivatives to update tracer values.
! Tracers are still in umol tracer/cm3 porewater or solids
!
             o2(k) = o2(k)  + (dt(ng)/24.0_r8/3600.0_r8)*do2dt(k)
             no3(k) = no3(k)  + (dt(ng)/24.0_r8/3600.0_r8)*dno3dt(k)
             nh4(k) = nh4(k)  + (dt(ng)/24.0_r8/3600.0_r8)*dnh4dt(k)
             odu(k) = odu(k)  + (dt(ng)/24.0_r8/3600.0_r8)*dodudt(k)
             oc1(k) = oc1(k) + (dt(ng)/24.0_r8/3600.0_r8)*doc1dt(k)
             oc2(k) = oc2(k) + (dt(ng)/24.0_r8/3600.0_r8)*doc2dt(k)
!
! Map bed tracer values back into ROMS variable
! Convert units from umol tracers/cm3 porewater or solids 
! to mmol tracer/m2 of seabed
! Also, calculate o2uptake (cumulative oxygen uptake by seabed 
! in mmol O2/m2 bed (produced oxygen (+), consumed oxygen (-))
!
             DO itrc=1,NSBT
                IF (idsedbio(itrc,2).eq.iOxyg) THEN
                   o2(k)=o2(k)*1000.0_r8
                   o2(k)=o2(k)*pwConv(k)
                   o2uptake(i,j)=o2uptake(i,j)+(o2(k)-bed_tracer(i,j,k,itrc)) 
                   bed_tracer(i,j,k,itrc)=o2(k)
                ELSEIF (idsedbio(itrc,2).eq.iMaOM) THEN
                   oc2(k)=oc2(k)*1000.0_r8
                   oc2(k)=oc2(k)*sConv(k)
                   oc2(k)=oc2(k)*r_om2 ! Convert from carbon to nitrogen
                   bed_tracer(i,j,k,itrc)=oc2(k) 
                ELSEIF (idsedbio(itrc,2).eq.iMiOM) THEN
                   oc1(k)=oc1(k)*1000.0_r8
                   oc1(k)=oc1(k)*sConv(k)
                   oc1(k)=oc1(k)*r_om1 ! Convert from carbon to nitrogen
                   bed_tracer(i,j,k,itrc)=oc1(k) 
                ELSEIF (idsedbio(itrc,2).eq.iNO3_) THEN
                   no3(k)=no3(k)*1000.0_r8
                   no3(k)=no3(k)*pwConv(k)
                   bed_tracer(i,j,k,itrc)=no3(k)
                ELSEIF (idsedbio(itrc,2).eq.iNH4_) THEN
                   nh4(k)=nh4(k)*1000.0_r8
                   nh4(k)=nh4(k)*pwConv(k)
                   bed_tracer(i,j,k,itrc)=nh4(k) 
                ELSEIF (idsedbio(itrc,2).eq.iODU) THEN
                   odu(k)=odu(k)*1000.0_r8
                   odu(k)=odu(k)*pwConv(k)
                   bed_tracer(i,j,k,itrc)=odu(k) 
                ELSE
!                     Error, tracer not associated with a biological tracer
                      write (6,*) "Debugging: sedtr_reactions" 
                      write (6,*) "DBG: ", itrc, idsedbio(itrc,2)
                ENDIF
             END DO
          END DO   
       END DO I_LOOP
      END DO J_LOOP
!
!-----------------------------------------------------------------------
!  Apply periodic or gradient boundary conditions to property arrays.
!-----------------------------------------------------------------------
!
      DO itrc=1,NSBT
        CALL bc_r3d_tile (ng, tile,                                     &
     &                    LBi, UBi, LBj, UBj, 1, Nbed,                  &
     &                    bed_tracer(:,:,:,itrc))
        CALL bc_r3d_tile (ng, tile,                                     &
     &                    LBi, UBi, LBj, UBj, 1, Nbed,                  &
     &                    boxcon(:,:,:,itrc))
        CALL bc_r3d_tile (ng, tile,                                     &
     &                    LBi, UBi, LBj, UBj, 1, Nbed,                  &
     &                    carrem(:,:,:,itrc))
      END DO
       CALL bc_r2d_tile (ng, tile,                                     &
     &                    LBi, UBi, LBj, UBj,                          &
     &                    o2uptake(:,:))
# ifdef DISTRIBUTE
      CALL mp_exchange4d (ng, tile, iNLM, 2,                            &
     &                    LBi, UBi, LBj, UBj, 1, Nbed, 1, NSBT,         &
     &                    NghostPoints, EWperiodic, NSperiodic,         &
     &                    bed_tracer)
      CALL mp_exchange4d (ng, tile, iNLM, 2,                            &
     &                    LBi, UBi, LBj, UBj, 1, Nbed, 1, NSBT,         &
     &                    NghostPoints, EWperiodic, NSperiodic,         &
     &                    boxcon)
      CALL mp_exchange4d (ng, tile, iNLM, 2,                            &
     &                    LBi, UBi, LBj, UBj, 1, Nbed, 1, NSBT,         &
     &                    NghostPoints, EWperiodic, NSperiodic,         &
     &                    carrem)
      CALL mp_exchange2d (ng, tile, iNLM, 2,                            &
     &                    LBi, UBi, LBj, UBj,                           &
     &                    NghostPoints, EWperiodic, NSperiodic,         &
     &                    o2uptake)
# endif
# endif
      RETURN
      END SUBROUTINE sedtr_reactions_tile
#endif
      END MODULE sedtr_reactions_mod
