#include "cppdefs.h"

      MODULE sedtr_biodiff_mod

#if defined NONLINEAR && defined SEDIMENT && defined SEDBIO_COUP
!
!svn $Id: sed_bed_mod.F 2011 2009-12-20 17:34:23Z arango $
!==================================================== C. R. Sherwood ===
!  Copyright (c) 2002-2010 The ROMS/TOMS Group      Hernan G. Arango   !
!    Licensed under a MIT/X style license                              !
!    See License_ROMS.txt                                              !
!=======================================================================
!                                                                      !
!     This routine computes mixing (biodiffusion) of tracers used in   ! 
!     the sediment bed by the sedbio coupling routines.                !
!     ckharris@vims.edu, moriarty@vims.edu, 2017                       !
!                                                                      !
!=======================================================================
!
      implicit none

      PRIVATE
      PUBLIC  :: sedtr_biodiff

      CONTAINS
!
!***********************************************************************
      SUBROUTINE sedtr_biodiff (ng, tile)
!***********************************************************************
!
      USE mod_param
      USE mod_forces
      USE mod_grid
      USE mod_ocean
      USE mod_sedbed
      USE mod_stepping
# ifdef BBL_MODEL
      USE mod_bbl
# endif
!
!  Imported variable declarations.
!
      integer, intent(in) :: ng, tile
!
!  Local variable declarations.
!
# include "tile.h"
!
# ifdef PROFILE
      CALL wclock_on (ng, iNLM, 16)
# endif
# if defined SEDBIO_COUP && defined SED_BIODIFF
      CALL sedtr_biodiff_tile (ng, tile,                                &
     &                   LBi, UBi, LBj, UBj,                            &
     &                   IminS, ImaxS, JminS, JmaxS,                    &
     &                   nstp(ng), nnew(ng),                            &
# ifdef WET_DRY
     &                   GRID(ng) % rmask_wet,                          &
# endif
     &                   OCEAN(ng) % t,                                 &
# if defined SED_MORPH
     &                   SEDBED(ng) % bed_thick,                        &
# endif
     &                   SEDBED(ng) % bed,                              &
     &                   SEDBED(ng) % bed_tracer)
# endif
# ifdef PROFILE
      CALL wclock_off (ng, iNLM, 16)
# endif
      RETURN
      END SUBROUTINE sedtr_biodiff
!
!***********************************************************************
      SUBROUTINE sedtr_biodiff_tile (ng, tile,                          &
     &                         LBi, UBi, LBj, UBj,                      &
     &                         IminS, ImaxS, JminS, JmaxS,              &
     &                         nstp, nnew,                              &
# ifdef WET_DRY
     &                         rmask_wet,                               &
# endif
     &                         t,                                       &
# if defined SED_MORPH
     &                         bed_thick,                               &
# endif
# if defined SEDBIO_COUP && defined SED_BIODIFF
     &                         bed,                                     &
     &                         bed_tracer)
# else
     &                         bed)
# endif
!***********************************************************************
!
      USE mod_param
      USE mod_scalars
      USE mod_sediment
!
      USE bc_3d_mod, ONLY : bc_r3d_tile
# if defined EW_PERIODIC || defined NS_PERIODIC
      USE exchange_2d_mod, ONLY : exchange_r2d_tile
# endif
# ifdef DISTRIBUTE
      USE mp_exchange_mod, ONLY : mp_exchange3d, mp_exchange4d
# endif
!
!  Imported variable declarations.
!
      integer, intent(in) :: ng, tile
      integer, intent(in) :: LBi, UBi, LBj, UBj
      integer, intent(in) :: IminS, ImaxS, JminS, JmaxS
      integer, intent(in) :: nstp, nnew
!
# ifdef ASSUMED_SHAPE
#  ifdef WET_DRY
      real(r8), intent(in) :: rmask_wet(LBi:,LBj:)
#  endif
#  if defined SED_MORPH
      real(r8), intent(inout):: bed_thick(LBi:,LBj:,:)
#  endif
      real(r8), intent(inout) :: t(LBi:,LBj:,:,:,:)
      real(r8), intent(inout) :: bed(LBi:,LBj:,:,:)
# if defined SEDBIO_COUP && defined SED_BIODIFF
      real(r8), intent(inout) :: bed_tracer(LBi:,LBj:,:,:)
# endif
# else
#  ifdef WET_DRY
      real(r8), intent(in) :: rmask_wet(LBi:UBi,LBj:UBj)
#  endif
#  if defined SED_MORPH
      real(r8), intent(inout):: bed_thick(LBi:UBi,LBj:UBj,2)
#  endif
      real(r8), intent(inout) :: t(LBi:UBi,LBj:UBj,N(ng),3,NT(ng))
      real(r8), intent(inout) :: bed(LBi:UBi,LBj:UBj,Nbed,MBEDP)
# if defined SEDBIO_COUP && defined SED_BIODIFF
      real(r8), intent(inout) :: bed_tracer(LBi:UBi,LBj:UBj,Nbed,NSBT)
# endif

# endif
!
!  Local variable declarations.
!
# ifdef DISTRIBUTE
#  ifdef EW_PERIODIC
      logical :: EWperiodic=.TRUE.
#  else
      logical :: EWperiodic=.FALSE.
#  endif
#  ifdef NS_PERIODIC
      logical :: NSperiodic=.TRUE.
#  else
      logical :: NSperiodic=.FALSE.
#  endif
# endif
      integer :: i, j, k, itrc
      real(r8), parameter :: eps = 1.0E-14_r8
      real(r8), parameter :: cmy2ms = 3.1688765E-12_r8 
!                            multiply cm2/yr by this to get m2/s

      real(r8) :: cff, cff1, cff2, cff3

      real(r8), dimension(IminS:ImaxS,NST) :: dep_mass

      integer :: iu,il,lp,ii
      real(r8) :: rtemp, zs, zm, zp, Zbmx
      real(r8), dimension(Nbed) :: zb
      real(r8), dimension(Nbed) :: zc
      real(r8), dimension(Nbed) :: dzui
      real(r8), dimension(Nbed) :: dzli
      real(r8), dimension(Nbed) :: dzmi
# if defined SEDBIO_COUP && defined SED_BIODIFF
      real(r8), dimension(Nbed,NSBT) :: Db
      real(r8), dimension(Nbed,NSBT) :: Dc
# endif
      real(r8), dimension(Nbed) :: a
      real(r8), dimension(Nbed) :: d
      real(r8), dimension(Nbed) :: b
      real(r8), dimension(Nbed) :: cc
      real(r8), dimension(Nbed) :: dd

# include "set_bounds.h"
# if defined SEDBIO_COUP && defined SED_BIODIFF
!     Depths for biodiffusion based on Pastor et al. (2011; Biogeosci.)
      Zbmx=0.13_r8
!
!-----------------------------------------------------------------------
! Compute mixing of biological tracers in the sediment bed
!-----------------------------------------------------------------------
!
      J_LOOP : DO j=Jstr,Jend
        I_LOOP : DO i=Istr,Iend

!       Set up the vertical grid
        IF (Nbed.GT.2) THEN
!           Compute cumulative depth (depth at bottom of layers) 
!           and depths of bed centers  
            zb(1)=bed(i,j,1,ithck)
            zc(1)=0.5*(bed(i,j,1,ithck))
            DO k=2,Nbed
              zb(k)=zb(k-1)+bed(i,j,k,ithck)
              zc(k)=zb(k-1)+0.5_r8*bed(i,j,k,ithck)
            END DO
!  Calculate finite differences
            dzui(1)=1.0_r8/(zc(2)-zc(1))
            dzli(1)=1.E35_r8       ! should not be needed
            dzmi(1)=1.0_r8/bed(i,j,1,ithck)
            DO k=2,Nbed-1
              dzui(k)=1.0_r8/(zc(k+1)-zc(k))
              dzli(k)=1.0_r8/(zc(k)-zc(k-1))
              !dzmi(k)=1.0_r8/(zb(k+1)-zb(k))
              ! equivalent:
              dzmi(k)=1.0/bed(i,j,k,ithck)
            ENDDO
            dzui(Nbed)=1.0E35_r8 ! should not be needed
            dzli(Nbed)=1.0_r8/(zc(Nbed)-zc(Nbed-1))
            dzmi(Nbed)=1.0_r8/bed(i,j,Nbed,ithck)
!
!           LOOP on TRACERS
!
            DO itrc=1,NSBT
!              Set mixing coefficient profile:
!              -Diffusion coefficient Db may change from Dbmx to Dbmm
!               with depth into the seabed (e.g. to mimic the 
!               reduction in bioturbation beneath surface sediments
!              -Note that each tracer may have its own Dbmx, Dbmm
               DO k=1,Nbed
                  IF (zb(k).LE.Zbmx) THEN
                     Db(k,itrc)=Dbmx(NST+itrc,1)
                  ELSEIF (zb(k).GE.2.0_r8*Zbmx) THEN
                     Db(k,itrc)=Dbmm(1,1)
                  ELSE
                     Db(k,itrc)=Dbmm(1,1)+                              &
     &                (2.0_r8*Zbmx-zb(k))*(Dbmx(itrc,1)-Dbmm(1,1))/Zbmx
                  ENDIF
               ENDDO
!
!  Tridiagonal terms
               DO k=2,Nbed-1
                  b(k)= -dt(ng)*dzmi(k)*Db(k-1,itrc)*dzli(k)
                  d(k)=1.0_r8+dt(ng)*dzmi(k)*                           &
     &                 ( Db(k-1,itrc)*dzli(k)+Db(k,itrc)*dzui(k) )
                  a(k)= -dt(ng)*dzmi(k)*Db(k,itrc)*dzui(k)
               ENDDO
!  No-flux boundary conditions at top and bottom
               b(1)= 999.9_r8  ! should not be needed
               d(1)= 1.0_r8 +dt(ng)*dzmi(1)*Db(1,itrc)*dzui(1)
               a(1)= -dt(ng)*dzmi(1)*Db(1,itrc)*dzui(1)
               b(Nbed)=  -dt(ng)*dzmi(Nbed)*Db(Nbed-1,itrc)*dzli(Nbed)
               d(Nbed)= 1.0_r8 +                                        &
     &                  dt(ng)*dzmi(Nbed)*Db(Nbed-1,itrc)*dzli(Nbed) 
               a(Nbed)= 999.9_r8 ! should not be needed
!                                                                       !
!   Calculate mixing for each tracer                                    !
!   ...make working copies of tracer CONCENTRATION (bed_tracer stores   !
!   mass-integral of tracer in bed layer; convert to tracer units per m3!
!   and account for porosity (differently for particulate and dissolved !
!   tracers).                                                           !
              DO k=1,Nbed
                IF (itrc.le.NSBP) THEN
!               Tracer is a particulate
                   cc(k) = bed_tracer(i,j,k,itrc)/bed(i,j,k,ithck)/     &
     &                     (1.0_r8 - bed(i,j,k,iporo)) 
                ELSE
!               Tracer is dissolved in pore water
                   cc(k) = bed_tracer(i,j,k,itrc)/bed(i,j,k,ithck)/     &
     &                     bed(i,j,k,iporo) 
                ENDIF
                dd(k)= d(k)
              ENDDO
!   Solve a tridiagonal system of equations using Thomas' algorithm
!   Anderson, Tannehill, and Pletcher (1984) pp. 549-550
!   ...establish upper triangular matrix
              il = 1
              iu = Nbed
              lp = il+1
              DO k = lp,iu
                rtemp = b(k)/dd(k-1)
                dd(k)= dd(k)-rtemp*a(k-1);
                cc(k)= cc(k)-rtemp*cc(k-1);
              ENDDO
!   ...back substitution
              cc(iu) = cc(iu)/dd(iu)
              DO k  = lp,iu
                ii = iu-k+il;
                cc(ii) = (cc(ii)-a(ii)*cc(ii+1))/dd(ii);
              ENDDO
!   ...solution stored in cc; copy out but convert to depth-integral of !
!      tracer within that bed layer                                     !
              DO k = 1,Nbed
                IF (itrc.le.NSBP) THEN
!               Tracer is a particulate
                   bed_tracer(i,j,k,itrc)=cc(k)*bed(i,j,k,ithck)*       &
     &                                    (1.0_r8 - bed(i,j,k,iporo))
                ELSE
!               Tracer is dissolved in pore water
                   bed_tracer(i,j,k,itrc)=cc(k)*bed(i,j,k,ithck)*       &
     &                                    bed(i,j,k,iporo)
                ENDIF
              ENDDO           
            ENDDO ! Tracer loop
          END IF !NBED.GT.2
        END DO I_LOOP
      END DO J_LOOP
!
!-----------------------------------------------------------------------
!  Apply periodic or gradient boundary conditions to property arrays.
!-----------------------------------------------------------------------
!
      DO itrc=1,NSBT
        CALL bc_r3d_tile (ng, tile,                                     &
     &                    LBi, UBi, LBj, UBj, 1, Nbed,                  &
     &                    bed_tracer(:,:,:,itrc))
      END DO
# ifdef DISTRIBUTE
      CALL mp_exchange4d (ng, tile, iNLM, 2,                            &
     &                    LBi, UBi, LBj, UBj, 1, Nbed, 1, NSBT,         &
     &                    NghostPoints, EWperiodic, NSperiodic,         &
     &                    bed_tracer)
# endif
# endif             
      RETURN
      END SUBROUTINE sedtr_biodiff_tile
#endif
      END MODULE sedtr_biodiff_mod
