#include "cppdefs.h"

      MODULE sed_bed_mod

#if defined NONLINEAR && defined SEDIMENT && !defined COHESIVE_BED
!
!svn $Id: sed_bed.F 2256 2012-02-29 19:35:16Z ckharris $
!==================================================== John C. Warner ===
!  Copyright (c) 2002-2010 The ROMS/TOMS Group      Hernan G. Arango   !
!    Licensed under a MIT/X style license                              !
!    See License_ROMS.txt                                              !
!=======================================================================
!                                                                      !
!  This routine computes sediment bed layer stratigraphy.              !
!                                                                      !
!  Warner, J.C., C.R. Sherwood, R.P. Signell, C.K. Harris, and H.G.    !
!    Arango, 2008:  Development of a three-dimensional,  regional,     !
!    coupled wave, current, and sediment-transport model, Computers    !
!    & Geosciences, 34, 1284-1306.                                     !
!                                                                      !
! This version of sed_bed.F was modified to account for fine resolution!
! near the seabed-water interface and coarser resolution deeper in the !
! seabed (moriarty@vims.edu, 2017)                                     !
!                                                                      !
!=======================================================================
!
      implicit none

      PRIVATE
      PUBLIC  :: sed_bed

      CONTAINS
!
!***********************************************************************
      SUBROUTINE sed_bed (ng, tile)
!***********************************************************************
!
      USE mod_param
      USE mod_forces
      USE mod_grid
      USE mod_ocean
      USE mod_sedbed
      USE mod_stepping
# ifdef BBL_MODEL
      USE mod_bbl
# endif
!
!  Imported variable declarations.
!
      integer, intent(in) :: ng, tile
!
!  Local variable declarations.
!
# include "tile.h"
!
# ifdef PROFILE
      CALL wclock_on (ng, iNLM, 16)
# endif
      CALL sed_bed_tile (ng, tile,                                      &
     &                   LBi, UBi, LBj, UBj,                            &
     &                   IminS, ImaxS, JminS, JmaxS,                    &
     &                   nstp(ng), nnew(ng),                            &
# ifdef WET_DRY
     &                   GRID(ng) % rmask_wet,                          &
# endif
# ifdef SEDBIO_COUP
     &                   GRID(ng) % Hz,                                 &
# endif
# ifdef BBL_MODEL
     &                   BBL(ng) % bustrc,                              &
     &                   BBL(ng) % bvstrc,                              &
     &                   BBL(ng) % bustrw,                              &
     &                   BBL(ng) % bvstrw,                              &
     &                   BBL(ng) % bustrcwmax,                          &
     &                   BBL(ng) % bvstrcwmax,                          &
# else
     &                   FORCES(ng) % bustr,                            &
     &                   FORCES(ng) % bvstr,                            &
# endif
     &                   OCEAN(ng) % t,                                 &
# ifdef SUSPLOAD
     &                   SEDBED(ng) % ero_flux,                         &
     &                   SEDBED(ng) % settling_flux,                    &
# endif
# if defined SED_MORPH
     &                   SEDBED(ng) % bed_thick,                        &
# endif
     &                   SEDBED(ng) % bed,                              &
     &                   SEDBED(ng) % bed_frac,                         &
     &                   SEDBED(ng) % bed_mass,                         &
# ifdef SEDBIO_COUP
     &                   SEDBED(ng) % bed_tracer,                       &
     &                   SEDBED(ng) % sdflux,                           &
     &                   SEDBED(ng) % saflux,                           &
# endif
     &                   SEDBED(ng) % bottom)
# ifdef PROFILE
      CALL wclock_off (ng, iNLM, 16)
# endif
      RETURN
      END SUBROUTINE sed_bed
!
!***********************************************************************
      SUBROUTINE sed_bed_tile (ng, tile,                                &
     &                         LBi, UBi, LBj, UBj,                      &
     &                         IminS, ImaxS, JminS, JmaxS,              &
     &                         nstp, nnew,                              &
# ifdef WET_DRY
     &                         rmask_wet,                               &
# endif
# ifdef SEDBIO_COUP
     &                         Hz,                                      &
# endif
# ifdef BBL_MODEL
     &                         bustrc, bvstrc,                          &
     &                         bustrw, bvstrw,                          &
     &                         bustrcwmax, bvstrcwmax,                  &
# else
     &                         bustr, bvstr,                            &
# endif
     &                         t,                                       &
# ifdef SUSPLOAD
     &                         ero_flux, settling_flux,                 &
# endif
# if defined SED_MORPH
     &                         bed_thick,                               &
# endif
     &                         bed, bed_frac, bed_mass,                 &
# if defined SEDBIO_COUP
     &                         bed_tracer, sdflux, saflux,              &
# endif
     &                         bottom)
!***********************************************************************
!
      USE mod_param
      USE mod_scalars
      USE mod_sediment
!
      USE bc_3d_mod, ONLY : bc_r3d_tile
# if defined EW_PERIODIC || defined NS_PERIODIC
      USE exchange_2d_mod, ONLY : exchange_r2d_tile
# endif
# ifdef DISTRIBUTE
      USE mp_exchange_mod, ONLY : mp_exchange3d, mp_exchange4d
# endif
!
!  Imported variable declarations.
!
      integer, intent(in) :: ng, tile
      integer, intent(in) :: LBi, UBi, LBj, UBj
      integer, intent(in) :: IminS, ImaxS, JminS, JmaxS
      integer, intent(in) :: nstp, nnew
!
# ifdef ASSUMED_SHAPE
#  ifdef WET_DRY
      real(r8), intent(in) :: rmask_wet(LBi:,LBj:)
#  endif
#  ifdef SEDBIO_COUP
      real(r8), intent(in) :: Hz(LBi:,LBj:,:)
#  endif
#  ifdef BBL_MODEL
      real(r8), intent(in) :: bustrc(LBi:,LBj:)
      real(r8), intent(in) :: bvstrc(LBi:,LBj:)
      real(r8), intent(in) :: bustrw(LBi:,LBj:)
      real(r8), intent(in) :: bvstrw(LBi:,LBj:)
      real(r8), intent(in) :: bustrcwmax(LBi:,LBj:)
      real(r8), intent(in) :: bvstrcwmax(LBi:,LBj:)
#  else
      real(r8), intent(in) :: bustr(LBi:,LBj:)
      real(r8), intent(in) :: bvstr(LBi:,LBj:)
#  endif
#  if defined SED_MORPH
      real(r8), intent(inout):: bed_thick(LBi:,LBj:,:)
#  endif
      real(r8), intent(inout) :: t(LBi:,LBj:,:,:,:)
#  ifdef SUSPLOAD
      real(r8), intent(inout) :: ero_flux(LBi:,LBj:,:)
      real(r8), intent(inout) :: settling_flux(LBi:,LBj:,:)
#  endif
      real(r8), intent(inout) :: bed(LBi:,LBj:,:,:)
      real(r8), intent(inout) :: bed_frac(LBi:,LBj:,:,:)
      real(r8), intent(inout) :: bed_mass(LBi:,LBj:,:,:,:)
#  ifdef SEDBIO_COUP
      real(r8), intent(inout) :: bed_tracer(LBi:,LBj:,:,:)
      real(r8), intent(inout) :: sdflux(LBi:,LBj:,:)
      real(r8), intent(inout) :: saflux(LBi:,LBj:,:)
#  endif
      real(r8), intent(inout) :: bottom(LBi:,LBj:,:)
# else
#  ifdef WET_DRY
      real(r8), intent(in) :: rmask_wet(LBi:UBi,LBj:UBj)
#  endif
#  ifdef SEDBIO_COUP
      real(r8), intent(in) :: Hz(LBi:UBi,LBj:UBj,UBk)
#  endif
#  ifdef BBL_MODEL
      real(r8), intent(in) :: bustrc(LBi:UBi,LBj:UBj)
      real(r8), intent(in) :: bvstrc(LBi:UBi,LBj:UBj)
      real(r8), intent(in) :: bustrw(LBi:UBi,LBj:UBj)
      real(r8), intent(in) :: bvstrw(LBi:UBi,LBj:UBj)
      real(r8), intent(in) :: bustrcwmax(LBi:UBi,LBj:UBj)
      real(r8), intent(in) :: bvstrcwmax(LBi:UBi,LBj:UBj)
#  else
      real(r8), intent(in) :: bustr(LBi:UBi,LBj:UBj)
      real(r8), intent(in) :: bvstr(LBi:UBi,LBj:UBj)
#  endif
#  if defined SED_MORPH
      real(r8), intent(inout):: bed_thick(LBi:UBi,LBj:UBj,2)
#  endif
      real(r8), intent(inout) :: t(LBi:UBi,LBj:UBj,N(ng),3,NT(ng))
#  ifdef SUSPLOAD
#  ifdef SEDBIO_COUP
      real(r8), intent(inout) :: bed_tracer(LBi:UBi,LBj:UBj,Nbed,NSBT)
      real(r8), intent(inout) :: sdflux(LBi:UBi,LBj:UBj,NSBT)
      real(r8), intent(inout) :: saflux(LBi:UBi,LBj:UBj,NSBT)
      real(r8), intent(inout) :: ero_flux(LBi:UBi,LBj:UBj,NST+NSBT) 
      real(r8), intent(inout) :: settling_flux(LBi:UBi,LBj:UBj,NST+NSBT) 
#  else
      real(r8), intent(inout) :: ero_flux(LBi:UBi,LBj:UBj,NST)
      real(r8), intent(inout) :: settling_flux(LBi:UBi,LBj:UBj,NST)
#  endif
#  endif
      real(r8), intent(inout) :: bed(LBi:UBi,LBj:UBj,Nbed,MBEDP)
      real(r8), intent(inout) :: bed_frac(LBi:UBi,LBj:UBj,Nbed,NST)
      real(r8), intent(inout) :: bed_mass(LBi:UBi,LBj:UBj,Nbed,1:2,NST)
      real(r8), intent(inout) :: bottom(LBi:UBi,LBj:UBj,MBOTP)
# endif
!
!  Local variable declarations.
!
# ifdef DISTRIBUTE
#  ifdef EW_PERIODIC
      logical :: EWperiodic=.TRUE.
#  else
      logical :: EWperiodic=.FALSE.
#  endif
#  ifdef NS_PERIODIC
      logical :: NSperiodic=.TRUE.
#  else
      logical :: NSperiodic=.FALSE.
#  endif
# endif
      integer :: Ksed, i, ised, j, k, ks
      integer :: bnew

      real(r8), parameter :: eps = 1.0E-14_r8

      real(r8) :: cff, cff1, cff2, cff3
      real(r8) :: thck_avail, thck_to_add

      real(r8), dimension(IminS:ImaxS,NST) :: dep_mass
# ifdef SEDBIO_COUP
      integer :: indxs,iised
      real(r8), dimension(IminS:ImaxS,NSBP) :: dep_mass_om
      real(r8), dimension(IminS:ImaxS) :: ndep_thck_tot
!     Coefficient for estimating diffusion across 
!     seabed-water interface:  Units: m2/s
!     1.08e-9 is value from Rhone delta, Toussaint et al., 2014.
      real(r8), parameter :: D_sw = 1.08E-9_r8
# endif
      real(r8), dimension(IminS:ImaxS,JminS:JmaxS) :: tau_w
!
!Paramters for making three layer bed scheme with
!     a repository, N1-1 coarse layers of thickness newlayer_thick2,
!     and N0 fine layers, including a first layer with 
!     thickness firstlayer_thick 
!
      integer,parameter :: N0=20
      integer :: N1,indnl,kk
      real(r8), parameter :: newlayer_thick2 = 0.01_r8
      real(r8), parameter :: firstlayer_thick = 0.0001_r8

# include "set_bounds.h"

# ifdef BEDLOAD
      bnew=nnew
# else
      bnew=nstp
# endif
!
!-----------------------------------------------------------------------
!  Set Number of bottom, thick layers
!-----------------------------------------------------------------------
!
     N1=MAX(Nbed-N0,0)
!
!-----------------------------------------------------------------------
! Compute sediment bed layer stratigraphy.
!-----------------------------------------------------------------------
!
# if defined BEDLOAD_MPM || defined SUSPLOAD
#  ifdef BBL_MODEL
      DO j=Jstr-1,Jend+1
        DO i=Istr-1,Iend+1
          tau_w(i,j)=SQRT(bustrcwmax(i,j)*bustrcwmax(i,j)+              &
     &                    bvstrcwmax(i,j)*bvstrcwmax(i,j))
#   ifdef WET_DRY
          tau_w(i,j)=tau_w(i,j)*rmask_wet(i,j)
#   endif
        END DO
      END DO
#  else
#   ifdef EW_PERIODIC
#    define I_RANGE Istr-1,Iend+1
#   else
#    define I_RANGE MAX(Istr-1,1),MIN(Iend+1,Lm(ng))
#   endif
#   ifdef NS_PERIODIC
#    define J_RANGE Jstr-1,Jend+1
#   else
#    define J_RANGE MAX(Jstr-1,1),MIN(Jend+1,Mm(ng))
#   endif
      DO i=I_RANGE
        DO j=J_RANGE
          tau_w(i,j)=0.5_r8*SQRT((bustr(i,j)+bustr(i+1,j))*             &
     &                           (bustr(i,j)+bustr(i+1,j))+             &
     &                           (bvstr(i,j)+bvstr(i,j+1))*             &
     &                           (bvstr(i,j)+bvstr(i,j+1)))
#   ifdef WET_DRY
          tau_w(i,j)=tau_w(i,j)*rmask_wet(i,j)
#   endif
        END DO
      END DO
#   undef I_RANGE
#   undef J_RANGE
#  endif
# endif
!
!-----------------------------------------------------------------------
!  Update bed properties according to ero_flux and dep_flux.
!-----------------------------------------------------------------------
!
# ifdef SUSPLOAD
!
! J_LOOP adds or subtracts material from the surface seabed layer
! based on fluxes calculated in sed_flux.F. It may also add or 
! subtract layer(s) if the surface layer becomes too thick. 
!
      J_LOOP : DO j=Jstr,Jend
# ifdef SEDBIO_COUP
!
! Define active layer thickness to be small if accounting for 
! biogeochemistry (i.e. if SEDBIO_COUP is defined)
!
          bottom(i,j,iactv)=MIN(firstlayer_thick, bottom(i,j,iactv))
!
! Define SED_LOOP to loop through tracers, in addition to sediment
! and define ndep_thck_tot, the thickness of new deposition or erosion.
!
        DO i=Istr,Iend
           ndep_thck_tot(i)=0.0_r8
        END DO
        SED_LOOP: DO ised=1,NST+NSBT
            IF (ised.gt.NST) THEN
               indxs=idsedbio(ised-NST,1)-2
            END IF
# endif

# ifndef SEDBIO_COUP
        SED_LOOP: DO ised=1,NST
# endif
!
!  SED_LOOP:
!  The deposition and resuspension of sediment on the bottom "bed"
!  is due to precepitation flux FC(:,0), already computed, and the
!  resuspension (erosion, hence called ero_flux). The resuspension is
!  applied to the bottom-most grid box value qc(:,1) so the total mass
!  is conserved. Restrict "ero_flux" so that "bed" cannot go negative
!  after both fluxes are applied.
!
          DO i=Istr,Iend
# ifdef SEDBIO_COUP
            IF ((ised.gt.NST).AND.(indxs.gt.0)) THEN
            dep_mass_om(i,indxs)=0.0_r8
            ELSE IF (ised.le.NST) THEN
# endif
            dep_mass(i,ised)=0.0_r8
# ifdef SEDBIO_COUP
            ENDIF
# endif

#  ifdef SED_MORPH
!
! Apply morphology factor.
!
            ero_flux(i,j,ised)=ero_flux(i,j,ised)*morph_fac(ised,ng)
            settling_flux(i,j,ised)=settling_flux(i,j,ised)*            &
     &                              morph_fac(ised,ng)
#  endif
            IF ((ero_flux(i,j,ised)-settling_flux(i,j,ised)).lt.        &
     &           0.0_r8) THEN
!
!  If, for a given sediment class, there is net deposition 
!  and if the surface layer is 'too old' or 'too thick'
!  then store ready-to-be-deposited material in a temporary array, 
!  dep_mass.
!
              IF ((time(ng).gt.(bed(i,j,1,iaged)+1.1_r8*dt(ng))).and.   &
     &            (bed(i,j,1,ithck).gt.                                 &
     &            (firstlayer_thick+bottom(i,j,iactv)))) THEN
# ifdef SEDBIO_COUP
                IF ((ised.gt.NST).AND.(indxs.gt.0)) THEN
                    dep_mass_om(i,indxs)=settling_flux(i,j,ised)-       &
     &                           ero_flux(i,j,ised)
                ELSE IF (ised.le.NST) THEN
# endif
                dep_mass(i,ised)=settling_flux(i,j,ised)-               &
     &                           ero_flux(i,j,ised)
# ifdef SEDBIO_COUP
                END IF
# endif
              END IF
            END IF
!
!  Update bed mass arrays by adding or subtracting new deposition or
!  erosion, respectively.
!
# ifdef SEDBIO_COUP
!  Also calculate ndep_thck_tot,the thickness of new deposition/erosion
!  and saflux, the amount of POM that is deposited/eroded.
!
            IF ((ised.gt.NST).AND.(indxs.gt.0)) THEN
               saflux(i,j,indxs)=saflux(i,j,indxs)+                     &
     &                      MAX(-bed_tracer(i,j,1,indxs),               &
     &                      -(ero_flux(i,j,ised)-                       &
     &                      settling_flux(i,j,ised)))
               bed_tracer(i,j,1,indxs)=MAX(bed_tracer(i,j,1,indxs)      &
     &                      -(ero_flux(i,j,ised)-                       &
     &                      settling_flux(i,j,ised)),0.0_r8)
            ELSE IF (ised.le.NST) THEN
# endif
            bed_mass(i,j,1,nnew,ised)=MAX(bed_mass(i,j,1,bnew,ised)-    &
     &                                   (ero_flux(i,j,ised)-           &
     &                                    settling_flux(i,j,ised)),     &
     &                                    0.0_r8)
# ifdef SEDBIO_COUP
            ndep_thck_tot(i)=ndep_thck_tot(i)+(bed_mass(i,j,1,nnew,ised)&
     &           - bed_mass(i,j,1,bnew,ised))/                          &
     &           (Srho(ised,ng)*(1.0_r8-bed(i,j,1,iporo)))
# endif
              DO k=2,Nbed
                bed_mass(i,j,k,nnew,ised)=bed_mass(i,j,k,nstp,ised)
              END DO
# ifdef SEDBIO_COUP
            END IF
# endif
          END DO
        END DO SED_LOOP
# ifdef SEDBIO_COUP
!
!       Update changed dissolved bed tracers due to erosion/deposition.  
!       Note that units of dissolved tracers are mmol (Tracer) /m2 
!       for both bed_tracer and t(...,nnew,...)
!       Biogeochemical tracers are accounted for in this loop if they 
!       have an equivalent water column tracer 
!       (i.e. if (idsedbio(NSBP+ised,2).ne.0) )
!
        DO i=Istr,Iend
           IF (ndep_thck_tot(i).gt.0) THEN
!
!          If the net change in seabed thickness is deposition, then 
!          assign dissolved tracers within the new porewater to have
!          the same concentration as overlying water, (so that 
!          the concentrations are in equilibrium during deposition)
!
              DO ised=1,NSBD
                 IF (idsedbio(NSBP+ised,2).ne.0) THEN
!                cff1 is equilibrium concentration in mmol T/m3 water
                  cff1=(bed_tracer(i,j,1,NSBP+ised)+                    &
     &                 t(i,j,1,nnew,idsedbio(NSBP+ised,2))) /           &
     &                 (Hz(i,j,1)+ndep_thck_tot(i)*bed(i,j,1,iporo))
                  saflux(i,j,NSBP+ised)=saflux(i,j,NSBP+ised)           &
     &                +cff1*ndep_thck_tot(i)*bed(i,j,1,iporo)
                  bed_tracer(i,j,1,NSBP+ised)=                          &
     &                bed_tracer(i,j,1,NSBP+ised)+                      &
     &                cff1*ndep_thck_tot(i)*bed(i,j,1,iporo)
                  t(i,j,1,nnew,idsedbio(NSBP+ised,2))=                  &
     &                t(i,j,1,nnew,idsedbio(NSBP+ised,2))-              &
     &                cff1*ndep_thck_tot(i)*bed(i,j,1,iporo)
                 ENDIF
              END DO
           ELSE
!
!          If net change in seabed thickness is erosion, then 
!          remove the inventory of dissolved tracer that was in the 
!          porewater of the eroded layer.
!          Note that ndep_thck_tot is negative!
!
              DO ised=1,NSBD
                 IF (bed(i,j,1,ithck).lt.eps) THEN
                   saflux(i,j,NSBP+ised)=saflux(i,j,NSBP+ised)-         &
     &                bed_tracer(i,j,1,NSBP+ised)
                   t(i,j,1,nnew,idsedbio(NSBP+ised,2))=                 &
     &                t(i,j,1,nnew,idsedbio(NSBP+ised,2))+              &
     &                bed_tracer(i,j,1,NSBP+ised)
                   bed_tracer(i,j,1,NSBP+ised)=0.0_r8
                 ELSE
                   cff1=bed_tracer(i,j,1,NSBP+ised)*ndep_thck_tot(i)/   &
     &                   bed(i,j,1,ithck)
                   saflux(i,j,NSBP+ised)=saflux(i,j,NSBP+ised)+cff1
                   bed_tracer(i,j,1,NSBP+ised)=                         &
     &                   bed_tracer(i,j,1,NSBP+ised)+cff1
                   t(i,j,1,nnew,idsedbio(NSBP+ised,2))=                 &
     &                   t(i,j,1,nnew,idsedbio(NSBP+ised,2))-cff1
                 END IF
              END DO
           END IF
        END DO
# endif
!
! Recalculate bed thickness and fractions for all layers.
!
        DO i=Istr,Iend
          DO k=1,Nbed
            cff3=0.0_r8
            DO ised=1,NST
              cff3=cff3+bed_mass(i,j,k,nnew,ised)
            END DO
            IF (cff3.eq.0.0_r8) THEN
              cff3=eps
            END IF
            bed(i,j,k,ithck)=0.0_r8
            DO ised=1,NST
              bed_frac(i,j,k,ised)=bed_mass(i,j,k,nnew,ised)/cff3
              bed(i,j,k,ithck)=MAX(bed(i,j,k,ithck)+                    &
     &                         bed_mass(i,j,k,nnew,ised)/               &
     &                         (Srho(ised,ng)*                          &
     &                          (1.0_r8-bed(i,j,k,iporo))),0.0_r8)
            END DO
          END DO
        END DO
!
!  If first time step of deposit, create new layer and combine bottom
!  two bed layers.
!
        DO i=Istr,Iend
          cff=0.0_r8
          IF (Nbed.gt.1) THEN
            DO ised=1,NST
              cff=cff+dep_mass(i,ised)
            END DO
            IF ((cff.gt.0.0_r8).or.(bed(i,j,1,ithck).gt.(2.1_r8*MAX(    &
     &         bottom(i,j,iactv),firstlayer_thick)))) THEN
!
!  If net deposition ocurred here (i.e. cff.gt.0), or the surficial
!  seabed layer is too thick, then go through this loop.  
!  In this loop, make a new surface layer by combining bottom layers 
!  if net deposition (cff) is greater than zero or if surface layer 
!  thickness is bigger than 2.1 times either the active layer or the 
!  new layer thickness. 
!  The loop chooses to combine the bottom-most two too-thin layers
!
              indnl=0
              kk=Nbed+1
              DO WHILE (indnl==0)
                  kk=kk-1
                IF (bed(i,j,kk,ithck).lt.newlayer_thick2) THEN
                  indnl=kk
                ELSEIF (kk.le.N0) THEN
                  indnl=Nbed
                END IF
              END DO
              bed(i,j,indnl,iporo)=( (bed(i,j,indnl-1,ithck)  *           &
     &                                bed(i,j,indnl-1,iporo)) +           &
     &                               (bed(i,j,indnl  ,ithck)  *           &
     &                                bed(i,j,indnl  ,iporo)) )           &
     &                              /(bed(i,j,indnl-1,ithck)  +           &
     &                                bed(i,j,indnl  ,ithck))
              bed(i,j,indnl,iaged)=( (bed(i,j,indnl-1,ithck)  *           &
     &                                bed(i,j,indnl-1,iaged)) +           &
     &                               (bed(i,j,indnl  ,ithck)  *           &
     &                                bed(i,j,indnl  ,iaged)) )           &
     &                              /(bed(i,j,indnl-1,ithck)  +           &
     &                                bed(i,j,indnl  ,ithck))
              DO ised=1,NST
                bed_mass(i,j,indnl,nnew,ised)=                           &
     &                             bed_mass(i,j,indnl-1,nnew,ised)+      &
     &                             bed_mass(i,j,indnl  ,nnew,ised)
              END DO
# ifdef SEDBIO_COUP
              DO ised=1,NSBT
                 bed_tracer(i,j,indnl,ised)=bed_tracer(i,j,indnl-1,ised) &
     &                +bed_tracer(i,j,indnl,ised)
              END DO
# endif
!
!             Push layers above layer indnl-1 down.
!
              DO k=indnl-1,2,-1
                bed(i,j,k,iporo)=bed(i,j,k-1,iporo)
                bed(i,j,k,iaged)=bed(i,j,k-1,iaged)
                DO ised =1,NST
                  bed_mass(i,j,k,nnew,ised)=bed_mass(i,j,k-1,nnew,ised)
                END DO
# ifdef SEDBIO_COUP
                DO ised=1,NSBT
                   bed_tracer(i,j,k,ised)=bed_tracer(i,j,k-1,ised)
                END DO
# endif
              END DO
!
!  Set new top layer parameters. The IF loop below seperates cases for 
!  which a new surface layer is forming because of deposition vs. a 
!  too-thick surface layer
!
              bed(i,j,1,iaged)=time(ng)
              DO ised=1,NST
                IF (cff.gt.eps) THEN
                  bed_mass(i,j,1,nnew,ised)=MAX(0.0_r8,dep_mass(i,ised))
                  bed_mass(i,j,2,nnew,ised)=MAX(0.0_r8,                   &
     &                       bed_mass(i,j,2,nnew,ised)-dep_mass(i,ised))
                ELSE
                  bed_mass(i,j,1,nnew,ised)=MAX(0.0_r8,                   &
     &                       (bed_mass(i,j,2,nnew,ised)/2.0_r8))
                  bed_mass(i,j,2,nnew,ised)=MAX(0.0_r8,                   &
     &                       (bed_mass(i,j,2,nnew,ised)/2.0_r8))
                END IF
              END DO 
# ifdef SEDBIO_COUP
!
! Same as above loop, but for sedbio tracers
!
              DO ised=1,NSBT
                IF (idsedbio(ised,1).gt.0) THEN
!                  loop for particulate matter
                  IF (cff.gt.eps) THEN
                  bed_tracer(i,j,1,ised)=MAX(0.0_r8,dep_mass_om(i,ised))
                  bed_tracer(i,j,2,ised)=MAX(bed_tracer(i,j,2,ised)-      &
     &                  dep_mass_om(i,ised),0.0_r8)
                  ELSE
                   bed_tracer(i,j,1,ised)=MAX(0.0_r8,                     &
     &                                    bed_tracer(i,j,2,ised)/2.0_r8)
                   bed_tracer(i,j,2,ised)=bed_tracer(i,j,2,ised)-         &
     &                                    bed_tracer(i,j,1,ised)
                  END IF
                ELSE
                 IF (idsedbio(ised,2).ne.0) THEN 
!                  loop for dissolved constituent; units (mmol T/m2 bed)
                  IF (cff.gt.eps) THEN
                      cff1=                                               &
     &                   MAX(t(i,j,1,nnew,idsedbio(ised,2))/Hz(i,j,1)     &
     &                   *ndep_thck_tot(i)*bed(i,j,2,iporo),0.0_r8)
                  ELSE
                      cff1=MAX(0.0_r8,bed_tracer(i,j,2,ised)/2.0_r8)
                  END IF
                 ELSE
!                No equivalent water column tracer, so no additions
!                due to deposition.
!
                  IF (cff.gt.eps) THEN
                       cff1=0.0_r8
                  ELSE
                      cff1=MAX(0.0_r8,bed_tracer(i,j,2,ised)/2.0_r8)
                  END IF
                 ENDIF
                   bed_tracer(i,j,2,ised)=bed_tracer(i,j,2,ised)-cff1     
                   bed_tracer(i,j,1,ised)=cff1
                END IF
              ENDDO
# endif
            END IF
          END IF !NBED=1
!
! Recalculate thickness and fractions for all layers.
!
          DO k=1,Nbed
            cff3=0.0_r8
            DO ised=1,NST
              cff3=cff3+bed_mass(i,j,k,nnew,ised)
            END DO
            IF (cff3.eq.0.0_r8) THEN
              cff3=eps
            END IF
            bed(i,j,k,ithck)=0.0_r8
            DO ised=1,NST
              bed_frac(i,j,k,ised)=bed_mass(i,j,k,nnew,ised)/cff3
              bed(i,j,k,ithck)=MAX(bed(i,j,k,ithck)+                    &
     &                         bed_mass(i,j,k,nnew,ised)/               &
     &                         (Srho(ised,ng)*                          &
     &                          (1.0_r8-bed(i,j,k,iporo))),0.0_r8)
            END DO
          END DO
        END DO
      END DO J_LOOP
!
!  End of Suspended Sediment only section.
!
# endif
!
!  J_LOOP2 : 
!  Ensure top bed layer thickness is greater or equal than active layer
!  thickness. If need to add sed to top layer, then entrain from lower
!  levels. Create new layers at bottom to maintain Nbed.
!  If SEDBIO_COUP is defined, also do seabed-water diffusion.
!
      J_LOOP2 : DO j=Jstr,Jend
        DO i=Istr,Iend
!
!  Calculate active layer thickness, bottom(i,j,iactv).
!
          bottom(i,j,iactv)=MAX(0.0_r8,                                 &
     &                          0.007_r8*                               &
     &                          (tau_w(i,j)-bottom(i,j,itauc))*rho0)+   &
     &                          6.0_r8*bottom(i,j,isd50)
# ifdef SEDBIO_COUP
! 
!     For seabed biogeochemistry, need a thin surface layer to resolve
!     gradients over short spatial scales and for seabed-water column
!     diffusion.  Do not let active layer thickness exceed
!     firstlayer_thick.  This also seems reasonable as long as we are 
!     considering fine grained muds, we would not expect really thick 
!     active layers. 
!
          bottom(i,j,iactv)=MIN(firstlayer_thick, bottom(i,j,iactv))
# endif

!
# ifdef SED_MORPH
!
! Apply morphology factor.
!
          bottom(i,j,iactv)=MAX(bottom(i,j,iactv)*morph_fac(1,ng),      &
     &                          bottom(i,j,iactv))
# endif
          IF (bottom(i,j,iactv).gt.bed(i,j,1,ithck)) THEN
            IF (Nbed.eq.1) THEN
              bottom(i,j,iactv)=bed(i,j,1,ithck)
            ELSE
              thck_to_add=bottom(i,j,iactv)-bed(i,j,1,ithck)
              thck_avail=0.0_r8
!             Ksed is index of the bottom-most seabed layer that
!             will be incorporated into the surface seabed layer 
!             to ensure that the active layer is thick enough. 
              Ksed=1
              DO k=2,Nbed
                IF (thck_avail.lt.thck_to_add) THEN
                  thck_avail=thck_avail+bed(i,j,k,ithck)
                  Ksed=k
                END IF
              END DO
!
!  Catch here if there was not enough bed material.
!
              IF (thck_avail.lt.thck_to_add) THEN
                bottom(i,j,iactv)=bed(i,j,1,ithck)+thck_avail
                thck_to_add=thck_avail
              END IF
!
!  Catch here if the new fractional layer will be very thin
!  To fix this, add the entire layer instead of leaving a thin layer
!  and add thickness of next layer to thck_avail.
!
             IF (thck_avail-thck_to_add.lt.(newlayer_thick(ng)/5.0_r8)) &
     &             THEN         
                thck_to_add=thck_avail
                If (Ksed.lt.Nbed) THEN
                   Ksed=Ksed+1
                   thck_avail=thck_avail+bed(i,j,Ksed,ithck)
                END IF
             END IF 
!
!  Update bed mass of top layer and fractional layer.
!
              cff2=MAX(thck_avail-thck_to_add,0.0_r8)/                  &
     &             MAX(bed(i,j,Ksed,ithck),eps)
              DO ised=1,NST
                cff1=0.0_r8
                DO k=1,Ksed
                  cff1=cff1+bed_mass(i,j,k,nnew,ised)
                END DO
                cff3=cff2*bed_mass(i,j,Ksed,nnew,ised)
                bed_mass(i,j,1   ,nnew,ised)=cff1-cff3
                bed_mass(i,j,Ksed,nnew,ised)=cff3
              END DO
# ifdef SEDBIO_COUP
              DO ised=1,NSBT
                 cff1=0.0_r8
                 Do k=1,Ksed
                    cff1=cff1+bed_tracer(i,j,k,ised)
                 ENDDO
                 bed_tracer(i,j,1,ised)=cff1-                           &
     &                bed_tracer(i,j,Ksed,ised)*cff2
                 bed_tracer(i,j,Ksed,ised)=                             &
     &                bed_tracer(i,j,Ksed,ised)*cff2
              ENDDO
# endif
!
!  Upate properties of seabed layers from surface to layer Kbed
!  & Pull all layers closer to the surface.
!
              bed(i,j,Ksed,ithck)=MAX(thck_avail-thck_to_add,0.0_r8)
              cff3=0.0_r8
              DO ised=1,NST
                cff3=cff3+bed_mass(i,j,1,nnew,ised)
              END DO
              IF (cff3.eq.0.0_r8) THEN
                cff3=eps
              END IF
              DO ised=1,NST
                bed_frac(i,j,1,ised)=bed_mass(i,j,1,nnew,ised)/cff3
              END DO
              bed(i,j,1,ithck)=bed(i,j,1,ithck)+thck_to_add
              ks=Ksed-2
              DO k=Ksed,Nbed
                bed(i,j,k-ks,ithck)=bed(i,j,k,ithck)
                bed(i,j,k-ks,iporo)=bed(i,j,k,iporo)
                bed(i,j,k-ks,iaged)=bed(i,j,k,iaged)
                DO ised=1,NST
                  bed_frac(i,j,k-ks,ised)=bed_frac(i,j,k,ised)
                  bed_mass(i,j,k-ks,nnew,ised)=bed_mass(i,j,k,nnew,ised)
                END DO
# ifdef SEDBIO_COUP
                DO ised=1,NSBT
                  bed_tracer(i,j,k-ks,ised)=bed_tracer(i,j,k,ised)
                END DO
# endif                  
              END DO
!
!  Add new layers onto the bottom. Split what was in the bottom layer to
!  fill these new empty cells. ("ks" is the number of new layers).
!  New layers should be newlayer_thick2 thickness
!  if there is enough sediment in bottom most layer
!
              ks=Ksed-2
              k=Nbed
              IF (bed(i,j,k,ithck).lt.                                  & 
     &             (newlayer_thick2*(REAL(ks+1,r8)))) THEN
!                cff is proportion of thick layer in a SINGLE new layer
!                cff1 is fraction of original thick layer that will stay
!                     in the thick layer
                 cff=1.0_r8/REAL(ks+1,r8)
                 cff1=cff
              ELSE
                 cff=newlayer_thick2/bed(i,j,k,ithck)
                 cff1=1.0_r8 - REAL(ks,r8)*cff
              END IF
              bed(i,j,k,ithck)=bed(i,j,k-ks,ithck)*cff1
              bed(i,j,k,iaged)=bed(i,j,k-ks,iaged)
              DO ised=1,NST
                 bed_frac(i,j,k,ised)=bed_frac(i,j,k-ks,ised)
                 bed_mass(i,j,k,nnew,ised)=                             &
     &                bed_mass(i,j,k-ks,nnew,ised)*cff1
              END DO
# ifdef SEDBIO_COUP
              DO ised=1,NSBT
                 bed_tracer(i,j,k,ised)=                                &
     &                bed_tracer(i,j,k-ks,ised)*cff1
              END DO
# endif
!             for other ks layers, use cff              
              DO k=Nbed-1,Nbed-ks,-1
                bed(i,j,k,ithck)=bed(i,j,Nbed-ks,ithck)*cff
                bed(i,j,k,iaged)=bed(i,j,Nbed-ks,iaged)
                DO ised=1,NST
                  bed_frac(i,j,k,ised)=bed_frac(i,j,Nbed-ks,ised)
                  bed_mass(i,j,k,nnew,ised)=                            &
     &                             bed_mass(i,j,Nbed-ks,nnew,ised)*cff
                END DO
# ifdef SEDBIO_COUP
                DO ised=1,NSBT
                  bed_tracer(i,j,k,ised)=                               &
     &                 bed_tracer(i,j,Nbed-ks,ised)*cff
                END DO
# endif
              END DO
            END IF  ! Nbed > 1
          END IF                ! increase top bed layer
        END DO
# ifdef SEDBIO_COUP
!       Need to update dissolved bed tracers due to diffusion across
!                 the seabed-water interface.  Also, calculate sdflux, 
!                 the cumulative diffusive uptake of tracers (mmol T/m2).
!                 sdflux increases for diffusion into the seabed.
!       Note that t(i,j,1,nnew,ind) is in units of kg/m2 or mmol/m2
!                 but t(i,j,1,3,ind) is in kg/m3 or mmol/m3 
!       Method of calculating diffusion is determined by the user 
!                 variable in ocean.in
        IF (INT(user(3)).eq.1) THEN
!
!       Method 1: Assume top seabed layer and bottom water column 
!                 layer are in equilibrium (i.e. set porewater 
!                 concentrations in the top bed layer and in the 
!                 overlying water cell equal to each other.
!                 Consistent with Soetaert et al. (1996)
!
        DO i=Istr,Iend
              DO ised=1,NSBD
                 IF (idsedbio(NSBP+ised,2).gt.0) THEN
!                Do only if there is an equivalent water column tracer
                 cff1=t(i,j,1,nnew,idsedbio(NSBP+ised,2)) +             &
     &                   bed_tracer(i,j,1,NSBP+ised)
                 cff2=Hz(i,j,1)/(Hz(i,j,1)+                             &
     &                bed(i,j,1,ithck)*bed(i,j,1,iporo))
! Note that porosity^(1.3) is porosity in m water / m bulk
                 sdflux(i,j,NSBP+ised)=sdflux(i,j,NSBP+ised)+           &
     &               (cff1*(1.0_r8-cff2) - bed_tracer(i,j,1,NSBP+ised))
                 t(i,j,1,nnew,idsedbio(NSBP+ised,2))=cff1*cff2
                 bed_tracer(i,j,1,NSBP+ised)=cff1*(1.0_r8-cff2)
                 ENDIF
              END DO
        END DO
        ELSEIF (INT(user(3)).eq.2) THEN
!
!       Method 2: Use Fickian diffusion to estimate diffusive fluxes
!                 of porewater across the seabed-water interface.
!                 For now assume that diffusion occurs over a length
!                 scale equal to the thickness of the surfacial seabed 
!                 layer.  In other words, either 1. diffusion occurs 
!                 over the surficial seabed layer (but the diffusion-
!                 dominated portion of the water column, i.e. the 
!                 viscous sublayer, is negligable), or 2. diffusion 
!                 occurs over the top half of the seabed layer, and 
!                 over an equivalent distance in the water column. 
!
        DO i=Istr,Iend
              DO ised=1,NSBD
                 IF (idsedbio(NSBP+ised,2).gt.0) THEN
!                Do only if there is an equivalent water column tracer
!                cff1 = dc/dz in mmol T/m3/m.  
                 cff1=(t(i,j,1,nnew,idsedbio(NSBP+ised,2))/Hz(i,j,1) -  &
     &                   bed_tracer(i,j,1,NSBP+ised)/(bed(i,j,1,ithck)  &
     &                   *bed(i,j,1,iporo)))
                 cff1=cff1/bed(i,j,1,ithck)
!                 cff1=cff1/(bed(i,j,1,ithck)*                           &
!     &                   (bed(i,j,1,iporo)**(0.3333)))
!                cff2 = diffusive flux out of bed 
!                     = -poro*DiffCoef*dC/dz*dt 
!                units are:  (m2/s)*(mmol T /m4)*s = mmol T /m2
                 cff2=-bed(i,j,1,iporo)*D_sw*cff1*dt(ng)
                   IF (cff2.gt.eps) THEN
!                  If diffusion is out of bed
!                  Make sure that diffusion doesn't go past equilibrium
                      cff2=MIN(cff2,                                    &
     &                     bed_tracer(i,j,1,NSBP+ised)-                 &
     &                     (t(i,j,1,nnew,idsedbio(NSBP+ised,2)) +       &
     &                     bed_tracer(i,j,1,NSBP+ised))*                &
     &                     (1.0_r8-(Hz(i,j,1)/(Hz(i,j,1)+               &
     &                     bed(i,j,1,ithck)*                            &
     &                     bed(i,j,1,iporo)))) )
!                      cff2=MIN(cff2,bed_tracer(i,j,1,NSBP+ised))
                   ELSE IF (cff2.lt.eps) THEN
!                  If diffusion is into of bed
!                  Make sure tracers don't go negative
                      cff2=-1.0_r8*MIN(ABS(cff2),                       &
     &                      t(i,j,1,nnew,idsedbio(NSBP+ised,2))-        &
     &                      (t(i,j,1,nnew,idsedbio(NSBP+ised,2)) +      &
     &                      bed_tracer(i,j,1,NSBP+ised))*               &
     &                      (Hz(i,j,1)/(Hz(i,j,1)+                      &
     &                      bed(i,j,1,ithck)*                           &
     &                      bed(i,j,1,iporo))) )
!                      cff2=-1.0_r8*MIN(ABS(cff2),                      &
!     &                      t(i,j,1,nnew,idsedbio(NSBP+ised,2)))
                   ELSE 
                      cff2=0.0_r8
                   END IF
!
!                Now, apply diffusion:
!
                 sdflux(i,j,NSBP+ised)=sdflux(i,j,NSBP+ised)-cff2
                 t(i,j,1,nnew,idsedbio(NSBP+ised,2))=                   &
     &                t(i,j,1,nnew,idsedbio(NSBP+ised,2)) +             &
     &                cff2
                 bed_tracer(i,j,1,NSBP+ised)=                           &
     &                bed_tracer(i,j,1,NSBP+ised) -                     &
     &                cff2 
                ENDIF
              END DO
        END DO
        ENDIF
# endif
      END DO J_LOOP2
!
! J_LOOP3 - Make sure that top N0 layers are thin (~= newlayer_thick)
! and that layers below that, except for the bottom one are moderately
! thin (~= newlayer_thick2)
!
      J_LOOP3 : DO j=Jstr,Jend
        DO i=Istr,Iend
          DO k=1,Nbed
              indnl=0
!           First, find layer that is too thin
            IF ((k.le.N0).and.                                          &
     &         (bed(i,j,k,ithck).gt.newlayer_thick(ng))) THEN
                   indnl=k
            ELSEIF ((k.lt.N1).and.                                      &
     &         (bed(i,j,k,ithck).gt.newlayer_thick2)) THEN
                   indnl=k
            ELSE
                   indnl=Nbed
            END IF
!           Second, if there is a 'too-thin' layer, 
!           combine bottom two layers, 
!           push other layers down and then split off new layer
          IF (indnl.lt.(Nbed-1)) THEN
                bed(i,j,Nbed,iporo)=                                    &
     &               (bed(i,j,Nbed-1,iporo)*                            &
     &               bed(i,j,Nbed-1,ithck) +                            &
     &               bed(i,j,Nbed  ,iporo)*                             &
     &               bed(i,j,Nbed  ,ithck))/                            &
     &               (bed(i,j,Nbed-1,ithck)+                            &
     &               bed(i,j,Nbed,ithck))
                bed(i,j,Nbed,iaged)=                                    &
     &               (bed(i,j,Nbed-1,iaged)*                            &
     &               bed(i,j,Nbed-1,ithck) +                            &
     &               bed(i,j,Nbed  ,iaged)*                             &
     &               bed(i,j,Nbed  ,ithck))/                            &
     &               (bed(i,j,Nbed-1,ithck)+                            &
     &               bed(i,j,Nbed,ithck))
                bed(i,j,Nbed,ithck)=bed(i,j,Nbed-1,ithck)+              &
     &                              bed(i,j,Nbed,ithck)
                DO ised =1,NST
                  bed_mass(i,j,Nbed,nnew,ised)=                         &
     &               bed_mass(i,j,Nbed-1,nnew,ised)+                    &
     &               bed_mass(i,j,Nbed,nnew,ised)
                END DO
# ifdef SEDBIO_COUP
                DO ised=1,NSBT
                   bed_tracer(i,j,Nbed,ised)=                            &
      &               bed_tracer(i,j,Nbed-1,ised)+                       &
      &               bed_tracer(i,j,Nbed,ised)
                ENDDO
# endif
            DO kk=Nbed-1,indnl+1,-1
                bed(i,j,kk,iporo)=bed(i,j,kk-1,iporo)
                bed(i,j,kk,iaged)=bed(i,j,kk-1,iaged)
                DO ised =1,NST
                 bed_mass(i,j,kk,nnew,ised)=bed_mass(i,j,kk-1,nnew,ised)
                END DO
# ifdef SEDBIO_COUP
                DO ised=1,NSBT
                   bed_tracer(i,j,kk,ised)=bed_tracer(i,j,kk-1,ised)
                ENDDO
# endif
            END DO
          END IF
!             Split off new layer (indnl) from layer below (indnl+1)
          IF (indnl.lt.Nbed) THEN
            IF (indnl.le.N0) THEN
              cff=MIN(newlayer_thick(ng)/bed(i,j,indnl,ithck),0.5_r8)
              cff1=1.0_r8 - cff
            ELSE IF (indnl.lt.(Nbed-1)) THEN
              cff=MIN(newlayer_thick2/bed(i,j,indnl,ithck),0.5_r8)
              cff1=1.0_r8 - cff
            ELSE
              cff=newlayer_thick2/bed(i,j,indnl,ithck)
              cff1=1.0_r8 +((1.0_r8-cff)*bed(i,j,indnl,ithck)/          &
     &             bed(i,j,indnl+1,ithck))
            END IF
            IF (indnl.lt.(Nbed-1)) THEN
              bed(i,j,indnl+1,iaged)=bed(i,j,indnl+1,iaged)
              DO ised=1,NST
                 bed_frac(i,j,indnl+1,ised)=bed_frac(i,j,indnl+1,ised)
              END DO
            ELSE
              bed(i,j,indnl+1,iaged)=                                   &
     &               (bed(i,j,indnl+1,iaged)*                           &
     &               bed(i,j,indnl+1,ithck)*cff1 +                      &
     &               bed(i,j,indnl  ,iaged)*                            &
     &               bed(i,j,indnl  ,ithck)*cff)/                       &
     &               (bed(i,j,indnl+1,ithck)*cff1+                      &
     &               bed(i,j,indnl,ithck)*cff)
              bed(i,j,indnl+1,iporo)=                                   &
     &               (bed(i,j,indnl+1,iporo)*                           &
     &               bed(i,j,indnl+1,ithck)*cff1 +                      &
     &               bed(i,j,indnl  ,iporo)*                            &
     &               bed(i,j,indnl  ,ithck)*cff)/                       &
     &               (bed(i,j,indnl+1,ithck)*cff1+                      &
     &               bed(i,j,indnl,ithck)*cff)
              DO ised=1,NST
                 bed_frac(i,j,indnl+1,ised)=                            &
     &               (bed_frac(i,j,indnl+1,ised)*                       &
     &               bed(i,j,indnl+1,ithck)*cff1 +                      &
     &               bed_frac(i,j,indnl  ,ised)*                        &
     &               bed(i,j,indnl  ,ithck)*cff)/                       &
     &               (bed(i,j,indnl+1,ithck)*cff1+                      &
     &               bed(i,j,indnl,ithck)*cff)
              END DO
            END IF
              DO ised=1,NST
                 bed_mass(i,j,indnl+1,nnew,ised)=                       &
     &                bed_mass(i,j,indnl+1,nnew,ised)*cff1
              END DO
              bed(i,j,indnl+1,ithck)=bed(i,j,indnl+1,ithck)*cff1
# ifdef SEDBIO_COUP
              DO ised=1,NSBT
                 bed_tracer(i,j,indnl+1,ised)=                          &
     &                bed_tracer(i,j,indnl+1,ised)*cff1
              END DO
# endif
                bed(i,j,indnl,ithck)=bed(i,j,indnl,ithck)*cff
                bed(i,j,indnl,iaged)=bed(i,j,indnl,iaged)
                bed(i,j,indnl,iporo)=bed(i,j,indnl,iporo)
                DO ised=1,NST
                  bed_frac(i,j,indnl,ised)=bed_frac(i,j,indnl,ised)
                  bed_mass(i,j,indnl,nnew,ised)=                         &
     &                             bed_mass(i,j,indnl,nnew,ised)*cff
                END DO
# ifdef SEDBIO_COUP
                DO ised=1,NSBT
                   bed_tracer(i,j,indnl,ised)=                           &
     &                  bed_tracer(i,j,indnl,ised)*cff
                END DO
# endif
          END IF
        END DO  !k loop
        END DO  !i loop
      END DO J_LOOP3
!
! Recalculate thickness and fractions for all layers.
!
          DO k=1,Nbed
            cff3=0.0_r8
            DO ised=1,NST
              cff3=cff3+bed_mass(i,j,k,nnew,ised)
            END DO
            IF (cff3.eq.0.0_r8) THEN
              cff3=eps
            END IF
            bed(i,j,k,ithck)=0.0_r8
            DO ised=1,NST
              bed_frac(i,j,k,ised)=bed_mass(i,j,k,nnew,ised)/cff3
              bed(i,j,k,ithck)=MAX(bed(i,j,k,ithck)+                    &
     &                         bed_mass(i,j,k,nnew,ised)/               &
     &                         (Srho(ised,ng)*                          &
     &                          (1.0_r8-bed(i,j,k,iporo))),0.0_r8)
            END DO
          END DO
!
!-----------------------------------------------------------------------
! Store old bed thickness.
!-----------------------------------------------------------------------
!
# if defined SED_MORPH
      DO j=JstrR,JendR
        DO i=IstrR,IendR
            bed_thick(i,j,nnew)=0.0_r8
            DO k=1,Nbed
              bed_thick(i,j,nnew)=bed_thick(i,j,nnew)+                  &
     &                            bed(i,j,k,ithck)
            END DO
          END DO
        END DO
#  if defined EW_PERIODIC || defined NS_PERIODIC
        CALL exchange_r2d_tile (ng, tile,                               &
     &                          LBi, UBi, LBj, UBj,                     &
     &                          bed_thick(:,:,nnew))
#  endif
# endif
# ifdef SEDBIO_COUP
#  ifdef TS_MPDATA
!
! Update tracer array
!
        DO j=JstrR,JendR
           DO i=IstrR,IendR
              DO ised=1,NSBD
                 t(i,j,1,3,idsedbio(NSBP+ised,2))=                      &
     &                t(i,j,1,nnew,idsedbio(NSBP+ised,2))/Hz(i,j,1)
              ENDDO
           ENDDO
        ENDDO
#  endif
# endif
!
!-----------------------------------------------------------------------
!  Apply periodic or gradient boundary conditions to property arrays.
!-----------------------------------------------------------------------
!
      DO ised=1,NST
        CALL bc_r3d_tile (ng, tile,                                     &
     &                    LBi, UBi, LBj, UBj, 1, Nbed,                  &
     &                    bed_frac(:,:,:,ised))
        CALL bc_r3d_tile (ng, tile,                                     &
     &                    LBi, UBi, LBj, UBj, 1, Nbed,                  &
     &                    bed_mass(:,:,:,nnew,ised))
      END DO
# ifdef SEDBIO_COUP
      DO ised=1,NSBT
           CALL bc_r3d_tile (ng, tile,                                  &
     &                    LBi, UBi, LBj, UBj, 1, Nbed,                  &
     &                    bed_tracer(:,:,:,ised))
      ENDDO
# endif
# ifdef DISTRIBUTE
      CALL mp_exchange4d (ng, tile, iNLM, 2,                            &
     &                    LBi, UBi, LBj, UBj, 1, Nbed, 1, NST,          &
     &                    NghostPoints, EWperiodic, NSperiodic,         &
     &                    bed_frac,                                     &
     &                    bed_mass(:,:,:,nnew,:))
# ifdef SEDBIO_COUP
      CALL mp_exchange4d (ng, tile, iNLM, 2,                            &
     &                    LBi, UBi, LBj, UBj, 1, Nbed, 1, NSBT,         &
     &                    NghostPoints, EWperiodic, NSperiodic,         &
     &                    bed_tracer)
# endif
# endif

      DO i=1,MBEDP
        CALL bc_r3d_tile (ng, tile,                                     &
     &                    LBi, UBi, LBj, UBj, 1, Nbed,                  &
     &                    bed(:,:,:,i))
      END DO
# ifdef DISTRIBUTE
      CALL mp_exchange4d (ng, tile, iNLM, 1,                            &
     &                    LBi, UBi, LBj, UBj, 1, Nbed, 1, MBEDP,        &
     &                    NghostPoints, EWperiodic, NSperiodic,         &
     &                    bed)
# endif

      RETURN
      END SUBROUTINE sed_bed_tile
#endif
      END MODULE sed_bed_mod
