package com.frevvo.forms.cli.shell;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.FileWriter;
import java.io.IOException;

import asg.cliche.Command;

import com.frevvo.forms.cli.ApiHelper;
import com.frevvo.forms.cli.core.FeedShell;
import com.frevvo.forms.client.Helper;
import com.frevvo.forms.client.SubmissionEntry;
import com.frevvo.forms.client.SubmissionFeed;
import com.google.gdata.data.Link;
import com.google.gdata.data.media.MediaSource;
import com.google.gdata.util.ServiceException;

/**
 * Shell context for all commands related to submission feeds
 */
public class SubmissionFeedShell extends
		FeedShell<SubmissionFeed, SubmissionEntry, SubmissionEntryShell> {
	public static final String PATH_ELEMENT = "subs";

	public SubmissionFeedShell(SubmissionFeed feed) {
		super(PATH_ELEMENT, feed, SubmissionFeed.class);
	}

	@Command(name = "docs", description = "MERGE xml documents to console (e.g. 'docs')")
	public String docs() throws IOException, ServiceException {

		SubmissionFeed subs = getFeed();
		StringBuilder sb = new StringBuilder();
		docs(sb, subs);

		return "Downloaded " + subs.getEntries().size()
				+ " submission documents(s):\n" + sb.toString();
	}

	@Command(name = "docs", description = "MERGE xml documents to file (e.g. 'docs /var/all_docs.xml')")
	public String docs(String filename) throws IOException, ServiceException {

		SubmissionFeed subs = getFeed();
		StringBuilder sb = new StringBuilder();
		docs(sb, subs);

		File file = File.createTempFile(filename + "_", "_docs.xml");
		FileWriter fw = new FileWriter(file);
		try {
			fw.write(sb.toString());
			return "Downloaded " + subs.getEntries().size()
					+ " submission documents(s) to file : "
					+ file.getAbsolutePath();
		} catch (Exception e) {
			return "Could not download documents to file "
					+ file.getAbsolutePath() + ": " + e.getMessage();
		} finally {
			fw.close();
		}
	}

	@Command(name = "snapshots", description = "DOWNLOAD ALL PDF snapshots into a temp folder (e.g.: 'snapshots')")
	public String snapshots() throws IOException, ServiceException {
		return snapshots(null, null);
	}

	@Command(name = "snapshots", description = "DOWNLOAD ALL PDF snapshots into a temp folder (e.g.: 'snapshots \"application/pdf\"')")
	public String snapshots(String folder) throws IOException, ServiceException {
		return snapshots(folder, null);
	}
	
	@Command(name = "snapshots", description = "DOWNLOAD ALL PDF snapshots into a temp folder (e.g.: 'snapshots \"application/pdf\"')")
	public String snapshots(String folder, String contentType) throws IOException, ServiceException {
		StringBuilder sb = new StringBuilder();
		
		File folderFile = null;
		if( folder == null )
			folderFile = ApiHelper.createTempFolder("Snapshots_");
		else 
			folderFile = new File(folder);
		
		sb.append("Downloading snapshots to folder: " + folderFile +"\n");
		
		if( !folderFile.exists() ){
			sb.append(" Folder " + folderFile.getPath() + " doesnt exist: creating ...\n");
			folderFile.mkdirs();
		}
		
		if( contentType == null )
			contentType = "application/pdf";
		
		SubmissionFeed subs = getFeed();
		int count = 0;
		for (SubmissionFeed page = subs; page != null
				&& page.getEntries().size() > 0; page = page.getNext()) {
			for (SubmissionEntry sub : subs.getEntries()) {
				Link link = ApiHelper.getSubmissionSnapshotLink(sub, "application/pdf");
				if( link == null ){
					sb.append("Submission ").append(ApiHelper.getName(sub) + " (" + sub.getId()).append(") has no PDF snapshot. Ignoring...\n");
					continue;
				}
				
				String name = ApiHelper.getName(sub);
				String fileName = ApiHelper.nameToFileName(name + "_" + sub.getId() + "_" + sub.getUpdated().toUiString()) + ".pdf"; // TODO what if not a pdf?
				try {
					File file = new File(folderFile, fileName);
					FileOutputStream fos = new FileOutputStream(file);
					try {
						MediaSource ms = getService().getMedia(link.getHref());
						Helper.readStream(ms.getInputStream(), fos);
	
						sb.append("\tDownloaded submission PDF #" + (++count) + " " + fileName + " : "
								+ "\n");
					} catch (Exception e) {
						sb.append("\tCould not download submission PDF " + fileName
								+ ": " + e.getMessage() + "\n");
					} finally {
						fos.close();
					}
				} catch (Exception e) {
					sb.append("\tCould not download submission PDF " + fileName + ": "
							+ e.getMessage() + "\n");
				}
			}
		}
		return "Downloaded " + count + " submission PDF(s):\n"
				+ sb.toString();
	}

	@Override
	protected SubmissionEntry createEntry(String name, String description)
			throws IOException, ServiceException {
		return null; // not supported
	}

	@Override
	protected SubmissionEntryShell createEntryShell(SubmissionEntry entry) {
		return new SubmissionEntryShell(entry);
	}

	@Override
	protected String print(SubmissionFeed feed) {
		try {
			return ApiHelper.print(feed);
		} catch (ServiceException e) {
			return "Could not print submission entries: " + e.getMessage();
		}
	}

	protected StringBuilder docs(StringBuilder sb, SubmissionFeed feed)
			throws IOException, ServiceException {
		sb.append("<submissions>\n");
		while (feed != null && feed.getEntries().size() > 0) {
			for (SubmissionEntry entry : feed.getEntries()) {
				sb.append(format("\t<submission id=\"{0}\">\n", entry.getId()));
				try {
					for (Link l : entry.getDocumentLinks("text/xml")) {
						sb.append(format(
								"\t\t<document contentType=\"{0}\">\n",
								l.getType()));
						try {
							MediaSource ms = getService().getMedia(l.getHref());
							ByteArrayOutputStream baos = new ByteArrayOutputStream();
							Helper.readStream(ms.getInputStream(), baos);
							sb.append(new String(baos.toByteArray(), "utf-8"));
						} finally {
							sb.append("\t\t</document>\n");
						}
					}
				} finally {
					sb.append("\t</submission>\n");
				}
			}
			feed = feed.getNext();
		}
		sb.append("</submissions>");
		return sb;
	}
}
