package com.frevvo.forms.cli.shell;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;

import asg.cliche.Command;

import com.frevvo.forms.cli.ApiHelper;
import com.frevvo.forms.cli.core.FeedShell;
import com.frevvo.forms.client.FormTypeEntry;
import com.frevvo.forms.client.FormTypeFeed;
import com.frevvo.forms.client.Helper;
import com.google.gdata.data.MediaContent;
import com.google.gdata.data.media.MediaSource;
import com.google.gdata.data.media.MediaStreamSource;
import com.google.gdata.util.ServiceException;

/**
 * Shell context for all commands related to form feeds
 */
public class FormTypeFeedShell extends
		FeedShell<FormTypeFeed, FormTypeEntry, FormTypeEntryShell> {
	public static final String PATH_ELEMENT = "forms";

	public FormTypeFeedShell(FormTypeFeed feed) {
		super(PATH_ELEMENT, feed, FormTypeFeed.class);
	}

	@Command(name = "down", description = "DOWNLOAD ALL forms into temp folder (e.g.: 'down')")
	public String download() throws IOException, ServiceException {

		FormTypeFeed forms = getFeed();
		StringBuilder sb = new StringBuilder();
		for (FormTypeEntry form : forms.getEntries()) {
			String name = ApiHelper.getName(form);
			String fileName = ApiHelper.nameToFileName(name);
			try {
				File file = File.createTempFile(fileName + "_", "_form.zip");
				FileOutputStream fos = new FileOutputStream(file);
				try {
					MediaContent mc = (MediaContent) form.getContent();
					MediaSource ms = getService().getMedia(mc);

					Helper.readStream(ms.getInputStream(), fos);

					sb.append("\tDownloaded formtype " + fileName + " : "
							+ file + "\n");
				} catch (Exception e) {
					sb.append("\tCould not download formtype " + fileName
							+ ": " + e.getMessage() + "\n");
				} finally {
					fos.close();
				}
			} catch (Exception e) {
				sb.append("\tCould not download formtype " + fileName + ": "
						+ e.getMessage() + "\n");
			}
		}
		return "Downloaded " + forms.getEntries().size() + " formtypes(s):\n"
				+ sb.toString();
	}

	@Command(name = "up", description = "UPLOAD an application (e.g. 'upload /var/contacts_apps.zip')")
	public String upload(String filePath) throws IOException, ServiceException {
		File f = new File(filePath);
		if (!f.exists())
			return "File " + filePath + " doesnt exist";

		try {
			// upload
			FileInputStream fis = new FileInputStream(f);
			try {
				FormTypeFeed forms = getFeed();
				MediaStreamSource ms = new MediaStreamSource(fis,
						"application/zip");
				FormTypeEntry form = forms.insert(ms);

				go(createEntryShell(form));
				return "FormType " + ApiHelper.getName(form)
						+ " successfully uploaded";
			} finally {
				fis.close();
			}
		} catch (ServiceException ex) {
			return "Could not upload formtype "
					+ filePath
					+ ": are you trying to upload an formtype with an id that already exists";
		} catch (Exception ex) {
			return "Could not upload formtype " + filePath + ": "
					+ ex.getMessage();
		}
	}

	@Override
	protected FormTypeEntry createEntry(String name, String description)
			throws IOException, ServiceException {
		return ApiHelper.createFormTypeEntry(getService(), getFeed(), name,
				description);
	}

	@Override
	protected FormTypeEntryShell createEntryShell(FormTypeEntry entry) {
		return new FormTypeEntryShell(entry);
	}

	@Override
	protected String print(FormTypeFeed feed) {
		return ApiHelper.print(feed);
	}
}
