package com.frevvo.forms.cli.shell;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import asg.cliche.Command;

import com.frevvo.forms.cli.ApiHelper;
import com.frevvo.forms.cli.core.EntryShell;
import com.frevvo.forms.client.ControlTypeFeed;
import com.frevvo.forms.client.DocumentTypeFeed;
import com.frevvo.forms.client.FormTypeEntry;
import com.frevvo.forms.client.Helper;
import com.frevvo.forms.client.SubmissionFeed;
import com.google.gdata.data.Link;
import com.google.gdata.data.MediaContent;
import com.google.gdata.data.media.MediaSource;
import com.google.gdata.data.media.MediaStreamSource;
import com.google.gdata.util.ServiceException;

/**
 * Shell context for all commands related to form entries
 */
public class FormTypeEntryShell extends EntryShell<FormTypeEntry> {
	public FormTypeEntryShell(FormTypeEntry entry) {
		super(entry.getTitle().getPlainText(), entry);
	}

	@Command(name = "readonly", description = "UPDATE the readonly property (e.g. 'readonly {true|false}')")
	public String readonly(boolean value) {
		try {
			FormTypeEntry form = getEntry();
			form.setReadOnly(value);
			form = form.update();
			setEntry(form);
		} catch (Exception e) {
			return "Could not update formtype's visibility: " + e.getMessage();
		}
		return "Readonly updated to " + value;
	}

	@Command(name = "snapshot", description = "OPEN form's PDF snapshot")
	public String snapshot() {

		FormTypeEntry form = getEntry();
		Link snapshotLink = form.getSnapshotLink("application/pdf");
		openLink(snapshotLink);

		return "Opening snapshot : " + snapshotLink.getHref();
	}

	@Command(name = "new", description = "CREATE new form instance (e.g. 'new')")
	public String createInstanc() throws ServiceException, IOException {
		FormTypeEntry form = getEntry();

		Map<String, Object> params = new HashMap<String, Object>();
		params.put(FormTypeEntry.FORMTYPE_DATA_PARAMETER, null);
		URL formInstanceUrl = form.createFormInstance(params, null);

		return go(new FormInstanceShell(form, formInstanceUrl));
	}

	@Command(name = "new", description = "CREATE new form instance (e.g. 'new p1=a,p2=10')")
	public String createInstanceFromData(String... data) {
		try {
			FormTypeEntry form = getEntry();

			Map<String, Object> d = toFormData(data);
			Map<String, Object> params = new HashMap<String, Object>();
			params.put(FormTypeEntry.FORMTYPE_DATA_PARAMETER, d);

			URL formInstanceUrl = form.createFormInstance(params, null);

			return go(new FormInstanceShell(getEntry(), formInstanceUrl));
		} catch (Exception e) {
			return "Could not create " + getEntry().getKind() + " instance: "
					+ e.getMessage();
		}
	}

	@Command(name = "newXml", description = "CREATE new form instance from XML (e.g. 'newXml /var/doc1.xml,/var/doc2.xml')")
	public String createInstanceFromXml(String... xmlDocPaths)
			throws ServiceException, IOException {
		try {
			List<MediaSource> mss = toMediaSources(xmlDocPaths);

			FormTypeEntry entry = getEntry();

			URL formInstanceUrl = entry.createFormInstance(null, mss);

			return go(new FormInstanceShell(entry, formInstanceUrl));
		} catch (Exception e) {
			return "Could not create " + getEntry().getKind() + " instance: "
					+ e.getMessage();
		}
	}

	@Command(name = "use", description = "USE form (e.g. 'use')")
	public String use() throws ServiceException {

		FormTypeEntry form = getEntry();
		Link formLink = form.getFormTypeLink(null);
		Link l = openLink(formLink);

		return "Using form: " + l.getHref();
	}

	@Command(name = "edit", description = "EDIT form (e.g. 'edit')")
	public String edit() throws ServiceException {

		FormTypeEntry form = getEntry();
		Link editorLink = form.getFormTypeEditorLink(null);
		Link l = openLink(editorLink);

		return "Editing form: " + l.getHref();
	}

	@Command(name = "schema", description = "DOWNLOAD the form XSD (e.g. 'schema')")
	public String schema() throws MalformedURLException, IOException,
		ServiceException {
		
		String name = ApiHelper.getName(getEntry());
		String fileName = ApiHelper.nameToFileName(name);
		try {
			File file = File.createTempFile(fileName + "_", "_form.xsd");
			FileOutputStream fos = new FileOutputStream(file);
			try {
				FormTypeEntry form =getEntry();
				
				Link schemaLink = form.getFormTypeSchemaLink();
				MediaSource schemaSource = getService().getMedia(schemaLink.getHref());
				Helper.readStream(schemaSource.getInputStream(), fos);

				return "Downloaded formtype schema: " + file;
			} catch (Exception e) {
				return "Could not download formtype schema: " + e.getMessage();
			} finally {
				fos.close();
			}
		} catch (Exception e) {
			return "Could not download formtype schema: " + e.getMessage();
		}
	}
	
	@Command(name = "controls", description = "LIST form controls (e.g. 'controls')")
	public String controls() throws MalformedURLException, IOException,
			ServiceException {

		FormTypeEntry form = getEntry();
		ControlTypeFeed controls = form.getControlTypeFeed();

		return go(new ControlTypeFeedShell(controls));
	}

	@Command(name = "docs", description = "LIST form documents (e.g. 'docs')")
	public String doctypes() throws MalformedURLException, IOException,
			ServiceException {

		FormTypeEntry form = getEntry();
		DocumentTypeFeed feed = form.getDocumentTypeFeed();

		return go(new DocumentTypeFeedShell(form, feed));
	}

	@Command(name = "subs", description = "LIST form submissions (e.g. 'subs')")
	public String subs() throws MalformedURLException, IOException,
			ServiceException {

		FormTypeEntry form = getEntry();
		SubmissionFeed subs = form.getSubmissionFeed();

		return go(new SubmissionFeedShell(subs));
	}

	@Command(name = "down", description = "DOWNLOAD form (e.g. 'down')")
	public String download() throws IOException, ServiceException {
		String name = ApiHelper.getName(getEntry());
		String fileName = ApiHelper.nameToFileName(name);
		try {
			File file = File.createTempFile(fileName + "_", "_form.zip");
			FileOutputStream fos = new FileOutputStream(file);
			try {
				FormTypeEntry form = getEntry();
				MediaContent mc = (MediaContent) form.getContent();
				MediaSource ms = getService().getMedia(mc);
				Helper.readStream(ms.getInputStream(), fos);

				return "Downloaded formtype: " + file;
			} catch (Exception e) {
				return "Could not download formtype: " + e.getMessage();
			} finally {
				fos.close();
			}
		} catch (Exception e) {
			return "Could not download formtype: " + e.getMessage();
		}
	}

	@Override
	protected String print(FormTypeEntry entry) {
		return ApiHelper.print(getEntry());
	}

	private Map<String, Object> toFormData(String[] data) {
		Map<String, Object> d = new HashMap<String, Object>();
		if (data != null) {
			for (String c : data) {
				String[] pair = c.split(":|=");
				if (pair.length == 2)
					d.put(pair[0], pair[1]);
			}
		}
		return d;
	}

	private List<MediaSource> toMediaSources(String[] xmlDocPaths)
			throws FileNotFoundException {
		List<MediaSource> mss = new ArrayList<MediaSource>();
		if (xmlDocPaths != null) {
			for (int i = 0; i < xmlDocPaths.length; i++) {
				FileInputStream fis = new FileInputStream(xmlDocPaths[i]);
				MediaStreamSource ms = new MediaStreamSource(fis,
						"application/xml");
				ms.setName("Document" + i);
				mss.add(ms);
			}
		}
		return mss;
	}
}