package com.frevvo.forms.cli.shell;

import java.net.URL;
import java.util.concurrent.atomic.AtomicInteger;

import asg.cliche.Command;

import com.frevvo.forms.cli.core.BaseShell;
import com.frevvo.forms.client.FormTypeEntry;
import com.frevvo.forms.client.FormsService;
import com.frevvo.forms.client.Helper;

/**
 * Shell context for all commands related to form instances
 */
public class FormInstanceShell extends BaseShell {
	public static final String PATH_ELEMENT = "";

	static private AtomicInteger count = new AtomicInteger(0);

	private FormTypeEntry entry;
	private URL instanceUrl;
	private boolean submitted;

	public FormInstanceShell(FormTypeEntry entry, URL instanceUrl) {
		super(PATH_ELEMENT);
		this.entry = entry;
		this.instanceUrl = instanceUrl;
		count.incrementAndGet();
	}

	protected String getPathElement() {
		return "Instance #" + count.get();
	}

	public FormsService getService() {
		return (FormsService) getEntry().getService();
	}

	public FormTypeEntry getEntry() {
		return entry;
	}

	public URL getInstanceUrl() {
		return instanceUrl;
	}

	public boolean isSubmitted() {
		return submitted;
	}

	@Command(name = "submit", description = "SUBMIT this form instance (e.g. 'submit')")
	public String submit() {
		if (isSubmitted())
			return "This form instance was already submitted/cancelled";
		try {
			URL url = getInstanceUrl();
			Helper.submitInstance(getService(), url);
			submitted = true;
		} catch (Exception e) {
			return "Could not submit form instance : " + e.getMessage()
					+ "(url: " + getInstanceUrl() + ")";
		}
		return "Form instance submitted: " + getInstanceUrl();
	}

	@Command(name = "reset", description = "RESET this form instance (e.g. 'reset')")
	public String reset() {
		if (isSubmitted())
			return "This form instance was already submitted/cancelled";
		try {
			URL url = getInstanceUrl();
			Helper.cancelInstance(getService(), url);
			submitted = true;
		} catch (Exception e) {
			return "Could not cancel form instance : " + e.getMessage()
					+ "(url: " + getInstanceUrl() + ")";
		}
		return "Form instance cancelled: " + getInstanceUrl();
	}

	@Override
	@Command(name = "print", description = "PRINT this instance URL (e.g. 'print')")
	public String print() {
		FormTypeEntry form = getEntry();

		String formName = form.getTitle().getPlainText();

		return String.format("%s instance #%d [name=%s, url=%s]", getEntry()
				.getKind(), count.get(), formName, getInstanceUrl());
	}
}