package com.frevvo.forms.cli.shell;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;

import asg.cliche.Command;

import com.frevvo.forms.cli.ApiHelper;
import com.frevvo.forms.cli.core.FeedShell;
import com.frevvo.forms.client.ApplicationEntry;
import com.frevvo.forms.client.ApplicationFeed;
import com.frevvo.forms.client.Helper;
import com.google.gdata.data.MediaContent;
import com.google.gdata.data.media.MediaSource;
import com.google.gdata.data.media.MediaStreamSource;
import com.google.gdata.util.ServiceException;

/**
 * Shell context for all commands related to application feeds
 */
public class ApplicationFeedShell extends
		FeedShell<ApplicationFeed, ApplicationEntry, ApplicationEntryShell> {
	public static final String PATH_ELEMENT = "apps";

	public ApplicationFeedShell(ApplicationFeed feed) {
		super(PATH_ELEMENT, feed, ApplicationFeed.class);
	}

	@Command(name = "down", description = "DOWNLOAD ALL applications into temp folder (e.g.: 'download')")
	public String downloadAll() throws IOException, ServiceException {
		ApplicationFeed apps = getFeed();

		StringBuilder sb = new StringBuilder();
		for (ApplicationEntry app : apps.getEntries()) {
			String name = ApiHelper.getName(app);
			String fileName = ApiHelper.nameToFileName(name);
			try {
				File file = File.createTempFile(fileName + "_", "_app.zip");
				FileOutputStream fos = new FileOutputStream(file);
				try {
					MediaContent mc = (MediaContent) app.getContent();
					MediaSource ms = getService().getMedia(mc);

					Helper.readStream(ms.getInputStream(), fos);

					sb.append("\tDownloaded application " + name + ": " + file
							+ "\n");
				} catch (Exception e) {
					sb.append("\tCould not download application " + fileName
							+ ": " + e.getMessage() + "\n");
				} finally {
					fos.close();
				}
			} catch (Exception e) {
				sb.append("\tCould not download application " + fileName + ": "
						+ e.getMessage() + "\n");
			}
		}
		return "Downloaded " + apps.getEntries().size() + " application(s):\n"
				+ sb.toString();
	}

	@Command(name = "up", description = "UPLOAD an application (e.g. 'up /var/contacts_apps.zip')")
	public String upload(String filePath) throws IOException, ServiceException {
		File f = new File(filePath);
		if (!f.exists())
			return "File " + filePath + " doesnt exist";

		ApplicationFeed apps = getFeed();

		try {
			// upload application
			FileInputStream fis = new FileInputStream(f);
			try {
				MediaStreamSource ms = new MediaStreamSource(fis,
						"application/zip");
				ApplicationEntry entry = apps.insert(ms);

				go(createEntryShell(entry));
				return "Application " + ApiHelper.getName(entry)
						+ " successfully uploaded";
			} finally {
				fis.close();
			}
		} catch (ServiceException ex) {
			return "Could not upload application "
					+ filePath
					+ ": are you trying to upload an app with an id that already exists";
		} catch (Exception ex) {
			return "Could not upload application " + filePath + ": "
					+ ex.getMessage();
		}
	}

	@Override
	protected ApplicationEntryShell createEntryShell(ApplicationEntry entry) {
		return new ApplicationEntryShell(entry);
	}

	@Override
	protected ApplicationEntry createEntry(String name, String description)
			throws ServiceException, IOException {
		return ApiHelper.createApplicationEntry(getService(), getFeed(), name,
				description);
	}

	@Override
	protected String print(ApplicationFeed feed) {
		return ApiHelper.print(feed);
	}
}
