package com.frevvo.forms.cli.core;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import asg.cliche.Command;

import com.frevvo.forms.cli.ApiHelper;
import com.frevvo.forms.client.FormsService;
import com.google.gdata.data.BaseEntry;
import com.google.gdata.data.BaseFeed;
import com.google.gdata.data.Link;
import com.google.gdata.util.ServiceException;

public abstract class FeedShell<F extends BaseFeed<F, E>, E extends BaseEntry<E>, S extends EntryShell<E>>
		extends BaseShell {
	public static final int DEFAULT_MAX_RESULTS = 20;

	private F feed;
	private Class<F> feedClass;

	public FeedShell(String pathElement, F feed, Class<F> feedCls) {
		super(pathElement);
		this.feed = feed;
		this.feedClass = feedCls;
	}

	@Override
	@Command(description = "PRINT current feed (e.g.: 'print')")
	public final String print() {
		return print(getFeed());
	}

	@Command(description = "REFRESH current feed (e.g.: 'refresh')")
	public final String refresh() {
		try {
			F feed = getFeed();
			setFeed(feed.getSelf());
			return "\nFeed refreshed...\n" + print();
		} catch (Exception e) {
			return "Could not refresh feed: " + e.getMessage();
		}
	}

	@Command(description = "CREATE a new entry (e.g.: 'create {name}')")
	public String create(String name) {
		try {
			E entry = createEntry(name, null);
			if (entry == null)
				return "Create entry not supported for this feed";

			return go(createEntryShell(entry));
		} catch (Exception e) {
			return "Could not create new application : " + e.getMessage();
		}
	}

	@Command(description = "SELECT an entry (e.g.: 'entry {entry#}')")
	public String select(int i) throws IOException, ServiceException {
		// find entry in feed by index
		F feed = getFeed();
		E entry = ApiHelper.findByIndex(feed, i);
		if (entry == null)
			return "Could not find entry with index " + i;

		return go(createEntryShell(entry));
	}

	@Command(description = "DELETE an entry (e.g. 'delete {entry#}')")
	public String delete(int i) {
		F feed = getFeed();
		E entry = ApiHelper.findByIndex(feed, i);
		if (entry == null)
			return "Could not find entry with index " + i;

		try {
			entry.delete();

			return "Entry " + ApiHelper.getName(entry) + " deleted";
		} catch (IOException e) {
			return "Could not delete " + ApiHelper.getName(entry) + ": "
					+ e.getMessage();
		} catch (Exception e) {
			return "Could not delete " + ApiHelper.getName(entry) + ": "
					+ e.getMessage();
		}
	}

	@Command(description = "SHOW all feed links (e.g. 'links')")
	public String links() throws IOException {
		return ApiHelper.print(getFeed().getLinks());
	}

	@Command(description = "SHOW feed links for given rel attr (e.g. 'links {relAttr}')")
	public String links(String rel) throws IOException {
		F f = getFeed();
		return ApiHelper.print(f.getLinks(rel, null));
	}

	@Command(description = "OPEN a link in the browser (e.g. 'openLink {link#}')")
	public String link(int i) {
		if (i > getFeed().getLinks().size() - 1)
			return "Invalid link index " + i + ". Use index from 0 to "
					+ (getFeed().getLinks().size() - 1);

		Link l = openLink(getFeed().getLinks().get(i));

		return "Openning link: " + l.getHref();
	}

	@Command(description = "NEXT feed page (e.g. 'next')")
	public String next() {
		F feed;
		try {
			feed = ApiHelper.next(getService(), getFeed(), getFeedClass());
			if (feed == null)
				return "This is already the last page";
			setFeed(feed);
			return print(feed);
		} catch (Exception e) {
			return "Could not get the next feed page: " + e.getMessage();
		}
	}

	@Command(description = "PREVIOUS feed page (e.g. 'previous')")
	public String prev() {
		F feed;
		try {
			feed = ApiHelper.previous(getService(), getFeed(), getFeedClass());
			if (feed == null)
				return "This is already the first page";
			setFeed(feed);
			return print(feed);
		} catch (Exception e) {
			return "Could not get the next feed page: " + e.getMessage();
		}
	}

	protected F getFeed() {
		return feed;
	}

	protected Class<F> getFeedClass() {
		return feedClass;
	}

	protected F setFeed(F feed) {
		this.feed = feed;

		int index = cliGetShell().getPath().size() - 1;
		cliGetShell().getPath().set(index, getPathElement());

		return feed;
	}

	@Override
	protected List<String> getPathDetails() {
		List<String> details = new ArrayList<String>();
		// details.add("page=" + ApiHelper.getCurrentPage(getFeed()));
		// if (getQuery().getStartIndex() > 0)
		// details.add("start=" + getQuery().getStartIndex());
		// if (getQuery().getMaxResults() > 0)
		// details.add("max=" + getQuery().getMaxResults());
		// if (getQuery().getPublishedMin() != null)
		// details.add("publishedMin=" + getQuery().getPublishedMin());
		// if (getQuery().getPublishedMax() != null)
		// details.add("publishedMax=" + getQuery().getPublishedMax());
		// if (getQuery().getUpdatedMin() != null)
		// details.add("updatedMin=" + getQuery().getUpdatedMin());
		// if (getQuery().getUpdatedMax() != null)
		// details.add("updatedMax=" + getQuery().getUpdatedMax());
		// if (getQuery().getAuthor() != null)
		// details.add("author=" + getQuery().getAuthor());
		return details;
	}

	protected FormsService getService() {
		return (FormsService) getFeed().getService();
	}

	protected abstract String print(F feed);

	protected abstract E createEntry(String name, String description)
			throws IOException, ServiceException;

	protected abstract S createEntryShell(E entry);
}