package com.frevvo.forms.cli.core;

import java.io.IOException;
import java.util.Collections;
import java.util.List;

import asg.cliche.Command;

import com.frevvo.forms.cli.ApiHelper;
import com.frevvo.forms.client.FormsService;
import com.google.gdata.data.BaseEntry;
import com.google.gdata.data.Link;
import com.google.gdata.data.PlainTextConstruct;

public abstract class EntryShell<E extends BaseEntry<E>> extends BaseShell {
	private E entry;

	public EntryShell(String pathElement, E entry) {
		super(pathElement);
		this.entry = entry;
	}

	@Command(description="PRINT this entry (e.g. 'print')")
	@Override
	public final String print() {
		return print(getEntry());
	}

	@Command(description="REFRESH this entry (e.g. 'refresh')")
	public String refresh() {
		try {
			E entry = getEntry();
			entry = entry.getSelf();
			setEntry(entry);
			return "\nEntry refreshed...\n" + print();
		} catch (Exception e) {
			return "Could not refresh entry: " + e.getMessage();
		}
	}

	@Command(description="UPDATE this entry's name (e.g. 'name {newName}')")
	public String name(String newName) {
		try {
			E entry = getEntry();
			entry.setTitle(new PlainTextConstruct(newName));
			entry = entry.update();
			setEntry(entry);
			return "Entry's name updated";
		} catch (Exception e) {
			return "Could not update entry's name : " + e.getMessage();
		}
	}

	@Command(description="UPDATE this entry's description (e.g. 'descr {newDescription}')")
	public String descr(String newValue) {
		try {
			E entry = getEntry();
			entry.setSummary(new PlainTextConstruct(newValue));
			entry = entry.update();
			setEntry(entry);
			return "Entry's description updated";
		} catch (Exception e) {
			return "Could not update entry's description : " + e.getMessage();
		}
	}

	@Command(description="DELETE this entry (e.g. 'delete')")
	public String delete() {
		try {
			getEntry().delete();
			return "Entry " + ApiHelper.getName(getEntry()) + " deleted";
		} catch (IOException e) {
			return "Could not delete " + ApiHelper.getName(getEntry()) + ": "
					+ e.getMessage();
		} catch (Exception e) {
			return "Could not delete " + ApiHelper.getName(getEntry()) + ": "
					+ e.getMessage();
		}
	}

	@Command(description="SHOW all entry links (e.g. 'links')")
	public String links() throws IOException {
		return ApiHelper.print(getEntry().getLinks());
	}

	@Command(description="SHOW feed links for given rel attr (e.g. 'links {relAttr}')")
	public String links(String rel) throws IOException {
		E e = getEntry();
		return ApiHelper.print(e.getLinks(rel, null));
	}

	@Command(description="OPEN a link in the browser (e.g. 'link {link#}')")
	public String link(int i) {
		if (i < 1 || i > getEntry().getLinks().size())
			return "Invalid link index " + i + ". Use index from 1 to "
					+ (getEntry().getLinks().size());

		Link l = openLink(getEntry().getLinks().get(i - 1));

		return "Openning link: " + l.getHref();
	}

	protected E getEntry() {
		return entry;
	}

	protected void setEntry(E entry) {
		this.entry = entry;
	}

	protected FormsService getService() {
		return (FormsService) getEntry().getService();
	}

	@Override
	protected List<String> getPathDetails() {
		return Collections.singletonList(ApiHelper.getName(getEntry()));
	}

	protected abstract String print(E entry);
}