package com.frevvo.forms.cli;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.CommandLineParser;
import org.apache.commons.cli.HelpFormatter;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.ParseException;
import org.apache.commons.cli.PosixParser;

import asg.cliche.Command;
import asg.cliche.Shell;
import asg.cliche.ShellDependent;
import asg.cliche.ShellFactory;

import com.frevvo.forms.cli.util.Util;
import com.frevvo.forms.client.ApplicationFeed;
import com.frevvo.forms.client.FormTypeEntry;
import com.frevvo.forms.client.FormTypeFeed;
import com.frevvo.forms.client.FormsService;
import com.frevvo.forms.client.SubmissionEntry;
import com.frevvo.forms.client.SubmissionFeed;
import com.frevvo.forms.client.SubmissionQuery;
import com.frevvo.forms.client.UserEntry;
import com.google.gdata.client.Query.CategoryFilter;
import com.google.gdata.data.DateTime;
import com.google.gdata.data.Link;
import com.google.gdata.data.media.MediaSource;
import com.google.gdata.data.media.MediaStreamSource;
import com.google.gdata.util.ResourceNotFoundException;
import com.google.gdata.util.ServiceException;

public class Tutorial implements ShellDependent {
	static public final String SERVER_URL = "http://localhost:8082";
	static public final String APP_RESOURCE = "/Contacts_app.zip";
	static public final Pattern ORDER_BY_PATTERN = Pattern
			.compile("(\\$?\\w+)(\\s+(asc|desc))?");

	static public String getFormTypeId(String user) {
		return "_pkMVwBH8EeCl2et9BuDRPg!_lJ8_ERH8EeCl2et9BuDRPg!" + user;
	}

	public static void main(String[] args) throws IOException, ServiceException {
		Options options = new Options();
		options.addOption("s", "server", true, "serverl url (e.g. "
				+ SERVER_URL + ")");
		options.addOption("t", "tenant", true,
				"tenant (required, e.g. doe.com)");
		options.addOption("u", "user", true, "username");
		options.addOption("p", "password", true, "password");
		options.addOption("a", "appfile", true,
				"override the default application file to use (*_app.zip)");
		options.addOption("f", "formtype", true,
				"override the default formtype id/name to use");

		CommandLineParser parser = new PosixParser();
		try {
			CommandLine line = parser.parse(options, args);
			// server argument
			String server = line.getOptionValue('s');
			if (server == null || server.length() == 0) {
				showUsage(options, null);
			} else
				server = server.trim();

			String tenant = line.getOptionValue('t');
			if (tenant == null || tenant.length() == 0) {
				showUsage(options, null);
			} else
				tenant = tenant.trim();

			String user = line.getOptionValue('u');
			if (user == null || user.length() == 0) {
				showUsage(options, null);
			} else
				user = user.trim();

			String pwd = line.getOptionValue('p');
			if (pwd == null || pwd.length() == 0) {
				showUsage(options, null);
			} else
				pwd = pwd.trim();

			InputStream appStream = null;
			String appFile = line.getOptionValue('a');
			if (appFile == null || appFile.length() == 0) {
				appStream = Tutorial.class.getResourceAsStream(APP_RESOURCE);
				System.out.println("Using embedded Contacts application ...");
			} else {
				System.out.println("Using custom application " + appFile
						+ " ...");
				appStream = new FileInputStream(appFile);
			}

			String formTypeId = line.getOptionValue('f');
			if (formTypeId == null || formTypeId.length() == 0) {
				if (appFile != null && appFile.length() > 0) {
					System.out
							.println("You must provide a formtype id when providing your own application file");
					showUsage(options, null);
				}
				formTypeId = getFormTypeId(user);
			}

			URL serverUrl = new URL(server);
			new Tutorial(serverUrl.getProtocol(), serverUrl.getHost(),
					serverUrl.getPort(), tenant, user, pwd, formTypeId,
					appFile, appStream).commandLoop();
			System.out.println("done.");
		} catch (ParseException exp) {
			showUsage(options, exp);
		}
	}

	private static void showUsage(Options options, ParseException exp) {
		HelpFormatter formatter = new HelpFormatter();
		formatter.printHelp(Tutorial.class.getName(), options);
		System.exit(1);
	}

	private String tenant;
	private String formTypeId;
	private String protocol;
	private String host;
	private int port;
	private String username;
	private String password;

	private FormsService service;
	private Shell theShell;

	// query params
	private Integer startIndex;
	private Integer maxResults;
	private String orderBy;
	private String[] filter;
	private DateTime updatedMax;
	private DateTime updatedMin;
	private CategoryFilter categoryFilter;
	private InputStream appStream;
	private String appFile;

	public Tutorial(String protocol, String host, int port, String tenant,
			String user, String pwd, String formTypeId, String appFile,
			InputStream appStream) {
		this.protocol = protocol;
		this.host = host;
		this.port = port;
		this.tenant = tenant;
		this.username = user;
		this.password = pwd;
		this.formTypeId = formTypeId;
		this.appFile = appFile;
		this.appStream = appStream;
	}

	public String getProtocol() {
		return protocol;
	}

	public String getHost() {
		return host;
	}

	public int getPort() {
		return port;
	}

	public String getTenant() {
		return tenant;
	}

	public String getUsername() {
		return username;
	}

	public String getPassword() {
		return password;
	}

	public String getContactFormId() {
		return formTypeId;
	}

	public String getAppFile() {
		return appFile;
	}

	public InputStream getAppStream() {
		return appStream;
	}

	public String[] getFilter() {
		return filter;
	}

	public void setFilter(String... filter) {
		this.filter = filter;
	}

	public String getOrderBy() {
		return orderBy;
	}

	public void setOrderBy(String orderBy) {
		this.orderBy = orderBy;
	}

	public FormsService getService() {
		return service;
	}

	public void cliSetShell(Shell theShell) {
		this.theShell = theShell;
	}

	@Command(header = "List all existing submissions...")
	public String list() {
		try {
			FormTypeEntry contactForm = getContactForm(getContactFormId());
			SubmissionFeed contacts = getSubmissions(contactForm
					.getSubmissionFeedLink());
			return ApiHelper.print(contacts,this.maxResults);
		} catch (Exception e) {
			return "Could not get the submissions: " + e.getMessage();
		}
	}

	@Command
	public String create() {
		try {
			FormTypeEntry contactForm = getContactForm(getContactFormId());
			Link contactFormLink = contactForm.getFormTypePopupLink(null);
			Util.openURL(contactFormLink.getHref());
			return "Opening add contact form...";
		} catch (Exception e) {
			return "Could not create the submissions: " + e.getMessage();
		}
	}

	@Command
	public String createFromXml(String... xmlFiles) {
		try {
			FormTypeEntry contactForm = getContactForm(getContactFormId());
			List<MediaSource> mss = new ArrayList<MediaSource>();
			try {
				if (xmlFiles != null) {
					for (String xmlFile : xmlFiles) {
						if (!new File(xmlFile).exists())
							System.out.println("The xml file " + xmlFile
									+ "doesnt exist. Ignoring...");
						InputStream is = new FileInputStream(xmlFile);
						MediaStreamSource ms = new MediaStreamSource(is,
								"application/xml");
						ms.setName("contactxml");
						mss.add(ms);
					}
				}

				URL form = contactForm.createFormInstance(null, mss);
				Util.openURL(form.toString());
				return "Opening contact form initialized with xml document(s) ...";
			} finally {
				for (MediaSource ms : mss) {
					ms.getInputStream().close();
				}
			}
		} catch (Exception e) {
			return "Instantiate form contact form: " + e.getMessage();
		}
	}

	@Command
	public String view(int i) {
		try {
			FormTypeEntry contactForm = getContactForm(getContactFormId());
			SubmissionFeed contacts = getSubmissions(contactForm
					.getSubmissionFeedLink());

			if (i < 1 || i > contacts.getEntries().size())
				return "Invalid contact #: " + i;

			// Get the contact by index
			SubmissionEntry contact = contacts.getEntries().get(i - 1);

			// open a readonly form in the browser
			Map<String, Object> params = new HashMap<String, Object>();
			params.put(FormTypeEntry.FORMTYPE_READONLY_PARAMETER, Boolean.TRUE);

			Link contactFormLink = contact.getFormTypeLink(params);
			Util.openURL(contactFormLink.getHref());
			return "Viewing contact #" + i + " ...";
		} catch (Exception e) {
			return "Could not view the contact #" + i + ": " + e.getMessage();
		}
	}

	@Command
	public String viewPdf(int i) {
		try {
			FormTypeEntry contactForm = getContactForm(getContactFormId());
			SubmissionFeed contacts = getSubmissions(contactForm
					.getSubmissionFeedLink());

			if (i < 1 || i > contacts.getEntries().size())
				return "Invalid contact #: " + i;

			// Get the contact by index
			SubmissionEntry contact = contacts.getEntries().get(i - 1);

			// open the PDF in the browser
			Link contactPdfLink = ApiHelper.getSubmissionSnapshotLink(contact);

			if (contactPdfLink == null)
				return "Contact #" + i + " has no PDF snapshot";
			Util.openURL(contactPdfLink.getHref());
			return "Viewing contact #" + i + " PDF ...";
		} catch (Exception e) {
			return "Could not view PDF for the contact #" + i + ": "
					+ e.getMessage();
		}
	}

	@Command
	public String viewXsd() {
		try {
			FormTypeEntry contactForm = getContactForm(getContactFormId());
			Link contactXsd = contactForm.getFormTypeSchemaLink();
			if (contactXsd != null && contactXsd.getHref() != null) {
				Util.openURL(contactXsd.getHref());
			}
			return "Viewing XML Schema for form "
					+ ApiHelper.getName(contactForm) + " [rel="
					+ contactXsd.getRel() + ",type=" + contactXsd.getType()
					+ ",href=" + contactXsd.getHref() + "] ...\n";
		} catch (Exception e) {
			return "Could not view XML Schema for form: " + e.getMessage();
		}
	}

	@Command
	public String viewXml(int i) {
		try {
			FormTypeEntry contactForm = getContactForm(getContactFormId());
			SubmissionFeed contacts = getSubmissions(contactForm
					.getSubmissionFeedLink());

			if (i < 1 || i > contacts.getEntries().size())
				return "Invalid contact #: " + i;

			// Get the contact by index
			SubmissionEntry contact = contacts.getEntries().get(i - 1);

			List<Link> docs = contact.getDocumentLinks("text/xml");
			if (docs == null || docs.size() == 0)
				return "Contact #" + i + " has no XML document";

			StringBuffer sb = new StringBuffer();
			for (Link doc : docs) {
				// open the document in the browser
				Util.openURL(doc.getHref());
				// or download it
				// MediaSource source = getService().getMedia(doc.getHref());
				// saveXML(source.getInputStream());

				sb.append("Viewing document " + docs.indexOf(doc) + " [rel="
						+ doc.getRel() + ",type=" + doc.getType() + ",href="
						+ doc.getHref() + "] ...\n");
			}
			return sb.toString();
		} catch (Exception e) {
			return "Could not view PDF for the contact #" + i + ": "
					+ e.getMessage();
		}
	}

	@Command
	public String edit(int i) {
		try {
			FormTypeEntry contactForm = getContactForm(getContactFormId());
			SubmissionFeed contacts = getSubmissions(contactForm
					.getSubmissionFeedLink());

			if (i < 1 || i > contacts.getEntries().size())
				return "Invalid contact #: " + i;

			SubmissionEntry contact = contacts.getEntries().get(i - 1);

			Map<String, Object> params = new HashMap<String, Object>();
			params.put(FormTypeEntry.FORMTYPE_READONLY_PARAMETER, Boolean.FALSE);

			Link contactFormLink = contact.getFormTypeLink(params);
			Util.openURL(contactFormLink.getHref());

			return "Editing contact #" + i + " ...";
		} catch (Exception e) {
			return "Could not edit the contacts: " + e.getMessage();
		}
	}

	@Command(abbrev = "d", description = "Delete a given contact: e.g. delete 1")
	public String delete(int i) {
		try {
			FormTypeEntry contactForm = getContactForm(getContactFormId());
			SubmissionFeed contacts = getSubmissions(contactForm
					.getSubmissionFeedLink());

			if (i < 1 || i > contacts.getEntries().size())
				return "Invalid contact #: " + i;

			SubmissionEntry contact = contacts.getEntries().get(i - 1);

			contact.delete();

			return "Deleted contact " + contact.getId() + "\n" + list();
		} catch (Exception e) {
			return "Could not edit the contacts: " + e.getMessage();
		}
	}

	@Command(description = "Sets the start index (pagination) when using positive value, unsets when negative. E.g. start-index 10 to set or start-index -1 to unset")
	public String startIndex(Integer value) {
		if (value == null || value <= 0) {
			value = null;
		}
		this.startIndex = value;

		updatePrompt();
		return "start-index query param updated to " + value + " ...\n"
				+ list();
	}

	@Command(description = "Sets the max results (pagination) when using positive value, unsets when nsegative. E.g. max-results 10 to set or max-results -1 to unset (note the server may override this)")
	public String maxResults(Integer value) {
		if (value == null || value < 0)
			value = null;

		this.maxResults = value;

		updatePrompt();
		return "max-results query param updated to " + value
				+ " (note that the server may override this) ...\n" + list();
	}

	@Command(description = "Set the updated-max query param, i.e. return all contacts submitted at most at this date (use the format seen in the results from the list command)")
	public String updatedMax(String value) {
		try {
			if (value == null || value.length() == 0)
				this.updatedMax = null;
			else
				this.updatedMax = DateTime.parseDateTimeChoice(value);
			updatePrompt();
			return "updated-max query param updated to " + this.updatedMax
					+ " ...\n" + list();
		} catch (NumberFormatException e) {
			return "Invalid date format: " + e.getMessage();
		}
	}

	@Command(description = "Set the updated-min query param, i.e. return all contacts submitted at least at from this date (use the format seen in the results from the list command)")
	public String updatedMin(String value) {
		try {
			if (value == null || value.length() == 0)
				this.updatedMin = null;
			else
				this.updatedMin = DateTime.parseDateTimeChoice(value);
			updatePrompt();
			return "updated-min query param updated to " + this.updatedMin
					+ " ...\n" + list();
		} catch (NumberFormatException e) {
			return "Invalid date format: " + e.getMessage();
		}
	}

	@Command
	public String filter(String... filter) {
		if (filter != null
				&& (filter.length == 0 || filter.length == 1
						&& filter[0].length() == 0))
			filter = null;
		setFilter(filter);
		updatePrompt();
		return list();
	}

	@Command(description = "")
	public String orderBy(String orderBy) {
		if (orderBy != null && orderBy.length() == 0)
			orderBy = null;
		else {
			Matcher m = ORDER_BY_PATTERN.matcher(orderBy);
			if (!m.matches())
				return "Invalid order by syntax. Expecting: <field> (asc|desc)?";
		}
		setOrderBy(orderBy);
		updatePrompt();
		return list();
	}

	/**
	 * This is the main entry point. We have the connection details (protocol,
	 * host, port, tenant, user and password and will establish a session with
	 * frevvo.
	 *
	 * @throws IOException
	 * @throws ServiceException
	 */
	private void commandLoop() throws IOException, ServiceException {
		// create the FormsService to hold the session with frevvo
		System.out.println("Connecting to tenant " + getTenant() + " at "
				+ getHost() + ":" + getPort() + " ...");
		FormsService s = new FormsService(getProtocol(), getHost(), getPort(),
				null);
		try {
			// login to frevvo
			Map<String, String> params = new HashMap<String,String>();
			params.put("autoLogin", "true");
			s.loginAs(getUsername(), getUsername() + '@' + getTenant(), getPassword(), false, null, null, null, null, params);

//			s.login(getUsername() + '@' + getTenant(), getPassword());

			// save the service to be used in subsequent interactions
			this.service = s;

			// Auto-upload the contact application, if not already there
			if (getContactForm(getContactFormId()) == null) {
				uploadContactApplication(s);
			}

			// start the interactive shell
			ShellFactory.createConsoleShell(getPrompt(), null, this)
					.commandLoop();
		} finally {
			this.service = null;
			s.logout();
		}
	}

	private void uploadContactApplication(FormsService s) throws IOException,
			ServiceException {
		if (appFile == null)
			System.out
					.println("Contacts application doesnt exist. Uploading ... ");
		else
			System.out
					.println("Application doesnt exist. Uploading application "
							+ appFile + " ...");

		// oops the contacts app doesnt exist, let's upload it
		UserEntry user = ApiHelper.getUserEntry(s);
		ApplicationFeed apps = user.getApplicationFeed();
		InputStream is = getAppStream();
		if (is == null)
			throw new IOException("Could not find application file ");

		try {
			apps.uploadApplication(is, true,false);
		} finally {
			System.out.println("Application uploaded!");
			is.close();
		}
	}

	/**
	 * Get the Contact FormTypeEntry by ID
	 *
	 * @param formTypeId
	 * @return
	 * @throws IOException
	 * @throws ServiceException
	 */
	protected FormTypeEntry getContactForm(String formTypeId)
			throws IOException, ServiceException {
		FormsService s = getService();
		try {
			URL url = s.getEntryURL(FormTypeEntry.class, formTypeId);
			return s.getEntry(url, FormTypeEntry.class);
		} catch (ResourceNotFoundException e) {
			// let's try by name
			return ApiHelper.getFormTypeByName(getService(), formTypeId);
		}
	}

	/**
	 * An alternate, less efficient and more error prone way to find the Contact
	 * Form by name
	 *
	 * @param formTypeName
	 * @return
	 * @throws IOException
	 * @throws ServiceException
	 */
	protected FormTypeEntry getContactForm2(String formTypeName)
			throws IOException, ServiceException {
		FormsService s = getService();
		try {
			URL formsUrl = s.getFeedURL(FormTypeFeed.class);
			FormTypeFeed forms = s.getFeed(formsUrl, FormTypeFeed.class);
			for (FormTypeEntry form : forms.getEntries()) {
				if (formTypeName.equals(form.getTitle().getPlainText()))
					return form;
			}
			return null;
		} catch (ResourceNotFoundException e) {
			return null;
		}
	}

	private String getPrompt() {
		String prompt = getProtocol() + "://" + getUsername() + "@" + getHost()
				+ ":" + getPort() + "/" + tenant;
		if (startIndex != null)
			prompt += "[start-index=" + startIndex + "]";
		if (maxResults != null)
			prompt += "[max-results=" + maxResults + "]";
		if (updatedMin != null)
			prompt += "[updated-min=" + updatedMin + "]";
		if (updatedMax != null)
			prompt += "[updated-max=" + updatedMax + "]";
		if (orderBy != null)
			prompt += "[orderBy=" + orderBy + "]";
		if (filter != null) {
			for (String f : filter) {
				prompt += "[filter=" + f + "]";
			}
		}
		return prompt;
	}

	private SubmissionFeed getSubmissions(Link contactsLink)
			throws IOException, ServiceException {
		URL contactsUrl = new URL(contactsLink.getHref());
		SubmissionQuery q = new SubmissionQuery(contactsUrl);
		if (getFilter() != null)
			q.setFilter(getFilter());
		if (getOrderBy() != null)
			q.setOrderby(getOrderBy());
		if (startIndex != null && startIndex >= 1)
			q.setStartIndex(startIndex);
		if (maxResults != null && maxResults >= 0)
			q.setMaxResults(maxResults);
		if (updatedMax != null)
			q.setUpdatedMax(updatedMax);
		if (updatedMin != null)
			q.setUpdatedMin(updatedMin);
		return getService().getFeed(q, SubmissionFeed.class);
	}

	private void updatePrompt() {
		theShell.getPath().set(theShell.getPath().size() - 1, getPrompt());
	}
}
