package com.frevvo.forms.cli;

import java.io.DataInputStream;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;

import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.CommandLineParser;
import org.apache.commons.cli.DefaultParser;
import org.apache.commons.cli.HelpFormatter;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.ParseException;

import com.frevvo.forms.cli.core.BaseShell;
import com.frevvo.forms.cli.shell.TaskFeedShell;
import com.frevvo.forms.cli.shell.UserEntryShell;
import com.frevvo.forms.client.FormsService;
import com.frevvo.forms.client.TaskFeed;
import com.frevvo.forms.client.TaskQuery;
import com.frevvo.forms.client.UserEntry;
import com.frevvo.forms.client.util.AutoLoginUserInfo;
import com.google.gdata.util.ResourceNotFoundException;
import com.google.gdata.util.ServiceException;

import asg.cliche.Command;
import asg.cliche.Param;
import asg.cliche.Shell;
import asg.cliche.ShellFactory;

public class Main extends BaseShell {
	static public final String SERVER_URL = "http://localhost:8082";

	public static void main(String[] args) throws IOException,
			ServiceException, ParseException {
		Options options = new Options();
		options.addOption("s", "server", true, "serverl url (e.g. "
				+ SERVER_URL + ")");
		options.addOption("t", "tenant", true,
				"tenant (optional, e.g. doe.com)");
		// options.addOption("d", "download", true,
		// "download path used as the base folder for downloads");
		// options.addOption("u", "user", true, "username (e.g. john)");
		// options.addOption("p", "password", true, "password");

		CommandLineParser parser = new DefaultParser();
		try {
			CommandLine line = parser.parse(options, args);
			// server argument
			String server = line.getOptionValue('s');
			if (server == null || server.length() == 0) {
				showUsage(options, null);
			} else
				server = server.trim();

			String tenant = line.getOptionValue('t');

			URL serverUrl = new URL(server);

			ShellFactory.createConsoleShell(server, null,
					new Main(serverUrl, tenant)).commandLoop();
			System.out.println("done.");
		} catch (ParseException exp) {
			showUsage(options, exp);
		}
	}

	private static String readPassword(String prompt) throws IOException {
		DataInputStream in = new DataInputStream(System.in);
		System.out.print(prompt);
		return in.readLine();
	}

	private static void showUsage(Options options, ParseException exp) {
		HelpFormatter formatter = new HelpFormatter();
		formatter.printHelp(Main.class.getName(), options);
		System.exit(1);
	}

	private URL serverUrl;
	private String defaultTenant;
	private Shell shell;

	public Main(URL serverUrl, String defaultTenant) {
		super("");
		this.serverUrl = serverUrl;
		this.defaultTenant = defaultTenant;
	}

	public URL getServerUrl() {
		return serverUrl;
	}

	public String getDefaultTenant() {
		if( defaultTenant == null || defaultTenant.length() == 0 )
			return "d";
		return defaultTenant;
	}

	public void cliSetShell(Shell s) {
		this.shell = s;
	}

	public Shell cliGetShell() {
		return this.shell;
	}

	public void cliEnterLoop() {
	}

	public void cliLeaveLoop() {
	}

	@Command
	@Override
	public String print() {
		StringBuilder sb = new StringBuilder();
		sb.append(String.format("%20s | %-60s%n", "SERVER", getServerUrl()));
		sb.append(String.format("%20s | %-60s%n", "TENANT", getDefaultTenant()));
		return sb.toString();
	}

	@Command(name = "login", description = "")
	public String login(@Param(name = "username") String username) {
		return login(username, null);
	}

	@Command(name = "login", description = "")
	public String login(@Param(name = "username") String username, @Param(name = "password") String password) {
		return loginAs(username, username, password);
	}

	@Command
	public String loginAs(@Param(name = "username") String username,
			@Param(name = "tenantAdmin") String tenantAdmin) {
		return loginAs(username, tenantAdmin, null);
	}
	@Command
	public String loginAs(@Param(name = "username") String username,
			@Param(name = "tenantAdmin") String tenantAdmin,
			@Param(name = "tenantAdminPassword") String tenantAdminPassword) {
		return loginAs(username, tenantAdmin, tenantAdminPassword, false, null);
	}

	@Command
	public String loginAs(@Param(name = "username") String username,
			@Param(name = "tenantAdmin") String tenantAdmin,
			@Param(name = "tenantAdminPassword") String tenantAdminPassword, 
			@Param(name="virtual") boolean virtual) {
		return loginAs(username, tenantAdmin, tenantAdminPassword, virtual, null);
	}
	
	@Command
	public String loginAs(@Param(name = "username") String username,
			@Param(name = "tenantAdmin") String tenantAdmin,
			@Param(name = "tenantAdminPassword") String tenantAdminPassword, 
			@Param(name="virtual") boolean virtual, 
			@Param(name="roles") String roles) {
		username = getUsername(username);
		String tenant = getTenant(tenantAdmin);
		if (tenant == null)
			tenant = getDefaultTenant();
		tenantAdmin = getUsername(tenantAdmin) + "@" + tenant;

		FormsService fs = null;
		try {
			fs = createService(serverUrl);

			// read password
			if (tenantAdminPassword == null
					|| tenantAdminPassword.length() == 0)
				tenantAdminPassword = readPassword(tenantAdmin
						+ "'s password: ");
			
			if( roles == null )
				roles = "";
			
			Map<String, String> params = new HashMap<String, String>(1);
			params.put("autoLogin", "true");
			AutoLoginUserInfo ui = fs.loginAs(username, tenantAdmin, tenantAdminPassword, virtual, Arrays.asList(roles.split(",| ")), null, null, null, params);

			try {
				URL userUrl = fs.getEntryURL(UserEntry.class, username);
				UserEntry user = fs.getEntry(userUrl, UserEntry.class);
				return go(new UserEntryShell(user, username + '@' + tenant, ui));
			}
			catch(ResourceNotFoundException e){
				URL tasksUrl = fs.getFeedURL(TaskFeed.class);
				TaskQuery tasksQuery = new TaskQuery(tasksUrl);
				TaskFeed tasks = fs.getFeed(tasksQuery, TaskFeed.class);

				URL feedUrl = new URL(tasks.getSelfLink().getHref());
				tasksQuery = new TaskQuery(feedUrl);

				return go(new TaskFeedShell(tasks, tasksQuery, username + '@' + tenant, ui));
			}
		} catch (Exception e) {
			if( fs != null )
				try {
					fs.logout();
				} catch (ServiceException e1) {
					// do nothing
				}
			return "Could not connect to " + serverUrl + ": " + e.getMessage();
		}
	}

	protected FormsService createService(URL serverUrl)
			throws MalformedURLException {
		return new FormsService(serverUrl.getProtocol(), serverUrl.getHost(),
				serverUrl.getPort(), null);
	}

	protected String getTenant(String userTenant) {
		if (userTenant.contains("@")) {
			if (userTenant.split("@").length > 1
					&& userTenant.split("@")[1].length() > 0)
				return userTenant.split("@")[1].trim();
		}
		return null;
	}

	protected String getUsername(String userTenant) {
		if (userTenant.contains("@")) {
			userTenant = userTenant.split("@")[0];
		}
		return userTenant.trim();
	}
}
