package com.frevvo.forms.cli;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import com.frevvo.forms.client.ApplicationEntry;
import com.frevvo.forms.client.ApplicationFeed;
import com.frevvo.forms.client.ControlTypeEntry;
import com.frevvo.forms.client.ControlTypeEntry.Option;
import com.frevvo.forms.client.ControlTypeFeed;
import com.frevvo.forms.client.DocumentTypeFeed;
import com.frevvo.forms.client.FormTypeEntry;
import com.frevvo.forms.client.FormTypeFeed;
import com.frevvo.forms.client.FormTypeQuery;
import com.frevvo.forms.client.FormsService;
import com.frevvo.forms.client.Helper;
import com.frevvo.forms.client.SchemaEntry;
import com.frevvo.forms.client.SchemaFeed;
import com.frevvo.forms.client.SubmissionEntry;
import com.frevvo.forms.client.SubmissionFeed;
import com.frevvo.forms.client.SubmissionQuery;
import com.frevvo.forms.client.TaskEntry;
import com.frevvo.forms.client.TaskFeed;
import com.frevvo.forms.client.TaskQuery;
import com.frevvo.forms.client.UserEntry;
import com.frevvo.forms.client.ext.Key;
import com.google.gdata.client.Query;
import com.google.gdata.client.Query.CategoryFilter;
import com.google.gdata.data.BaseEntry;
import com.google.gdata.data.BaseFeed;
import com.google.gdata.data.Category;
import com.google.gdata.data.ILink;
import com.google.gdata.data.Link;
import com.google.gdata.data.MediaContent;
import com.google.gdata.data.PlainTextConstruct;
import com.google.gdata.data.media.MediaSource;
import com.google.gdata.data.media.MediaStreamSource;
import com.google.gdata.util.ResourceNotFoundException;
import com.google.gdata.util.ServiceException;

public class ApiHelper {
	/*
	 * ************************************************************************
	 *
	 * ************************************************************************
	 */
	static public <T extends BaseEntry<T>> String getName(T entry) {
		if (entry.getTitle() != null)
			return entry.getTitle().getPlainText();
		return null;
	}

	static public <T extends BaseEntry<T>> String getDescription(T entry) {
		if (entry.getSummary() != null)
			return entry.getSummary().getPlainText();
		return null;
	}

	/*
	 * ************************************************************************
	 * PRINT ENTRY
	 * ************************************************************************
	 */
	static private <T extends BaseEntry<T>> StringBuffer print(StringBuffer sb,
			T e) {
		sb.append(String.format("%20s | %-60s%n", "ID", e.getId()));
		if (e.getSelfLink() != null)
			sb.append(String.format("%20s | %60s%n", "URL", e.getSelfLink()
					.getHref()));
		if (e.getPublished() != null)
			sb.append(String.format("%20s | %-60s%n", "PUBLISHED",
					e.getPublished()));
		sb.append(String.format("%20s | %-60s%n", "UPDATED", e.getUpdated()));
		sb.append(String.format("%20s | %-60s%n", "NAME", getName(e)));
		sb.append(String.format("%20s | %-60s%n", "DESCRIPTION",
				getDescription(e)));
		return sb;
	}

	static public <T extends BaseEntry<T>> String print(T e) {
		return print(new StringBuffer(), e).toString();
	}

	static public String print(SubmissionEntry e) {
		StringBuffer sb = print(new StringBuffer(), e);
		sb.append(String.format("%20s | %-60s%n", "KIND", e.getKind()));
		sb.append(String.format("%20s | %-60s%n", "STATE", e.getState()));
		sb.append(String.format("%20s | %-60s%n", "ERROR", e.isError()));
		sb.append(String.format("%20s | %-60s%n", "ERROR DESCR.",
				e.getErrorDescription()));
		if (e.getKeys().iterator().hasNext()) {
			sb.append(String.format("%20s | %-20s%-20s%-20s%n", "KEYS", "ID",
					"NAME", "VALUE"));
			for (Iterator<Key> i = e.getKeys().iterator(); i.hasNext();) {
				Key k = i.next();
				sb.append(String.format("%20s | %-20s%-20s%-20s%n", k.getId(),
						k.getName(), k.getValue()));
			}
		}
		return sb.toString();
	}

	static public String print(FormTypeEntry e) {
		StringBuffer sb = new StringBuffer();
		print(sb, e);
		sb.append(String.format("%20s | %-60s%n", "KIND", e.getKind()));
		sb.append(String.format("%20s | %-60s%n", "READONLY", e.isReadOnly()));
		sb.append(String.format("%20s | %-60s%n", "VISIBILITY",
				e.getVisibility()));
		if (e.getKeys().iterator().hasNext()) {
			sb.append(String.format("%20s | %-20s%-20s%-20s%n", "KEYS", "ID",
					"NAME", "VALUE"));
			for (Iterator<Key> i = e.getKeys().iterator(); i.hasNext();) {
				Key k = i.next();
				sb.append(String.format("%20s | %-20s%-20s%-20s%n", k.getId(),
						k.getName(), k.getValue()));
			}
		}
		return sb.toString();
	}

	static public String print(ControlTypeEntry e) {
		StringBuffer sb = new StringBuffer();
		print(sb, e);
		sb.append(String.format("%20s | %-60s%n", "CONTROL TYPE",
				e.getControlType()));
		sb.append(String.format("%20s | %-60s%n", "DISPLAY TYPE",
				e.getDisplayType()));
		sb.append(String.format("%20s | %-60s%n", "READONLY", e.isReadonly()));
		sb.append(String.format("%20s | %-60s%n", "REQUIRED", e.isRequired()));
		sb.append(String.format("%20s | %-60s%n", "COMMENTS?",
				e.isCommentEnabled()));
		if (e.getOptions().size() > 0) {
			sb.append(String.format("%20s | %-60s%n", "OPTIONS", ""));
			for (Option o : e.getOptions()) {
				sb.append(String.format("%20s | %-60s%n", "", o));
			}
		}
		return sb.toString();
	}

	static public String print(ApplicationEntry e) {
		StringBuffer sb = new StringBuffer();
		print(sb, e);
		return sb.toString();
	}

	static public String print(UserEntry e) {
		StringBuffer sb = new StringBuffer();
		print(sb, e);
		return sb.toString();
	}

	/*
	 * ************************************************************************
	 * PRINT FEED
	 * ************************************************************************
	 */
	static public <F extends BaseFeed<F, E>, E extends BaseEntry<E>> String print(
			F feed) {
		StringBuffer sb = new StringBuffer();
		if (feed.getSelfLink() != null)
			sb.append(String.format("%3s ! %-60s%n", "URL", feed.getSelfLink()
					.getHref()));
		sb.append(String.format("%3s | %-40s | %-40s%n", "#", "NAME",
				"DESCRIPTION"));
		for (int i = 0; i < feed.getEntries().size(); i++) {
			E e = feed.getEntries().get(i);
			sb.append(String.format("%3d | %-40s | %-40s%n", i + 1, getName(e),
					getDescription(e)));
		}
		return sb.toString();
	}
	
	
	static public String print(SchemaFeed feed) throws MalformedURLException, IOException, ServiceException {
		StringBuffer sb = new StringBuffer();
		if (feed.getSelfLink() != null)
			sb.append(String.format("%3s ! %-60s%n", "URL", feed.getSelfLink()
					.getHref()));
		sb.append(String.format("%3s | %-40s | %-40s | %-80s%n", "#", "NAME",
				"DESCRIPTION", "NAMESPACE"));
		for (int i = 0; i < feed.getEntries().size(); i++) {
			SchemaEntry e = feed.getEntries().get(i);
			sb.append(String.format("%3d | %-40s | %-40s | %-80s%n", i + 1, getName(e),
					getDescription(e), getNameSpace(e)));
		}
		return sb.toString();
	}
	
	static public String print(SchemaEntry entry) throws MalformedURLException, IOException, ServiceException  {
		StringBuffer sb = new StringBuffer();
		sb.append(String.format("%20s | %-60s%n", "ID", entry.getId()));
		if (entry.getSelfLink() != null)
			sb.append(String.format("%20s | %60s%n", "URL", entry.getSelfLink()
					.getHref()));
		if (entry.getPublished() != null)
			sb.append(String.format("%20s | %-60s%n", "PUBLISHED",
					entry.getPublished()));
		sb.append(String.format("%20s | %-60s%n", "UPDATED", entry.getUpdated()));
		sb.append(String.format("%20s | %-60s%n", "NAME", getName(entry)));
		sb.append(String.format("%20s | %-60s%n", "DESCRIPTION",
				getDescription(entry)));
		sb.append(String.format("%20s | %-60s%n", "NAMESPACE",
				getNameSpace(entry)));
		return sb.toString();
	}

	static public String print(SubmissionFeed feed) throws ServiceException {
		return print(feed,null);
	}
	static public String print(SubmissionFeed feed, Integer maxRecordsToPrinted) throws ServiceException {
		StringBuffer sb = new StringBuffer();
		// print self link
		if (feed.getSelfLink() != null)
			sb.append(String.format("%3s ! %-60s%n", "URL", feed.getSelfLink()
					.getHref()));

		// collect the header columns (good enough for small data sets)
		sb.append(String.format(
				"%3s | %-23s | %-23s | %-23s | %-6s | %-10s | %4s", "#", "ID",
				"CREATED", "UPDATED", "KIND", "STATE", "PDF?"));
		List<String> columns = new ArrayList<String>();
		for (SubmissionFeed page = feed; page != null
				&& page.getEntries().size() > 0; page = page.getNext()) {
			for (int i = 0; i < page.getEntries().size(); i++) {
				SubmissionEntry e = page.getEntries().get(i);
				for (Key k : e.getKeys()) {
					if (!columns.contains(k.getName())) {
						sb.append(String.format(" | %-15s ", k.getName()
								.toUpperCase()));
						columns.add(k.getName());
					}
				}
			}
		}
		sb.append("\n");
		int entriesPrinted = 0;
		// generate the rows
		int count = 0;
		for (SubmissionFeed page = feed; page != null
				&& page.getEntries().size() > 0; page = page.getNext()) {
			
			for (int i = 0; i < page.getEntries().size(); i++) {
				 if(maxRecordsToPrinted != null && ++entriesPrinted > maxRecordsToPrinted) {
					 break;
				 }
				SubmissionEntry e = page.getEntries().get(i);
				boolean hasSnapshot = ApiHelper.getSubmissionSnapshotLink(e) != null;

				sb.append(String.format(
						"%3d | %-23s | %-23s | %-23s | %-6s | %-10s | %4s",
						++count, e.getId(), e.getPublished(), e.getUpdated(),
						e.getKind(), e.getState(), hasSnapshot));
				for (int j = 0; j < columns.size(); j++) {
					String col = columns.get(j);
					for (Key k : e.getKeys()) {
						String value = "";
						if (col.equals(k.getName())) {
							value = k.getValue().substring(0,
									Math.min(k.getValue().length(), 20));
							sb.append(String.format(" | %-15s ", value));
						}
					}
				}
				sb.append("\n");
			}
		}
		return sb.toString();
	}

	static public String print(TaskFeed feed) {
		StringBuffer sb = new StringBuffer();
		if (feed.getSelfLink() != null)
			sb.append(String.format("%3s ! %-60s%n", "URL", feed.getSelfLink()
					.getHref()));
		sb.append(String.format("%3s | %-23s | %-23s | %-23s | %-6s | %-10s | %-6s",
				"#", "ID", "CREATED", "UPDATED", "KIND", "STATE", "NAME")).append("\n");
		for (int i = 0; i < feed.getEntries().size(); i++) {
			TaskEntry e = feed.getEntries().get(i);
			sb.append(String.format(
					"%3d | %-23s | %-23s | %-23s | %-6s | %-10s | %-6s", i + 1, e.getId(),
					e.getPublished(), e.getUpdated(), e.getKind(), e.getState(), getName(e))).append("\n");
		}
		return sb.toString();
	}

	static public String print(FormTypeFeed feed) {
		StringBuffer sb = new StringBuffer();
		if (feed.getSelfLink() != null)
			sb.append(String.format("%3s ! %-60s%n", "URL", feed.getSelfLink()
					.getHref()));
		sb.append(String.format("%3s | %-30s | %-6s | %-8s | %-15s | %-40s%n",
				"#", "UPDATED", "KIND", "READONLY", "VISIBILITY", "NAME"));
		for (int i = 0; i < feed.getEntries().size(); i++) {
			FormTypeEntry e = feed.getEntries().get(i);
			sb.append(String.format(
					"%3d | %-30s | %-6s | %-8s | %-15s | %-40s%n", i + 1,
					e.getUpdated(), e.getKind(), e.isReadOnly(),
					e.getVisibility(), getName(e)));
		}
		return sb.toString();
	}

	static public String print(ControlTypeFeed feed) {
		StringBuffer sb = new StringBuffer();
		if (feed.getSelfLink() != null)
			sb.append(String.format("%3s ! %-60s%n", "URL", feed.getSelfLink()
					.getHref()));
		sb.append(String.format(
				"%3s | %-18s | %-13s | %-10s | %-10s | %-10s | %-30s%n", "#",
				"CONTROL TYPE", "DISPLAY TYPE", "READONLY", "REQUIRED",
				"COMMENTS?", "NAME"));
		for (int i = 0; i < feed.getEntries().size(); i++) {
			ControlTypeEntry e = feed.getEntries().get(i);
			sb.append(String.format(
					"%3d | %-18s | %-13s | %-10s | %-10s | %-10s | %-30s%n",
					i + 1, e.getControlType(), e.getDisplayType(),
					e.isReadonly(), e.isRequired(), e.isCommentEnabled(),
					getName(e)));
		}
		return sb.toString();
	}

	static public <L extends ILink> String print(List<L> links) {
		StringBuffer sb = new StringBuffer();
		sb.append(String.format("%3s | %-24s | %-30s | %-60s%n", "#", "REL",
				"TYPE", "HREF"));
		for (int i = 0; i < links.size(); i++) {
			L l = links.get(i);
			sb.append(String.format("%3d | %-24s | %-30s | %-60s%n", i + 1,
					l.getRel(), l.getType(), l.getHref()));
		}
		return sb.toString();
	}

	/*
	 * ************************************************************************
	 *
	 * ************************************************************************
	 */
	static public <E extends BaseEntry<E>> E getEntry(FormsService s,
			Class<E> entryCls, String entryId) throws IOException,
			ServiceException {
		try {
			URL entryUrl = s.getEntryURL(entryCls, entryId);
			return s.getEntry(entryUrl, entryCls);
		} catch (ResourceNotFoundException e) {
			return null;
		}
	}

	static public <F extends BaseFeed<F, E>, E extends BaseEntry<E>> E findByIndex(
			F f, int entryIndex) {
		if (entryIndex < 1 || entryIndex > f.getEntries().size())
			return null;
		return f.getEntries().get(entryIndex - 1);
	}

	static public <F extends BaseFeed<F, E>, E extends BaseEntry<E>> E findById(
			F f, String id) {
		if (id == null)
			return null;
		for (E e : f.getEntries()) {
			if (id.equals(e.getId()))
				return e;
		}
		return null;
	}

	static public <F extends BaseFeed<F, E>, E extends BaseEntry<E>, Q extends Query> Q nextPage(
			F feed, Q query) {
		int start = getStartIndex(feed);
		int max = getMaxResults(feed);

		query.setStartIndex(start + max);
		return query;
	}

	static public <F extends BaseFeed<F, E>, E extends BaseEntry<E>, Q extends Query> Q previousPage(
			F feed, Q query) {
		int start = getStartIndex(feed);
		int max = getMaxResults(feed);

		query.setStartIndex(Math.max(0, start - max));
		return query;
	}

	static public <F extends BaseFeed<F, E>, E extends BaseEntry<E>> int getCurrentPage(
			F feed) {
		int start = getStartIndex(feed);
		int max = getMaxResults(feed);
		return Math.max(start / max + 1, 1);
	}

	static public <F extends BaseFeed<F, E>, E extends BaseEntry<E>> int getStartIndex(
			F feed) {
		Link self = feed.getSelfLink();

		Pattern p = Pattern.compile("start-index=([^&])");
		Matcher m = p.matcher(self.getHref());
		if (m.find())
			return Integer.valueOf(m.group(1));
		return 1;
	}

	static public <F extends BaseFeed<F, E>, E extends BaseEntry<E>> int getMaxResults(
			F feed) {
		Link self = feed.getSelfLink();
		Pattern p = Pattern.compile("max-results=([^&])");
		Matcher m = p.matcher(self.getHref());
		if (m.find())
			return Integer.valueOf(m.group(1));
		return feed.getEntries().size();
	}

	static public TaskFeed getTaskFeed(FormsService service)
			throws IOException, ServiceException {
		URL feedUrl = service.getFeedURL(TaskFeed.class);
		TaskQuery query = new TaskQuery(feedUrl);
		TaskFeed feed = service.getFeed(query, TaskFeed.class);
		return feed;
	}

	static public SubmissionFeed getSubmissionFeed(FormsService service,
			FormTypeEntry owner) throws IOException, ServiceException {
		URL feedUrl = service.getFeedURL(SubmissionFeed.class);
		SubmissionQuery query = new SubmissionQuery(feedUrl);
		if (owner != null) {
			// You can manually do this....
			// String formTypeId = owner.getId().split("!")[0];
			// query.setFilter("$formTypeId eq '" + formTypeId + "'");
			// or this...
			return owner.getSubmissionFeed();
		} else {
			SubmissionFeed feed = service.getFeed(query, SubmissionFeed.class);
			return feed;
		}
	}

	static public ApplicationFeed getApplicationFeed(FormsService service,
			UserEntry owner) throws IOException, ServiceException {
		if (owner != null) {
			return owner.getApplicationFeed();
		} else {
			URL feedUrl = service.getFeedURL(ApplicationFeed.class);
			return service.getFeed(feedUrl, ApplicationFeed.class);
		}
	}

	static public SchemaFeed getSchemaFeed(FormsService service,
			ApplicationEntry owner) throws IOException, ServiceException {
		if (owner != null) {
			return owner.getSchemaFeed();
		} else {
			URL feedUrl = service.getFeedURL(SchemaFeed.class);
			return service.getFeed(feedUrl, SchemaFeed.class);
		}
	}

	static public FormTypeFeed getFormTypeFeed(FormsService service,
			ApplicationEntry owner, String kind) throws IOException,
			ServiceException {
		if (FormsService.SCHEME_KIND_FORM.equals(kind)
				|| FormsService.SCHEME_KIND_FLOW.equals(kind)) {
			// we are filtering by kind, i.e. either only flows or onky formss
			if (owner != null) {
				// we are using the owner as the starting point
				if (FormsService.SCHEME_KIND_FORM.equals(kind))
					return owner.getFormTypeFeed();
				else
					return owner.getFlowTypeFeed();
			} else {
				// add a category filter to query only for forms, not flows
				URL feedUrl = service.getFeedURL(FormTypeFeed.class);
				FormTypeQuery query = new FormTypeQuery(feedUrl);
				CategoryFilter onlyFormsFilter = new CategoryFilter();
				Category formCatetory = new Category(kind);
				onlyFormsFilter.addCategory(formCatetory);
				query.addCategoryFilter(onlyFormsFilter);
				return service.getFeed(query, FormTypeFeed.class);
			}
		} else {
			// no filtering so
			URL feedUrl = service.getFeedURL(FormTypeFeed.class);
			return service.getFeed(feedUrl, FormTypeFeed.class);
		}
	}

	static public ApplicationEntry createApplicationEntry(FormsService service,
			ApplicationFeed feed, String name, String description)
			throws ServiceException, IOException {
		ApplicationEntry entry = feed.createEntry();
		if (name != null && name.length() > 0)
			entry.setTitle(new PlainTextConstruct(name));
		if (description != null && description.length() > 0)
			entry.setTitle(new PlainTextConstruct(description));
		entry = feed.insert(entry); // entry is stateless: new entry instance is
									// returned
		return entry;
	}

	static public FormTypeEntry createFormTypeEntry(FormsService service,
			FormTypeFeed feed, String name, String description)
			throws ServiceException, IOException {
		FormTypeEntry entry = feed.createEntry();
		if (name != null && name.length() > 0)
			entry.setTitle(new PlainTextConstruct(name));
		if (description != null && description.length() > 0)
			entry.setTitle(new PlainTextConstruct(description));
		entry = feed.insert(entry); // entry is stateless: new entry instance is
									// returned
		return entry;
	}

	static public FormTypeEntry uploadNewFormType(FormsService service,
			FormTypeFeed feed, InputStream is) throws ServiceException,
			IOException {
		MediaStreamSource ms = new MediaStreamSource(is, "application/zip");
		FormTypeEntry entry = feed.insert(ms);
		return entry;
	}

	static public SchemaEntry uploadNewSchema(FormsService service,
			SchemaFeed feed, InputStream is) throws ServiceException,
			IOException {
		MediaStreamSource ms = new MediaStreamSource(is, "application/zip");
		SchemaEntry entry = feed.insert(ms);
		return entry;
	}

	/*
	 * ************************************************************************
	 * DOWNLOAD
	 * ************************************************************************
	 */
	static private <E extends BaseEntry<E>> void download(FormsService fs, E e,
			OutputStream os) throws IOException, ServiceException {
		MediaContent mc = (MediaContent) e.getContent();
		MediaSource ms = fs.getMedia(mc);

		Helper.readStream(ms.getInputStream(), os);
	}

	static public void downloadFormType(FormsService service,
			FormTypeEntry entry, OutputStream os) throws IOException,
			ServiceException {
		download(service, entry, os);
	}

	static public void downloadApplication(FormsService service,
			ApplicationEntry entry, OutputStream os) throws IOException,
			ServiceException {
		download(service, entry, os);
	}

	static public <F extends BaseFeed<F, E>, E extends BaseEntry<E>> F next(
			FormsService service, F feed, Class<F> feedCls) throws IOException,
			ServiceException {
		// if( 1 == 2 ){
		// this will not work in 4.1.2 or earlier
		Link nextLink = feed.getNextLink();
		if (nextLink == null)
			return null;

		URL nextUrl = new URL(nextLink.getHref());
		feed = service.getFeed(nextUrl, feedCls);
		return feed;
		// }
		// else {
		// // this is the fallback for 4.1.2 or earlier
		// Link selfLink = feed.getSelfLink();
		// if (selfLink == null)
		// return null;
		// String selfHref = selfLink.getHref();
		// selfHref.replaceAll("start-index=[^&]*", "start-index="+);
		//
		// }
	}

	static public <F extends BaseFeed<F, E>, E extends BaseEntry<E>> F previous(
			FormsService service, F feed, Class<F> feedCls) throws IOException,
			ServiceException {
		Link nextLink = feed.getPreviousLink();
		if (nextLink == null)
			return null;

		URL nextUrl = new URL(nextLink.getHref());
		feed = service.getFeed(nextUrl, feedCls);
		return feed;
	}

	static public Link getFormTypeEditLink(FormTypeEntry entry)
			throws ServiceException {
		Link editorLink = entry.getFormTypeEditorLink(null);
		return editorLink;
	}

	static public Link getSubmissionUseLink(SubmissionEntry entry,
			boolean readonly) throws ServiceException {
		Map<String, Object> params = new HashMap<String, Object>();
		params.put(FormTypeEntry.FORMTYPE_READONLY_PARAMETER, readonly);
		Link l = entry.getFormTypePopupLink(params);
		return l;
	}

	static public Link getFormTypeUseLink(FormTypeEntry entry)
			throws ServiceException {
		Link formLink = entry.getFormTypeLink(null);
		return formLink;
	}

	static public Link getFormTypeSnapshotLink(FormTypeEntry entry) {
		Link snapshotLink = entry.getSnapshotLink("application/pdf");
		return snapshotLink;
	}

	// TODO Simplify snapshot link access in API
	static public Link getSubmissionSnapshotLink(SubmissionEntry entry) {
		return getSubmissionSnapshotLink(entry, null);
	}

	static public Link getSubmissionSnapshotLink(SubmissionEntry entry, String contentType) {
		List<Link> documentLinks = entry.getDocumentLinks();
		for (Link l : documentLinks) {
			if (l.getType().contains("frevvo-snapshot=true")
					&& (contentType == null || l.getType().contains(contentType)) )
				return l;
		}
		return null;
	}

	static public SubmissionFeed getFormTypeSubmissions(FormTypeEntry entry)
			throws MalformedURLException, IOException, ServiceException {
		SubmissionFeed feed = entry.getSubmissionFeed();
		return feed;
	}

	static public DocumentTypeFeed getFormTypeDocTypeFeed(FormTypeEntry entry)
			throws MalformedURLException, IOException, ServiceException {
		DocumentTypeFeed feed = entry.getDocumentTypeFeed();
		return feed;
	}

	static public ControlTypeFeed getFormTypeControls(FormTypeEntry entry)
			throws MalformedURLException, IOException, ServiceException {
		ControlTypeFeed feed = entry.getControlTypeFeed();
		return feed;
	}

	static public FormTypeEntry getFormTypeByName(FormsService service,
			String name) throws MalformedURLException, IOException,
			ServiceException {
		URL formTypesUrl = service.getFeedURL(FormTypeFeed.class);
		FormTypeFeed formTypes = service.getFeed(formTypesUrl,
				FormTypeFeed.class);
		for (FormTypeEntry formType : formTypes.getEntries()) {
			if (name.equals(getName(formType))) {
				return formType;
			}
		}
		return null;
	}

	static public FormTypeEntry setReadonly(FormTypeEntry entry, boolean value)
			throws IOException, ServiceException {
		entry.setReadOnly(value);
		entry = entry.update();
		return entry;
	}

	static public String nameToFileName(String name) {
		if (name == null)
			return null;
		return name.replaceAll("[^0-9a-zA-Z_-]", "");
	}

	static public void submitFormInstance(FormsService service,
			URL formInstanceUrl) throws ServiceException {
		Helper.submitInstance(service, formInstanceUrl);
	}

	static public void cancelFormInstance(FormsService service,
			URL formInstanceUrl) throws ServiceException {
		Helper.cancelInstance(service, formInstanceUrl);
	}

	static public boolean isFlow(FormTypeEntry entry) {
		return FormsService.SCHEME_KIND_FLOW.equals(entry.getKind());
	}

	static public boolean isForm(FormTypeEntry entry) {
		return FormsService.SCHEME_KIND_FORM.equals(entry.getKind());
	}

	public static URL createFormInstance(FormsService service,
			FormTypeEntry entry, Map<String, Object> data,
			List<MediaSource> documents) throws ServiceException {

		Map<String, Object> params = new HashMap<String, Object>();
		params.put(FormTypeEntry.FORMTYPE_DATA_PARAMETER, data);

		URL formInstanceUrl = entry.createFormInstance(params, documents);
		return formInstanceUrl;
	}

	public static UserEntry getUserEntry(FormsService service)
			throws IOException, ServiceException {
		// strip away the tenant from the user name if any
		String username = service.getUsername();
		if (username.contains("@"))
			username = username.split("@")[0];
		return getEntry(service, UserEntry.class, username);
	}

	public static File createTempFolder(String prefix) {
		prefix += UUID.randomUUID().toString();
		File f = new File(System.getProperty("java.io.tmpdir"), prefix);
		if( !f.exists() )
			f.mkdirs();
		return f;
	}
	
	public static String getNameSpace(SchemaEntry schemaEntry) throws MalformedURLException, IOException, ServiceException {
		return schemaEntry.getDocumentTypeFeed().getEntries().get(0).getTitle().getPlainText().split("#")[0];
	}
}
