#!/usr/bin/env node
"use strict";

var pathIsAbsolute = require("path-is-absolute");
var commander = require("commander");
var Module = require("module");
var babel = require("babel-core");
var inspect = require("util").inspect;
var path = require("path");
var repl = require("repl");
var util = require("babel-core").util;
var vm = require("vm");
var _ = require("lodash");

var program = new commander.Command("babel-node");

program.option("-e, --eval [script]", "Evaluate script");
program.option("-p, --print [code]", "Evaluate script and print result");
program.option("-i, --ignore [regex]", "Ignore all files that match this regex when using the require hook");
program.option("-x, --extensions [extensions]", "List of extensions to hook into [.es6,.js,.es,.jsx]");
program.option("-r, --stage [stage]", "Enable support for specific ECMAScript stages");
program.option("-w, --whitelist [whitelist]", "Whitelist of transformers separated by comma to ONLY use", util.list);
program.option("-b, --blacklist [blacklist]", "Blacklist of transformers separated by comma to NOT use", util.list);
program.option("-o, --optional [optional]", "List of optional transformers separated by comma to enable", util.list);

var pkg = require("../package.json");
program.version(pkg.version);
program.usage("[options] [ -e script | script.js ] [arguments]");
program.parse(process.argv);

//

babel.register({
  extensions: program.extensions,
  blacklist: program.blacklist,
  whitelist: program.whitelist,
  optional: program.optional,
  ignore: program.ignore,
  stage: program.stage
});

//

var _eval = function _eval(code, filename) {
  code = code.trim();
  if (!code) return undefined;

  code = babel.transform(code, {
    filename: filename,
    blacklist: program.blacklist,
    whitelist: program.whitelist,
    optional: program.optional,
    stage: program.stage
  }).code;

  return vm.runInThisContext(code, {
    filename: filename
  });
};

if (program.eval || program.print) {
  var code = program.eval;
  if (!code || code === true) code = program.print;

  global.__filename = "[eval]";
  global.__dirname = process.cwd();

  var _module = new Module(global.__filename);
  _module.filename = global.__filename;
  _module.paths = Module._nodeModulePaths(global.__dirname);

  global.exports = _module.exports;
  global.module = _module;
  global.require = _module.require.bind(_module);

  var result = _eval(code, global.__filename);
  if (program.print) {
    var output = _.isString(result) ? result : inspect(result);
    process.stdout.write(output + "\n");
  }
} else {
  if (program.args.length) {
    // slice all arguments up to the first filename since they're babel args that we handle
    var args = process.argv.slice(2);

    var i = 0;
    var ignoreNext = false;
    _.each(args, function (arg, i2) {
      if (ignoreNext) {
        ignoreNext = false;
        return;
      }

      if (arg[0] === "-") {
        var parsedArg = program[arg.slice(2)];
        if (parsedArg && parsedArg !== true) {
          ignoreNext = true;
        }
      } else {
        i = i2;
        return false;
      }
    });
    args = args.slice(i);

    // make the filename absolute
    var filename = args[0];
    if (!pathIsAbsolute(filename)) args[0] = path.join(process.cwd(), filename);

    // add back on node and concat the sliced args
    process.argv = ["node"].concat(args);
    process.execArgv.unshift(__filename);

    Module.runMain();
  } else {
    replStart();
  }
}

function replStart() {
  repl.start({
    prompt: "> ",
    input: process.stdin,
    output: process.stdout,
    eval: replEval,
    useGlobal: true
  });
}

function replEval(code, context, filename, callback) {
  var err;
  var result;

  try {
    if (code[0] === "(" && code[code.length - 1] === ")") {
      code = code.slice(1, -1); // remove "(" and ")"
    }

    result = _eval(code, filename);
  } catch (e) {
    err = e;
  }

  callback(err, result);
}