package com.book.pongv2;

import android.graphics.Color;
import android.graphics.Point;
import android.graphics.PointF;
import android.graphics.RectF;
import android.util.Log;

import com.book.simplegameenginev2.SGEntity;
import com.book.simplegameenginev2.SGTimer;
import com.book.simplegameenginev2.SGWorld;

import java.util.ArrayList;
import java.util.Random;

public class GameModel extends SGWorld
{
	public static final int     BALL_ID = 0;
	public static final int     OPPONENT_ID = 1;
	public static final int     PLAYER_ID = 2;
	public final static int     BALL_SIZE = 16;
	public final static int     DISTANCE_FROM_EDGE = 16;
	public final static int     PADDLE_BBOX_PADDING = 3;
	public final static int     PADDLE_HEIGHT = 98;
	public final static int     PADDLE_WIDTH = 23;

	public static final int     STATE_RESTART = 0;
	public static final int     STATE_RUNNING = 1;
	public static final int     STATE_GAME_OVER = 2;
	public static final int     STATE_GOAL = 3;
	public static final int     STATE_PAUSED = 4;

	public static final int     TRG_GAP_ID = 3;
	public static final int     TRG_LEFT_GOAL_ID = 4;
	public static final int     TRG_LOWER_WALL_ID = 5;
	public static final int     TRG_RIGHT_GOAL_ID = 6;
	public static final int     TRG_UPPER_WALL_ID = 7;

	public final static int     GOAL_WIDTH = 64;
	public final static int     WALL_HEIGHT = 64;

	private EntBall 		    mBall;
	private EntOpponent 	    mOpponent;
	private EntPlayer		    mPlayer;

	private int                 mCurrentState;
	private int                 mOpponentScore;
	private int                 mPlayerScore;
	Random                      mRandom = new Random();
	private StringBuilder       mStringBuilder = new StringBuilder();

	private ArrayList<SGEntity> mEntities = new ArrayList<SGEntity>();
	private TrgGap              mGap;
	private TrgLeftGoal         mLeftGoal;
	private TrgLowerWall        mLowerWall;
	private TrgRightGoal        mRightGoal;
	private TrgUpperWall        mUpperWall;

	private SGTimer             mGoalTimer;
	private SGTimer             mRestartTimer;

	public GameModel(Point worldDimensions)
	{
		super(worldDimensions);
	}

	public void setup()
	{
		mCurrentState = STATE_RESTART;

		Point worldDimensions = getDimensions();

		// Bola
		PointF tempPosition = new PointF((worldDimensions.x / 2) - (BALL_SIZE / 2),
										 (worldDimensions.y / 2) - (BALL_SIZE / 2));
		PointF tempDimensions = new PointF(BALL_SIZE, BALL_SIZE);
		mBall = new EntBall(this, tempPosition, tempDimensions);

		mEntities.add(mBall);

		// Paddle do jogador
		tempPosition.set(DISTANCE_FROM_EDGE,
						 (worldDimensions.y / 2) - (PADDLE_HEIGHT / 2));
		tempDimensions.set(PADDLE_WIDTH, PADDLE_HEIGHT);
		mPlayer = new EntPlayer(this, tempPosition, tempDimensions);
		mPlayer.setDebugColor(Color.GREEN);
		RectF bboxPadding = new RectF(0, 0, PADDLE_BBOX_PADDING, PADDLE_BBOX_PADDING);
		mPlayer.setBBoxPadding(bboxPadding);

		mEntities.add(mPlayer);

		// Paddle do oponente
		tempPosition.set(worldDimensions.x - (DISTANCE_FROM_EDGE + PADDLE_WIDTH),
				(worldDimensions.y / 2) - (PADDLE_HEIGHT / 2));
		tempDimensions.set(PADDLE_WIDTH, PADDLE_HEIGHT);
		mOpponent = new EntOpponent(this, tempPosition, tempDimensions);
		mOpponent.setDebugColor(Color.CYAN);
		mOpponent.setBBoxPadding(bboxPadding);

		mEntities.add(mOpponent);

		// Meta do jogador
		tempPosition.set(-GOAL_WIDTH, 0);
		tempDimensions.set(GOAL_WIDTH - BALL_SIZE, worldDimensions.y);
		mLeftGoal = new TrgLeftGoal(this, tempPosition, tempDimensions);
		mLeftGoal.addObservedEntity(mBall);
		mEntities.add(mLeftGoal);

		// Meta do oponente
		tempPosition.set(worldDimensions.x + BALL_SIZE, 0);
		tempDimensions.set(GOAL_WIDTH - BALL_SIZE, worldDimensions.y);
		mRightGoal = new TrgRightGoal(this, tempPosition, tempDimensions);
		mRightGoal.addObservedEntity(mBall);
		mEntities.add(mRightGoal);

		// Vão superior
		tempPosition.set(0, 0);
		tempDimensions.set(worldDimensions.x, TrgGap.GAP_SIZE);
		mGap = new TrgGap(this, tempPosition, tempDimensions);
		mGap.addObservedEntity(mOpponent);
		mGap.addObservedEntity(mPlayer);
		mEntities.add(mGap);

		// Parede inferior
		tempPosition.set(0, worldDimensions.y);
		tempDimensions.set(worldDimensions.x, WALL_HEIGHT);
		mLowerWall = new TrgLowerWall(this, tempPosition, tempDimensions);
		mLowerWall.addObservedEntity(mBall);
		mLowerWall.addObservedEntity(mOpponent);
		mLowerWall.addObservedEntity(mPlayer);
		mEntities.add(mLowerWall);

		// Parede superior
		tempPosition.set(0, -WALL_HEIGHT);
		tempDimensions.set(worldDimensions.x, WALL_HEIGHT);
		mUpperWall = new TrgUpperWall(this, tempPosition, tempDimensions);
		mUpperWall.addObservedEntity(mBall);
		mEntities.add(mUpperWall);

		// Timer de reinício
		mRestartTimer = new SGTimer(0.8f);

		// Timer de marcação de ponto
		mGoalTimer = new SGTimer(1.2f);
	}

	public void movePlayer(float x, float y)
	{
		mPlayer.move(0, y);

		PointF playerPosition = mPlayer.getPosition();
		PointF playerDimensions = mPlayer.getDimensions();
		Point worldDimensions = getDimensions();

		RectF tempBoundingBox = mPlayer.getBoundingBox();

		if(playerPosition.y < 0)
		{
			mPlayer.setPosition(playerPosition.x, 0);
		}
		else if(tempBoundingBox.bottom > worldDimensions.y)
		{
			mPlayer.setPosition(playerPosition.x, worldDimensions.y - (playerDimensions.y - PADDLE_BBOX_PADDING));
		}
	}

	@Override
	public void step(float elapsedTimeInSeconds)
	{
		if(elapsedTimeInSeconds > 1.0f)
		{
			elapsedTimeInSeconds = 0.1f;
		}

		if(mCurrentState == STATE_RUNNING)
		{
			for(SGEntity currentEntity : mEntities)
			{
				currentEntity.step(elapsedTimeInSeconds);
			}
		}
		else if(mCurrentState == STATE_GOAL)
		{
			if(!mGoalTimer.hasStarted())
			{
				mGoalTimer.start();
			}
			if(mGoalTimer.step(elapsedTimeInSeconds))
			{
				mGoalTimer.stopAndReset();

				if(mOpponentScore == 5)
				{
					Log.d("PongV2", "Fim de jogo!");
					mCurrentState = STATE_GAME_OVER;
				}
				else
				{
					mCurrentState = STATE_RESTART;
				}
			}
		}
		else if(mCurrentState == STATE_RESTART)
		{
			if(!mRestartTimer.hasStarted())
			{
				mRestartTimer.start();
				resetWorld();
			}
			if(mRestartTimer.step(elapsedTimeInSeconds))
			{
				mRestartTimer.stopAndReset();
				mCurrentState = STATE_RUNNING;
			}
		}

		else if(mCurrentState == STATE_GAME_OVER)
		{
		}
	}

	public void logScore()
	{
		mStringBuilder.delete(0, mStringBuilder.length());
		mStringBuilder.append("Placar: ");
		mStringBuilder.append(mPlayerScore);
		mStringBuilder.append(" X ");
		mStringBuilder.append(mOpponentScore);

		Log.d("PongV2", mStringBuilder.toString());
	}

	public void resetWorld()
	{
		float halfWorldWidth = getDimensions().x / 2;
		float halfWorldHeight = getDimensions().y / 2;
		float halfBallWidth = mBall.getDimensions().x / 2;
		float halfBallHeight = mBall.getDimensions().y / 2;
		float halfPaddleHeight = mPlayer.getDimensions().y / 2;
		float ballPositionX = halfWorldWidth - halfBallWidth;
		float ballPositionY = halfWorldHeight - halfBallHeight;
		float paddlePositionY = halfWorldHeight - halfPaddleHeight;

		mBall.setPosition(ballPositionX, ballPositionY);
		mOpponent.setPosition(mOpponent.getPosition().x, paddlePositionY);
		mPlayer.setPosition(mPlayer.getPosition().x, paddlePositionY);

		if(mRandom.nextInt(2) == 0)
		{
			mBall.setVelocity(-90.0f, 90.0f);
		}
		else
		{
			mBall.setVelocity(90.0f, 90.0f);
		}
	}

	public EntBall      getBall() { return mBall; }
	public int          getCurrentState() { return mCurrentState; }
	public ArrayList<SGEntity>
						getEntities() { return mEntities; }
	public EntOpponent  getOpponent() { return mOpponent; }
	public int          getOpponentScore() { return mOpponentScore; }
	public EntPlayer    getPlayer() { return mPlayer; }
	public int          getPlayerScore() { return mPlayerScore; }

	public void         setCurrentState(int state) { mCurrentState = state; }

	public void         increaseOpponentScore() { mOpponentScore++; }
	public void         increasePlayerScore() { mPlayerScore++; }

}

