/*---------------------------------------------------------------------------------------------
 *  Copyright (C) 2025 Posit Software, PBC. All rights reserved.
 *  Licensed under the Elastic License 2.0. See LICENSE.txt for license information.
 *--------------------------------------------------------------------------------------------*/

import * as vscode from 'vscode';
import * as positron from 'positron';
import * as xml from '../xml.js';
import { calculateSlidingWindow, filterNotebookContext, MAX_CELLS_FOR_ALL_CELLS_CONTEXT, getOriginalContentLength } from '../notebookContextFilter.js';
import { isRuntimeSessionReference } from '../utils.js';
import { log } from '../log.js';

/**
 * Maximum preview length per cell for confirmations (characters)
 */
const MAX_CELL_PREVIEW_LENGTH = 500;

/**
 * Maximum cell content length (1MB)
 */
export const MAX_CELL_CONTENT_LENGTH = 1_000_000;

/**
 * Validation result for cell indices
 */
export interface CellIndexValidation {
	valid: boolean;
	error?: string;
}

/**
 * Validates an array of cell indices against the total cell count.
 *
 * @param indices Array of cell indices to validate
 * @param cellCount Total number of cells in the notebook
 * @param allowEmpty Whether to allow an empty array (default: false)
 * @returns Validation result with error message if invalid
 */
export function validateCellIndices(
	indices: number[],
	cellCount: number,
	allowEmpty: boolean = false
): CellIndexValidation {
	// Check for empty array
	if (indices.length === 0) {
		if (allowEmpty) {
			return { valid: true };
		}
		return { valid: false, error: 'Cell indices array cannot be empty' };
	}

	// Validate each index
	for (const index of indices) {
		// Check if integer
		if (!Number.isInteger(index)) {
			return { valid: false, error: `Cell index must be an integer: ${index}` };
		}

		// Check if negative
		if (index < 0) {
			return { valid: false, error: `Cell index cannot be negative: ${index}` };
		}

		// Check if within bounds
		if (index >= cellCount) {
			return { valid: false, error: `Cell index ${index} is out of bounds (notebook has ${cellCount} cells, valid indices: 0-${cellCount - 1})` };
		}
	}

	return { valid: true };
}

/**
 * Validation result for permutation arrays
 */
export interface PermutationValidation {
	valid: boolean;
	error?: string;
	isIdentity?: boolean;
}

/**
 * Validates a permutation array for reordering cells.
 * A valid permutation must contain each index from 0 to cellCount-1 exactly once.
 *
 * @param newOrder The proposed new order array
 * @param cellCount The total number of cells in the notebook
 * @returns Validation result with error message if invalid, and whether it's an identity permutation
 */
export function validatePermutation(
	newOrder: number[],
	cellCount: number
): PermutationValidation {
	// Check length matches
	if (newOrder.length !== cellCount) {
		return {
			valid: false,
			error: `Permutation length (${newOrder.length}) must match cell count (${cellCount})`
		};
	}

	// Handle empty notebook case
	if (cellCount === 0) {
		return { valid: true, isIdentity: true };
	}

	// An identity permutation is just one where each index maps to itself. Aka a no-op.
	let isIdentity = true;

	// Check if any indices are outside of the valid range
	for (const [i, index] of newOrder.entries()) {
		if (!Number.isInteger(index) || index < 0 || index >= cellCount) {
			return {
				valid: false,
				error: `Invalid index in permutation: ${index} (must be integer between 0 and ${cellCount - 1})`
			};
		}

		// Check for identity at the same time
		if (index !== i) {
			isIdentity = false;
		}
	}

	// If identity permutation, no need to check further
	if (isIdentity) {
		return { valid: true, isIdentity: true };
	}

	// Make sure there are no duplicates and all indices are present
	const uniqueIndices = new Set(newOrder);

	if (uniqueIndices.size !== cellCount) {
		return {
			valid: false,
			error: 'Invalid permutation: must contain each index from 0 to cellCount-1 exactly once'
		};
	}

	return { valid: true, isIdentity: false };
}

/**
 * Fetches and formats cell content for preview in confirmation dialogs.
 * Truncates long content with ellipsis.
 *
 * @param uri The notebook URI (as string)
 * @param cellIndices Array of cell indices to preview
 * @returns Formatted preview string with cell content
 */
export async function getCellsPreview(
	uri: string,
	cellIndices: number[]
): Promise<string> {
	const previews: string[] = [];

	for (const cellIndex of cellIndices) {
		try {
			const cell = await positron.notebooks.getCell(uri, cellIndex);
			if (!cell) {
				previews.push(`Cell ${cellIndex}: [Cell not found]`);
				continue;
			}

			let content = cell.content.trim();
			if (content.length > MAX_CELL_PREVIEW_LENGTH) {
				content = content.substring(0, MAX_CELL_PREVIEW_LENGTH) + '...';
			}

			previews.push(`Cell ${cellIndex} (${cell.type}):\n${content}`);
		} catch (error) {
			previews.push(`Cell ${cellIndex}: [Error fetching cell]`);
		}
	}

	return previews.join('\n\n');
}

/**
 * Format cell status information for display in prompts
 */
function formatCellStatus(cell: positron.notebooks.NotebookCell): string {
	const statusParts: string[] = [];

	// Selection status
	statusParts.push(`Selection: ${cell.selectionStatus}`);

	// Execution status (only for code cells)
	if (cell.executionStatus !== undefined) {
		statusParts.push(`Execution: ${cell.executionStatus}`);
		if (cell.executionOrder !== undefined) {
			statusParts.push(`Order: [${cell.executionOrder}]`);
		}
		if (cell.lastRunSuccess !== undefined) {
			statusParts.push(`Last run: ${cell.lastRunSuccess ? 'success' : 'failed'}`);
		}
		if (cell.lastExecutionDuration !== undefined) {
			const durationMs = cell.lastExecutionDuration;
			const durationStr = durationMs < 1000
				? `${durationMs}ms`
				: `${(durationMs / 1000).toFixed(2)}s`;
			statusParts.push(`Duration: ${durationStr}`);
		}
	}

	// Output status
	statusParts.push(cell.hasOutput ? 'Has output' : 'No output');

	return statusParts.join(' | ');
}

/**
 * Options for formatting notebook cells
 */
export interface FormatCellsOptions {
	/** The notebook cells to format */
	cells: positron.notebooks.NotebookCell[];
	/** The prefix to use for cell labels (e.g., 'Selected Cell', 'Cell') */
	prefix: string;
	/** Whether to include cell content in the output. Defaults to true. */
	includeContent?: boolean;
}

/**
 * Format a collection of cells for display in prompts using XML format
 *
 * @param options Options for formatting cells
 * @returns A formatted XML string describing all cells, separated by single newlines
 */
export function formatCells(options: FormatCellsOptions): string {
	const { cells, prefix, includeContent = true } = options;

	if (cells.length === 0) {
		return prefix === 'Selected Cell' ? 'No cells currently selected' : '';
	}

	return cells.map((cell, idx) => {
		const statusInfo = formatCellStatus(cell);
		const cellLabel = cells.length === 1
			? prefix
			: `${prefix} ${idx + 1}`;

		// Check if cell content was truncated (has originalContentLength property)
		const originalLength = getOriginalContentLength(cell);
		const wasTruncated = originalLength !== undefined && originalLength > cell.content.length;

		const parts = [
			`<cell index="${cell.index}" type="${cell.type}">`,
			`  <label>${cellLabel}</label>`,
			`  <status>${statusInfo}</status>`,
			includeContent ? `<content>${cell.content}</content>` : '',
			wasTruncated ? `  <truncated original-length="${originalLength}" />` : '',
			`</cell>`
		];
		return parts.filter(Boolean).join('\n');
	}).join('\n');
}

/**
 * Convert notebook cell outputs to LanguageModel parts (text and image data).
 * Handles both text and image outputs, converting base64 image data to binary format.
 *
 * @param outputs Array of notebook cell outputs to convert
 * @param prefixText Optional text to prepend before the outputs
 * @returns Array of LanguageModel parts ready for use in tool results
 */
export function convertOutputsToLanguageModelParts(
	outputs: positron.notebooks.NotebookCellOutput[],
	prefixText?: string
): (vscode.LanguageModelTextPart | vscode.LanguageModelDataPart)[] {
	const resultParts: (vscode.LanguageModelTextPart | vscode.LanguageModelDataPart)[] = [];

	// Add prefix text if provided
	if (prefixText) {
		resultParts.push(new vscode.LanguageModelTextPart(prefixText));
	}

	// Convert each output to appropriate LanguageModel part
	for (const output of outputs) {
		if (output.mimeType.startsWith('image/')) {
			// Handle image outputs - convert base64 to binary
			if (!output.data) {
				resultParts.push(new vscode.LanguageModelTextPart('[Image data unavailable]'));
				continue;
			}
			const imageBuffer = Buffer.from(output.data, 'base64');
			const imageData = new Uint8Array(imageBuffer);
			resultParts.push(new vscode.LanguageModelDataPart(imageData, output.mimeType));
		} else {
			// Handle text outputs
			let textContent = output.data;
			// Add newline before text output if there are already parts (for readability)
			if (resultParts.length > 0) {
				textContent = '\n' + textContent;
			}
			resultParts.push(new vscode.LanguageModelTextPart(textContent));
		}
	}

	return resultParts;
}

/**
 * Options for serializing notebook context
 */
export interface NotebookContextSerializationOptions {
	/** Optional anchor for sliding window. Defaults: last selected cell → last executed cell → 0 */
	anchorIndex?: number;
	/** Default: false. If true, wraps everything in <notebook-context> node (for suggestions format) */
	wrapInNotebookContext?: boolean;
}

/**
 * Serialized notebook context components
 */
export interface SerializedNotebookContext {
	/** Kernel information XML node */
	kernelInfo: string;
	/** Cell count information XML node (used internally in wrapped format) */
	cellCountInfo?: string;
	/** Selected cells XML (may be empty if no selection) */
	selectedCellsInfo: string;
	/** All cells XML (present if cells available after filtering) */
	allCellsInfo?: string;
	/** Context note XML */
	contextNote: string;
	/** Full wrapped context (if wrapInNotebookContext is true) */
	fullContext?: string;
	/** Whether this serialized context represents a full notebook (true) or a windowed context (false). Undefined when allCellsInfo is not present. */
	isFullNotebookContext?: boolean;
}

/**
 * Serialize notebook context to XML format with integrated filtering logic.
 *
 * This function serves as the single source of truth for notebook context serialization
 * across notebook suggestions, chat pane prompts, and inline chat. It handles filtering
 * internally and generates consistent XML components.
 *
 * @param context The notebook context to serialize
 * @param options Serialization options
 * @returns Serialized notebook context components
 */
export function serializeNotebookContext(
	context: positron.notebooks.NotebookContext,
	options: NotebookContextSerializationOptions = {}
): SerializedNotebookContext {
	const { anchorIndex, wrapInNotebookContext = false } = options;

	// Get all cells from context (may already be filtered)
	const allCells = context.allCells || [];
	const totalCells = context.cellCount;
	const canIncludeFullNotebook = totalCells < MAX_CELLS_FOR_ALL_CELLS_CONTEXT;

	// Helper function to find executed cells (used in multiple places)
	const getExecutedCells = (cells: positron.notebooks.NotebookCell[]) => {
		const codeCells = cells.filter(c => c.type === positron.notebooks.NotebookCellType.Code);
		return codeCells.filter(c => c.executionOrder !== undefined);
	};

	// Determine anchor index for sliding window if not provided
	let effectiveAnchorIndex: number;
	if (anchorIndex !== undefined) {
		effectiveAnchorIndex = anchorIndex;
	} else if (context.selectedCells.length > 0) {
		// Use last selected cell index
		effectiveAnchorIndex = Math.max(...context.selectedCells.map(cell => cell.index));
	} else {
		// Try to find last executed cell
		const executedCells = getExecutedCells(allCells);
		if (executedCells.length > 0) {
			effectiveAnchorIndex = Math.max(...executedCells.map(c => c.index));
		} else {
			// Fallback to 0
			effectiveAnchorIndex = 0;
		}
	}

	// Apply filtering logic to determine which cells to include
	let cellsToInclude: positron.notebooks.NotebookCell[];

	if (canIncludeFullNotebook) {
		// Small notebooks: include all cells
		cellsToInclude = allCells.length > 0 ? allCells : [];
	} else if (context.selectedCells.length === 0 && allCells.length === 0) {
		// Large notebooks without selection and no allCells: no cells to include
		cellsToInclude = [];
	} else if (context.selectedCells.length === 0) {
		// Large notebooks without selection: use sliding window around executed cells
		const executedCells = getExecutedCells(allCells);
		if (executedCells.length > 0 || effectiveAnchorIndex !== 0) {
			const { startIndex, endIndex } = calculateSlidingWindow(allCells.length, effectiveAnchorIndex);
			cellsToInclude = allCells.slice(startIndex, endIndex);
		} else {
			// No executed cells, use first MAX_CELLS_FOR_ALL_CELLS_CONTEXT cells
			cellsToInclude = allCells.slice(0, MAX_CELLS_FOR_ALL_CELLS_CONTEXT);
		}
	} else {
		// Large notebooks with selection: use sliding window around anchor
		if (allCells.length > 0) {
			const { startIndex, endIndex } = calculateSlidingWindow(allCells.length, effectiveAnchorIndex);
			cellsToInclude = allCells.slice(startIndex, endIndex);
		} else {
			cellsToInclude = [];
		}
	}

	// Generate kernel info XML (using xml.node for consistency)
	const kernelInfo = context.kernelId
		? xml.node('kernel', '', {
			language: context.kernelLanguage || 'unknown',
			id: context.kernelId
		})
		: xml.node('kernel', 'No kernel attached');

	// Generate cell count info XML
	const cellCountInfo = xml.node('cell-count', '', {
		total: context.cellCount,
		selected: context.selectedCells.length,
		included: cellsToInclude.length
	});

	// Generate selected cells XML
	const selectedCellsInfo = formatCells({ cells: context.selectedCells, prefix: 'Selected Cell' });

	// Generate all cells XML if available
	let allCellsInfo: string | undefined;
	let formattedCells: string | undefined;
	if (cellsToInclude.length > 0) {
		const description = canIncludeFullNotebook
			? 'All cells in notebook (notebook has fewer than 20 cells)'
			: 'Context window around selected/recent cells (notebook has 20+ cells)';
		// Format cells once and reuse
		formattedCells = formatCells({ cells: cellsToInclude, prefix: 'Cell' });
		allCellsInfo = xml.node('all-cells', formattedCells, {
			description
		});
	}

	// Generate context note XML
	let contextNote: string;
	if (cellsToInclude.length > 0) {
		if (canIncludeFullNotebook) {
			contextNote = xml.node('note', 'All cells are provided above because this notebook has fewer than 20 cells.');
		} else {
			contextNote = xml.node('note', 'A context window around the selected/recent cells is provided above. Use the GetNotebookCells tool to retrieve additional cells by index when needed.');
		}
	} else {
		contextNote = xml.node('note', 'Only selected cells are shown above to conserve tokens. Use the GetNotebookCells tool to retrieve additional cells by index when needed.');
	}

	// Build result
	const result: SerializedNotebookContext = {
		kernelInfo,
		cellCountInfo,
		selectedCellsInfo,
		allCellsInfo,
		contextNote,
		isFullNotebookContext: allCellsInfo ? canIncludeFullNotebook : undefined
	};

	// Optionally wrap in notebook-context node
	if (wrapInNotebookContext) {
		const contextMode = canIncludeFullNotebook
			? 'Full notebook (< 20 cells, all cells provided below)'
			: 'Context window around selected/recent cells (notebook has 20+ cells)';

		const contextModeNode = xml.node('context-mode', contextMode);
		const notebookInfo = xml.node('notebook-info', `${kernelInfo}\n${cellCountInfo}`);

		const parts: string[] = [xml.node('notebook-context', `${notebookInfo}\n${contextModeNode}`)];

		if (context.selectedCells.length > 0) {
			parts.push(xml.node('selected-cells', selectedCellsInfo));
		}

		if (allCellsInfo) {
			parts.push(allCellsInfo);
		}

		parts.push(contextNote);

		result.fullContext = parts.join('\n\n');
	}

	return result;
}

/**
 * Checks if notebook mode feature is enabled in workspace configuration.
 *
 * @returns True if notebook mode is enabled, false otherwise
 */
export function isNotebookModeEnabled(): boolean {
	return vscode.workspace
		.getConfiguration('positron.notebook')
		.get('enabled', false);
}

/**
 * Extracts notebook URIs from chat request references.
 * Looks for URIs in two places:
 * 1. activeSession.notebookUri property in reference values (from RuntimeSessionReference)
 * 2. Direct .ipynb file URIs in reference values
 *
 * @param request The chat request containing references
 * @returns Array of notebook URI strings found in the request
 */
function extractAttachedNotebookUris(request: vscode.ChatRequest): string[] {
	const uris: string[] = [];

	for (const ref of request.references) {
		const value = ref.value;

		// Check for RuntimeSessionReference with activeSession.notebookUri
		if (isRuntimeSessionReference(value)) {
			const notebookUri = value.activeSession.notebookUri;
			// Match original behavior: accept any string (including empty strings)
			if (typeof notebookUri === 'string') {
				uris.push(notebookUri);
			}
			continue;
		}

		// Check for direct .ipynb file URI reference
		if (value instanceof vscode.Uri && value.path.endsWith('.ipynb')) {
			uris.push(value.toString());
		}
	}

	return uris;
}

/**
 * Check if this request is from inline chat (not the chat pane).
 * Chat pane has location2 === undefined, inline chat has location2 set.
 *
 * This is used to determine whether we should bypass the explicit notebook
 * attachment check for inline chat within Positron notebooks.
 */
function isInlineChat(request: vscode.ChatRequest): boolean {
	return request.location2 !== undefined;
}

/**
 * Checks if there is an attached notebook context without applying filtering or serialization.
 * Returns the raw notebook context if:
 * 1. Notebook mode feature is enabled
 * 2. A notebook editor is currently active
 * 3. That notebook's URI is attached as context
 *
 * This is useful for tool availability checks that don't need the full filtered/serialized context.
 *
 * @param request The chat request to check for attached notebook context
 * @returns The raw notebook context if attached, undefined otherwise
 */
async function getRawAttachedNotebookContext(
	request: vscode.ChatRequest
): Promise<positron.notebooks.NotebookContext | undefined> {
	// Check if notebook mode feature is enabled
	if (!isNotebookModeEnabled()) {
		return undefined;
	}

	// Get active editor's notebook context (unfiltered from main thread)
	const activeContext = await positron.notebooks.getContext();
	if (!activeContext) {
		return undefined;
	}

	// Extract attached notebook URIs
	const attachedNotebookUris = extractAttachedNotebookUris(request);

	// If no explicit references, check if this is inline chat in a Positron notebook
	if (attachedNotebookUris.length === 0) {
		// Only bypass reference check for inline chat (not chat pane)
		// Chat pane requires explicit attachment even if notebook is open
		if (isInlineChat(request)) {
			const activeEditor = vscode.window.activeNotebookEditor;
			if (activeEditor?.isPositronNotebook === true) {
				// Inline notebook chat - no explicit attachment needed
				return activeContext;
			}
		}
		return undefined;
	}

	// Chat pane with explicit attachment - check if active notebook is attached
	const isActiveNotebookAttached = attachedNotebookUris.includes(
		activeContext.uri
	);

	if (!isActiveNotebookAttached) {
		return undefined;
	}

	return activeContext;
}

/**
 * Checks if there is an attached notebook context.
 * Returns true if:
 * 1. Notebook mode feature is enabled
 * 2. A Positron notebook editor is currently active
 * 3. That notebook's URI is attached as context
 *
 * This is a lightweight synchronous check for tool availability that doesn't require
 * filtering or serialization of the notebook context.
 *
 * @param request The chat request to check for attached notebook context
 * @returns True if there is an attached notebook context, false otherwise
 */
export function hasAttachedNotebookContext(
	request: vscode.ChatRequest
): boolean {
	// Check if notebook mode feature is enabled
	if (!isNotebookModeEnabled()) {
		return false;
	}

	// Check if a Positron notebook editor is currently active
	const activeEditor = vscode.window.activeNotebookEditor;
	if (!activeEditor || activeEditor.isPositronNotebook !== true) {
		return false;
	}

	// Extract attached notebook URIs
	const attachedNotebookUris = extractAttachedNotebookUris(request);

	// If no explicit references, check if this is inline chat in a Positron notebook
	if (attachedNotebookUris.length === 0) {
		// Only bypass reference check for inline chat (not chat pane)
		// Chat pane requires explicit attachment even if notebook is open
		return isInlineChat(request);
	}

	// Chat pane with explicit attachment - check if active notebook is attached
	const activeNotebookUri = activeEditor.notebook.uri.toString();
	return attachedNotebookUris.includes(activeNotebookUri);
}

/**
 * Checks if notebook mode should be enabled based on attached context.
 * Returns filtered notebook context only if:
 * 1. A notebook editor is currently active
 * 2. That notebook's URI is attached as context
 *
 * Applies filtering to limit context size for large notebooks.
 *
 * This function handles errors internally and will return `undefined` if any error
 * occurs during context retrieval, filtering, or serialization. Errors are logged
 * automatically. Callers do not need to wrap this function in try-catch blocks.
 *
 * @param request The chat request to check for attached notebook context
 * @returns The serialized notebook context if available, or `undefined` if no context
 *   is available or if an error occurs
 */
export async function getAttachedNotebookContext(
	request: vscode.ChatRequest
): Promise<SerializedNotebookContext | undefined> {
	try {
		const activeContext = await getRawAttachedNotebookContext(request);
		if (!activeContext) {
			return undefined;
		}

		// Apply filtering before returning context
		const filteredContext = filterNotebookContext(activeContext);
		return serializeNotebookContext(filteredContext);
	} catch (err) {
		log.error('[getAttachedNotebookContext] Error getting notebook context:', err);
		return undefined;
	}
}

/**
 * Serialize notebook context as a user message for injection into the chat.
 *
 * This function formats the notebook context as XML suitable for a user message,
 * allowing the system prompt to remain static (and thus more cacheable) while
 * the dynamic notebook state is provided separately.
 *
 * The output includes:
 * - Notebook info (kernel, cell count)
 * - Context mode (full notebook vs sliding window)
 * - Selected cells (if any)
 * - All/windowed cells content
 * - Context note explaining what's included
 *
 * @param context The serialized notebook context from getAttachedNotebookContext()
 * @returns XML-formatted string for use as a user message
 */
export function serializeNotebookContextAsUserMessage(
	context: SerializedNotebookContext
): string {
	const parts: string[] = [];

	// Header explaining what this message contains
	parts.push('<notebook-state>');
	parts.push('The following is the current state of the notebook you are assisting with:');
	parts.push('');

	// Notebook info section
	parts.push('<notebook-info>');
	parts.push(context.kernelInfo);
	if (context.cellCountInfo) {
		parts.push(context.cellCountInfo);
	}

	// Determine context mode based on whether allCellsInfo exists
	if (context.allCellsInfo) {
		// Use explicit isFullNotebookContext field instead of parsing XML
		const isFullNotebookContext = context.isFullNotebookContext === true;
		const contextMode = isFullNotebookContext
			? 'Full notebook (< 20 cells, all cells provided below)'
			: 'Context window around selected/recent cells (notebook has 20+ cells)';
		parts.push(xml.node('context-mode', contextMode));
	} else {
		parts.push(xml.node('context-mode', 'Selected cells only (use GetNotebookCells for other cells)'));
	}
	parts.push('</notebook-info>');

	// Selected cells section (if any selected)
	if (context.selectedCellsInfo && context.selectedCellsInfo !== 'No cells currently selected') {
		parts.push('');
		parts.push('<selected-cells>');
		parts.push(context.selectedCellsInfo);
		parts.push('</selected-cells>');
	}

	// All cells section (if available)
	if (context.allCellsInfo) {
		parts.push('');
		parts.push(context.allCellsInfo);
	}

	// Context note
	parts.push('');
	parts.push(context.contextNote);

	parts.push('</notebook-state>');

	return parts.join('\n');
}
