/*---------------------------------------------------------------------------------------------
 *  Copyright (C) 2025-2026 Posit Software, PBC. All rights reserved.
 *  Licensed under the Elastic License 2.0. See LICENSE.txt for license information.
 *--------------------------------------------------------------------------------------------*/

import * as vscode from 'vscode';
import { getAllModelDefinitions } from './modelDefinitions.js';
import { DEFAULT_MAX_TOKEN_INPUT, DEFAULT_MAX_TOKEN_OUTPUT, MIN_TOKEN_LIMIT, DEFAULT_MODEL_CAPABILITIES } from './constants.js';
import { log } from './log.js';
import { getSettingNameForProvider } from './providerMetadata.js';

/**
 * Type definition for token limits configuration from user settings.
 */
export interface TokenLimits {
	maxInput: Record<string, number>;
	maxOutput: Record<string, number>;
}

/**
 * Retrieves user-configured token limits from workspace settings.
 * These settings allow users to override default token limits for specific models.
 *
 * @returns Object containing maxInput and maxOutput token limits by model ID
 */
export function getUserTokenLimits(): TokenLimits {
	const config = vscode.workspace.getConfiguration('positron.assistant');
	return {
		maxInput: config.get('maxInputTokens', {}),
		maxOutput: config.get('maxOutputTokens', {})
	};
}

/**
 * Finds the index of the best matching model for a pattern.
 * Prefers exact ID or name match over partial match. Matching is case-insensitive.
 *
 * @param models Array of models to search
 * @param pattern Pattern to match against model ID or name
 * @returns Index of matching model, or -1 if no match
 */
export function findMatchingModelIndex(
	models: vscode.LanguageModelChatInformation[],
	pattern: string
): number {
	const patternLower = pattern.toLowerCase();
	let firstPartialMatchIndex = -1;

	for (let i = 0; i < models.length; i++) {
		const model = models[i];
		const idLower = model.id.toLowerCase();
		const nameLower = model.name?.toLowerCase() ?? '';

		// Exact match on ID or name - use it immediately
		if (idLower === patternLower || nameLower === patternLower) {
			return i;
		}

		// Track first partial match (includes pattern in id or name)
		if (firstPartialMatchIndex === -1) {
			if (idLower.includes(patternLower) || nameLower.includes(patternLower)) {
				firstPartialMatchIndex = i;
			}
		}
	}

	return firstPartialMatchIndex;
}

/**
 * Resolves the maximum token count for a model with proper fallback hierarchy.
 *
 * Priority order:
 * 1. User override from workspace settings (maxInputTokens/maxOutputTokens)
 * 2. Model definition limits from getAllModelDefinitions()
 * 3. Provider-specific defaults
 * 4. Global defaults
 *
 * Includes validation to ensure minimum token limit with helpful warnings.
 *
 * @param id The model ID to resolve tokens for
 * @param type Whether to resolve 'input' or 'output' tokens
 * @param provider The provider ID (e.g., 'anthropic-api', 'posit-ai')
 * @param providerDefault Optional provider-specific default (overrides global default)
 * @param providerName Optional provider display name for logging
 * @returns The resolved token limit
 */
export function getMaxTokens(
	id: string,
	type: 'input' | 'output',
	provider: string,
	providerDefault?: number,
	providerName?: string
): number {
	const globalDefault = type === 'input' ? DEFAULT_MAX_TOKEN_INPUT : DEFAULT_MAX_TOKEN_OUTPUT;
	const defaultTokens = providerDefault ?? globalDefault;

	// Get model-specific limits from definitions
	const configuredModels = getAllModelDefinitions(provider);
	const fixedValue = type === 'input'
		? configuredModels?.find(m => m.identifier === id)?.maxInputTokens
		: configuredModels?.find(m => m.identifier === id)?.maxOutputTokens;
	let maxTokens = fixedValue ?? defaultTokens;

	// Apply user overrides from workspace settings
	const configKey = type === 'input' ? 'maxInputTokens' : 'maxOutputTokens';
	const tokensConfig: Record<string, number> = vscode.workspace.getConfiguration('positron.assistant').get(configKey, {});
	for (const [key, value] of Object.entries(tokensConfig)) {
		if (id.indexOf(key) !== -1 && value) {
			if (typeof value !== 'number') {
				log.warn(`[${providerName ?? provider}] Invalid ${configKey} '${value}' for ${key} (${id}); ignoring`);
				continue;
			}
			if (value < MIN_TOKEN_LIMIT) {
				log.warn(`[${providerName ?? provider}] Specified ${configKey} '${value}' for ${key} (${id}) is too low; using ${MIN_TOKEN_LIMIT} instead`);
				maxTokens = MIN_TOKEN_LIMIT;
			} else {
				maxTokens = value;
			}
			break;
		}
	}

	log.trace(`[${providerName ?? provider}] Setting ${configKey} for (${id}) to ${maxTokens}`);
	return maxTokens;
}

/**
 * Parameters for creating a language model chat information object.
 */
export interface CreateModelInfoParams {
	/** The unique model ID */
	id: string;
	/** The display name of the model */
	name: string;
	/** The provider/family of the model */
	family: string;
	/** The version identifier of the model */
	version: string;
	/** The provider ID for token resolution */
	provider: string;
	/** The provider display name for logging */
	providerName: string;
	/** Model capabilities */
	capabilities?: vscode.LanguageModelChatCapabilities;
	/** Optional default max input tokens (overrides model definition defaults) */
	defaultMaxInput?: number;
	/** Optional default max output tokens (overrides model definition defaults) */
	defaultMaxOutput?: number;
}

/**
 * Creates a standardized LanguageModelChatInformation object.
 *
 * This shared utility ensures consistent model information creation across all providers
 * while applying the proper token resolution. Default model selection is handled separately
 * by the markDefaultModel function to avoid duplication.
 *
 * @param params The parameters for creating the model information
 * @returns A complete LanguageModelChatInformation object with isDefault set to false
 */
export function createModelInfo(params: CreateModelInfoParams): vscode.LanguageModelChatInformation {
	const {
		id,
		name,
		family,
		version,
		provider,
		providerName,
		capabilities = DEFAULT_MODEL_CAPABILITIES,
		defaultMaxInput,
		defaultMaxOutput
	} = params;

	return {
		id,
		name,
		family,
		version,
		maxInputTokens: getMaxTokens(id, 'input', provider, defaultMaxInput, providerName),
		maxOutputTokens: getMaxTokens(id, 'output', provider, defaultMaxOutput, providerName),
		capabilities,
		isDefault: false,
		isUserSelectable: true,
	};
}

/**
 * Marks models as default, ensuring only one default per provider.
 *
 * Priority order:
 * 1. Individual provider setting (models.preference.<settingName>)
 * 2. Provider-specific defaultMatch pattern (exact match preferred, then partial)
 * 3. First model in list
 *
 * @param models Array of models to process
 * @param provider The provider ID (used for default model detection)
 * @param defaultMatch Optional fallback pattern to match against for default selection
 * @returns Array of models with exactly one marked as default
 */
export function markDefaultModel(
	models: vscode.LanguageModelChatInformation[],
	provider: string,
	defaultMatch?: string
): vscode.LanguageModelChatInformation[] {
	if (models.length === 0) {
		return models;
	}

	const config = vscode.workspace.getConfiguration('positron.assistant');
	let defaultModelIndex = -1;

	const settingName = getSettingNameForProvider(provider);

	// Try individual provider setting
	if (settingName) {
		const individualPref = config.get<string>(`models.preference.${settingName}`);
		if (individualPref) {
			defaultModelIndex = findMatchingModelIndex(models, individualPref);
		}
	}

	// Fall back to provider-specific defaultMatch
	if (defaultModelIndex === -1 && defaultMatch) {
		defaultModelIndex = findMatchingModelIndex(models, defaultMatch);
	}

	// Fall back to first model
	if (defaultModelIndex === -1) {
		defaultModelIndex = 0;
	}

	return models.map((model, index) => ({
		...model,
		isDefault: index === defaultModelIndex,
	}));
}
