"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.getCertificateCrlUrls = exports.CRL_SIGNATURE_OID_TO_CRYPTO_DIGEST_ALGORITHM = void 0;
exports.getCertificateDebugName = getCertificateDebugName;
exports.isShortLivedCertificate = isShortLivedCertificate;
exports.isCrlSignatureValid = isCrlSignatureValid;
exports.isCertificateRevoked = isCertificateRevoked;
exports.isIssuingDistributionPointExtensionValid = isIssuingDistributionPointExtensionValid;
const crypto_1 = __importDefault(require("crypto"));
const asn1_js_rfc5280_1 = __importDefault(require("asn1.js-rfc5280"));
const logger_1 = __importDefault(require("../logger"));
// TODO:
// Implement RSASSA-PSS signature verification
// https://snowflakecomputing.atlassian.net/browse/SNOW-2333028
exports.CRL_SIGNATURE_OID_TO_CRYPTO_DIGEST_ALGORITHM = {
    '1.2.840.113549.1.1.11': 'sha256',
    '1.2.840.113549.1.1.12': 'sha384',
    '1.2.840.113549.1.1.13': 'sha512',
    '1.2.840.10045.4.3.2': 'sha256',
    '1.2.840.10045.4.3.3': 'sha384',
    '1.2.840.10045.4.3.4': 'sha512',
};
function getCertificateDebugName(certificate) {
    return [
        `O:${certificate.subject.O}`,
        `CN:${certificate.subject.CN}`,
        `SN:${certificate.serialNumber}`,
    ].join(',');
}
const getCertificateCrlUrls = (certificateName, decodedCertificate) => {
    const logDebug = (msg, ...msgArgs) => (0, logger_1.default)().debug(`getCertificateCrlUrls[${certificateName}]: ${msg}`, ...msgArgs);
    const crlExtension = decodedCertificate.tbsCertificate.extensions?.find((ext) => ext.extnID === 'cRLDistributionPoints');
    if (!crlExtension) {
        logDebug('certificate doesnt have cRLDistributionPoints extension');
        return null;
    }
    const result = [];
    for (const entry of crlExtension.extnValue) {
        if (!entry.distributionPoint) {
            logDebug('skipping entry without distributionPoint %j', entry);
            continue;
        }
        for (const fullNameEntry of entry.distributionPoint.value) {
            if (fullNameEntry.type !== 'uniformResourceIdentifier') {
                logDebug('skipping non-uniformResourceIdentifier entry %j', fullNameEntry);
                continue;
            }
            if (fullNameEntry.value.startsWith('http')) {
                // Even though the spec allows multiple http urls, we only pick first one and don't handle redundancy
                result.push(fullNameEntry.value);
                break;
            }
            else {
                logDebug('skipping non-http value %j', fullNameEntry);
            }
        }
    }
    logDebug(`found URLs: ${result.join(',')}`);
    return result.length > 0 ? result : null;
};
exports.getCertificateCrlUrls = getCertificateCrlUrls;
/**
 * See Short-lived Subscriber Certificate section\
 * https://cabforum.org/working-groups/server/baseline-requirements/requirements/
 */
function isShortLivedCertificate(decodedCertificate) {
    const notBefore = new Date(decodedCertificate.tbsCertificate.validity.notBefore.value);
    const notAfter = new Date(decodedCertificate.tbsCertificate.validity.notAfter.value);
    let maximumValidityPeriod = 7 * 24 * 60 * 60 * 1000; // 7 days in milliseconds
    if (notBefore < new Date('2026-03-15T00:00:00.000Z')) {
        maximumValidityPeriod = 10 * 24 * 60 * 60 * 1000; // 10 days in milliseconds
    }
    maximumValidityPeriod += 60 * 1000; // Fix inclusion start and end time (1 minute)
    const certValidityPeriod = notAfter.getTime() - notBefore.getTime();
    return maximumValidityPeriod > certValidityPeriod;
}
function isCrlSignatureValid(crl, issuerPublicKey) {
    const signatureAlgOid = crl.signatureAlgorithm.algorithm.join('.');
    const digestAlg = exports.CRL_SIGNATURE_OID_TO_CRYPTO_DIGEST_ALGORITHM[signatureAlgOid];
    if (!digestAlg) {
        throw new Error(`Unsupported signature algorithm: ${signatureAlgOid}`);
    }
    const verify = crypto_1.default.createVerify(digestAlg);
    const tbsEncoded = asn1_js_rfc5280_1.default.TBSCertList.encode(crl.tbsCertList, 'der');
    verify.update(tbsEncoded);
    return verify.verify(issuerPublicKey, crl.signature.data);
}
function isCertificateRevoked(decodedCertificate, crl) {
    for (const revokedCert of crl.tbsCertList.revokedCertificates) {
        if (revokedCert.userCertificate.eq(decodedCertificate.tbsCertificate.serialNumber)) {
            return true;
        }
    }
    return false;
}
function isIssuingDistributionPointExtensionValid(crl, expectedCrlUrl) {
    const issuingDistributionPointExtension = crl.tbsCertList.crlExtensions?.find((ext) => ext.extnID === 'issuingDistributionPoint');
    if (!issuingDistributionPointExtension) {
        (0, logger_1.default)().debug(`CRL ${expectedCrlUrl} doesnt have issuingDistributionPoint extension, ignoring`);
        return true;
    }
    for (const fullNameEntry of issuingDistributionPointExtension.extnValue.distributionPoint.value) {
        if (fullNameEntry.type === 'uniformResourceIdentifier' &&
            fullNameEntry.value === expectedCrlUrl) {
            return true;
        }
    }
    return false;
}
//# sourceMappingURL=crl_utils.js.map