"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const attestation_aws_1 = require("./attestation_aws");
const types_1 = require("./types");
const errors_1 = require("../../errors");
const logger_1 = __importDefault(require("../../logger"));
const attestation_azure_1 = require("./attestation_azure");
const attestation_gcp_1 = require("./attestation_gcp");
const authentication_types_1 = __importDefault(require("../authentication_types"));
class AuthWorkloadIdentity {
    connectionConfig;
    tokenProvider;
    token;
    constructor(connectionConfig) {
        this.connectionConfig = connectionConfig;
    }
    updateBody(body) {
        body.data['AUTHENTICATOR'] = authentication_types_1.default.WORKLOAD_IDENTITY;
        body.data['PROVIDER'] = this.tokenProvider;
        body.data['TOKEN'] = this.token;
    }
    async authenticate() {
        const { workloadIdentityProvider: provider, workloadIdentityImpersonationPath: impersonationPath, } = this.connectionConfig;
        let token;
        if (impersonationPath &&
            provider !== types_1.WorkloadIdentityProvider.AWS &&
            provider !== types_1.WorkloadIdentityProvider.GCP) {
            throw new Error(`workloadIdentityImpersonationPath for ${provider} is not supported`);
        }
        if (provider === types_1.WorkloadIdentityProvider.AWS) {
            token = await (0, attestation_aws_1.getAwsAttestationToken)(impersonationPath);
        }
        else if (provider === types_1.WorkloadIdentityProvider.AZURE) {
            token = await (0, attestation_azure_1.getAzureAttestationToken)({
                managedIdentityClientId: this.connectionConfig.workloadIdentityAzureClientId,
                entraIdResource: this.connectionConfig.workloadIdentityAzureEntraIdResource,
            });
        }
        else if (provider === types_1.WorkloadIdentityProvider.GCP) {
            token = await (0, attestation_gcp_1.getGcpAttestationToken)(impersonationPath);
        }
        else if (provider === types_1.WorkloadIdentityProvider.OIDC) {
            if (this.connectionConfig.token) {
                token = this.connectionConfig.token;
            }
            else {
                throw (0, errors_1.createInvalidParameterError)(errors_1.ErrorCode.ERR_CONN_CREATE_INVALID_WORKLOAD_IDENTITY_PARAMETERS, `workloadIdentityProvider: OIDC requires token in connection options`);
            }
        }
        else {
            throw (0, errors_1.createInvalidParameterError)(errors_1.ErrorCode.ERR_CONN_CREATE_INVALID_WORKLOAD_IDENTITY_PARAMETERS, `workloadIdentityProvider must be one of: ${Object.values(types_1.WorkloadIdentityProvider).join(', ')}`);
        }
        (0, logger_1.default)().debug(`AuthWorkloadIdentity using provider=${provider}`);
        this.tokenProvider = provider;
        this.token = token;
    }
}
exports.default = AuthWorkloadIdentity;
//# sourceMappingURL=auth_workload_identity.js.map