// Ported from js-yaml v3.13.1:
// Copyright 2011-2015 by Vitaly Puzrin. All rights reserved. MIT license.
// https://github.com/nodeca/js-yaml/commit/665aadda42349dcae869f12040d9b10ef18d12da
// Copyright 2018-2024 the Deno authors. All rights reserved. MIT license.
import { Type } from "../type.ts";
// [ 64, 65, 66 ] -> [ padding, CR, LF ]
const BASE64_MAP = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/=\n\r";
function resolveYamlBinary(data) {
  if (data === null) return false;
  let code;
  let bitlen = 0;
  const max = data.length;
  const map = BASE64_MAP;
  // Convert one by one.
  for(let idx = 0; idx < max; idx++){
    code = map.indexOf(data.charAt(idx));
    // Skip CR/LF
    if (code > 64) continue;
    // Fail on illegal characters
    if (code < 0) return false;
    bitlen += 6;
  }
  // If there are any bits left, source was corrupted
  return bitlen % 8 === 0;
}
function constructYamlBinary(data) {
  // remove CR/LF & padding to simplify scan
  const input = data.replace(/[\r\n=]/g, "");
  const max = input.length;
  const map = BASE64_MAP;
  // Collect by 6*4 bits (3 bytes)
  const result = [];
  let bits = 0;
  for(let idx = 0; idx < max; idx++){
    if (idx % 4 === 0 && idx) {
      result.push(bits >> 16 & 0xff);
      result.push(bits >> 8 & 0xff);
      result.push(bits & 0xff);
    }
    bits = bits << 6 | map.indexOf(input.charAt(idx));
  }
  // Dump tail
  const tailbits = max % 4 * 6;
  if (tailbits === 0) {
    result.push(bits >> 16 & 0xff);
    result.push(bits >> 8 & 0xff);
    result.push(bits & 0xff);
  } else if (tailbits === 18) {
    result.push(bits >> 10 & 0xff);
    result.push(bits >> 2 & 0xff);
  } else if (tailbits === 12) {
    result.push(bits >> 4 & 0xff);
  }
  return new Uint8Array(result);
}
function representYamlBinary(object) {
  const max = object.length;
  const map = BASE64_MAP;
  // Convert every three bytes to 4 ASCII characters.
  let result = "";
  let bits = 0;
  for(let idx = 0; idx < max; idx++){
    if (idx % 3 === 0 && idx) {
      result += map[bits >> 18 & 0x3f];
      result += map[bits >> 12 & 0x3f];
      result += map[bits >> 6 & 0x3f];
      result += map[bits & 0x3f];
    }
    bits = (bits << 8) + object[idx];
  }
  // Dump tail
  const tail = max % 3;
  if (tail === 0) {
    result += map[bits >> 18 & 0x3f];
    result += map[bits >> 12 & 0x3f];
    result += map[bits >> 6 & 0x3f];
    result += map[bits & 0x3f];
  } else if (tail === 2) {
    result += map[bits >> 10 & 0x3f];
    result += map[bits >> 4 & 0x3f];
    result += map[bits << 2 & 0x3f];
    result += map[64];
  } else if (tail === 1) {
    result += map[bits >> 2 & 0x3f];
    result += map[bits << 4 & 0x3f];
    result += map[64];
    result += map[64];
  }
  return result;
}
function isBinary(obj) {
  return obj instanceof Uint8Array;
}
export const binary = new Type("tag:yaml.org,2002:binary", {
  construct: constructYamlBinary,
  kind: "scalar",
  predicate: isBinary,
  represent: representYamlBinary,
  resolve: resolveYamlBinary
});
//# sourceMappingURL=data:application/json;base64,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