"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const jsdom_1 = require("jsdom");
const fs_1 = __importDefault(require("fs"));
const path_1 = __importDefault(require("path"));
const vscode_api_1 = require("./vscode_api");
const test_data_1 = require("./test_data");
const __1 = require("..");
// load our html file
let htmlFile = fs_1.default.readFileSync(path_1.default.resolve("./index.html"), "utf8");
// change the script locations
htmlFile = htmlFile.replace("%LAUNCHER-UI%", path_1.default.resolve("launcher/launcher.js"));
htmlFile = htmlFile.replace("%SCRIPT%", path_1.default.resolve("dist/index.js"));
// process the html file
const processedDom = new jsdom_1.JSDOM(htmlFile);
// eject everything in head which is not needed for testing, preserve active-file meta
for (const child of processedDom.window.document.head.children) {
    if (child.localName.toLowerCase() === "meta" && child.getAttribute("name") === "active-file") {
        continue;
    }
    child.remove();
}
// re-serialize now that we have our sanitized htmlFile. We can use
// this when loading copies of this DOM for testing
htmlFile = processedDom.serialize();
function newDom(api_1) {
    return __awaiter(this, arguments, void 0, function* (api, activeFile = "") {
        const newHtmlFile = htmlFile.replace("%ACTIVE-FILE%", activeFile);
        const processedDom = new jsdom_1.JSDOM(newHtmlFile, {
            url: `file://${path_1.default.resolve("./.")}`,
            runScripts: "dangerously",
            resources: "usable"
        });
        Object.assign(processedDom.window, {
            acquireVsCodeApi: () => {
                if (api) {
                    api.postBack = (message) => {
                        processedDom.window.postMessage(message, "*");
                    };
                }
                return api || (0, vscode_api_1.acquireVsCodeApi)();
            }
        });
        Object.assign(processedDom.window.Range.prototype, {
            getBoundingClientRect: () => {
                return {
                    x: 0,
                    y: 0,
                    width: 0,
                    height: 0,
                    top: 0,
                    right: 0,
                    bottom: 0,
                    left: 0,
                    toJSON: jest.fn()
                };
            }
        });
        yield new Promise((resolve) => {
            processedDom.window.addEventListener("load", () => {
                resolve();
            });
        });
        return processedDom;
    });
}
const BROWSE_PATH_SCRIPT = "/path/to/code.R";
const BROWSE_PATH_DIRECTORY = "/path/to/";
function createDefaultApi(postMessage) {
    const state = {};
    const api = (0, vscode_api_1.acquireVsCodeApi)(state);
    api.postBack =
        postMessage ||
            (() => {
                /* do nothing */
            });
    api.postMessage = (message) => __awaiter(this, void 0, void 0, function* () {
        var _a, _b, _c;
        switch (message.event) {
            case "launch":
                console.log({
                    launchSpec: message.launchSpec,
                    jobOptions: message.jobOptions
                });
                break;
            case "get_info":
                (_a = api.postBack) === null || _a === void 0 ? void 0 : _a.call(api, {
                    type: "get_info",
                    message: JSON.stringify(test_data_1.launcherInfo, null, 2)
                });
                break;
            case "browse_path":
                (_b = api.postBack) === null || _b === void 0 ? void 0 : _b.call(api, {
                    type: "browse_path",
                    id: message.id,
                    message: message.directory ? BROWSE_PATH_DIRECTORY : BROWSE_PATH_SCRIPT,
                    canceled: false
                });
                break;
            case "get_info_for_script": {
                let response = "{}";
                switch (message.ext) {
                    case "r":
                        response = JSON.stringify(test_data_1.rScriptInfo, null, 2);
                        break;
                    case "py":
                        response = JSON.stringify(test_data_1.pyScriptInfo, null, 2);
                        break;
                }
                (_c = api.postBack) === null || _c === void 0 ? void 0 : _c.call(api, {
                    type: "get_info_for_script",
                    id: message.id,
                    message: response
                });
                break;
            }
        }
    });
    return { state, api };
}
const baseState = () => {
    return {
        input: {
            default_names: ["RStudio Job"],
            show_clusters: "non-interactive",
            show_workbenches: false,
            workbenches: ["RStudio"],
            launch_spec: null
        },
        uiOptions: {
            fieldset_class: "",
            form_group_class: "form-group",
            input_addon_class: "input-group-addon",
            input_hint_class: "form-text text-muted",
            input_hint_error_class: "form-text text-muted input-error",
            job_name_label: "Job Name",
            job_name_placeholder: "New Job",
            label_class: "col-sm-3 control-label",
            link_label_class: "",
            select_class: "form-control",
            show_legend: true,
            table_layout: true,
            textbox_class: "form-control",
            textbox_error_class: "form-control input-error"
        },
        launchSpec: {
            name: "RStudio Job",
            cluster: "Local",
            placement_constraints: [],
            resource_limits: [],
            queues: [],
            container_image: "",
            default_image: "",
            container_images: [""]
        },
        jobOptions: { script: "", workDir: "", args: [] },
        userModified: {},
        outputArgs: []
    };
};
test("smoke test (should not time out)", () => __awaiter(void 0, void 0, void 0, function* () {
    const { api } = createDefaultApi();
    const dom = yield newDom(api);
    yield new Promise((resolve) => {
        dom.window.addEventListener("message", (message) => {
            if (message.data.type !== "get_info") {
                return;
            }
            resolve();
        });
    });
}));
describe("initialization", () => {
    test("base state", () => __awaiter(void 0, void 0, void 0, function* () {
        const { state, api } = createDefaultApi();
        const { window } = yield newDom(api);
        const { document } = window;
        yield new Promise((resolve) => {
            window.addEventListener("message", (message) => {
                if (message.data.type !== "get_info") {
                    return;
                }
                resolve();
            });
        });
        expect(state).toEqual(baseState());
        const jobNameEl = document.body.querySelector("#rstudio_label_job_name");
        expect(jobNameEl.value).toBe("RStudio Job");
    }));
    test("VALID active R file", () => __awaiter(void 0, void 0, void 0, function* () {
        const fileName = "script_file";
        const validPath = `/path/to/${fileName}.R`;
        const { state, api } = createDefaultApi();
        const { window } = yield newDom(api, validPath);
        const { document } = window;
        yield Promise.all([
            new Promise((resolve) => {
                window.addEventListener("message", (message) => {
                    if (message.data.type !== "get_info") {
                        return;
                    }
                    resolve();
                });
            }),
            new Promise((resolve) => {
                window.addEventListener("message", (message) => {
                    if (message.data.type !== "get_info_for_script") {
                        return;
                    }
                    resolve();
                });
            })
        ]);
        const expectedArgs = [
            {
                name: "r-version",
                value: "(System Default)"
            },
            {
                name: undefined,
                value: "-s"
            },
            {
                name: undefined,
                value: "--no-save"
            },
            {
                name: undefined,
                value: "--no-restore"
            }
        ];
        // if initializing with a file, the state should be
        // immediately updated with the new information
        const base = baseState();
        expect(state).toEqual(Object.assign(Object.assign({}, base), { jobOptions: Object.assign(Object.assign({}, base.jobOptions), { args: expectedArgs, script: validPath, workDir: test_data_1.pyScriptInfo.workingDirectory }), launchSpec: Object.assign(Object.assign({}, base.launchSpec), { name: fileName }), outputArgs: [[expectedArgs[0]], expectedArgs.slice(1)] }));
        // verify the UI elements display what they should
        const jobNameEl = document.body.querySelector("#rstudio_label_job_name");
        expect(jobNameEl.value).toBe("script_file");
        const scriptNameEl = document.body.querySelector("#launcher-script-path");
        expect(scriptNameEl.textContent).toBe(validPath);
        const versionSelectorEl = document.body.querySelector("#r-version-selector");
        expect(versionSelectorEl.value).toBe("(System Default)");
    }));
    test("VALID active PY file", () => __awaiter(void 0, void 0, void 0, function* () {
        const fileName = "script_file";
        const validPath = `/path/to/${fileName}.py`;
        const { state, api } = createDefaultApi();
        const { window } = yield newDom(api, validPath);
        const { document } = window;
        yield Promise.all([
            new Promise((resolve) => {
                window.addEventListener("message", (message) => {
                    if (message.data.type !== "get_info") {
                        return;
                    }
                    resolve();
                });
            }),
            new Promise((resolve) => {
                window.addEventListener("message", (message) => {
                    if (message.data.type !== "get_info_for_script") {
                        return;
                    }
                    resolve();
                });
            })
        ]);
        const expectedArgs = [
            {
                name: "py-version",
                value: "(System Default)"
            },
            {
                name: undefined,
                value: "-u"
            }
        ];
        // if initializing with a file, the state should be
        // immediately updated with the new information
        const base = baseState();
        expect(state).toEqual(Object.assign(Object.assign({}, base), { jobOptions: Object.assign(Object.assign({}, base.jobOptions), { args: expectedArgs, script: validPath, workDir: test_data_1.pyScriptInfo.workingDirectory }), launchSpec: Object.assign(Object.assign({}, base.launchSpec), { name: fileName }), outputArgs: [[expectedArgs[0]], expectedArgs.slice(1)] }));
        // verify the UI elements display what they should
        const jobNameEl = document.body.querySelector("#rstudio_label_job_name");
        expect(jobNameEl.value).toBe("script_file");
        const scriptNameEl = document.body.querySelector("#launcher-script-path");
        expect(scriptNameEl.textContent).toBe(validPath);
        const versionSelectorEl = document.body.querySelector("#py-version-selector");
        expect(versionSelectorEl.value).toBe("(System Default)");
    }));
    /*
    test('VALID active IPYNB file', async () => {
      
      const fileName = 'script_file';
      const validPath = `/path/to/${fileName}.ipynb`;
  
      const { state, api } = createDefaultApi();
      const { window } = await newDom(api, validPath);
      const { document } = window;
  
      await Promise.all([
        new Promise<void>(resolve => {
          window.addEventListener('message', message => {
            if(message.data.type !== 'get_info') { return; }
            resolve();
          });
        }),
        new Promise<void>(resolve => {
          window.addEventListener('message', message => {
            if(message.data.type !== 'get_info_for_script') { return; }
            resolve();
          });
        })
      ]);
  
      const expectedArgs: Arg[] = [];
  
      // if initializing with a file, the state should be
      // immediately updated with the new information
      const base = baseState();
      expect(state).toEqual({
        ...base,
        jobOptions: {
          ...base.jobOptions,
          args: expectedArgs,
          script: validPath
        },
        launchSpec: {
          ...base.launchSpec,
          name: fileName
        },
        outputArgs: [ [] ]
      });
  
      // verify the UI elements display what they should
      const jobNameEl = document.body.querySelector('#rstudio_label_job_name') as HTMLInputElement;
      expect(jobNameEl.value).toBe('script_file');
      
      const scriptNameEl = document.body.querySelector('#launcher-script-path') as HTMLDivElement;
      expect(scriptNameEl.textContent).toBe(validPath);
      
      // there should be no version selector on screen for ipynb
      expect(
        document.body.querySelector('select[id$="-version-selector"]') as HTMLSelectElement
      ).toBe(null);
      
      // there should be arguments controls
      expect(
        document.body.querySelector('div.argument-summary') as HTMLDivElement
      ).not.toBe(null);
    });
  
    test('VALID active QMD file', async () => {
      
      const fileName = 'script_file';
      const validPath = `/path/to/${fileName}.Qmd`;
  
      const { state, api } = createDefaultApi();
      const { window } = await newDom(api, validPath);
      const { document } = window;
  
      await Promise.all([
        new Promise<void>(resolve => {
          window.addEventListener('message', message => {
            if(message.data.type !== 'get_info') { return; }
            resolve();
          });
        }),
        new Promise<void>(resolve => {
          window.addEventListener('message', message => {
            if(message.data.type !== 'get_info_for_script') { return; }
            resolve();
          });
        })
      ]);
  
      const expectedArgs: Arg[] = [];
  
      // if initializing with a file, the state should be
      // immediately updated with the new information
      const base = baseState();
      expect(state).toEqual({
        ...base,
        jobOptions: {
          ...base.jobOptions,
          args: expectedArgs,
          script: validPath
        },
        launchSpec: {
          ...base.launchSpec,
          name: fileName
        },
        outputArgs: [ [] ]
      });
  
      // verify the UI elements display what they should
      const jobNameEl = document.body.querySelector('#rstudio_label_job_name') as HTMLInputElement;
      expect(jobNameEl.value).toBe('script_file');
      
      const scriptNameEl = document.body.querySelector('#launcher-script-path') as HTMLDivElement;
      expect(scriptNameEl.textContent).toBe(validPath);
      
      // there should be no version selector on screen for Qmd
      expect(
        document.body.querySelector('select[id$="-version-selector"]') as HTMLSelectElement
      ).toBe(null);
      
      // there should be arguments controls
      expect(
        document.body.querySelector('div.argument-summary') as HTMLDivElement
      ).not.toBe(null);
    });
  
    test('VALID active RMD file', async () => {
      
      const fileName = 'script_file';
      const validPath = `/path/to/${fileName}.rmd`;
  
      const { state, api } = createDefaultApi();
      const { window } = await newDom(api, validPath);
      const { document } = window;
  
      await Promise.all([
        new Promise<void>(resolve => {
          window.addEventListener('message', message => {
            if(message.data.type !== 'get_info') { return; }
            resolve();
          });
        }),
        new Promise<void>(resolve => {
          window.addEventListener('message', message => {
            if(message.data.type !== 'get_info_for_script') { return; }
            resolve();
          });
        })
      ]);
  
      const expectedArgs: Arg[] = [];
  
      // if initializing with a file, the state should be
      // immediately updated with the new information
      const base = baseState();
      expect(state).toEqual({
        ...base,
        jobOptions: {
          ...base.jobOptions,
          args: expectedArgs,
          script: validPath
        },
        launchSpec: {
          ...base.launchSpec,
          name: fileName
        },
        outputArgs: [ [] ]
      });
  
      // verify the UI elements display what they should
      const jobNameEl = document.body.querySelector('#rstudio_label_job_name') as HTMLInputElement;
      expect(jobNameEl.value).toBe('script_file');
      
      const scriptNameEl = document.body.querySelector('#launcher-script-path') as HTMLDivElement;
      expect(scriptNameEl.textContent).toBe(validPath);
      
      // there should be no version selector on screen for rmd
      expect(
        document.body.querySelector('select[id$="-version-selector"]') as HTMLSelectElement
      ).toBe(null);
      
      // there should be arguments controls
      expect(
        document.body.querySelector('div.argument-summary') as HTMLDivElement
      ).not.toBe(null);
    });
    */
    test("INVALID active file", () => __awaiter(void 0, void 0, void 0, function* () {
        const fileName = "one_of_those_vs_code_output_streams";
        const validPath = fileName;
        const { state, api } = createDefaultApi();
        const { window } = yield newDom(api, validPath);
        const { document } = window;
        // we don't await get_info_for_script here because it shouldn't
        // be called if the active-file has an invalid extension
        yield new Promise((resolve) => {
            window.addEventListener("message", (message) => {
                if (message.data.type !== "get_info") {
                    return;
                }
                resolve();
            });
        });
        // invalid script file on init should be a no-op over the base state
        expect(state).toEqual(baseState());
        // verify the UI elements display what they should
        const jobNameEl = document.body.querySelector("#rstudio_label_job_name");
        expect(jobNameEl.value).toBe("RStudio Job");
        const scriptNameEl = document.body.querySelector("#launcher-script-path");
        expect(scriptNameEl.textContent).toBe("");
    }));
    test("modified base state", () => __awaiter(void 0, void 0, void 0, function* () {
        const scriptName = "complex_job";
        const validPath = `/path/to/${scriptName}.R`;
        const { state, api } = createDefaultApi();
        state.jobOptions = {
            script: validPath
        };
        state.outputArgs = [[{ name: "r-version", value: "3.6.3" }], [{ value: "--no-save" }, { value: "--made-up-arg" }]];
        state.launchSpec = {
            name: "Custom Job",
            cluster: "Kubernetes",
            placement_constraints: [],
            resource_limits: [
                { type: "cpuCount", value: "2" },
                { type: "memory", value: "600" }
            ],
            queues: [],
            container_image: "workbench/default-image",
            default_image: "",
            container_images: [""]
        };
        const { window } = yield newDom(api);
        const { document } = window;
        yield Promise.all([
            new Promise((resolve) => {
                window.addEventListener("message", (message) => {
                    if (message.data.type !== "get_info") {
                        return;
                    }
                    resolve();
                });
            }),
            new Promise((resolve) => {
                window.addEventListener("message", (message) => {
                    if (message.data.type !== "get_info_for_script") {
                        return;
                    }
                    resolve();
                });
            })
        ]);
        // verify the UI elements display what they should
        const jobNameEl = document.body.querySelector("#rstudio_label_job_name");
        expect(jobNameEl.value).toBe("Custom Job");
        // we are using kubernetes test get_info data, so verify the contents
        // of the displayed cluster option elements
        const clusterOptionsCpus = document.body.querySelector("#rstudio_label_cpus");
        expect(clusterOptionsCpus.value).toBe("2");
        const clusterOptionsMemory = document.body.querySelector("#rstudio_label_memory");
        expect(clusterOptionsMemory.value).toBe("0.59");
        const scriptNameEl = document.body.querySelector("#launcher-script-path");
        expect(scriptNameEl.textContent).toBe(validPath);
        const argSummaryEl = document.body.querySelector("div.argument-summary");
        expect(argSummaryEl.textContent).toBe("--no-save --made-up-arg");
        const argumentEls = document.body.querySelectorAll("input.form-control.argument");
        expect(argumentEls.length).toBe(2);
    }));
});
test("test that state saves with user updates", () => __awaiter(void 0, void 0, void 0, function* () {
    var _a, _b, _c, _d, _e, _f, _g, _h, _j;
    const fileName = "script_file";
    const validPath = `/path/to/${fileName}.R`;
    const { state, api } = createDefaultApi();
    const { window } = yield newDom(api, validPath);
    const { document } = window;
    yield Promise.all([
        new Promise((resolve) => {
            window.addEventListener("message", (message) => {
                if (message.data.type !== "get_info") {
                    return;
                }
                resolve();
            });
        }),
        new Promise((resolve) => {
            window.addEventListener("message", (message) => {
                if (message.data.type !== "get_info_for_script") {
                    return;
                }
                resolve();
            });
        })
    ]);
    // change the default name -- this is so we can test that usermodified
    // will correctly mask changes to this when the script is changed
    const jobNameEl = document.body.querySelector("#rstudio_label_job_name");
    jobNameEl.value = "Custom Job";
    jobNameEl.dispatchEvent(new window.Event("change"));
    expect((_a = state.launchSpec) === null || _a === void 0 ? void 0 : _a.name).toBe("Custom Job");
    const clusterEl = document.body.querySelector("#rstudio_label_cluster");
    clusterEl.value = "Kubernetes";
    clusterEl.dispatchEvent(new window.Event("change"));
    expect((_b = state.launchSpec) === null || _b === void 0 ? void 0 : _b.cluster).toBe("Kubernetes");
    // the following fields use 'blur' as a trigger for updates
    const clusterOptionsCpus = document.body.querySelector("#rstudio_label_cpus");
    clusterOptionsCpus.value = "2";
    clusterOptionsCpus.dispatchEvent(new window.Event("blur"));
    expect((_c = state.launchSpec) === null || _c === void 0 ? void 0 : _c.resource_limits[0].value).toBe("2");
    const clusterOptionsMemory = document.body.querySelector("#rstudio_label_memory");
    clusterOptionsMemory.value = "0.8"; // Under the 1024 MB limit from test data.
    clusterOptionsMemory.dispatchEvent(new window.Event("blur"));
    expect((_d = state.launchSpec) === null || _d === void 0 ? void 0 : _d.resource_limits[1].value).toBe("819.2");
    const scriptBrowseEl = document.body.querySelector("#launcher-browse-script-path");
    let onBrowse = new Promise((resolve) => {
        window.addEventListener("message", (message) => {
            if (message.data.type !== "browse_path") {
                return;
            }
            resolve();
        });
    });
    scriptBrowseEl.click();
    yield onBrowse;
    const scriptNameEl = document.body.querySelector("#launcher-script-path");
    expect(scriptNameEl.textContent).toBe(BROWSE_PATH_SCRIPT);
    expect((_e = state.jobOptions) === null || _e === void 0 ? void 0 : _e.script).toBe(BROWSE_PATH_SCRIPT);
    expect((_f = state.userModified) === null || _f === void 0 ? void 0 : _f.script).toBe(true);
    // this should not have changed since we already modified it
    expect((_g = state.launchSpec) === null || _g === void 0 ? void 0 : _g.name).toBe("Custom Job");
    const workDirBrowseEl = document.body.querySelector("#launcher-browse-workdir-path");
    onBrowse = new Promise((resolve) => {
        window.addEventListener("message", (message) => {
            if (message.data.type !== "browse_path") {
                return;
            }
            resolve();
        });
    });
    workDirBrowseEl.click();
    yield onBrowse;
    const workDirPathEl = document.body.querySelector("#launcher-workdir-path");
    expect(workDirPathEl.textContent).toBe(BROWSE_PATH_DIRECTORY);
    expect((_h = state.jobOptions) === null || _h === void 0 ? void 0 : _h.workDir).toBe(BROWSE_PATH_DIRECTORY);
    expect((_j = state.userModified) === null || _j === void 0 ? void 0 : _j.workDir).toBe(true);
}));
test("test that warning appears when non-local config is selected", () => __awaiter(void 0, void 0, void 0, function* () {
    var _a, _b, _c;
    const fileName = "script_file";
    const validPath = `/path/to/${fileName}.R`;
    const { state, api } = createDefaultApi();
    const { window } = yield newDom(api, validPath);
    const { document } = window;
    yield Promise.all([
        new Promise((resolve) => {
            window.addEventListener("message", (message) => {
                if (message.data.type !== "get_info") {
                    return;
                }
                resolve();
            });
        }),
        new Promise((resolve) => {
            window.addEventListener("message", (message) => {
                if (message.data.type !== "get_info_for_script") {
                    return;
                }
                resolve();
            });
        })
    ]);
    // change the cluster to not "Local"
    const clusterEl = document.body.querySelector("#rstudio_label_cluster");
    clusterEl.value = "Kubernetes";
    clusterEl.dispatchEvent(new window.Event("change"));
    expect((_a = state.launchSpec) === null || _a === void 0 ? void 0 : _a.cluster).toBe("Kubernetes");
    // verify that the warning is still not visible before changing the path
    const warningEl = document.body.querySelector("#version-path-warning");
    expect(warningEl.style.display).toBe("none");
    // change the version to a non-default value
    const versionSelectorEl = document.body.querySelector("#r-version-selector");
    versionSelectorEl.value = "4.2.1";
    versionSelectorEl.dispatchEvent(new window.Event("change"));
    // verify warning is visible and contains a mention of the cluster name and the version
    expect(warningEl.style.display).not.toBe("none");
    expect((_b = warningEl.textContent) === null || _b === void 0 ? void 0 : _b.includes("Kubernetes")).toBe(true);
    expect((_c = warningEl.textContent) === null || _c === void 0 ? void 0 : _c.includes("4.2.1")).toBe(true);
}));
describe("arguments", () => {
    test("modification hidden by default", () => __awaiter(void 0, void 0, void 0, function* () {
        const fileName = "script_file";
        const validPath = `/path/to/${fileName}.R`;
        const { state, api } = createDefaultApi();
        const { window } = yield newDom(api, validPath);
        const { document } = window;
        state.jobOptions = {
            script: validPath
        };
        yield Promise.all([
            new Promise((resolve) => {
                window.addEventListener("message", (message) => {
                    if (message.data.type !== "get_info") {
                        return;
                    }
                    resolve();
                });
            }),
            new Promise((resolve) => {
                window.addEventListener("message", (message) => {
                    if (message.data.type !== "get_info_for_script") {
                        return;
                    }
                    resolve();
                });
            })
        ]);
        const formGridEl = document.body.querySelector("#form-grid");
        expect(Array.from(formGridEl.classList).includes("show-arguments")).toBe(false);
        const argumentEls = formGridEl.querySelectorAll(".argument");
        for (const argumentEl of argumentEls) {
            expect(window.getComputedStyle(argumentEl).display === "none");
        }
    }));
    test("arguments show correctly", () => __awaiter(void 0, void 0, void 0, function* () {
        const fileName = "script_file";
        const validPath = `/path/to/${fileName}.R`;
        const { state, api } = createDefaultApi();
        const { window } = yield newDom(api, validPath);
        const { document } = window;
        state.jobOptions = {
            script: validPath
        };
        yield Promise.all([
            new Promise((resolve) => {
                window.addEventListener("message", (message) => {
                    if (message.data.type !== "get_info") {
                        return;
                    }
                    resolve();
                });
            }),
            new Promise((resolve) => {
                window.addEventListener("message", (message) => {
                    if (message.data.type !== "get_info_for_script") {
                        return;
                    }
                    resolve();
                });
            })
        ]);
        const modifyArgumentsButtomEl = document.body.querySelector("div.argument-summary + button.browse-button");
        modifyArgumentsButtomEl.click();
        const formGridEl = document.body.querySelector("#form-grid");
        expect(Array.from(formGridEl.classList).includes("show-arguments")).toBe(true);
        const argumentEls = formGridEl.querySelectorAll(".argument");
        for (const argumentEl of argumentEls) {
            expect(window.getComputedStyle(argumentEl).display !== "none");
        }
    }));
    test("modify multiple", () => __awaiter(void 0, void 0, void 0, function* () {
        var _a, _b;
        const fileName = "script_file";
        const validPath = `/path/to/${fileName}.R`;
        const { state, api } = createDefaultApi();
        const { window } = yield newDom(api, validPath);
        const { document } = window;
        state.jobOptions = {
            script: validPath
        };
        yield Promise.all([
            new Promise((resolve) => {
                window.addEventListener("message", (message) => {
                    if (message.data.type !== "get_info") {
                        return;
                    }
                    resolve();
                });
            }),
            new Promise((resolve) => {
                window.addEventListener("message", (message) => {
                    if (message.data.type !== "get_info_for_script") {
                        return;
                    }
                    resolve();
                });
            })
        ]);
        const modifyArgumentsButtomEl = document.body.querySelector("div.argument-summary + button.browse-button");
        modifyArgumentsButtomEl.click();
        const argumentInputEls = Array.from(document.body.querySelectorAll("#form-grid input.form-control.argument"));
        expect(argumentInputEls.length).toBe(test_data_1.rScriptInfo.scriptArgs.length);
        for (let a = 0; a < argumentInputEls.length; a++) {
            const argumentInputEl = argumentInputEls[a];
            argumentInputEl.value = `--arg${a}`;
            argumentInputEl.dispatchEvent(new window.Event("change"));
        }
        const argSummaryEl = document.body.querySelector("div.argument-summary");
        expect(argSummaryEl.textContent).toBe(test_data_1.rScriptInfo.scriptArgs.map((_, index) => `--arg${index}`).join(" "));
        // outputArgs[1] is the array of arguments associated with the general-purpose list
        expect((_a = state.outputArgs) === null || _a === void 0 ? void 0 : _a[1]).toEqual(test_data_1.rScriptInfo.scriptArgs.map((_, index) => {
            return { name: undefined, value: `--arg${index}` };
        }));
        // perform an arg reset and re-verify that all args are correct
        const resetButtonEl = document.body.querySelector("#form-grid button.reset-arguments");
        resetButtonEl.click();
        expect(argSummaryEl.textContent).toBe(test_data_1.rScriptInfo.scriptArgs.map((arg) => arg.name).join(" "));
        // outputArgs[1] is the array of arguments associated with the general-purpose list
        expect((_b = state.outputArgs) === null || _b === void 0 ? void 0 : _b[1]).toEqual(test_data_1.rScriptInfo.scriptArgs.map((arg) => {
            return { name: undefined, value: arg.name };
        }));
    }));
    test("add new", () => __awaiter(void 0, void 0, void 0, function* () {
        var _a, _b;
        const fileName = "script_file";
        const validPath = `/path/to/${fileName}.R`;
        const { state, api } = createDefaultApi();
        const { window } = yield newDom(api, validPath);
        const { document } = window;
        state.jobOptions = {
            script: validPath
        };
        yield Promise.all([
            new Promise((resolve) => {
                window.addEventListener("message", (message) => {
                    if (message.data.type !== "get_info") {
                        return;
                    }
                    resolve();
                });
            }),
            new Promise((resolve) => {
                window.addEventListener("message", (message) => {
                    if (message.data.type !== "get_info_for_script") {
                        return;
                    }
                    resolve();
                });
            })
        ]);
        const modifyArgumentsButtomEl = document.body.querySelector("div.argument-summary + button.browse-button");
        modifyArgumentsButtomEl.click();
        // verify lengths match before adding new
        expect(document.body.querySelectorAll("#form-grid input.form-control.argument").length).toBe(test_data_1.rScriptInfo.scriptArgs.length);
        const addArgumentButtonEl = document.body.querySelector("#form-grid button.add-argument");
        addArgumentButtonEl.click();
        // verify that there are now scriptArgs.length + 1 elements queried
        expect(document.body.querySelectorAll("#form-grid input.form-control.argument").length).toBe(test_data_1.rScriptInfo.scriptArgs.length + 1);
        // verify that the summary has not changed yet, adding a new blank argument does not affect summary
        const argSummaryEl = document.body.querySelector("div.argument-summary");
        expect(argSummaryEl.textContent).toBe(test_data_1.rScriptInfo.scriptArgs.map((arg) => arg.name).join(" "));
        // modify the final element
        const newArg = { name: "--new-arg" };
        const lastArgumentEl = document.body.querySelector("#form-grid input.form-control.argument:last-of-type");
        lastArgumentEl.value = newArg.name;
        lastArgumentEl.dispatchEvent(new window.Event("change"));
        // verify summary and state
        expect(argSummaryEl.textContent).toBe([...test_data_1.rScriptInfo.scriptArgs, newArg].map((arg) => arg.name).join(" "));
        // outputArgs[1] is the array of arguments associated with the general-purpose list
        expect((_a = state.outputArgs) === null || _a === void 0 ? void 0 : _a[1]).toEqual([...test_data_1.rScriptInfo.scriptArgs, newArg].map((arg) => {
            return { name: undefined, value: arg.name };
        }));
        // perform an arg reset and re-verify that all args are correct
        const resetButtonEl = document.body.querySelector("#form-grid button.reset-arguments");
        resetButtonEl.click();
        expect(argSummaryEl.textContent).toBe(test_data_1.rScriptInfo.scriptArgs.map((arg) => arg.name).join(" "));
        // outputArgs[1] is the array of arguments associated with the general-purpose list
        expect((_b = state.outputArgs) === null || _b === void 0 ? void 0 : _b[1]).toEqual(test_data_1.rScriptInfo.scriptArgs.map((arg) => {
            return { name: undefined, value: arg.name };
        }));
    }));
    test("remove one", () => __awaiter(void 0, void 0, void 0, function* () {
        var _a, _b;
        const fileName = "script_file";
        const validPath = `/path/to/${fileName}.R`;
        const { state, api } = createDefaultApi();
        const { window } = yield newDom(api, validPath);
        const { document } = window;
        state.jobOptions = {
            script: validPath
        };
        yield Promise.all([
            new Promise((resolve) => {
                window.addEventListener("message", (message) => {
                    if (message.data.type !== "get_info") {
                        return;
                    }
                    resolve();
                });
            }),
            new Promise((resolve) => {
                window.addEventListener("message", (message) => {
                    if (message.data.type !== "get_info_for_script") {
                        return;
                    }
                    resolve();
                });
            })
        ]);
        const modifyArgumentsButtomEl = document.body.querySelector("div.argument-summary + button.browse-button");
        modifyArgumentsButtomEl.click();
        // verify lengths match before adding new
        expect(document.body.querySelectorAll("#form-grid input.form-control.argument").length).toBe(test_data_1.rScriptInfo.scriptArgs.length);
        const deleteArgumentButtonEl = document.body.querySelector("#form-grid button.delete-argument");
        deleteArgumentButtonEl.click();
        // verify that there are now scriptArgs.length + 1 elements queried
        expect(document.body.querySelectorAll("#form-grid input.form-control.argument").length).toBe(test_data_1.rScriptInfo.scriptArgs.length - 1);
        // verify summary and state
        // use slice(1) because we've removed the first argument
        const argSummaryEl = document.body.querySelector("div.argument-summary");
        expect(argSummaryEl.textContent).toBe(test_data_1.rScriptInfo.scriptArgs
            .slice(1)
            .map((arg) => arg.name)
            .join(" "));
        // outputArgs[1] is the array of arguments associated with the general-purpose list
        expect((_a = state.outputArgs) === null || _a === void 0 ? void 0 : _a[1]).toEqual(test_data_1.rScriptInfo.scriptArgs.slice(1).map((arg) => {
            return { name: undefined, value: arg.name };
        }));
        // perform an arg reset and re-verify that all args are correct
        const resetButtonEl = document.body.querySelector("#form-grid button.reset-arguments");
        resetButtonEl.click();
        expect(argSummaryEl.textContent).toBe(test_data_1.rScriptInfo.scriptArgs.map((arg) => arg.name).join(" "));
        // outputArgs[1] is the array of arguments associated with the general-purpose list
        expect((_b = state.outputArgs) === null || _b === void 0 ? void 0 : _b[1]).toEqual(test_data_1.rScriptInfo.scriptArgs.map((arg) => {
            return { name: undefined, value: arg.name };
        }));
    }));
    test("remove all", () => __awaiter(void 0, void 0, void 0, function* () {
        var _a, _b;
        const fileName = "script_file";
        const validPath = `/path/to/${fileName}.R`;
        const { state, api } = createDefaultApi();
        const { window } = yield newDom(api, validPath);
        const { document } = window;
        state.jobOptions = {
            script: validPath
        };
        yield Promise.all([
            new Promise((resolve) => {
                window.addEventListener("message", (message) => {
                    if (message.data.type !== "get_info") {
                        return;
                    }
                    resolve();
                });
            }),
            new Promise((resolve) => {
                window.addEventListener("message", (message) => {
                    if (message.data.type !== "get_info_for_script") {
                        return;
                    }
                    resolve();
                });
            })
        ]);
        const modifyArgumentsButtomEl = document.body.querySelector("div.argument-summary + button.browse-button");
        modifyArgumentsButtomEl.click();
        // verify lengths match before adding new
        expect(document.body.querySelectorAll("#form-grid input.form-control.argument").length).toBe(test_data_1.rScriptInfo.scriptArgs.length);
        // delete all arguments
        const deleteArgumentButtonEls = Array.from(document.body.querySelectorAll("#form-grid button.delete-argument"));
        for (const deleteArgumentButtonEl of deleteArgumentButtonEls) {
            deleteArgumentButtonEl.click();
        }
        // verify that there are now ZERO  elements queried
        expect(document.body.querySelectorAll("#form-grid input.form-control.argument").length).toBe(0);
        // verify summary and state
        const argSummaryEl = document.body.querySelector("div.argument-summary");
        expect(argSummaryEl.textContent).toBe("");
        // outputArgs[1] is the array of arguments associated with the general-purpose list
        expect((_a = state.outputArgs) === null || _a === void 0 ? void 0 : _a[1]).toEqual([]);
        // perform an arg reset and re-verify that all args are correct
        const resetButtonEl = document.body.querySelector("#form-grid button.reset-arguments");
        resetButtonEl.click();
        expect(argSummaryEl.textContent).toBe(test_data_1.rScriptInfo.scriptArgs.map((arg) => arg.name).join(" "));
        // outputArgs[1] is the array of arguments associated with the general-purpose list
        expect((_b = state.outputArgs) === null || _b === void 0 ? void 0 : _b[1]).toEqual(test_data_1.rScriptInfo.scriptArgs.map((arg) => {
            return { name: undefined, value: arg.name };
        }));
    }));
});
describe("managed credentials", () => {
    test("checkbox section hidden by default", () => __awaiter(void 0, void 0, void 0, function* () {
        const { api } = createDefaultApi();
        const { window } = yield newDom(api);
        const { document } = window;
        yield new Promise((resolve) => {
            window.addEventListener("message", (message) => {
                if (message.data.type !== "get_info") {
                    return;
                }
                resolve();
            });
        });
        // Verify that the managed credentials section is hidden by default
        const managedCredsSection = document.body.querySelector("#managed-credentials");
        expect(managedCredsSection).not.toBe(null);
        expect(managedCredsSection.classList.contains("hidden")).toBe(true);
        // Verify the checkbox exists but is not visible
        const managedCredsCheckbox = document.body.querySelector("#managed-credentials-enabled");
        expect(managedCredsCheckbox).not.toBe(null);
        expect(managedCredsCheckbox.type).toBe("checkbox");
    }));
    test("checkbox shows when enabled via message", () => __awaiter(void 0, void 0, void 0, function* () {
        const { api } = createDefaultApi();
        const { window } = yield newDom(api);
        const { document } = window;
        yield new Promise((resolve) => {
            window.addEventListener("message", (message) => {
                if (message.data.type !== "get_info") {
                    return;
                }
                resolve();
            });
        });
        // Send managed_credentials enabled message
        window.postMessage({
            type: "managed_credentials",
            enabled: true
        }, "*");
        // Wait a bit for the message to be processed
        yield new Promise((resolve) => setTimeout(resolve, 10));
        // Verify that the managed credentials section is now visible
        const managedCredsSection = document.body.querySelector("#managed-credentials");
        expect(managedCredsSection.classList.contains("hidden")).toBe(false);
        // Verify the checkbox is checked by default when enabled
        const managedCredsCheckbox = document.body.querySelector("#managed-credentials-enabled");
        expect(managedCredsCheckbox.checked).toBe(true);
    }));
    test("checkbox hides when disabled via message", () => __awaiter(void 0, void 0, void 0, function* () {
        const { api } = createDefaultApi();
        const { window } = yield newDom(api);
        const { document } = window;
        yield new Promise((resolve) => {
            window.addEventListener("message", (message) => {
                if (message.data.type !== "get_info") {
                    return;
                }
                resolve();
            });
        });
        // First enable the checkbox
        window.postMessage({
            type: "managed_credentials",
            enabled: true
        }, "*");
        yield new Promise((resolve) => setTimeout(resolve, 10));
        // Verify it's visible
        const managedCredsSection = document.body.querySelector("#managed-credentials");
        expect(managedCredsSection.classList.contains("hidden")).toBe(false);
        // Now disable it
        window.postMessage({
            type: "managed_credentials",
            enabled: false
        }, "*");
        yield new Promise((resolve) => setTimeout(resolve, 10));
        // Verify it's hidden again
        expect(managedCredsSection.classList.contains("hidden")).toBe(true);
    }));
    test("checkbox user interaction updates state", () => __awaiter(void 0, void 0, void 0, function* () {
        var _a, _b, _c;
        const { api, state } = createDefaultApi();
        const { window } = yield newDom(api);
        const { document } = window;
        yield new Promise((resolve) => {
            window.addEventListener("message", (message) => {
                if (message.data.type !== "get_info") {
                    return;
                }
                resolve();
            });
        });
        // Enable managed credentials
        window.postMessage({
            type: "managed_credentials",
            enabled: true
        }, "*");
        yield new Promise((resolve) => setTimeout(resolve, 10));
        const managedCredsCheckbox = document.body.querySelector("#managed-credentials-enabled");
        // Initially should be checked
        expect(managedCredsCheckbox.checked).toBe(true);
        expect((_a = state.jobOptions) === null || _a === void 0 ? void 0 : _a.managedCredentialsEnabled).toBe(true);
        // Uncheck the checkbox
        managedCredsCheckbox.checked = false;
        managedCredsCheckbox.dispatchEvent(new window.Event("click"));
        // State should be updated
        expect((_b = state.jobOptions) === null || _b === void 0 ? void 0 : _b.managedCredentialsEnabled).toBe(false);
        // Check it again
        managedCredsCheckbox.checked = true;
        managedCredsCheckbox.dispatchEvent(new window.Event("click"));
        // State should be updated again
        expect((_c = state.jobOptions) === null || _c === void 0 ? void 0 : _c.managedCredentialsEnabled).toBe(true);
    }));
    test("user modification prevents automatic updates", () => __awaiter(void 0, void 0, void 0, function* () {
        const { api } = createDefaultApi();
        const { window } = yield newDom(api);
        const { document } = window;
        yield new Promise((resolve) => {
            window.addEventListener("message", (message) => {
                if (message.data.type !== "get_info") {
                    return;
                }
                resolve();
            });
        });
        // Enable managed credentials
        window.postMessage({
            type: "managed_credentials",
            enabled: true
        }, "*");
        yield new Promise((resolve) => setTimeout(resolve, 10));
        const managedCredsCheckbox = document.body.querySelector("#managed-credentials-enabled");
        // Initially should be checked
        expect(managedCredsCheckbox.checked).toBe(true);
        // User unchecks the checkbox
        managedCredsCheckbox.checked = false;
        managedCredsCheckbox.dispatchEvent(new window.Event("click"));
        // Now send another enable message
        window.postMessage({
            type: "managed_credentials",
            enabled: true
        }, "*");
        yield new Promise((resolve) => setTimeout(resolve, 10));
        // The checkbox should remain unchecked because user has modified it
        expect(managedCredsCheckbox.checked).toBe(false);
        // But the section should still be visible
        const managedCredsSection = document.body.querySelector("#managed-credentials");
        expect(managedCredsSection.classList.contains("hidden")).toBe(false);
    }));
    test("checkbox state persists in validation with enabled credentials", () => __awaiter(void 0, void 0, void 0, function* () {
        var _a, _b;
        const fileName = "script_file";
        const validPath = `/path/to/${fileName}.R`;
        const { api, state } = createDefaultApi();
        const { window } = yield newDom(api, validPath);
        const { document } = window;
        yield Promise.all([
            new Promise((resolve) => {
                window.addEventListener("message", (message) => {
                    if (message.data.type !== "get_info") {
                        return;
                    }
                    resolve();
                });
            }),
            new Promise((resolve) => {
                window.addEventListener("message", (message) => {
                    if (message.data.type !== "get_info_for_script") {
                        return;
                    }
                    resolve();
                });
            })
        ]);
        // Enable managed credentials
        window.postMessage({
            type: "managed_credentials",
            enabled: true
        }, "*");
        yield new Promise((resolve) => setTimeout(resolve, 10));
        // Get initial state
        const managedCredsCheckbox = document.body.querySelector("#managed-credentials-enabled");
        expect(managedCredsCheckbox.checked).toBe(true);
        expect((_a = state.jobOptions) === null || _a === void 0 ? void 0 : _a.managedCredentialsEnabled).toBe(true);
        // Change job name to trigger validation
        const jobNameEl = document.body.querySelector("#rstudio_label_job_name");
        jobNameEl.value = "Test Job";
        jobNameEl.dispatchEvent(new window.Event("change"));
        // Managed credentials should still be enabled
        expect((_b = state.jobOptions) === null || _b === void 0 ? void 0 : _b.managedCredentialsEnabled).toBe(true);
        // Start button should be enabled (valid config with managed credentials)
        const startButtonEl = document.body.querySelector("#submit");
        expect(startButtonEl === null || startButtonEl === void 0 ? void 0 : startButtonEl.classList.contains("disabled")).toBe(false);
    }));
    test("checkbox checked state reflects in job launch", () => __awaiter(void 0, void 0, void 0, function* () {
        const fileName = "script_file";
        const validPath = `/path/to/${fileName}.R`;
        let capturedLaunchMessage = null;
        const { api } = createDefaultApi();
        // Override the postMessage to capture the launch message
        const originalPostMessage = api.postMessage;
        api.postMessage = (message) => {
            if (message.event === "launch") {
                capturedLaunchMessage = message;
            }
            return originalPostMessage.call(api, message);
        };
        const { window } = yield newDom(api, validPath);
        const { document } = window;
        yield Promise.all([
            new Promise((resolve) => {
                window.addEventListener("message", (message) => {
                    if (message.data.type !== "get_info") {
                        return;
                    }
                    resolve();
                });
            }),
            new Promise((resolve) => {
                window.addEventListener("message", (message) => {
                    if (message.data.type !== "get_info_for_script") {
                        return;
                    }
                    resolve();
                });
            })
        ]);
        // Enable managed credentials and check the checkbox
        window.postMessage({
            type: "managed_credentials",
            enabled: true
        }, "*");
        yield new Promise((resolve) => setTimeout(resolve, 10));
        const managedCredsCheckbox = document.body.querySelector("#managed-credentials-enabled");
        expect(managedCredsCheckbox.checked).toBe(true);
        // Uncheck the checkbox manually
        managedCredsCheckbox.checked = false;
        managedCredsCheckbox.dispatchEvent(new window.Event("click"));
        // Click the start button
        const startButtonEl = document.body.querySelector("#submit");
        startButtonEl.click();
        // Verify that managedCredentialsEnabled is false in the job options
        expect(capturedLaunchMessage).not.toBe(null);
        expect(capturedLaunchMessage.jobOptions.managedCredentialsEnabled).toBe(false);
    }));
    test("setting change from enabled to disabled hides section", () => __awaiter(void 0, void 0, void 0, function* () {
        const { api } = createDefaultApi();
        const { window } = yield newDom(api);
        const { document } = window;
        yield new Promise((resolve) => {
            window.addEventListener("message", (message) => {
                if (message.data.type !== "get_info") {
                    return;
                }
                resolve();
            });
        });
        // First enable managed credentials
        window.postMessage({
            type: "managed_credentials",
            enabled: true
        }, "*");
        yield new Promise((resolve) => setTimeout(resolve, 10));
        // Verify it's visible and checked
        const managedCredsSection = document.body.querySelector("#managed-credentials");
        const managedCredsCheckbox = document.body.querySelector("#managed-credentials-enabled");
        expect(managedCredsSection.classList.contains("hidden")).toBe(false);
        expect(managedCredsCheckbox.checked).toBe(true);
        // Now disable via setting change
        window.postMessage({
            type: "managed_credentials",
            enabled: false
        }, "*");
        yield new Promise((resolve) => setTimeout(resolve, 10));
        // Verify it's hidden (this is what should happen)
        expect(managedCredsSection.classList.contains("hidden")).toBe(true);
        // The checkbox state becomes irrelevant when section is hidden
    }));
    test("setting change after user modification preserves user choice when re-enabled", () => __awaiter(void 0, void 0, void 0, function* () {
        const { api } = createDefaultApi();
        const { window } = yield newDom(api);
        const { document } = window;
        yield new Promise((resolve) => {
            window.addEventListener("message", (message) => {
                if (message.data.type !== "get_info") {
                    return;
                }
                resolve();
            });
        });
        // Enable managed credentials
        window.postMessage({
            type: "managed_credentials",
            enabled: true
        }, "*");
        yield new Promise((resolve) => setTimeout(resolve, 10));
        const managedCredsSection = document.body.querySelector("#managed-credentials");
        const managedCredsCheckbox = document.body.querySelector("#managed-credentials-enabled");
        // User unchecks the checkbox
        managedCredsCheckbox.checked = false;
        managedCredsCheckbox.dispatchEvent(new window.Event("click"));
        // Disable the setting
        window.postMessage({
            type: "managed_credentials",
            enabled: false
        }, "*");
        yield new Promise((resolve) => setTimeout(resolve, 10));
        expect(managedCredsSection.classList.contains("hidden")).toBe(true);
        // Re-enable the setting
        window.postMessage({
            type: "managed_credentials",
            enabled: true
        }, "*");
        yield new Promise((resolve) => setTimeout(resolve, 10));
        // Section should be visible again, but checkbox should remain unchecked (user's choice)
        expect(managedCredsSection.classList.contains("hidden")).toBe(false);
        expect(managedCredsCheckbox.checked).toBe(false);
    }));
});
describe("valid config", () => {
    // validateState logic:
    //   valid when state.launchSpec?.name is defined and state.jobOptions?.script is defined
    test("start button enabled on valid config", () => __awaiter(void 0, void 0, void 0, function* () {
        const { api } = createDefaultApi();
        const { window } = yield newDom(api);
        const { document } = window;
        yield new Promise((resolve) => {
            window.addEventListener("message", (message) => {
                if (message.data.type !== "get_info") {
                    return;
                }
                resolve();
            });
        });
        // verify that the start button is diabled
        const startButtonEl = document.body.querySelector("#submit");
        expect(startButtonEl === null || startButtonEl === void 0 ? void 0 : startButtonEl.classList.contains("disabled")).toBe(true);
        // set the job name and script
        const jobNameEl = document.body.querySelector("#rstudio_label_job_name");
        jobNameEl.value = "Custom Job";
        jobNameEl.dispatchEvent(new window.Event("change"));
        const scriptBrowseEl = document.body.querySelector("#launcher-browse-script-path");
        const onBrowse = new Promise((resolve) => {
            window.addEventListener("message", (message) => {
                if (message.data.type !== "browse_path") {
                    return;
                }
                resolve();
            });
        });
        scriptBrowseEl.click();
        yield onBrowse;
        yield new Promise((resolve) => {
            window.addEventListener("message", (message) => {
                if (message.data.type !== "get_info_for_script") {
                    return;
                }
                resolve();
            });
        });
        // it should now be possible to launch the job
        expect(startButtonEl === null || startButtonEl === void 0 ? void 0 : startButtonEl.classList.contains("disabled")).toBe(false);
    }));
    test("start button disabled on invalid config", () => __awaiter(void 0, void 0, void 0, function* () {
        const fileName = "script_file";
        const validPath = `/path/to/${fileName}.R`;
        const { api, state } = createDefaultApi();
        const { window } = yield newDom(api, validPath);
        const { document } = window;
        state.launchSpec = {
            name: "Custom Job",
            cluster: "Local",
            placement_constraints: [],
            resource_limits: [],
            queues: [],
            container_image: "",
            default_image: "",
            container_images: [""]
        };
        yield Promise.all([
            new Promise((resolve) => {
                window.addEventListener("message", (message) => {
                    if (message.data.type !== "get_info") {
                        return;
                    }
                    resolve();
                });
            }),
            new Promise((resolve) => {
                window.addEventListener("message", (message) => {
                    if (message.data.type !== "get_info_for_script") {
                        return;
                    }
                    resolve();
                });
            })
        ]);
        // we're starting with a valid config
        // verify that the start button is enabled
        const startButtonEl = document.body.querySelector("#submit");
        expect(startButtonEl === null || startButtonEl === void 0 ? void 0 : startButtonEl.classList.contains("disabled")).toBe(false);
        // clear the job name
        const jobNameEl = document.body.querySelector("#rstudio_label_job_name");
        jobNameEl.value = "";
        jobNameEl.dispatchEvent(new window.Event("change"));
        // the state should now be invalid because the job has been cleared
        expect(startButtonEl === null || startButtonEl === void 0 ? void 0 : startButtonEl.classList.contains("disabled")).toBe(true);
    }));
    test("validate state funciton on invalid state", () => {
        expect((0, __1.validateState)(baseState())).toBe(false);
    });
    test("validate state funciton on valid state", () => {
        const validState = baseState();
        validState.jobOptions.script = BROWSE_PATH_SCRIPT;
        validState.launchSpec.name = "a non empty value";
        expect((0, __1.validateState)(validState)).toBe(true);
    });
});
//# sourceMappingURL=data:application/json;base64,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