"use strict";
/*
 * languages.ts
 *
 * Copyright (C) 2022 by Posit Software, PBC
 *
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.LANGUAGES = exports.R_MARKDOWN = exports.QUARTO = exports.JUPYTER = exports.PYTHON = exports.R = exports.commandLineArgType = void 0;
exports.clearAfter = clearAfter;
exports.extension = extension;
exports.detectFileType = detectFileType;
exports.getScriptName = getScriptName;
exports.validScriptExtension = validScriptExtension;
const index_1 = require("./index");
function clearAfter(parentEl, childEl) {
    const children = Array.from(parentEl.children).reverse();
    for (const child of children) {
        if (child === childEl) {
            break;
        }
        parentEl.removeChild(child);
    }
}
function createVersionSelectorBuilder({ id, valueName, valueListName, labelText }) {
    return ({ currents = [], defaults = [], set = () => { } }) => {
        var _a;
        // always source the versions list from the defaults -- it should never change
        const listValue = defaults.find((dflt) => dflt.name === valueListName);
        const list = ((_a = listValue === null || listValue === void 0 ? void 0 : listValue.value) !== null && _a !== void 0 ? _a : []);
        // pick the value from the currents, because the user can modify it
        const value = currents.find((dflt) => dflt.name === valueName);
        const label = document.createElement("label");
        label.classList.add("col-sm-3", "control-label");
        label.textContent = labelText;
        label.setAttribute("for", id);
        const select = document.createElement("select");
        select.classList.add("form-control");
        select.id = id;
        // The `<option>`s will be set when TomSelect is first initialized
        select.dataset.runtimes = JSON.stringify(list);
        const OTHER_TEXT = "Other...";
        const pathSelectionWrapper = document.createElement("div");
        pathSelectionWrapper.classList.add("form-grid");
        pathSelectionWrapper.style.gridColumn = "2 / -1";
        pathSelectionWrapper.style.display = "none";
        const pathInputEl = document.createElement("input");
        pathInputEl.classList.add("form-control");
        pathInputEl.type = "text";
        pathInputEl.style.width = "100%";
        pathInputEl.placeholder = "Path to executable";
        pathInputEl.id = `${id}-path`;
        pathInputEl.style.gridColumn = "1 / 3";
        const pathBrowseEl = document.createElement("button");
        pathBrowseEl.classList.add("secondary", "browse-button");
        pathBrowseEl.textContent = "Browse";
        pathSelectionWrapper.append(pathInputEl);
        pathSelectionWrapper.append(pathBrowseEl);
        pathBrowseEl.addEventListener("click", () => {
            (0, index_1.handleOnBrowse)(false, (event) => {
                const { message } = event.data;
                pathInputEl.value = message;
                pathInputEl.dispatchEvent(new Event("change"));
            });
        });
        pathInputEl.addEventListener("change", () => {
            set([
                {
                    name: `${valueName}-other`,
                    value: pathInputEl.value
                },
                {
                    name: valueName,
                    value: OTHER_TEXT
                }
            ]);
        });
        select.addEventListener("change", () => {
            pathSelectionWrapper.style.display = select.value !== OTHER_TEXT ? "none" : "";
            set([
                {
                    name: valueName,
                    value: select.value
                }
            ]);
        });
        // Populate other version path if it was specified
        const otherVersionIndex = currents.findIndex((dflt) => dflt.name === `${valueName}-other`);
        if (otherVersionIndex !== -1) {
            const otherVersionPath = currents[otherVersionIndex].value;
            if (otherVersionPath) {
                pathInputEl.value = otherVersionPath;
                select.value = OTHER_TEXT;
                pathInputEl.dispatchEvent(new Event("change"));
            }
        }
        else if (value === null || value === void 0 ? void 0 : value.value) {
            select.value = value.value;
        }
        // include a spacer div for the unused grid column
        // include a warning placeholder
        const warningEl = document.createElement("div");
        warningEl.id = "version-path-warning";
        warningEl.style.gridColumn = "2 / -1";
        warningEl.style.color = "var(--vscode-editorWarning-foreground)";
        warningEl.style.display = "none";
        return [label, select, document.createElement("div"), pathSelectionWrapper, warningEl];
    };
}
// create element builders/renderers for r versions and python verisons
const buildRVersionSelector = createVersionSelectorBuilder({
    id: "r-version-selector",
    valueListName: "r-versions",
    valueName: "r-version",
    labelText: "R version"
});
const buildPyVersionSelector = createVersionSelectorBuilder({
    id: "py-version-selector",
    valueListName: "py-versions",
    valueName: "py-version",
    labelText: "Python version"
});
function buildArgumentListEditor({ currents = [], defaults = [], set = () => { } }) {
    const result = [];
    let first;
    let last;
    // construct the linked list from defaults array
    const createLinkedList = (defaults) => {
        first = undefined;
        last = undefined;
        if (defaults.length) {
            first = { name: defaults[0].name, value: defaults[0].value };
            last = first;
            for (const dflt of defaults.slice(1)) {
                const current = Object.assign(Object.assign({}, dflt), { last });
                last.next = current;
                last = current;
            }
        }
    };
    createLinkedList(currents);
    // flatten the linked list to an array.
    // called when passing args array to `set` callback
    const argList = () => {
        const result = [];
        if (!last) {
            return result;
        }
        let current = last;
        while (current) {
            if (current.name || current.value) {
                // only include arguments that are not completely empty
                // we're traversing backwards, so we're using `unshift` to
                // preserve visual order
                result.unshift({ name: current.name, value: current.value });
            }
            current = current.last;
        }
        return result;
    };
    // create a mechanism for showing/hiding arguments UI
    const argLabel = document.createElement("label");
    argLabel.classList.add("col-sm-3", "control-label");
    argLabel.style.width = "unset";
    argLabel.textContent = "Arguments";
    const argSummary = document.createElement("div");
    argSummary.classList.add("form-control", "argument-summary");
    argSummary.style.width = "unset";
    argSummary.style.opacity = "60%";
    const updateArgPreview = () => {
        argSummary.textContent = argList()
            .map((arg) => { var _a; return (_a = arg.value) === null || _a === void 0 ? void 0 : _a.trim(); })
            .join(" ");
    };
    updateArgPreview();
    // hijack the set callback to trigger updates to the args preview
    // whenever the callback is called -- easier than remembering to call both
    const oldSet = set;
    set = (args) => {
        updateArgPreview();
        oldSet(args);
    };
    let showingArgs = false;
    const argShowHideButton = document.createElement("button");
    argShowHideButton.classList.add("browse-button", "secondary");
    argShowHideButton.addEventListener("click", () => {
        toggleArgs();
    });
    const toggleArgs = (override) => {
        var _a;
        showingArgs = override !== null && override !== void 0 ? override : !showingArgs;
        argShowHideButton.textContent = showingArgs ? "Hide" : "Modify";
        (_a = argShowHideButton.parentElement) === null || _a === void 0 ? void 0 : _a.classList.toggle("show-arguments", showingArgs);
        set(argList());
    };
    toggleArgs(false);
    result.push(argLabel, argSummary, argShowHideButton);
    const removers = new Set();
    // create the UI elements which manage a particular argument
    const createListItem = (argument) => {
        const valueInputEl = document.createElement("input");
        valueInputEl.type = "text";
        valueInputEl.placeholder = "Flag / Argument";
        valueInputEl.value = argument.value || "";
        valueInputEl.classList.add("form-control", "argument");
        valueInputEl.style.width = "unset";
        valueInputEl.style.gridColumn = "1 / 3";
        valueInputEl.addEventListener("change", () => {
            argument.value = valueInputEl.value;
            set(argList());
        });
        const deleteArgEl = document.createElement("button");
        deleteArgEl.classList.add("delete-button", "argument", "delete-argument");
        deleteArgEl.innerHTML = "&#x2715;";
        const remover = () => {
            if (argument.last) {
                argument.last.next = argument.next;
            }
            if (argument.next) {
                argument.next.last = argument.last;
            }
            // ensure that the last argument is always kept updated when removing
            // this is important so we can append new arguments correctly
            if (argument === last) {
                last = argument.last;
            }
            valueInputEl.remove();
            deleteArgEl.remove();
            removers.delete(remover);
            set(argList());
        };
        deleteArgEl.addEventListener("click", remover);
        removers.add(remover);
        return [valueInputEl, deleteArgEl];
    };
    const drawList = () => {
        let current = first;
        const result = [];
        while (current) {
            result.push(...createListItem(current));
            last = current;
            current = current.next;
        }
        return result;
    };
    result.push(...drawList());
    const addButton = document.createElement("button");
    addButton.classList.add("secondary", "argument", "add-argument");
    addButton.style.gridColumn = "1 / 3";
    addButton.textContent = "+ Add argument";
    addButton.addEventListener("click", () => {
        const argument = { last };
        if (last) {
            last.next = argument;
        }
        createListItem(argument).forEach((el) => addButton.insertAdjacentElement("beforebegin", el));
        last = argument;
        set(argList());
    });
    result.push(addButton);
    const resetButton = document.createElement("button");
    resetButton.classList.add("delete-button", "argument", "reset-arguments");
    resetButton.textContent = "Reset";
    resetButton.addEventListener("click", () => {
        // remove everything in the list first, then clear the removers cache
        // so they aren't called again another another reset
        removers.forEach((remover) => remover());
        removers.clear();
        // reset the linked list based off the defaults, not the currents
        createLinkedList(defaults);
        drawList().forEach((el) => addButton.insertAdjacentElement("beforebegin", el));
        // re-update the currents and redraw the preview
        set(argList());
    });
    result.push(resetButton);
    return result;
}
exports.commandLineArgType = [
    {
        name: "Command Line",
        type: "list",
        render: buildArgumentListEditor
    }
];
exports.R = "r";
exports.PYTHON = "py";
exports.JUPYTER = "ipynb";
exports.QUARTO = "qmd";
exports.R_MARKDOWN = "rmd";
exports.LANGUAGES = new Map()
    .set(exports.R, {
    name: "R",
    ext: exports.R,
    controls: [
        {
            name: "R Version",
            type: "select",
            render: buildRVersionSelector
        },
        ...exports.commandLineArgType
    ]
})
    .set(exports.PYTHON, {
    name: "Python",
    ext: exports.PYTHON,
    controls: [
        {
            name: "Python Version",
            type: "select",
            render: buildPyVersionSelector
        },
        ...exports.commandLineArgType
    ]
});
/*
  .set(JUPYTER, {
    name: 'Jupyter Notebook',
    ext: JUPYTER,
    controls: commandLineArgType
  })
  .set(QUARTO, {
    name: 'Quarto Markdown',
    ext: QUARTO,
    controls: commandLineArgType
  })
  .set(R_MARKDOWN, {
    name: 'R Markdown',
    ext: R_MARKDOWN,
    controls: commandLineArgType
  });
  */
function extension(path) {
    if (!path) {
        return "";
    }
    const ext = path.toLowerCase().split(".").at(-1);
    if (!ext) {
        return "";
    }
    return exports.LANGUAGES.has(ext) ? ext : "";
}
function detectFileType(path) {
    return exports.LANGUAGES.get(extension(path));
}
function getScriptName(path) {
    var _a;
    return ((_a = path.split("/").at(-1)) === null || _a === void 0 ? void 0 : _a.split(".").slice(0, -1).join("")) || "New Workbench Job";
}
function validScriptExtension(path) {
    var _a, _b;
    // output streams in VS Code count as files, and we don't want to
    // trigger changes when the user selects one of these.
    // Only continue if the active file has one of the extension types
    // we care about, defined in LANGAUGES.keys
    const langs = Array.from(exports.LANGUAGES.keys());
    // at(-1) accesses the element at the end of the array, which is the
    // extension in this case. LowerCase guarantees we can find it even if
    // the user has decided to change its capitalization (for whatever reason)
    return !path.includes("extension-output") && langs.includes((_b = (_a = path.split(".").at(-1)) === null || _a === void 0 ? void 0 : _a.toLowerCase()) !== null && _b !== void 0 ? _b : "");
}
//# sourceMappingURL=data:application/json;base64,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