"use strict";
/*
 * index.ts
 *
 * Copyright (C) 2022 by Posit Software, PBC
 *
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.getApi = getApi;
exports.handleOnBrowse = handleOnBrowse;
exports.validateState = validateState;
exports.run = run;
const credentials_1 = require("./credentials");
const interface_1 = require("./interface");
const languages_1 = require("./languages");
function sizeToContent(el, immediate = false, trim = 0) {
    if (!el) {
        return;
    }
    const range = new Range();
    range.selectNodeContents(el);
    el.style.transition = immediate ? '' : 'height 500ms var(--ease-out)';
    el.style.height = `${range.getBoundingClientRect().height + trim}px`;
    el.style.boxSizing = 'content-box';
    el.style.overflowY = 'hidden';
    range.detach();
}
function resetSize(el) {
    if (!el) {
        return;
    }
    el.style.transition = '';
    el.style.height = '';
    el.style.boxSizing = '';
}
// message ID for postMessage association
let mid = 0;
// convert to our own version of the WebviewApi for explicit events
let api;
function getApi() {
    if (!api) {
        api = acquireVsCodeApi();
    }
    return api;
}
function handleOnBrowse(directory = false, callback) {
    const vscode = getApi();
    const messageId = ++mid;
    const browseListener = (event) => {
        const { type, id, message } = event.data;
        // message id must match, otherwise this may have come from a
        // different browse_path event (unlikely)
        if (type !== 'browse_path' || id !== messageId) {
            return;
        }
        window.removeEventListener('message', browseListener);
        if (!message) {
            return;
        }
        callback(event);
    };
    window.addEventListener('message', browseListener);
    vscode.postMessage({ event: 'browse_path', id: messageId, directory });
}
function validateState(state) {
    var _a, _b;
    return !!(((_a = state.launchSpec) === null || _a === void 0 ? void 0 : _a.name) &&
        (0, languages_1.validScriptExtension)(((_b = state.jobOptions) === null || _b === void 0 ? void 0 : _b.script) || ''));
}
;
function run() {
    const vscode = getApi();
    const viewState = vscode.getState() || {};
    // ---- Create and wire up the launcher ----
    const launcherUiEl = document.getElementById('launcher-ui');
    const mainEl = document.getElementById('main');
    const startButton = document.getElementById('submit');
    const workbenches = ['RStudio'];
    try {
        const launcherInput = Object.assign(Object.assign({ default_names: workbenches.map(workbench => `${workbench} Job`), show_clusters: ShowClusterType.NonInteractive, show_workbenches: false, workbenches }, viewState.input), { launch_spec: viewState.launchSpec || null });
        viewState.input = launcherInput;
        const launcherUiOptions = Object.assign(Object.assign(Object.assign({}, interface_1.DEFAULT_LAUNCHER_UI_OPTIONS), { job_name_label: 'Job Name', job_name_placeholder: 'New Job', table_layout: true }), viewState.uiOptions);
        viewState.uiOptions = launcherUiOptions;
        const launcher = new LauncherUI(launcherUiEl, () => {
            return new Promise(resolve => {
                const infoListener = (event) => {
                    const { type, message } = event.data;
                    if (type !== 'get_info') {
                        return;
                    }
                    // only want this to trigger once, so remove it once
                    // we receive the info that we requested
                    window.removeEventListener('message', infoListener);
                    resolve(message);
                };
                window.addEventListener('message', infoListener);
                vscode.postMessage({ event: 'get_info' });
            });
        }, launcherInput, launcherUiOptions);
        const saveState = () => {
            viewState.launchSpec = launcher.getLaunchSpec();
            viewState.jobOptions = Object.assign({}, jobOptions);
            viewState.userModified = Object.assign({}, userModified);
            viewState.outputArgs = outputArgs.map(args => args.map(arg => { return Object.assign({}, arg); }));
            showPathWarning();
            vscode.setState(Object.assign({}, viewState));
            const valid = validateState(viewState);
            startButton.classList.toggle('disabled', !valid);
            return valid;
        };
        launcher.setOnLaunchSpecModified(() => {
            sizeToContent(launcherUiEl);
            saveState();
        });
        launcher.setOnUiRender(() => {
            var _a;
            resetSize(launcherUiEl);
            sizeToContent(launcherUiEl, true);
            // hack in the default name if the user has already changed it
            if ((_a = launcherInput.launch_spec) === null || _a === void 0 ? void 0 : _a.name) {
                launcher.setJobName(launcherInput.launch_spec.name);
                // just setting the job name won't trigger a save -- have to trigger it manually
                saveState();
            }
            setCurrentScript(jobOptions.script);
            // Add a keydown handler that prevents the form element from submitting itself (Enter key). 
            // We don't want to allow this in this UI, which only allows submission through a direct 
            // interaction with the submit button 
            const formEl = launcherUiEl.querySelector('form.form-horizontal');
            formEl.addEventListener('keydown', evt => {
                if (evt.key === 'Enter') {
                    evt.preventDefault();
                }
            });
            // add an event listener to the job name field so that we can store it's state immediately
            // following any user changes to it.
            const jobNameEl = launcherUiEl.querySelector('#rstudio_label_job_name');
            jobNameEl.addEventListener('change', () => { saveState(); });
        });
        launcher.initialize();
        // ---- Wire up the other fields ----
        const activeFileInit = document.head.querySelector('meta[name="active-file"]').getAttribute('content');
        const outputArgs = viewState.outputArgs || [];
        const jobOptions = Object.assign({ script: (0, languages_1.validScriptExtension)(activeFileInit || '') ? activeFileInit : '', workDir: '', args: [] }, viewState.jobOptions);
        const userModified = Object.assign({}, viewState.userModified);
        const scriptOptionsEl = document.getElementById('script-options');
        resetSize(scriptOptionsEl);
        sizeToContent(scriptOptionsEl, true, -4);
        const showPathWarning = () => {
            var _a, _b, _c;
            // id comes from a placeholder added by version selector builder
            const warningEl = document.getElementById('version-path-warning');
            // if there is no version selector available, then bail immediately
            if (!warningEl) {
                return;
            }
            // show a path warning if the cluster is set to non-local and the user
            // selects an R version that is anything other than (System Default)
            // TODO: compare image name of the current k8s session (if applicable) for improved 
            // warning accuracy, as the IDE does. Right now we don't have access to this information
            const version = ((_a = outputArgs.flat().find(arg => { var _a; return (_a = arg.name) === null || _a === void 0 ? void 0 : _a.endsWith('-version'); })) === null || _a === void 0 ? void 0 : _a.value) || '(System Default)';
            if (((_b = viewState.launchSpec) === null || _b === void 0 ? void 0 : _b.cluster) !== 'Local' && version !== '(System Default)') {
                warningEl.textContent = `Ensure version selection "${version}" exists in "${(_c = viewState.launchSpec) === null || _c === void 0 ? void 0 : _c.cluster}"`;
                warningEl.style.display = '';
            }
            else {
                warningEl.style.display = 'none';
            }
            sizeToContent(scriptOptionsEl, false, -4);
        };
        const setCurrentScript = (path = 'New Workbench Job') => {
            var _a, _b;
            if (!(0, languages_1.validScriptExtension)(path)) {
                return;
            }
            // only auto-update the name if the user has not modified it
            if (!((_a = viewState.launchSpec) === null || _a === void 0 ? void 0 : _a.name) ||
                ((_b = viewState.launchSpec) === null || _b === void 0 ? void 0 : _b.name) === launcherInput.default_names[0]) {
                launcher.setJobName((0, languages_1.getScriptName)(path));
            }
            scriptPathInputEl.textContent = path;
            scriptPathInputEl.dispatchEvent(new Event('change'));
            const currentLanguage = (0, languages_1.detectFileType)(jobOptions.script);
            jobOptions.script = path;
            saveState();
            const language = (0, languages_1.detectFileType)(path);
            // clear args if it is a different language type
            if ((currentLanguage === null || currentLanguage === void 0 ? void 0 : currentLanguage.ext) !== (language === null || language === void 0 ? void 0 : language.ext)) {
                outputArgs.splice(0);
                jobOptions.args = [];
            }
            scriptPathInputEl.dataset.lang = (language === null || language === void 0 ? void 0 : language.ext) || '';
            const gridFormEl = document.getElementById('form-grid');
            const separatorEl = document.getElementById('arg-list');
            (0, languages_1.clearAfter)(gridFormEl, separatorEl);
            // unrecognized languages should not display r-version selector
            // or generic arguments list. Still resize in case there was
            // previously content which affected the height of the container
            if (!language) {
                sizeToContent(scriptOptionsEl, false, -4);
                return;
            }
            const messageId = ++mid;
            const scriptInfoListener = (event) => {
                const { type, id, message } = event.data;
                // message id must match, otherwise this may have come from a
                // different get_info_for_script event (unlikely)
                if (type !== 'get_info_for_script' || id !== messageId) {
                    return;
                }
                window.removeEventListener('message', scriptInfoListener);
                let versionArgs = [];
                let scriptArgs = [];
                if (message) {
                    try {
                        const args = JSON.parse(message);
                        if (args.versionInfo) {
                            versionArgs = args.versionInfo;
                        }
                        if (args.scriptArgs) {
                            scriptArgs = args.scriptArgs;
                        }
                        if (!userModified.workDir && typeof args.workingDirectory === 'string') {
                            jobOptions.workDir = args.workingDirectory;
                            workdirPathInputEl.textContent = jobOptions.workDir;
                        }
                    }
                    catch (_) { /* do nothing */ }
                }
                const valuesOnly = (args) => {
                    return args.map(arg => { return { value: `${(arg.name || '').trim()} ${(arg.value || '').trim()}`.trim() }; });
                };
                // defaults can be modified by the render functions
                // to strip out already-consumed arguments
                const defaultArgs = [versionArgs.slice(), valuesOnly(scriptArgs)];
                for (let a = 0; a < language.controls.length; a++) {
                    const control = language.controls[a];
                    HTMLElement.prototype.append.apply(gridFormEl, control.render({
                        currents: (outputArgs[a] || defaultArgs[a]).map(arg => { return Object.assign({}, arg); }),
                        defaults: defaultArgs[a],
                        set: args => {
                            outputArgs[a] = args;
                            jobOptions.args = outputArgs.flat();
                            sizeToContent(scriptOptionsEl, false, -4);
                            saveState();
                        }
                    }));
                }
                sizeToContent(scriptOptionsEl, false, -4);
            };
            window.addEventListener('message', scriptInfoListener);
            vscode.postMessage({ event: 'get_info_for_script', id: messageId, ext: language.ext, script: jobOptions.script });
        };
        const scriptPathInputEl = document.getElementById('launcher-script-path');
        const scriptPathBrowseEl = document.getElementById('launcher-browse-script-path');
        const workdirPathInputEl = document.getElementById('launcher-workdir-path');
        workdirPathInputEl.textContent = jobOptions.workDir;
        const workdirPathBrowseEl = document.getElementById('launcher-browse-workdir-path');
        const browseHandler = (targetEl, keyName, directory = false) => {
            return () => {
                handleOnBrowse(directory, (event) => {
                    const { message } = event.data;
                    targetEl.textContent = message;
                    userModified[keyName] = true;
                    targetEl.dispatchEvent(new Event('change'));
                    if (keyName !== 'script') {
                        jobOptions[keyName] = message;
                        saveState();
                        return;
                    }
                    setCurrentScript(message);
                });
            };
        };
        scriptPathInputEl.addEventListener('click', browseHandler(scriptPathInputEl, 'script'));
        scriptPathBrowseEl.addEventListener('click', browseHandler(scriptPathInputEl, 'script'));
        workdirPathInputEl.addEventListener('click', browseHandler(workdirPathInputEl, 'workDir', true));
        workdirPathBrowseEl.addEventListener('click', browseHandler(workdirPathInputEl, 'workDir', true));
        window.addEventListener('message', event => {
            const { type, message: path } = event.data;
            if (type !== 'active_file') {
                return;
            }
            if (!(0, languages_1.validScriptExtension)(path)) {
                return;
            }
            // this means the user has not changed the file manually, so we
            // can update the script path without overwriting their choice
            if (!userModified.script) {
                setCurrentScript(path);
            }
        });
        // Wire up the managed credentials input.
        const managedCreds = new credentials_1.ManagedCredentialsInput((enabled) => {
            jobOptions.managedCredentialsEnabled = enabled;
            saveState();
        });
        // Handle managed credentials messages (both during initialization and at
        // runtime when settings change).
        window.addEventListener('message', (event) => {
            if (event.data.type !== 'managed_credentials') {
                return;
            }
            managedCreds.enable(event.data.enabled);
        });
        startButton.addEventListener('click', () => {
            const valid = saveState();
            if (!valid) {
                return;
            }
            // Add script name to end of arguments
            const ext = (0, languages_1.extension)(jobOptions.script);
            if (ext === languages_1.R) {
                jobOptions.args.push({ name: '-f', value: jobOptions.script });
            }
            else if (ext === languages_1.PYTHON) {
                jobOptions.args.push({ value: jobOptions.script });
            }
            vscode.postMessage({
                event: 'launch',
                launchSpec: Object.assign({}, viewState.launchSpec),
                jobOptions: Object.assign({}, viewState.jobOptions)
            });
        });
    }
    catch (err) {
        mainEl.classList.toggle('error', true);
        console.error(err);
        const errorEl = document.getElementById('error');
        errorEl.textContent = 'There was an error initializing the job launcher';
    }
}
//# sourceMappingURL=data:application/json;base64,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