"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.getDefaultCacheDir = getDefaultCacheDir;
exports.isFileNotFoundError = isFileNotFoundError;
exports.createCacheDirIfNotExists = createCacheDirIfNotExists;
exports.writeCacheFile = writeCacheFile;
const path_1 = __importDefault(require("path"));
const promises_1 = __importDefault(require("fs/promises"));
const os_1 = __importDefault(require("os"));
// NOTE:
// Besides this file, there are other entrypoints implementing reading/writing cache files:
// - global_config.js
// - authentication/secure_storage/json_credential_manager.js
//
// We should refactor the code so every place is using utils from this file
function getDefaultCacheDir() {
    switch (process.platform) {
        case 'win32':
            return path_1.default.join(os_1.default.homedir(), 'AppData', 'Local', 'Snowflake', 'Caches');
        case 'linux':
            return path_1.default.join(os_1.default.homedir(), '.cache', 'snowflake');
        case 'darwin':
            return path_1.default.join(os_1.default.homedir(), 'Library');
        default:
            throw new Error(`Unsupported platform: ${process.platform}`);
    }
}
function isFileNotFoundError(error) {
    return error && typeof error === 'object' && 'code' in error && error.code === 'ENOENT';
}
async function createCacheDirIfNotExists(cacheDir) {
    const options = { recursive: true };
    if (process.platform !== 'win32') {
        options.mode = 0o755;
    }
    await promises_1.default.mkdir(cacheDir, options);
    if (process.platform !== 'win32') {
        await promises_1.default.chmod(cacheDir, 0o700);
    }
}
async function writeCacheFile(filePath, content) {
    const dirName = path_1.default.dirname(filePath);
    await createCacheDirIfNotExists(dirName);
    await promises_1.default.writeFile(filePath, content);
    if (process.platform !== 'win32') {
        await promises_1.default.chmod(filePath, 0o600);
    }
}
//# sourceMappingURL=disk_cache.js.map