// Copyright 2018-2024 the Deno authors. All rights reserved. MIT license.
// This module is browser compatible.
/**
 * Returns the first element that is the largest value of the given function or
 * undefined if there are no elements.
 *
 * @template T The type of the elements in the array.
 *
 * @param array The array to find the maximum element in.
 * @param selector The function to get the value to compare from each element.
 *
 * @returns The first element that is the largest value of the given function or
 * undefined if there are no elements.
 *
 * @example Basic usage
 * ```ts
 * import { maxBy } from "@std/collections/max-by";
 * import { assertEquals } from "@std/assert/assert-equals";
 *
 * const people = [
 *   { name: "Anna", age: 34 },
 *   { name: "Kim", age: 42 },
 *   { name: "John", age: 23 },
 * ];
 *
 * const personWithMaxAge = maxBy(people, (person) => person.age);
 *
 * assertEquals(personWithMaxAge, { name: "Kim", age: 42 });
 * ```
 */ export function maxBy(array, selector) {
  let max;
  let maxValue;
  for (const current of array){
    const currentValue = selector(current);
    if (maxValue === undefined || currentValue > maxValue) {
      max = current;
      maxValue = currentValue;
    }
  }
  return max;
}
//# sourceMappingURL=data:application/json;base64,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