// Copyright 2018-2024 the Deno authors. All rights reserved. MIT license.
export function compareNumber(a, b) {
  if (isNaN(a) || isNaN(b)) throw new Error("Comparison against non-numbers");
  return a === b ? 0 : a < b ? -1 : 1;
}
export function checkIdentifier(v1 = [], v2 = []) {
  // NOT having a prerelease is > having one
  // But NOT having a build is < having one
  if (v1.length && !v2.length) return -1;
  if (!v1.length && v2.length) return 1;
  return 0;
}
export function compareIdentifier(v1 = [], v2 = []) {
  const length = Math.max(v1.length, v2.length);
  for(let i = 0; i < length; i++){
    const a = v1[i];
    const b = v2[i];
    // same length is equal
    if (a === undefined && b === undefined) return 0;
    // longer > shorter
    if (b === undefined) return 1;
    // shorter < longer
    if (a === undefined) return -1;
    // string > number
    if (typeof a === "string" && typeof b === "number") return 1;
    // number < string
    if (typeof a === "number" && typeof b === "string") return -1;
    if (a < b) return -1;
    if (a > b) return 1;
  // If they're equal, continue comparing segments.
  }
  return 0;
}
/**
 * A single `0`, or a non-zero digit followed by zero or more digits.
 */ const NUMERIC_IDENTIFIER = "0|[1-9]\\d*";
/**
 * Zero or more digits, followed by a letter or hyphen, and then zero or more letters, digits, or hyphens.
 */ const NON_NUMERIC_IDENTIFIER = "\\d*[a-zA-Z-][a-zA-Z0-9-]*";
/**
 * Three dot-separated numeric identifiers.
 */ const VERSION_CORE = `(?<major>${NUMERIC_IDENTIFIER})\\.(?<minor>${NUMERIC_IDENTIFIER})\\.(?<patch>${NUMERIC_IDENTIFIER})`;
/**
 * A numeric identifier, or a non-numeric identifier.
 */ const PRERELEASE_IDENTIFIER = `(?:${NUMERIC_IDENTIFIER}|${NON_NUMERIC_IDENTIFIER})`;
/**
 * A hyphen, followed by one or more dot-separated pre-release version identifiers.
 * @example "-pre.release"
 */ const PRERELEASE = `(?:-(?<prerelease>${PRERELEASE_IDENTIFIER}(?:\\.${PRERELEASE_IDENTIFIER})*))`;
/**
 * Any combination of digits, letters, or hyphens.
 */ const BUILD_IDENTIFIER = "[0-9A-Za-z-]+";
/**
 * A plus sign, followed by one or more period-separated build metadata identifiers.
 * @example "+build.meta"
 */ const BUILD = `(?:\\+(?<buildmetadata>${BUILD_IDENTIFIER}(?:\\.${BUILD_IDENTIFIER})*))`;
/**
 * A version, followed optionally by a pre-release version and build metadata.
 */ const FULL_VERSION = `v?${VERSION_CORE}${PRERELEASE}?${BUILD}?`;
export const FULL_REGEXP = new RegExp(`^${FULL_VERSION}$`);
/**
 * A comparator.
 * @example `=`, `<`, `<=`, `>`, `>=`
 */ const COMPARATOR = "(?:<|>)?=?";
/**
 * A wildcard identifier.
 * @example "x", "X", "*"
 */ const WILDCARD_IDENTIFIER = `x|X|\\*`;
const XRANGE_IDENTIFIER = `${NUMERIC_IDENTIFIER}|${WILDCARD_IDENTIFIER}`;
/**
 * A version that can contain wildcards.
 * @example "x", "1.x", "1.x.x", "1.2.x", "*", "1.*", "1.*.*", "1.2.*"
 */ export const XRANGE = `[v=\\s]*(?<major>${XRANGE_IDENTIFIER})(?:\\.(?<minor>${XRANGE_IDENTIFIER})(?:\\.(?<patch>${XRANGE_IDENTIFIER})${PRERELEASE}?${BUILD}?)?)?`;
/**
 * An operator (`~`, `~>`, `^`, `=`, `<`, `<=`, `>`, or `>=`), followed by an x-range.
 * @example "~1.x.x", "^1.2.*", ">=1.2.3"
 */ export const OPERATOR_XRANGE_REGEXP = new RegExp(`^(?<operator>~>?|\\^|${COMPARATOR})\\s*${XRANGE}$`);
/**
 * An empty string or a comparator (`=`, `<`, `<=`, `>`, or `>=`), followed by a version.
 * @example ">1.2.3"
 */ export const COMPARATOR_REGEXP = new RegExp(`^(?<operator>${COMPARATOR})\\s*(${FULL_VERSION})$|^$`);
/**
 * Returns true if the value is a valid SemVer number.
 *
 * Must be a number. Must not be NaN. Can be positive or negative infinity.
 * Can be between 0 and MAX_SAFE_INTEGER.
 * @param value The value to check
 * @returns True if its a valid semver number
 */ export function isValidNumber(value) {
  return typeof value === "number" && !Number.isNaN(value) && (!Number.isFinite(value) || 0 <= value && value <= Number.MAX_SAFE_INTEGER);
}
export const MAX_LENGTH = 256;
/**
 * Returns true if the value is a valid semver pre-release or build identifier.
 *
 * Must be a string. Must be between 1 and 256 characters long. Must match
 * the regular expression /[0-9A-Za-z-]+/.
 * @param value The value to check
 * @returns True if the value is a valid semver string.
 */ export function isValidString(value) {
  return typeof value === "string" && value.length > 0 && value.length <= MAX_LENGTH && /[0-9A-Za-z-]+/.test(value);
}
const NUMERIC_IDENTIFIER_REGEXP = new RegExp(`^${NUMERIC_IDENTIFIER}$`);
export function parsePrerelease(prerelease) {
  return prerelease.split(".").filter(Boolean).map((id)=>{
    if (NUMERIC_IDENTIFIER_REGEXP.test(id)) {
      const number = Number(id);
      if (isValidNumber(number)) return number;
    }
    return id;
  });
}
export function parseBuild(buildmetadata) {
  return buildmetadata.split(".").filter(Boolean);
}
export function parseNumber(input, errorMessage) {
  const number = Number(input);
  if (!isValidNumber(number)) throw new TypeError(errorMessage);
  return number;
}
export function isWildcardComparator(c) {
  return Number.isNaN(c.major) && Number.isNaN(c.minor) && Number.isNaN(c.patch) && (c.prerelease === undefined || c.prerelease.length === 0) && (c.build === undefined || c.build.length === 0);
}
//# sourceMappingURL=data:application/json;base64,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