import { deepAssign } from "../../_util/deep_assign.ts";
export class TOMLParseError extends Error {
}
export class Scanner {
    source;
    #whitespace = /[ \t]/;
    #position = 0;
    constructor(source) {
        this.source = source;
    }
    char(index = 0) {
        return this.source[this.#position + index] ?? "";
    }
    slice(start, end) {
        return this.source.slice(this.#position + start, this.#position + end);
    }
    next(count) {
        if (typeof count === "number") {
            for (let i = 0; i < count; i++) {
                this.#position++;
            }
        }
        else {
            this.#position++;
        }
    }
    nextUntilChar(options = { comment: true }) {
        if (options.inline) {
            while (this.#whitespace.test(this.char()) && !this.eof()) {
                this.next();
            }
        }
        else {
            while (!this.eof()) {
                const char = this.char();
                if (this.#whitespace.test(char) || this.isCurrentCharEOL()) {
                    this.next();
                }
                else if (options.comment && this.char() === "#") {
                    while (!this.isCurrentCharEOL() && !this.eof()) {
                        this.next();
                    }
                }
                else {
                    break;
                }
            }
        }
        if (!this.isCurrentCharEOL() && /\s/.test(this.char())) {
            const escaped = "\\u" + this.char().charCodeAt(0).toString(16);
            throw new TOMLParseError(`Contains invalid whitespaces: \`${escaped}\``);
        }
    }
    eof() {
        return this.position() >= this.source.length;
    }
    position() {
        return this.#position;
    }
    isCurrentCharEOL() {
        return this.char() === "\n" || this.slice(0, 2) === "\r\n";
    }
}
function success(body) {
    return {
        ok: true,
        body,
    };
}
function failure() {
    return {
        ok: false,
    };
}
export const Utils = {
    unflat(keys, values = {}, cObj) {
        const out = {};
        if (keys.length === 0) {
            return cObj;
        }
        else {
            if (!cObj) {
                cObj = values;
            }
            const key = keys[keys.length - 1];
            if (typeof key === "string") {
                out[key] = cObj;
            }
            return this.unflat(keys.slice(0, -1), values, out);
        }
    },
    deepAssignWithTable(target, table) {
        if (table.key.length === 0) {
            throw new Error("Unexpected key length");
        }
        const value = target[table.key[0]];
        if (typeof value === "undefined") {
            Object.assign(target, this.unflat(table.key, table.type === "Table" ? table.value : [table.value]));
        }
        else if (Array.isArray(value)) {
            if (table.type === "TableArray" && table.key.length === 1) {
                value.push(table.value);
            }
            else {
                const last = value[value.length - 1];
                Utils.deepAssignWithTable(last, {
                    type: table.type,
                    key: table.key.slice(1),
                    value: table.value,
                });
            }
        }
        else if (typeof value === "object" && value !== null) {
            Utils.deepAssignWithTable(value, {
                type: table.type,
                key: table.key.slice(1),
                value: table.value,
            });
        }
        else {
            throw new Error("Unexpected assign");
        }
    },
};
function or(parsers) {
    return function Or(scanner) {
        for (const parse of parsers) {
            const result = parse(scanner);
            if (result.ok) {
                return result;
            }
        }
        return failure();
    };
}
function join(parser, separator) {
    const Separator = character(separator);
    return function Join(scanner) {
        const first = parser(scanner);
        if (!first.ok) {
            return failure();
        }
        const out = [first.body];
        while (!scanner.eof()) {
            if (!Separator(scanner).ok) {
                break;
            }
            const result = parser(scanner);
            if (result.ok) {
                out.push(result.body);
            }
            else {
                throw new TOMLParseError(`Invalid token after "${separator}"`);
            }
        }
        return success(out);
    };
}
function kv(keyParser, separator, valueParser) {
    const Separator = character(separator);
    return function Kv(scanner) {
        const key = keyParser(scanner);
        if (!key.ok) {
            return failure();
        }
        const sep = Separator(scanner);
        if (!sep.ok) {
            throw new TOMLParseError(`key/value pair doesn't have "${separator}"`);
        }
        const value = valueParser(scanner);
        if (!value.ok) {
            throw new TOMLParseError(`Value of key/value pair is invalid data format`);
        }
        return success(Utils.unflat(key.body, value.body));
    };
}
function merge(parser) {
    return function Merge(scanner) {
        const result = parser(scanner);
        if (!result.ok) {
            return failure();
        }
        const body = {};
        for (const record of result.body) {
            if (typeof body === "object" && body !== null) {
                deepAssign(body, record);
            }
        }
        return success(body);
    };
}
function repeat(parser) {
    return function Repeat(scanner) {
        const body = [];
        while (!scanner.eof()) {
            const result = parser(scanner);
            if (result.ok) {
                body.push(result.body);
            }
            else {
                break;
            }
            scanner.nextUntilChar();
        }
        if (body.length === 0) {
            return failure();
        }
        return success(body);
    };
}
function surround(left, parser, right) {
    const Left = character(left);
    const Right = character(right);
    return function Surround(scanner) {
        if (!Left(scanner).ok) {
            return failure();
        }
        const result = parser(scanner);
        if (!result.ok) {
            throw new TOMLParseError(`Invalid token after "${left}"`);
        }
        if (!Right(scanner).ok) {
            throw new TOMLParseError(`Not closed by "${right}" after started with "${left}"`);
        }
        return success(result.body);
    };
}
function character(str) {
    return function character(scanner) {
        scanner.nextUntilChar({ inline: true });
        if (scanner.slice(0, str.length) === str) {
            scanner.next(str.length);
        }
        else {
            return failure();
        }
        scanner.nextUntilChar({ inline: true });
        return success(undefined);
    };
}
const Patterns = {
    BARE_KEY: /[A-Za-z0-9_-]/,
    FLOAT: /[0-9_\.e+\-]/i,
    END_OF_VALUE: /[ \t\r\n#,}]/,
};
export function BareKey(scanner) {
    scanner.nextUntilChar({ inline: true });
    if (!scanner.char() || !Patterns.BARE_KEY.test(scanner.char())) {
        return failure();
    }
    const acc = [];
    while (scanner.char() && Patterns.BARE_KEY.test(scanner.char())) {
        acc.push(scanner.char());
        scanner.next();
    }
    const key = acc.join("");
    return success(key);
}
function EscapeSequence(scanner) {
    if (scanner.char() === "\\") {
        scanner.next();
        switch (scanner.char()) {
            case "b":
                scanner.next();
                return success("\b");
            case "t":
                scanner.next();
                return success("\t");
            case "n":
                scanner.next();
                return success("\n");
            case "f":
                scanner.next();
                return success("\f");
            case "r":
                scanner.next();
                return success("\r");
            case "u":
            case "U": {
                const codePointLen = scanner.char() === "u" ? 4 : 6;
                const codePoint = parseInt("0x" + scanner.slice(1, 1 + codePointLen), 16);
                const str = String.fromCodePoint(codePoint);
                scanner.next(codePointLen + 1);
                return success(str);
            }
            case '"':
                scanner.next();
                return success('"');
            case "\\":
                scanner.next();
                return success("\\");
            default:
                scanner.next();
                return success(scanner.char());
        }
    }
    else {
        return failure();
    }
}
export function BasicString(scanner) {
    scanner.nextUntilChar({ inline: true });
    if (scanner.char() === '"') {
        scanner.next();
    }
    else {
        return failure();
    }
    const acc = [];
    while (scanner.char() !== '"' && !scanner.eof()) {
        if (scanner.char() === "\n") {
            throw new TOMLParseError("Single-line string cannot contain EOL");
        }
        const escapedChar = EscapeSequence(scanner);
        if (escapedChar.ok) {
            acc.push(escapedChar.body);
        }
        else {
            acc.push(scanner.char());
            scanner.next();
        }
    }
    if (scanner.eof()) {
        throw new TOMLParseError(`Single-line string is not closed:\n${acc.join("")}`);
    }
    scanner.next();
    return success(acc.join(""));
}
export function LiteralString(scanner) {
    scanner.nextUntilChar({ inline: true });
    if (scanner.char() === "'") {
        scanner.next();
    }
    else {
        return failure();
    }
    const acc = [];
    while (scanner.char() !== "'" && !scanner.eof()) {
        if (scanner.char() === "\n") {
            throw new TOMLParseError("Single-line string cannot contain EOL");
        }
        acc.push(scanner.char());
        scanner.next();
    }
    if (scanner.eof()) {
        throw new TOMLParseError(`Single-line string is not closed:\n${acc.join("")}`);
    }
    scanner.next();
    return success(acc.join(""));
}
export function MultilineBasicString(scanner) {
    scanner.nextUntilChar({ inline: true });
    if (scanner.slice(0, 3) === '"""') {
        scanner.next(3);
    }
    else {
        return failure();
    }
    if (scanner.char() === "\n") {
        scanner.next();
    }
    const acc = [];
    while (scanner.slice(0, 3) !== '"""' && !scanner.eof()) {
        if (scanner.slice(0, 2) === "\\\n") {
            scanner.next();
            scanner.nextUntilChar({ comment: false });
            continue;
        }
        const escapedChar = EscapeSequence(scanner);
        if (escapedChar.ok) {
            acc.push(escapedChar.body);
        }
        else {
            acc.push(scanner.char());
            scanner.next();
        }
    }
    if (scanner.eof()) {
        throw new TOMLParseError(`Multi-line string is not closed:\n${acc.join("")}`);
    }
    if (scanner.char(3) === '"') {
        acc.push('"');
        scanner.next();
    }
    scanner.next(3);
    return success(acc.join(""));
}
export function MultilineLiteralString(scanner) {
    scanner.nextUntilChar({ inline: true });
    if (scanner.slice(0, 3) === "'''") {
        scanner.next(3);
    }
    else {
        return failure();
    }
    if (scanner.char() === "\n") {
        scanner.next();
    }
    const acc = [];
    while (scanner.slice(0, 3) !== "'''" && !scanner.eof()) {
        acc.push(scanner.char());
        scanner.next();
    }
    if (scanner.eof()) {
        throw new TOMLParseError(`Multi-line string is not closed:\n${acc.join("")}`);
    }
    if (scanner.char(3) === "'") {
        acc.push("'");
        scanner.next();
    }
    scanner.next(3);
    return success(acc.join(""));
}
const symbolPairs = [
    ["true", true],
    ["false", false],
    ["inf", Infinity],
    ["+inf", Infinity],
    ["-inf", -Infinity],
    ["nan", NaN],
    ["+nan", NaN],
    ["-nan", NaN],
];
export function Symbols(scanner) {
    scanner.nextUntilChar({ inline: true });
    const found = symbolPairs.find(([str]) => scanner.slice(0, str.length) === str);
    if (!found) {
        return failure();
    }
    const [str, value] = found;
    scanner.next(str.length);
    return success(value);
}
export const DottedKey = join(or([BareKey, BasicString, LiteralString]), ".");
export function Integer(scanner) {
    scanner.nextUntilChar({ inline: true });
    const first2 = scanner.slice(0, 2);
    if (first2.length === 2 && /0(?:x|o|b)/i.test(first2)) {
        scanner.next(2);
        const acc = [first2];
        while (/[0-9a-f_]/i.test(scanner.char()) && !scanner.eof()) {
            acc.push(scanner.char());
            scanner.next();
        }
        if (acc.length === 1) {
            return failure();
        }
        return success(acc.join(""));
    }
    const acc = [];
    if (/[+-]/.test(scanner.char())) {
        acc.push(scanner.char());
        scanner.next();
    }
    while (/[0-9_]/.test(scanner.char()) && !scanner.eof()) {
        acc.push(scanner.char());
        scanner.next();
    }
    if (acc.length === 0 || (acc.length === 1 && /[+-]/.test(acc[0]))) {
        return failure();
    }
    const int = parseInt(acc.filter((char) => char !== "_").join(""));
    return success(int);
}
export function Float(scanner) {
    scanner.nextUntilChar({ inline: true });
    let position = 0;
    while (scanner.char(position) &&
        !Patterns.END_OF_VALUE.test(scanner.char(position))) {
        if (!Patterns.FLOAT.test(scanner.char(position))) {
            return failure();
        }
        position++;
    }
    const acc = [];
    if (/[+-]/.test(scanner.char())) {
        acc.push(scanner.char());
        scanner.next();
    }
    while (Patterns.FLOAT.test(scanner.char()) && !scanner.eof()) {
        acc.push(scanner.char());
        scanner.next();
    }
    if (acc.length === 0) {
        return failure();
    }
    const float = parseFloat(acc.filter((char) => char !== "_").join(""));
    if (isNaN(float)) {
        return failure();
    }
    return success(float);
}
export function DateTime(scanner) {
    scanner.nextUntilChar({ inline: true });
    let dateStr = scanner.slice(0, 10);
    if (/^\d{4}-\d{2}-\d{2}/.test(dateStr)) {
        scanner.next(10);
    }
    else {
        return failure();
    }
    const acc = [];
    while (/[ 0-9TZ.:-]/.test(scanner.char()) && !scanner.eof()) {
        acc.push(scanner.char());
        scanner.next();
    }
    dateStr += acc.join("");
    const date = new Date(dateStr.trim());
    if (isNaN(date.getTime())) {
        throw new TOMLParseError(`Invalid date string "${dateStr}"`);
    }
    return success(date);
}
export function LocalTime(scanner) {
    scanner.nextUntilChar({ inline: true });
    let timeStr = scanner.slice(0, 8);
    if (/^(\d{2}):(\d{2}):(\d{2})/.test(timeStr)) {
        scanner.next(8);
    }
    else {
        return failure();
    }
    const acc = [];
    if (scanner.char() === ".") {
        acc.push(scanner.char());
        scanner.next();
    }
    else {
        return success(timeStr);
    }
    while (/[0-9]/.test(scanner.char()) && !scanner.eof()) {
        acc.push(scanner.char());
        scanner.next();
    }
    timeStr += acc.join("");
    return success(timeStr);
}
export function ArrayValue(scanner) {
    scanner.nextUntilChar({ inline: true });
    if (scanner.char() === "[") {
        scanner.next();
    }
    else {
        return failure();
    }
    const array = [];
    while (!scanner.eof()) {
        scanner.nextUntilChar();
        const result = Value(scanner);
        if (result.ok) {
            array.push(result.body);
        }
        else {
            break;
        }
        scanner.nextUntilChar({ inline: true });
        if (scanner.char() === ",") {
            scanner.next();
        }
        else {
            break;
        }
    }
    scanner.nextUntilChar();
    if (scanner.char() === "]") {
        scanner.next();
    }
    else {
        throw new TOMLParseError("Array is not closed");
    }
    return success(array);
}
export function InlineTable(scanner) {
    scanner.nextUntilChar();
    const pairs = surround("{", join(Pair, ","), "}")(scanner);
    if (!pairs.ok) {
        return failure();
    }
    const table = {};
    for (const pair of pairs.body) {
        deepAssign(table, pair);
    }
    return success(table);
}
export const Value = or([
    MultilineBasicString,
    MultilineLiteralString,
    BasicString,
    LiteralString,
    Symbols,
    DateTime,
    LocalTime,
    Float,
    Integer,
    ArrayValue,
    InlineTable,
]);
export const Pair = kv(DottedKey, "=", Value);
export function Block(scanner) {
    scanner.nextUntilChar();
    const result = merge(repeat(Pair))(scanner);
    if (result.ok) {
        return success({
            type: "Block",
            value: result.body,
        });
    }
    else {
        return failure();
    }
}
export const TableHeader = surround("[", DottedKey, "]");
export function Table(scanner) {
    scanner.nextUntilChar();
    const header = TableHeader(scanner);
    if (!header.ok) {
        return failure();
    }
    scanner.nextUntilChar();
    const block = Block(scanner);
    return success({
        type: "Table",
        key: header.body,
        value: block.ok ? block.body.value : {},
    });
}
export const TableArrayHeader = surround("[[", DottedKey, "]]");
export function TableArray(scanner) {
    scanner.nextUntilChar();
    const header = TableArrayHeader(scanner);
    if (!header.ok) {
        return failure();
    }
    scanner.nextUntilChar();
    const block = Block(scanner);
    return success({
        type: "TableArray",
        key: header.body,
        value: block.ok ? block.body.value : {},
    });
}
export function Toml(scanner) {
    const blocks = repeat(or([Block, TableArray, Table]))(scanner);
    if (!blocks.ok) {
        return failure();
    }
    const body = {};
    for (const block of blocks.body) {
        switch (block.type) {
            case "Block": {
                deepAssign(body, block.value);
                break;
            }
            case "Table": {
                Utils.deepAssignWithTable(body, block);
                break;
            }
            case "TableArray": {
                Utils.deepAssignWithTable(body, block);
                break;
            }
        }
    }
    return success(body);
}
export function ParserFactory(parser) {
    return function parse(tomlString) {
        const scanner = new Scanner(tomlString);
        let parsed = null;
        let err = null;
        try {
            parsed = parser(scanner);
        }
        catch (e) {
            err = e instanceof Error ? e : new Error("[non-error thrown]");
        }
        if (err || !parsed || !parsed.ok || !scanner.eof()) {
            const position = scanner.position();
            const subStr = tomlString.slice(0, position);
            const lines = subStr.split("\n");
            const row = lines.length;
            const column = (() => {
                let count = subStr.length;
                for (const line of lines) {
                    if (count > line.length) {
                        count -= line.length + 1;
                    }
                    else {
                        return count;
                    }
                }
                return count;
            })();
            const message = `Parse error on line ${row}, column ${column}: ${err ? err.message : `Unexpected character: "${scanner.char()}"`}`;
            throw new TOMLParseError(message);
        }
        return parsed.body;
    };
}
export const parse = ParserFactory(Toml);
//# sourceMappingURL=data:application/json;base64,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