import { ascend } from "./_comparators.ts";
import { BSNode } from "./bs_node.ts";
export * from "./_comparators.ts";
export class BSTree {
    compare;
    root = null;
    _size = 0;
    constructor(compare = ascend) {
        this.compare = compare;
    }
    static from(collection, options) {
        let result;
        let unmappedValues = [];
        if (collection instanceof BSTree) {
            result = new BSTree(options?.compare ?? collection.compare);
            if (options?.compare || options?.map) {
                unmappedValues = collection;
            }
            else {
                const nodes = [];
                if (collection.root) {
                    result.root = BSNode.from(collection.root);
                    nodes.push(result.root);
                }
                while (nodes.length) {
                    const node = nodes.pop();
                    const left = node.left
                        ? BSNode.from(node.left)
                        : null;
                    const right = node.right
                        ? BSNode.from(node.right)
                        : null;
                    if (left) {
                        left.parent = node;
                        nodes.push(left);
                    }
                    if (right) {
                        right.parent = node;
                        nodes.push(right);
                    }
                }
            }
        }
        else {
            result = (options?.compare
                ? new BSTree(options.compare)
                : new BSTree());
            unmappedValues = collection;
        }
        const values = options?.map
            ? Array.from(unmappedValues, options.map, options.thisArg)
            : unmappedValues;
        for (const value of values)
            result.insert(value);
        return result;
    }
    get size() {
        return this._size;
    }
    findNode(value) {
        let node = this.root;
        while (node) {
            const order = this.compare(value, node.value);
            if (order === 0)
                break;
            const direction = order < 0 ? "left" : "right";
            node = node[direction];
        }
        return node;
    }
    rotateNode(node, direction) {
        const replacementDirection = direction === "left"
            ? "right"
            : "left";
        if (!node[replacementDirection]) {
            throw new TypeError(`cannot rotate ${direction} without ${replacementDirection} child`);
        }
        const replacement = node[replacementDirection];
        node[replacementDirection] = replacement[direction] ?? null;
        if (replacement[direction])
            replacement[direction].parent = node;
        replacement.parent = node.parent;
        if (node.parent) {
            const parentDirection = node === node.parent[direction]
                ? direction
                : replacementDirection;
            node.parent[parentDirection] = replacement;
        }
        else {
            this.root = replacement;
        }
        replacement[direction] = node;
        node.parent = replacement;
    }
    insertNode(Node, value) {
        if (!this.root) {
            this.root = new Node(null, value);
            this._size++;
            return this.root;
        }
        else {
            let node = this.root;
            while (true) {
                const order = this.compare(value, node.value);
                if (order === 0)
                    break;
                const direction = order < 0 ? "left" : "right";
                if (node[direction]) {
                    node = node[direction];
                }
                else {
                    node[direction] = new Node(node, value);
                    this._size++;
                    return node[direction];
                }
            }
        }
        return null;
    }
    removeNode(value) {
        let removeNode = this.findNode(value);
        if (removeNode) {
            const successorNode = !removeNode.left || !removeNode.right
                ? removeNode
                : removeNode.findSuccessorNode();
            const replacementNode = successorNode.left ??
                successorNode.right;
            if (replacementNode)
                replacementNode.parent = successorNode.parent;
            if (!successorNode.parent) {
                this.root = replacementNode;
            }
            else {
                successorNode.parent[successorNode.directionFromParent()] =
                    replacementNode;
            }
            if (successorNode !== removeNode) {
                removeNode.value = successorNode.value;
                removeNode = successorNode;
            }
            this._size--;
        }
        return removeNode;
    }
    insert(value) {
        return !!this.insertNode(BSNode, value);
    }
    remove(value) {
        return !!this.removeNode(value);
    }
    find(value) {
        return this.findNode(value)?.value ?? null;
    }
    min() {
        return this.root ? this.root.findMinNode().value : null;
    }
    max() {
        return this.root ? this.root.findMaxNode().value : null;
    }
    clear() {
        this.root = null;
        this._size = 0;
    }
    isEmpty() {
        return this.size === 0;
    }
    *lnrValues() {
        const nodes = [];
        let node = this.root;
        while (nodes.length || node) {
            if (node) {
                nodes.push(node);
                node = node.left;
            }
            else {
                node = nodes.pop();
                yield node.value;
                node = node.right;
            }
        }
    }
    *rnlValues() {
        const nodes = [];
        let node = this.root;
        while (nodes.length || node) {
            if (node) {
                nodes.push(node);
                node = node.right;
            }
            else {
                node = nodes.pop();
                yield node.value;
                node = node.left;
            }
        }
    }
    *nlrValues() {
        const nodes = [];
        if (this.root)
            nodes.push(this.root);
        while (nodes.length) {
            const node = nodes.pop();
            yield node.value;
            if (node.right)
                nodes.push(node.right);
            if (node.left)
                nodes.push(node.left);
        }
    }
    *lrnValues() {
        const nodes = [];
        let node = this.root;
        let lastNodeVisited = null;
        while (nodes.length || node) {
            if (node) {
                nodes.push(node);
                node = node.left;
            }
            else {
                const lastNode = nodes[nodes.length - 1];
                if (lastNode.right && lastNode.right !== lastNodeVisited) {
                    node = lastNode.right;
                }
                else {
                    yield lastNode.value;
                    lastNodeVisited = nodes.pop();
                }
            }
        }
    }
    *lvlValues() {
        const children = [];
        let cursor = this.root;
        while (cursor) {
            yield cursor.value;
            if (cursor.left)
                children.push(cursor.left);
            if (cursor.right)
                children.push(cursor.right);
            cursor = children.shift() ?? null;
        }
    }
    *[Symbol.iterator]() {
        yield* this.lnrValues();
    }
}
//# sourceMappingURL=data:application/json;base64,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