// Copyright 2018-2024 the Deno authors. All rights reserved. MIT license.
// This module is browser compatible.
import { bytesToUuid } from "./_common.ts";
const UUID_RE = /^[0-9a-f]{8}-[0-9a-f]{4}-1[0-9a-f]{3}-[89ab][0-9a-f]{3}-[0-9a-f]{12}$/i;
/**
 * Determines whether a string is a valid
 * {@link https://www.rfc-editor.org/rfc/rfc9562.html#section-5.1 | UUIDv1}.
 *
 * @param id UUID value.
 *
 * @returns `true` if the string is a valid UUIDv1, otherwise `false`.
 *
 * @example Usage
 * ```ts
 * import { validate } from "@std/uuid/v1";
 * import { assert, assertFalse } from "@std/assert";
 *
 * assert(validate("ea71fc60-a713-11ee-af61-8349da24f689"));
 * assertFalse(validate("fac8c1e0-ad1a-4204-a0d0-8126ae84495d"));
 * ```
 */ export function validate(id) {
  return UUID_RE.test(id);
}
let _nodeId;
let _clockseq;
let _lastMSecs = 0;
let _lastNSecs = 0;
export function generate(options = {}, buf, offset) {
  let i = buf && offset || 0;
  const b = buf ?? [];
  let { node = _nodeId, clockseq = _clockseq } = options;
  if (node === undefined || clockseq === undefined) {
    // deno-lint-ignore no-explicit-any
    const seedBytes = options.random ?? options.rng ?? crypto.getRandomValues(new Uint8Array(16));
    if (node === undefined) {
      node = _nodeId = [
        seedBytes[0] | 0x01,
        seedBytes[1],
        seedBytes[2],
        seedBytes[3],
        seedBytes[4],
        seedBytes[5]
      ];
    }
    if (clockseq === undefined) {
      clockseq = _clockseq = (seedBytes[6] << 8 | seedBytes[7]) & 0x3fff;
    }
  }
  let { msecs = new Date().getTime(), nsecs = _lastNSecs + 1 } = options;
  const dt = msecs - _lastMSecs + (nsecs - _lastNSecs) / 10000;
  if (dt < 0 && options.clockseq === undefined) {
    clockseq = clockseq + 1 & 0x3fff;
  }
  if ((dt < 0 || msecs > _lastMSecs) && options.nsecs === undefined) {
    nsecs = 0;
  }
  if (nsecs > 10000) {
    throw new Error("Can't create more than 10M uuids/sec");
  }
  if (node.length !== 6) {
    throw new Error("Cannot create UUID. The node option must be an array of 6 bytes");
  }
  _lastMSecs = msecs;
  _lastNSecs = nsecs;
  _clockseq = clockseq;
  // We have to add this value because "msecs" here is the number of
  // milliseconds since January 1, 1970, not since October 15, 1582.
  // This is also the milliseconds from October 15, 1582 to January 1, 1970.
  msecs += 12219292800000;
  const tl = ((msecs & 0xfffffff) * 10000 + nsecs) % 0x100000000;
  b[i++] = tl >>> 24 & 0xff;
  b[i++] = tl >>> 16 & 0xff;
  b[i++] = tl >>> 8 & 0xff;
  b[i++] = tl & 0xff;
  const tmh = msecs / 0x100000000 * 10000 & 0xfffffff;
  b[i++] = tmh >>> 8 & 0xff;
  b[i++] = tmh & 0xff;
  b[i++] = tmh >>> 24 & 0xf | 0x10;
  b[i++] = tmh >>> 16 & 0xff;
  b[i++] = clockseq >>> 8 | 0x80;
  b[i++] = clockseq & 0xff;
  for(let n = 0; n < 6; ++n){
    b[i + n] = node[n];
  }
  return buf ?? bytesToUuid(b);
}
//# sourceMappingURL=data:application/json;base64,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
// denoCacheMetadata=11373624257968709047,4157984131322792248