/*---------------------------------------------------------------------------------------------
 *  Copyright (C) 2025 Posit Software, PBC. All rights reserved.
 *  Licensed under the Elastic License 2.0. See LICENSE.txt for license information.
 *--------------------------------------------------------------------------------------------*/

import * as vscode from 'vscode';
import { getEnabledProviders } from './config.js';
import { log } from './extension.js';

export interface ModelDefinition {
	name: string;
	identifier: string;
	maxInputTokens?: number;
	maxOutputTokens?: number;
}

/**
 * Get custom models from VS Code settings for a specific provider.
 */
export function getCustomModels(providerId: string): ModelDefinition[] {
	const config = vscode.workspace.getConfiguration('positron.assistant');
	const customModels = config.get<Record<string, ModelDefinition[]>>('models.custom', {});
	return customModels[providerId] || [];
}

/**
 * Check whether the provider IDs in the custom models are valid providers.
 */
export async function verifyProvidersInCustomModels() {
	const config = vscode.workspace.getConfiguration('positron.assistant');
	const customModels = config.get<Record<string, ModelDefinition[]>>('models.custom', {});
	const enabledProviders = await getEnabledProviders();

	const invalidProviders = Object.keys(customModels)
		// Note: 'copilot' is a special case, where we don't support customModels
		.filter(providerId => !enabledProviders.includes(providerId) || providerId === 'copilot');
	if (invalidProviders.length === 0) {
		return;
	}

	const message = vscode.l10n.t('Custom models contain unsupported providers: {0}. Please review your configuration for \'positron.assistant.models.custom\'', invalidProviders.map(p => `'${p}'`).join(', '));
	log.warn(message);
	const settingsAction = vscode.l10n.t('Open Settings');
	const selectedAction = await vscode.window.showWarningMessage(message, settingsAction);
	if (selectedAction === settingsAction) {
		await vscode.commands.executeCommand('workbench.action.openSettings', 'positron.assistant.models.custom');
	}
}

/**
 * Built-in model definitions that serve as fallback defaults when no user configuration
 * is provided and dynamic model querying is not available or fails.
 */
const builtInModelDefinitions = new Map<string, ModelDefinition[]>([
	['posit-ai', [
		{
			name: 'Claude Sonnet 4.5',
			identifier: 'claude-sonnet-4-5',
			maxInputTokens: 200_000, // reference: https://docs.anthropic.com/en/docs/about-claude/models/all-models#model-comparison-table
			maxOutputTokens: 64_000, // reference: https://docs.anthropic.com/en/docs/about-claude/models/all-models#model-comparison-table
		},
		{
			name: 'Claude Opus 4.1',
			identifier: 'claude-opus-4-1',
			maxInputTokens: 200_000, // reference: https://docs.anthropic.com/en/docs/about-claude/models/all-models#model-comparison-table
			maxOutputTokens: 32_000, // reference: https://docs.anthropic.com/en/docs/about-claude/models/all-models#model-comparison-table
		},
		{
			name: 'Claude Haiku 4.5',
			identifier: 'claude-haiku-4-5',
			maxInputTokens: 200_000, // reference: https://docs.anthropic.com/en/docs/about-claude/models/all-models#model-comparison-table
			maxOutputTokens: 64_000, // reference: https://docs.anthropic.com/en/docs/about-claude/models/all-models#model-comparison-table
		},
	]],
	['google', [
		{
			name: 'Gemini 2.5 Flash',
			identifier: 'gemini-2.5-pro-exp-03-25',
			maxOutputTokens: 65_536, // reference: https://ai.google.dev/gemini-api/docs/models#gemini-2.5-flash-preview
		},
		{
			name: 'Gemini 2.0 Flash',
			identifier: 'gemini-2.0-flash-exp',
			maxOutputTokens: 8_192, // reference: https://ai.google.dev/gemini-api/docs/models#gemini-2.0-flash
		},
	]],
	// Model listing reference: https://docs.snowflake.com/en/user-guide/snowflake-cortex/aisql#regional-availability
	['snowflake-cortex', [
		{
			name: 'Claude Sonnet 4',
			identifier: 'claude-4-sonnet',
			maxInputTokens: 200_000, // Snowflake Cortex AI model context limit
		},
		{
			name: 'Claude Sonnet 4.5',
			identifier: 'claude-sonnet-4-5',
			maxInputTokens: 200_000, // Snowflake Cortex AI model context limit
		},
		{
			name: 'Claude Haiku 4.5',
			identifier: 'claude-haiku-4-5',
			maxInputTokens: 200_000, // Snowflake Cortex AI model context limit
		},
		{
			name: 'GPT-5',
			identifier: 'openai-gpt-5',
			maxInputTokens: 128_000, // Typical GPT-5 context window
		},
		{
			name: 'GPT-4.1',
			identifier: 'openai-gpt-4.1',
			maxInputTokens: 128_000, // Typical GPT-5 context window
		},
	]]
]);

/**
 * Get all available model definitions for a provider, with intelligent fallback hierarchy:
 * 1. Custom models (from settings) - highest priority
 * 2. Built-in model definitions - fallback when no user config
 */
export function getAllModelDefinitions(providerId: string): ModelDefinition[] {
	const models = getCustomModels(providerId);
	if (models.length > 0) {
		return models;
	}
	return builtInModelDefinitions.get(providerId) || [];
}
