import { loadConfig } from "@smithy/node-config-provider";
import { NodeHttpHandler } from "@smithy/node-http-handler";
import { HttpRequest } from "@smithy/protocol-http";
import { sdkStreamMixin } from "@smithy/util-stream";
import { ENDPOINT_SELECTORS, IMDSv1_DISABLED_SELECTORS } from "./ConfigLoaders";
export class MetadataService {
    disableFetchToken;
    config;
    constructor(options = {}) {
        this.config = (async () => {
            const profile = options?.profile || process.env.AWS_PROFILE;
            return {
                endpoint: options.endpoint ?? (await loadConfig(ENDPOINT_SELECTORS, { profile })()),
                httpOptions: {
                    timeout: options?.httpOptions?.timeout || 0,
                },
                ec2MetadataV1Disabled: options?.ec2MetadataV1Disabled ?? (await loadConfig(IMDSv1_DISABLED_SELECTORS, { profile })()),
            };
        })();
        this.disableFetchToken = options?.disableFetchToken || false;
    }
    async request(path, options) {
        const { endpoint, ec2MetadataV1Disabled, httpOptions } = await this.config;
        const handler = new NodeHttpHandler({
            requestTimeout: httpOptions?.timeout,
            connectionTimeout: httpOptions?.timeout,
        });
        const endpointUrl = new URL(endpoint);
        const headers = options.headers || {};
        if (this.disableFetchToken && ec2MetadataV1Disabled) {
            throw new Error("IMDSv1 is disabled and fetching token is disabled, cannot make the request.");
        }
        if (!this.disableFetchToken) {
            try {
                headers["x-aws-ec2-metadata-token"] = await this.fetchMetadataToken();
            }
            catch (err) {
                if (ec2MetadataV1Disabled) {
                    throw err;
                }
            }
        }
        const request = new HttpRequest({
            method: options.method || "GET",
            headers: headers,
            hostname: endpointUrl.hostname,
            path: endpointUrl.pathname + path,
            protocol: endpointUrl.protocol,
        });
        try {
            const { response } = await handler.handle(request, {});
            if (response.statusCode === 200 && response.body) {
                return sdkStreamMixin(response.body).transformToString();
            }
            else {
                throw new Error(`Request failed with status code ${response.statusCode}`);
            }
        }
        catch (error) {
            throw new Error(`Error making request to the metadata service: ${error}`);
        }
    }
    async fetchMetadataToken() {
        const { endpoint, httpOptions } = await this.config;
        const handler = new NodeHttpHandler({
            requestTimeout: httpOptions?.timeout,
            connectionTimeout: httpOptions?.timeout,
        });
        const endpointUrl = new URL(endpoint);
        const tokenRequest = new HttpRequest({
            method: "PUT",
            headers: {
                "x-aws-ec2-metadata-token-ttl-seconds": "21600",
            },
            hostname: endpointUrl.hostname,
            path: "/latest/api/token",
            protocol: endpointUrl.protocol,
        });
        try {
            const { response } = await handler.handle(tokenRequest, {});
            if (response.statusCode === 200 && response.body) {
                return sdkStreamMixin(response.body).transformToString();
            }
            else {
                throw new Error(`Failed to fetch metadata token with status code ${response.statusCode}`);
            }
        }
        catch (error) {
            if (error?.statusCode === 400) {
                throw new Error(`Error fetching metadata token: ${error}`);
            }
            else if (error.message === "TimeoutError" || [403, 404, 405].includes(error.statusCode)) {
                this.disableFetchToken = true;
                throw new Error(`Error fetching metadata token: ${error}. disableFetchToken is enabled`);
            }
            throw new Error(`Error fetching metadata token: ${error}`);
        }
    }
}
