// Copyright 2018-2024 the Deno authors. All rights reserved. MIT license.
// This module is browser compatible.
/**
 * Returns the first element that is the smallest value of the given function or
 * undefined if there are no elements.
 *
 * @template T The type of the elements in the array.
 *
 * @param array The array to find the minimum element in.
 * @param selector The function to get the value to compare from each element.
 *
 * @returns The first element that is the smallest value of the given function
 * or undefined if there are no elements.
 *
 * @example Basic usage
 * ```ts
 * import { minBy } from "@std/collections/min-by";
 * import { assertEquals } from "@std/assert/assert-equals";
 *
 * const people = [
 *   { name: "Anna", age: 34 },
 *   { name: "Kim", age: 42 },
 *   { name: "John", age: 23 },
 * ];
 *
 * const personWithMinAge = minBy(people, (i) => i.age);
 *
 * assertEquals(personWithMinAge, { name: "John", age: 23 });
 * ```
 */ export function minBy(array, selector) {
  let min;
  let minValue;
  for (const current of array){
    const currentValue = selector(current);
    if (minValue === undefined || currentValue < minValue) {
      min = current;
      minValue = currentValue;
    }
  }
  return min;
}
//# sourceMappingURL=data:application/json;base64,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