"use strict";
/*
 * languages.ts
 *
 * Copyright (C) 2022 by Posit Software, PBC
 *
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.LANGUAGES = exports.R_MARKDOWN = exports.QUARTO = exports.JUPYTER = exports.PYTHON = exports.R = exports.commandLineArgType = void 0;
exports.clearAfter = clearAfter;
exports.extension = extension;
exports.detectFileType = detectFileType;
exports.getScriptName = getScriptName;
exports.validScriptExtension = validScriptExtension;
const index_1 = require("./index");
function clearAfter(parentEl, childEl) {
    const children = Array.from(parentEl.children).reverse();
    for (const child of children) {
        if (child === childEl) {
            break;
        }
        parentEl.removeChild(child);
    }
}
function createVersionSelectorBuilder({ id, valueName, valueListName, labelText }) {
    return ({ currents = [], defaults = [], set = () => { } }) => {
        var _a;
        // always source the versions list from the defaults -- it should never change
        const listValue = defaults.find((dflt) => dflt.name === valueListName);
        const list = ((_a = listValue === null || listValue === void 0 ? void 0 : listValue.value) !== null && _a !== void 0 ? _a : []);
        // pick the value from the currents, because the user can modify it
        const value = currents.find((dflt) => dflt.name === valueName);
        const label = document.createElement("label");
        label.classList.add("col-sm-3", "control-label");
        label.textContent = labelText;
        label.setAttribute("for", id);
        const select = document.createElement("select");
        select.classList.add("form-control");
        select.id = id;
        for (const value of list) {
            const option = document.createElement("option");
            option.value = value.id;
            option.textContent = value.name;
            option.dataset.language = value.language;
            if (value.path) {
                option.dataset.path = value.path;
            }
            if (value.venv) {
                option.dataset.venv = value.venv;
            }
            if (value.current) {
                option.selected = true;
            }
            select.appendChild(option);
        }
        const option = document.createElement("option");
        const OTHER_TEXT = "Other...";
        option.value = OTHER_TEXT;
        option.textContent = OTHER_TEXT;
        select.appendChild(option);
        const pathSelectionWrapper = document.createElement("div");
        pathSelectionWrapper.classList.add("form-grid");
        pathSelectionWrapper.style.gridColumn = "2 / -1";
        pathSelectionWrapper.style.display = "none";
        const pathInputEl = document.createElement("input");
        pathInputEl.classList.add("form-control");
        pathInputEl.type = "text";
        pathInputEl.style.width = "100%";
        pathInputEl.placeholder = "Path to executable";
        pathInputEl.id = `${id}-path`;
        pathInputEl.style.gridColumn = "1 / 3";
        const pathBrowseEl = document.createElement("button");
        pathBrowseEl.classList.add("secondary", "browse-button");
        pathBrowseEl.textContent = "Browse";
        pathSelectionWrapper.append(pathInputEl);
        pathSelectionWrapper.append(pathBrowseEl);
        pathBrowseEl.addEventListener("click", () => {
            (0, index_1.handleOnBrowse)(false, (event) => {
                const { message } = event.data;
                pathInputEl.value = message;
                pathInputEl.dispatchEvent(new Event("change"));
            });
        });
        pathInputEl.addEventListener("change", () => {
            set([
                {
                    name: `${valueName}-other`,
                    value: pathInputEl.value
                },
                {
                    name: valueName,
                    value: OTHER_TEXT
                }
            ]);
        });
        select.addEventListener("change", () => {
            pathSelectionWrapper.style.display = select.value !== OTHER_TEXT ? "none" : "";
            set([
                {
                    name: valueName,
                    value: select.value
                }
            ]);
        });
        // Populate other version path if it was specified
        const otherVersionIndex = currents.findIndex((dflt) => dflt.name === `${valueName}-other`);
        if (otherVersionIndex !== -1) {
            const otherVersionPath = currents[otherVersionIndex].value;
            if (otherVersionPath) {
                pathInputEl.value = otherVersionPath;
                select.value = OTHER_TEXT;
                pathInputEl.dispatchEvent(new Event("change"));
            }
        }
        else if (value === null || value === void 0 ? void 0 : value.value) {
            select.value = value.value;
        }
        // Trigger a change to select whatever version is currently displayed in the selector
        select.dispatchEvent(new Event("change"));
        // include a spacer div for the unused grid column
        // include a warning placeholder
        const warningEl = document.createElement("div");
        warningEl.id = "version-path-warning";
        warningEl.style.gridColumn = "2 / -1";
        warningEl.style.color = "var(--vscode-editorWarning-foreground)";
        warningEl.style.display = "none";
        return [label, select, document.createElement("div"), pathSelectionWrapper, warningEl];
    };
}
// create element builders/renderers for r versions and python verisons
const buildRVersionSelector = createVersionSelectorBuilder({
    id: "r-version-selector",
    valueListName: "r-versions",
    valueName: "r-version",
    labelText: "R version"
});
const buildPyVersionSelector = createVersionSelectorBuilder({
    id: "py-version-selector",
    valueListName: "py-versions",
    valueName: "py-version",
    labelText: "Python version"
});
function buildArgumentListEditor({ currents = [], defaults = [], set = () => { } }) {
    const result = [];
    let first;
    let last;
    // construct the linked list from defaults array
    const createLinkedList = (defaults) => {
        first = undefined;
        last = undefined;
        if (defaults.length) {
            first = { name: defaults[0].name, value: defaults[0].value };
            last = first;
            for (const dflt of defaults.slice(1)) {
                const current = Object.assign(Object.assign({}, dflt), { last });
                last.next = current;
                last = current;
            }
        }
    };
    createLinkedList(currents);
    // flatten the linked list to an array.
    // called when passing args array to `set` callback
    const argList = () => {
        const result = [];
        if (!last) {
            return result;
        }
        let current = last;
        while (current) {
            if (current.name || current.value) {
                // only include arguments that are not completely empty
                // we're traversing backwards, so we're using `unshift` to
                // preserve visual order
                result.unshift({ name: current.name, value: current.value });
            }
            current = current.last;
        }
        return result;
    };
    // create a mechanism for showing/hiding arguments UI
    const argLabel = document.createElement("label");
    argLabel.classList.add("col-sm-3", "control-label");
    argLabel.style.width = "unset";
    argLabel.textContent = "Arguments";
    const argSummary = document.createElement("div");
    argSummary.classList.add("form-control", "argument-summary");
    argSummary.style.width = "unset";
    argSummary.style.opacity = "60%";
    const updateArgPreview = () => {
        argSummary.textContent = argList()
            .map((arg) => { var _a; return (_a = arg.value) === null || _a === void 0 ? void 0 : _a.trim(); })
            .join(" ");
    };
    updateArgPreview();
    // hijack the set callback to trigger updates to the args preview
    // whenever the callback is called -- easier than remembering to call both
    const oldSet = set;
    set = (args) => {
        updateArgPreview();
        oldSet(args);
    };
    let showingArgs = false;
    const argShowHideButton = document.createElement("button");
    argShowHideButton.classList.add("browse-button", "secondary");
    argShowHideButton.addEventListener("click", () => {
        toggleArgs();
    });
    const toggleArgs = (override) => {
        var _a;
        showingArgs = override !== null && override !== void 0 ? override : !showingArgs;
        argShowHideButton.textContent = showingArgs ? "Hide" : "Modify";
        (_a = argShowHideButton.parentElement) === null || _a === void 0 ? void 0 : _a.classList.toggle("show-arguments", showingArgs);
        set(argList());
    };
    toggleArgs(false);
    result.push(argLabel, argSummary, argShowHideButton);
    const removers = new Set();
    // create the UI elements which manage a particular argument
    const createListItem = (argument) => {
        const valueInputEl = document.createElement("input");
        valueInputEl.type = "text";
        valueInputEl.placeholder = "Flag / Argument";
        valueInputEl.value = argument.value || "";
        valueInputEl.classList.add("form-control", "argument");
        valueInputEl.style.width = "unset";
        valueInputEl.style.gridColumn = "1 / 3";
        valueInputEl.addEventListener("change", () => {
            argument.value = valueInputEl.value;
            set(argList());
        });
        const deleteArgEl = document.createElement("button");
        deleteArgEl.classList.add("delete-button", "argument", "delete-argument");
        deleteArgEl.innerHTML = "&#x2715;";
        const remover = () => {
            if (argument.last) {
                argument.last.next = argument.next;
            }
            if (argument.next) {
                argument.next.last = argument.last;
            }
            // ensure that the last argument is always kept updated when removing
            // this is important so we can append new arguments correctly
            if (argument === last) {
                last = argument.last;
            }
            valueInputEl.remove();
            deleteArgEl.remove();
            removers.delete(remover);
            set(argList());
        };
        deleteArgEl.addEventListener("click", remover);
        removers.add(remover);
        return [valueInputEl, deleteArgEl];
    };
    const drawList = () => {
        let current = first;
        const result = [];
        while (current) {
            result.push(...createListItem(current));
            last = current;
            current = current.next;
        }
        return result;
    };
    result.push(...drawList());
    const addButton = document.createElement("button");
    addButton.classList.add("secondary", "argument", "add-argument");
    addButton.style.gridColumn = "1 / 3";
    addButton.textContent = "+ Add argument";
    addButton.addEventListener("click", () => {
        const argument = { last };
        if (last) {
            last.next = argument;
        }
        createListItem(argument).forEach((el) => addButton.insertAdjacentElement("beforebegin", el));
        last = argument;
        set(argList());
    });
    result.push(addButton);
    const resetButton = document.createElement("button");
    resetButton.classList.add("delete-button", "argument", "reset-arguments");
    resetButton.textContent = "Reset";
    resetButton.addEventListener("click", () => {
        // remove everything in the list first, then clear the removers cache
        // so they aren't called again another another reset
        removers.forEach((remover) => remover());
        removers.clear();
        // reset the linked list based off the defaults, not the currents
        createLinkedList(defaults);
        drawList().forEach((el) => addButton.insertAdjacentElement("beforebegin", el));
        // re-update the currents and redraw the preview
        set(argList());
    });
    result.push(resetButton);
    return result;
}
exports.commandLineArgType = [
    {
        name: "Command Line",
        type: "list",
        render: buildArgumentListEditor
    }
];
exports.R = "r";
exports.PYTHON = "py";
exports.JUPYTER = "ipynb";
exports.QUARTO = "qmd";
exports.R_MARKDOWN = "rmd";
exports.LANGUAGES = new Map()
    .set(exports.R, {
    name: "R",
    ext: exports.R,
    controls: [
        {
            name: "R Version",
            type: "select",
            render: buildRVersionSelector
        },
        ...exports.commandLineArgType
    ]
})
    .set(exports.PYTHON, {
    name: "Python",
    ext: exports.PYTHON,
    controls: [
        {
            name: "Python Version",
            type: "select",
            render: buildPyVersionSelector
        },
        ...exports.commandLineArgType
    ]
});
/*
  .set(JUPYTER, {
    name: 'Jupyter Notebook',
    ext: JUPYTER,
    controls: commandLineArgType
  })
  .set(QUARTO, {
    name: 'Quarto Markdown',
    ext: QUARTO,
    controls: commandLineArgType
  })
  .set(R_MARKDOWN, {
    name: 'R Markdown',
    ext: R_MARKDOWN,
    controls: commandLineArgType
  });
  */
function extension(path) {
    if (!path) {
        return "";
    }
    const ext = path.toLowerCase().split(".").at(-1);
    if (!ext) {
        return "";
    }
    return exports.LANGUAGES.has(ext) ? ext : "";
}
function detectFileType(path) {
    return exports.LANGUAGES.get(extension(path));
}
function getScriptName(path) {
    var _a;
    return ((_a = path.split("/").at(-1)) === null || _a === void 0 ? void 0 : _a.split(".").slice(0, -1).join("")) || "New Workbench Job";
}
function validScriptExtension(path) {
    var _a, _b;
    // output streams in VS Code count as files, and we don't want to
    // trigger changes when the user selects one of these.
    // Only continue if the active file has one of the extension types
    // we care about, defined in LANGAUGES.keys
    const langs = Array.from(exports.LANGUAGES.keys());
    // at(-1) accesses the element at the end of the array, which is the
    // extension in this case. LowerCase guarantees we can find it even if
    // the user has decided to change its capitalization (for whatever reason)
    return !path.includes("extension-output") && langs.includes((_b = (_a = path.split(".").at(-1)) === null || _a === void 0 ? void 0 : _a.toLowerCase()) !== null && _b !== void 0 ? _b : "");
}
//# sourceMappingURL=data:application/json;base64,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