"use strict";
/*
 * languages.ts
 *
 * Copyright (C) 2022 by Posit Software, PBC
 *
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.LANGUAGES = exports.R_MARKDOWN = exports.QUARTO = exports.JUPYTER = exports.PYTHON = exports.R = exports.commandLineArgType = void 0;
exports.clearAfter = clearAfter;
exports.extension = extension;
exports.detectFileType = detectFileType;
exports.getScriptName = getScriptName;
exports.validScriptExtension = validScriptExtension;
const index_1 = require("./index");
function clearAfter(parentEl, childEl) {
    const children = Array.from(parentEl.children).reverse();
    for (const child of children) {
        if (child === childEl) {
            break;
        }
        parentEl.removeChild(child);
    }
}
function createVersionSelectorBuilder({ id, valueName, valueListName, labelText }) {
    return ({ currents = [], defaults = [], set = () => { } }) => {
        var _a;
        // always source the versions list from the defaults -- it should never change
        const listValue = defaults.find(dflt => dflt.name === valueListName);
        const list = ((_a = listValue === null || listValue === void 0 ? void 0 : listValue.value) === null || _a === void 0 ? void 0 : _a.split(',').map(str => str.trim())) || [];
        // pick the value from the currents, because the user can modify it
        const value = currents.find(dflt => dflt.name === valueName);
        const label = document.createElement('label');
        label.classList.add('col-sm-3', 'control-label');
        label.textContent = labelText;
        label.setAttribute('for', id);
        const select = document.createElement('select');
        select.classList.add('form-control');
        select.id = id;
        for (const value of list) {
            const option = document.createElement('option');
            option.value = value;
            option.textContent = value;
            select.appendChild(option);
        }
        const option = document.createElement('option');
        const OTHER_TEXT = 'Other...';
        option.value = OTHER_TEXT;
        option.textContent = OTHER_TEXT;
        select.appendChild(option);
        const pathSelectionWrapper = document.createElement('div');
        pathSelectionWrapper.classList.add('form-grid');
        pathSelectionWrapper.style.gridColumn = '2 / -1';
        pathSelectionWrapper.style.display = 'none';
        const pathInputEl = document.createElement('input');
        pathInputEl.classList.add('form-control');
        pathInputEl.type = 'text';
        pathInputEl.style.width = '100%';
        pathInputEl.placeholder = 'Path to executable';
        pathInputEl.id = `${id}-path`;
        pathInputEl.style.gridColumn = '1 / 3';
        const pathBrowseEl = document.createElement('button');
        pathBrowseEl.classList.add('secondary', 'browse-button');
        pathBrowseEl.textContent = 'Browse';
        pathSelectionWrapper.append(pathInputEl);
        pathSelectionWrapper.append(pathBrowseEl);
        pathBrowseEl.addEventListener('click', () => {
            (0, index_1.handleOnBrowse)(false, (event) => {
                const { message } = event.data;
                pathInputEl.value = message;
                pathInputEl.dispatchEvent(new Event('change'));
            });
        });
        pathInputEl.addEventListener('change', () => {
            set([
                {
                    name: `${valueName}-other`,
                    value: pathInputEl.value
                },
                {
                    name: valueName,
                    value: OTHER_TEXT
                }
            ]);
        });
        select.addEventListener('change', () => {
            pathSelectionWrapper.style.display = select.value !== OTHER_TEXT ? 'none' : '';
            set([
                {
                    name: valueName,
                    value: select.value
                }
            ]);
        });
        // Populate other version path if it was specified
        const otherVersionIndex = currents.findIndex(dflt => dflt.name === `${valueName}-other`);
        if (otherVersionIndex !== -1) {
            const otherVersionPath = currents[otherVersionIndex].value;
            if (otherVersionPath) {
                pathInputEl.value = otherVersionPath;
                select.value = OTHER_TEXT;
                pathInputEl.dispatchEvent(new Event('change'));
            }
        }
        else if (value === null || value === void 0 ? void 0 : value.value) {
            select.value = value.value;
        }
        // Trigger a change to select whatever version is currently displayed in the selector
        select.dispatchEvent(new Event('change'));
        // include a spacer div for the unused grid column
        // include a warning placeholder
        const warningEl = document.createElement('div');
        warningEl.id = 'version-path-warning';
        warningEl.style.gridColumn = '2 / -1';
        warningEl.style.color = 'var(--vscode-editorWarning-foreground)';
        warningEl.style.display = 'none';
        return [label, select, document.createElement('div'), pathSelectionWrapper, warningEl];
    };
}
// create element builders/renderers for r versions and python verisons
const buildRVersionSelector = createVersionSelectorBuilder({
    id: 'r-version-selector',
    valueListName: 'r-versions',
    valueName: 'r-version',
    labelText: 'R version'
});
const buildPyVersionSelector = createVersionSelectorBuilder({
    id: 'py-version-selector',
    valueListName: 'py-versions',
    valueName: 'py-version',
    labelText: 'Python version'
});
function buildArgumentListEditor({ currents = [], defaults = [], set = () => { } }) {
    const result = [];
    let first;
    let last;
    // construct the linked list from defaults array
    const createLinkedList = (defaults) => {
        first = undefined;
        last = undefined;
        if (defaults.length) {
            first = { name: defaults[0].name, value: defaults[0].value };
            last = first;
            for (const dflt of defaults.slice(1)) {
                const current = Object.assign(Object.assign({}, dflt), { last });
                last.next = current;
                last = current;
            }
        }
    };
    createLinkedList(currents);
    // flatten the linked list to an array. 
    // called when passing args array to `set` callback
    const argList = () => {
        const result = [];
        if (!last) {
            return result;
        }
        let current = last;
        while (current) {
            if (current.name || current.value) {
                // only include arguments that are not completely empty 
                // we're traversing backwards, so we're using `unshift` to
                // preserve visual order
                result.unshift({ name: current.name, value: current.value });
            }
            current = current.last;
        }
        return result;
    };
    // create a mechanism for showing/hiding arguments UI
    const argLabel = document.createElement('label');
    argLabel.classList.add('col-sm-3', 'control-label');
    argLabel.style.width = 'unset';
    argLabel.textContent = 'Arguments';
    const argSummary = document.createElement('div');
    argSummary.classList.add('form-control', 'argument-summary');
    argSummary.style.width = 'unset';
    argSummary.style.opacity = '60%';
    const updateArgPreview = () => {
        argSummary.textContent = argList().map(arg => { var _a; return (_a = arg.value) === null || _a === void 0 ? void 0 : _a.trim(); }).join(' ');
    };
    updateArgPreview();
    // hijack the set callback to trigger updates to the args preview
    // whenever the callback is called -- easier than remembering to call both
    const oldSet = set;
    set = (args) => {
        updateArgPreview();
        oldSet(args);
    };
    let showingArgs = false;
    const argShowHideButton = document.createElement('button');
    argShowHideButton.classList.add('browse-button', 'secondary');
    argShowHideButton.addEventListener('click', () => {
        toggleArgs();
    });
    const toggleArgs = (override) => {
        var _a;
        showingArgs = override !== null && override !== void 0 ? override : !showingArgs;
        argShowHideButton.textContent = showingArgs ? 'Hide' : 'Modify';
        (_a = argShowHideButton.parentElement) === null || _a === void 0 ? void 0 : _a.classList.toggle('show-arguments', showingArgs);
        set(argList());
    };
    toggleArgs(false);
    result.push(argLabel, argSummary, argShowHideButton);
    const removers = new Set();
    // create the UI elements which manage a particular argument
    const createListItem = (argument) => {
        const valueInputEl = document.createElement('input');
        valueInputEl.type = 'text';
        valueInputEl.placeholder = 'Flag / Argument';
        valueInputEl.value = argument.value || '';
        valueInputEl.classList.add('form-control', 'argument');
        valueInputEl.style.width = 'unset';
        valueInputEl.style.gridColumn = '1 / 3';
        valueInputEl.addEventListener('change', () => {
            argument.value = valueInputEl.value;
            set(argList());
        });
        const deleteArgEl = document.createElement('button');
        deleteArgEl.classList.add('delete-button', 'argument', 'delete-argument');
        deleteArgEl.innerHTML = '&#x2715;';
        const remover = () => {
            if (argument.last) {
                argument.last.next = argument.next;
            }
            if (argument.next) {
                argument.next.last = argument.last;
            }
            // ensure that the last argument is always kept updated when removing
            // this is important so we can append new arguments correctly
            if (argument === last) {
                last = argument.last;
            }
            valueInputEl.remove();
            deleteArgEl.remove();
            removers.delete(remover);
            set(argList());
        };
        deleteArgEl.addEventListener('click', remover);
        removers.add(remover);
        return [valueInputEl, deleteArgEl];
    };
    const drawList = () => {
        let current = first;
        const result = [];
        while (current) {
            result.push(...createListItem(current));
            last = current;
            current = current.next;
        }
        return result;
    };
    result.push(...drawList());
    const addButton = document.createElement('button');
    addButton.classList.add('secondary', 'argument', 'add-argument');
    addButton.style.gridColumn = '1 / 3';
    addButton.textContent = '+ Add argument';
    addButton.addEventListener('click', () => {
        const argument = { last };
        if (last) {
            last.next = argument;
        }
        createListItem(argument).forEach(el => addButton.insertAdjacentElement('beforebegin', el));
        last = argument;
        set(argList());
    });
    result.push(addButton);
    const resetButton = document.createElement('button');
    resetButton.classList.add('delete-button', 'argument', 'reset-arguments');
    resetButton.textContent = 'Reset';
    resetButton.addEventListener('click', () => {
        // remove everything in the list first, then clear the removers cache
        // so they aren't called again another another reset
        removers.forEach(remover => remover());
        removers.clear();
        // reset the linked list based off the defaults, not the currents
        createLinkedList(defaults);
        drawList().forEach(el => addButton.insertAdjacentElement('beforebegin', el));
        // re-update the currents and redraw the preview
        set(argList());
    });
    result.push(resetButton);
    return result;
}
exports.commandLineArgType = [{
        name: 'Command Line',
        type: 'list',
        render: buildArgumentListEditor
    }];
exports.R = 'r';
exports.PYTHON = 'py';
exports.JUPYTER = 'ipynb';
exports.QUARTO = 'qmd';
exports.R_MARKDOWN = 'rmd';
exports.LANGUAGES = new Map()
    .set(exports.R, {
    name: 'R',
    ext: exports.R,
    controls: [
        {
            name: 'R Version',
            type: 'select',
            render: buildRVersionSelector
        },
        ...exports.commandLineArgType
    ]
})
    .set(exports.PYTHON, {
    name: 'Python',
    ext: exports.PYTHON,
    controls: [
        {
            name: 'Python Version',
            type: 'select',
            render: buildPyVersionSelector
        },
        ...exports.commandLineArgType
    ]
});
/*
.set(JUPYTER, {
  name: 'Jupyter Notebook',
  ext: JUPYTER,
  controls: commandLineArgType
})
.set(QUARTO, {
  name: 'Quarto Markdown',
  ext: QUARTO,
  controls: commandLineArgType
})
.set(R_MARKDOWN, {
  name: 'R Markdown',
  ext: R_MARKDOWN,
  controls: commandLineArgType
});
*/
function extension(path) {
    if (!path) {
        return '';
    }
    const ext = path.toLowerCase().split('.').at(-1);
    if (!ext) {
        return '';
    }
    return exports.LANGUAGES.has(ext) ? ext : '';
}
function detectFileType(path) {
    return exports.LANGUAGES.get(extension(path));
}
function getScriptName(path) {
    var _a;
    return ((_a = path.split('/').at(-1)) === null || _a === void 0 ? void 0 : _a.split('.').slice(0, -1).join('')) || 'New Workbench Job';
}
function validScriptExtension(path) {
    var _a, _b;
    // output streams in VS Code count as files, and we don't want to
    // trigger changes when the user selects one of these.
    // Only continue if the active file has one of the extension types
    // we care about, defined in LANGAUGES.keys
    const langs = Array.from(exports.LANGUAGES.keys());
    // at(-1) accesses the element at the end of the array, which is the
    // extension in this case. LowerCase guarantees we can find it even if 
    // the user has decided to change its capitalization (for whatever reason)
    return !path.includes("extension-output") && langs.includes((_b = (_a = path.split('.').at(-1)) === null || _a === void 0 ? void 0 : _a.toLowerCase()) !== null && _b !== void 0 ? _b : '');
}
//# sourceMappingURL=data:application/json;base64,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