"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const jsdom_1 = require("jsdom");
const fs_1 = __importDefault(require("fs"));
const path_1 = __importDefault(require("path"));
const vscode_api_1 = require("./vscode_api");
const test_data_1 = require("./test_data");
const __1 = require("..");
// load our html file
let htmlFile = fs_1.default.readFileSync(path_1.default.resolve('./index.html'), 'utf8');
// change the script locations
htmlFile = htmlFile.replace('%LAUNCHER-UI%', path_1.default.resolve('launcher/launcher.js'));
htmlFile = htmlFile.replace('%SCRIPT%', path_1.default.resolve('dist/index.js'));
// process the html file
const processedDom = new jsdom_1.JSDOM(htmlFile);
// eject everything in head which is not needed for testing, preserve active-file meta
for (const child of processedDom.window.document.head.children) {
    if (child.localName.toLowerCase() === 'meta' && child.getAttribute('name') === 'active-file') {
        continue;
    }
    child.remove();
}
// re-serialize now that we have our sanitized htmlFile. We can use
// this when loading copies of this DOM for testing
htmlFile = processedDom.serialize();
function newDom(api_1) {
    return __awaiter(this, arguments, void 0, function* (api, activeFile = '') {
        const newHtmlFile = htmlFile.replace('%ACTIVE-FILE%', activeFile);
        const processedDom = new jsdom_1.JSDOM(newHtmlFile, {
            url: `file://${path_1.default.resolve('./.')}`,
            runScripts: 'dangerously',
            resources: 'usable'
        });
        Object.assign(processedDom.window, {
            acquireVsCodeApi: () => {
                if (api) {
                    api.postBack = message => { processedDom.window.postMessage(message, '*'); };
                }
                return api || (0, vscode_api_1.acquireVsCodeApi)();
            }
        });
        Object.assign(processedDom.window.Range.prototype, {
            getBoundingClientRect: () => {
                return {
                    x: 0,
                    y: 0,
                    width: 0,
                    height: 0,
                    top: 0,
                    right: 0,
                    bottom: 0,
                    left: 0,
                    toJSON: jest.fn()
                };
            }
        });
        yield new Promise(resolve => {
            processedDom.window.addEventListener('load', () => {
                resolve();
            });
        });
        return processedDom;
    });
}
const BROWSE_PATH_SCRIPT = '/path/to/code.R';
const BROWSE_PATH_DIRECTORY = '/path/to/';
function createDefaultApi(postMessage) {
    const state = {};
    const api = (0, vscode_api_1.acquireVsCodeApi)(state);
    api.postBack = postMessage || (() => { });
    api.postMessage = (message) => __awaiter(this, void 0, void 0, function* () {
        var _a, _b, _c;
        switch (message.event) {
            case 'launch':
                console.log({
                    launchSpec: message.launchSpec,
                    jobOptions: message.jobOptions
                });
                break;
            case 'get_info':
                (_a = api.postBack) === null || _a === void 0 ? void 0 : _a.call(api, {
                    type: 'get_info',
                    message: JSON.stringify(test_data_1.launcherInfo, null, 2)
                });
                break;
            case 'browse_path':
                (_b = api.postBack) === null || _b === void 0 ? void 0 : _b.call(api, {
                    type: 'browse_path',
                    id: message.id,
                    message: message.directory ? BROWSE_PATH_DIRECTORY : BROWSE_PATH_SCRIPT,
                    canceled: false
                });
                break;
            case 'get_info_for_script': {
                let response = '{}';
                switch (message.ext) {
                    case 'r':
                        response = JSON.stringify(test_data_1.rScriptInfo, null, 2);
                        break;
                    case 'py':
                        response = JSON.stringify(test_data_1.pyScriptInfo, null, 2);
                        break;
                }
                (_c = api.postBack) === null || _c === void 0 ? void 0 : _c.call(api, {
                    type: 'get_info_for_script',
                    id: message.id,
                    message: response
                });
                break;
            }
        }
    });
    return { state, api };
}
const baseState = () => {
    return {
        input: {
            default_names: ['RStudio Job'],
            show_clusters: 'non-interactive',
            show_workbenches: false,
            workbenches: ['RStudio'],
            launch_spec: null
        },
        uiOptions: {
            fieldset_class: '',
            form_group_class: 'form-group',
            input_addon_class: 'input-group-addon',
            input_hint_class: 'form-text text-muted',
            input_hint_error_class: 'form-text text-muted input-error',
            job_name_label: 'Job Name',
            job_name_placeholder: 'New Job',
            label_class: 'col-sm-3 control-label',
            link_label_class: '',
            select_class: 'form-control',
            show_legend: true,
            table_layout: true,
            textbox_class: 'form-control',
            textbox_error_class: 'form-control input-error'
        },
        launchSpec: {
            name: 'RStudio Job',
            cluster: 'Local',
            placement_constraints: [],
            resource_limits: [],
            queues: [],
            container_image: '',
            default_image: '',
            container_images: ['']
        },
        jobOptions: { script: '', workDir: '', args: [] },
        userModified: {},
        outputArgs: []
    };
};
test('smoke test (should not time out)', () => __awaiter(void 0, void 0, void 0, function* () {
    const { api } = createDefaultApi();
    const dom = yield newDom(api);
    yield new Promise(resolve => {
        dom.window.addEventListener('message', message => {
            if (message.data.type !== 'get_info') {
                return;
            }
            resolve();
        });
    });
}));
describe('initialization', () => {
    test('base state', () => __awaiter(void 0, void 0, void 0, function* () {
        const { state, api } = createDefaultApi();
        const { window } = yield newDom(api);
        const { document } = window;
        yield new Promise(resolve => {
            window.addEventListener('message', message => {
                if (message.data.type !== 'get_info') {
                    return;
                }
                resolve();
            });
        });
        expect(state).toEqual(baseState());
        const jobNameEl = document.body.querySelector('#rstudio_label_job_name');
        expect(jobNameEl.value).toBe('RStudio Job');
    }));
    test('VALID active R file', () => __awaiter(void 0, void 0, void 0, function* () {
        const fileName = 'script_file';
        const validPath = `/path/to/${fileName}.R`;
        const { state, api } = createDefaultApi();
        const { window } = yield newDom(api, validPath);
        const { document } = window;
        yield Promise.all([
            new Promise(resolve => {
                window.addEventListener('message', message => {
                    if (message.data.type !== 'get_info') {
                        return;
                    }
                    resolve();
                });
            }),
            new Promise(resolve => {
                window.addEventListener('message', message => {
                    if (message.data.type !== 'get_info_for_script') {
                        return;
                    }
                    resolve();
                });
            })
        ]);
        const expectedArgs = [
            {
                name: "r-version",
                value: "(System Default)",
            },
            {
                name: undefined,
                value: "-s"
            },
            {
                name: undefined,
                value: "--no-save"
            },
            {
                name: undefined,
                value: "--no-restore"
            }
        ];
        // if initializing with a file, the state should be 
        // immediately updated with the new information
        const base = baseState();
        expect(state).toEqual(Object.assign(Object.assign({}, base), { jobOptions: Object.assign(Object.assign({}, base.jobOptions), { args: expectedArgs, script: validPath, workDir: test_data_1.pyScriptInfo.workingDirectory }), launchSpec: Object.assign(Object.assign({}, base.launchSpec), { name: fileName }), outputArgs: [
                [expectedArgs[0]], expectedArgs.slice(1)
            ] }));
        // verify the UI elements display what they should
        const jobNameEl = document.body.querySelector('#rstudio_label_job_name');
        expect(jobNameEl.value).toBe('script_file');
        const scriptNameEl = document.body.querySelector('#launcher-script-path');
        expect(scriptNameEl.textContent).toBe(validPath);
        const versionSelectorEl = document.body.querySelector('#r-version-selector');
        expect(versionSelectorEl.value).toBe('(System Default)');
    }));
    test('VALID active PY file', () => __awaiter(void 0, void 0, void 0, function* () {
        const fileName = 'script_file';
        const validPath = `/path/to/${fileName}.py`;
        const { state, api } = createDefaultApi();
        const { window } = yield newDom(api, validPath);
        const { document } = window;
        yield Promise.all([
            new Promise(resolve => {
                window.addEventListener('message', message => {
                    if (message.data.type !== 'get_info') {
                        return;
                    }
                    resolve();
                });
            }),
            new Promise(resolve => {
                window.addEventListener('message', message => {
                    if (message.data.type !== 'get_info_for_script') {
                        return;
                    }
                    resolve();
                });
            })
        ]);
        const expectedArgs = [
            {
                name: "py-version",
                value: "(System Default)",
            },
            {
                name: undefined,
                value: "-u"
            }
        ];
        // if initializing with a file, the state should be 
        // immediately updated with the new information
        const base = baseState();
        expect(state).toEqual(Object.assign(Object.assign({}, base), { jobOptions: Object.assign(Object.assign({}, base.jobOptions), { args: expectedArgs, script: validPath, workDir: test_data_1.pyScriptInfo.workingDirectory }), launchSpec: Object.assign(Object.assign({}, base.launchSpec), { name: fileName }), outputArgs: [
                [expectedArgs[0]], expectedArgs.slice(1)
            ] }));
        // verify the UI elements display what they should
        const jobNameEl = document.body.querySelector('#rstudio_label_job_name');
        expect(jobNameEl.value).toBe('script_file');
        const scriptNameEl = document.body.querySelector('#launcher-script-path');
        expect(scriptNameEl.textContent).toBe(validPath);
        const versionSelectorEl = document.body.querySelector('#py-version-selector');
        expect(versionSelectorEl.value).toBe('(System Default)');
    }));
    /*
    test('VALID active IPYNB file', async () => {
      
      const fileName = 'script_file';
      const validPath = `/path/to/${fileName}.ipynb`;
  
      const { state, api } = createDefaultApi();
      const { window } = await newDom(api, validPath);
      const { document } = window;
  
      await Promise.all([
        new Promise<void>(resolve => {
          window.addEventListener('message', message => {
            if(message.data.type !== 'get_info') { return; }
            resolve();
          });
        }),
        new Promise<void>(resolve => {
          window.addEventListener('message', message => {
            if(message.data.type !== 'get_info_for_script') { return; }
            resolve();
          });
        })
      ]);
  
      const expectedArgs: Arg[] = [];
  
      // if initializing with a file, the state should be
      // immediately updated with the new information
      const base = baseState();
      expect(state).toEqual({
        ...base,
        jobOptions: {
          ...base.jobOptions,
          args: expectedArgs,
          script: validPath
        },
        launchSpec: {
          ...base.launchSpec,
          name: fileName
        },
        outputArgs: [ [] ]
      });
  
      // verify the UI elements display what they should
      const jobNameEl = document.body.querySelector('#rstudio_label_job_name') as HTMLInputElement;
      expect(jobNameEl.value).toBe('script_file');
      
      const scriptNameEl = document.body.querySelector('#launcher-script-path') as HTMLDivElement;
      expect(scriptNameEl.textContent).toBe(validPath);
      
      // there should be no version selector on screen for ipynb
      expect(
        document.body.querySelector('select[id$="-version-selector"]') as HTMLSelectElement
      ).toBe(null);
      
      // there should be arguments controls
      expect(
        document.body.querySelector('div.argument-summary') as HTMLDivElement
      ).not.toBe(null);
    });
  
    test('VALID active QMD file', async () => {
      
      const fileName = 'script_file';
      const validPath = `/path/to/${fileName}.Qmd`;
  
      const { state, api } = createDefaultApi();
      const { window } = await newDom(api, validPath);
      const { document } = window;
  
      await Promise.all([
        new Promise<void>(resolve => {
          window.addEventListener('message', message => {
            if(message.data.type !== 'get_info') { return; }
            resolve();
          });
        }),
        new Promise<void>(resolve => {
          window.addEventListener('message', message => {
            if(message.data.type !== 'get_info_for_script') { return; }
            resolve();
          });
        })
      ]);
  
      const expectedArgs: Arg[] = [];
  
      // if initializing with a file, the state should be
      // immediately updated with the new information
      const base = baseState();
      expect(state).toEqual({
        ...base,
        jobOptions: {
          ...base.jobOptions,
          args: expectedArgs,
          script: validPath
        },
        launchSpec: {
          ...base.launchSpec,
          name: fileName
        },
        outputArgs: [ [] ]
      });
  
      // verify the UI elements display what they should
      const jobNameEl = document.body.querySelector('#rstudio_label_job_name') as HTMLInputElement;
      expect(jobNameEl.value).toBe('script_file');
      
      const scriptNameEl = document.body.querySelector('#launcher-script-path') as HTMLDivElement;
      expect(scriptNameEl.textContent).toBe(validPath);
      
      // there should be no version selector on screen for Qmd
      expect(
        document.body.querySelector('select[id$="-version-selector"]') as HTMLSelectElement
      ).toBe(null);
      
      // there should be arguments controls
      expect(
        document.body.querySelector('div.argument-summary') as HTMLDivElement
      ).not.toBe(null);
    });
  
    test('VALID active RMD file', async () => {
      
      const fileName = 'script_file';
      const validPath = `/path/to/${fileName}.rmd`;
  
      const { state, api } = createDefaultApi();
      const { window } = await newDom(api, validPath);
      const { document } = window;
  
      await Promise.all([
        new Promise<void>(resolve => {
          window.addEventListener('message', message => {
            if(message.data.type !== 'get_info') { return; }
            resolve();
          });
        }),
        new Promise<void>(resolve => {
          window.addEventListener('message', message => {
            if(message.data.type !== 'get_info_for_script') { return; }
            resolve();
          });
        })
      ]);
  
      const expectedArgs: Arg[] = [];
  
      // if initializing with a file, the state should be
      // immediately updated with the new information
      const base = baseState();
      expect(state).toEqual({
        ...base,
        jobOptions: {
          ...base.jobOptions,
          args: expectedArgs,
          script: validPath
        },
        launchSpec: {
          ...base.launchSpec,
          name: fileName
        },
        outputArgs: [ [] ]
      });
  
      // verify the UI elements display what they should
      const jobNameEl = document.body.querySelector('#rstudio_label_job_name') as HTMLInputElement;
      expect(jobNameEl.value).toBe('script_file');
      
      const scriptNameEl = document.body.querySelector('#launcher-script-path') as HTMLDivElement;
      expect(scriptNameEl.textContent).toBe(validPath);
      
      // there should be no version selector on screen for rmd
      expect(
        document.body.querySelector('select[id$="-version-selector"]') as HTMLSelectElement
      ).toBe(null);
      
      // there should be arguments controls
      expect(
        document.body.querySelector('div.argument-summary') as HTMLDivElement
      ).not.toBe(null);
    });
    */
    test('INVALID active file', () => __awaiter(void 0, void 0, void 0, function* () {
        const fileName = 'one_of_those_vs_code_output_streams';
        const validPath = fileName;
        const { state, api } = createDefaultApi();
        const { window } = yield newDom(api, validPath);
        const { document } = window;
        // we don't await get_info_for_script here because it shouldn't
        // be called if the active-file has an invalid extension
        yield new Promise(resolve => {
            window.addEventListener('message', message => {
                if (message.data.type !== 'get_info') {
                    return;
                }
                resolve();
            });
        });
        // invalid script file on init should be a no-op over the base state
        expect(state).toEqual(baseState());
        // verify the UI elements display what they should
        const jobNameEl = document.body.querySelector('#rstudio_label_job_name');
        expect(jobNameEl.value).toBe('RStudio Job');
        const scriptNameEl = document.body.querySelector('#launcher-script-path');
        expect(scriptNameEl.textContent).toBe('');
    }));
    test('modified base state', () => __awaiter(void 0, void 0, void 0, function* () {
        const scriptName = 'complex_job';
        const validPath = `/path/to/${scriptName}.R`;
        const { state, api } = createDefaultApi();
        state.jobOptions = {
            script: validPath,
        };
        state.outputArgs = [
            [{ name: 'r-version', value: '3.6.3' }],
            [{ value: '--no-save' }, { value: '--made-up-arg' }]
        ];
        state.launchSpec = {
            name: "Custom Job",
            cluster: "Kubernetes",
            placement_constraints: [],
            resource_limits: [
                { type: "cpuCount", value: "2" },
                { type: "memory", value: "600" }
            ],
            queues: [],
            container_image: "workbench/default-image",
            default_image: '',
            container_images: ['']
        };
        const { window } = yield newDom(api);
        const { document } = window;
        yield Promise.all([
            new Promise(resolve => {
                window.addEventListener('message', message => {
                    if (message.data.type !== 'get_info') {
                        return;
                    }
                    resolve();
                });
            }),
            new Promise(resolve => {
                window.addEventListener('message', message => {
                    if (message.data.type !== 'get_info_for_script') {
                        return;
                    }
                    resolve();
                });
            })
        ]);
        // verify the UI elements display what they should
        const jobNameEl = document.body.querySelector('#rstudio_label_job_name');
        expect(jobNameEl.value).toBe("Custom Job");
        // we are using kubernetes test get_info data, so verify the contents
        // of the displayed cluster option elements
        const clusterOptionsCpus = document.body.querySelector('#rstudio_label_cpus');
        expect(clusterOptionsCpus.value).toBe('2');
        const clusterOptionsMemory = document.body.querySelector('#rstudio_label_memory');
        expect(clusterOptionsMemory.value).toBe('0.59');
        const scriptNameEl = document.body.querySelector('#launcher-script-path');
        expect(scriptNameEl.textContent).toBe(validPath);
        const argSummaryEl = document.body.querySelector('div.argument-summary');
        expect(argSummaryEl.textContent).toBe('--no-save --made-up-arg');
        const argumentEls = document.body.querySelectorAll('input.form-control.argument');
        expect(argumentEls.length).toBe(2);
    }));
});
test('test that state saves with user updates', () => __awaiter(void 0, void 0, void 0, function* () {
    var _a, _b, _c, _d, _e, _f, _g, _h, _j;
    const fileName = 'script_file';
    const validPath = `/path/to/${fileName}.R`;
    const { state, api } = createDefaultApi();
    const { window } = yield newDom(api, validPath);
    const { document } = window;
    yield Promise.all([
        new Promise(resolve => {
            window.addEventListener('message', message => {
                if (message.data.type !== 'get_info') {
                    return;
                }
                resolve();
            });
        }),
        new Promise(resolve => {
            window.addEventListener('message', message => {
                if (message.data.type !== 'get_info_for_script') {
                    return;
                }
                resolve();
            });
        })
    ]);
    // change the default name -- this is so we can test that usermodified 
    // will correctly mask changes to this when the script is changed
    const jobNameEl = document.body.querySelector('#rstudio_label_job_name');
    jobNameEl.value = 'Custom Job';
    jobNameEl.dispatchEvent(new window.Event('change'));
    expect((_a = state.launchSpec) === null || _a === void 0 ? void 0 : _a.name).toBe('Custom Job');
    const clusterEl = document.body.querySelector('#rstudio_label_cluster');
    clusterEl.value = 'Kubernetes';
    clusterEl.dispatchEvent(new window.Event('change'));
    expect((_b = state.launchSpec) === null || _b === void 0 ? void 0 : _b.cluster).toBe('Kubernetes');
    // the following fields use 'blur' as a trigger for updates
    const clusterOptionsCpus = document.body.querySelector('#rstudio_label_cpus');
    clusterOptionsCpus.value = '2';
    clusterOptionsCpus.dispatchEvent(new window.Event('blur'));
    expect((_c = state.launchSpec) === null || _c === void 0 ? void 0 : _c.resource_limits[0].value).toBe('2');
    const clusterOptionsMemory = document.body.querySelector('#rstudio_label_memory');
    clusterOptionsMemory.value = '0.8'; // Under the 1024 MB limit from test data.
    clusterOptionsMemory.dispatchEvent(new window.Event('blur'));
    expect((_d = state.launchSpec) === null || _d === void 0 ? void 0 : _d.resource_limits[1].value).toBe('819.2');
    const scriptBrowseEl = document.body.querySelector('#launcher-browse-script-path');
    let onBrowse = new Promise(resolve => {
        window.addEventListener('message', message => {
            if (message.data.type !== 'browse_path') {
                return;
            }
            resolve();
        });
    });
    scriptBrowseEl.click();
    yield onBrowse;
    const scriptNameEl = document.body.querySelector('#launcher-script-path');
    expect(scriptNameEl.textContent).toBe(BROWSE_PATH_SCRIPT);
    expect((_e = state.jobOptions) === null || _e === void 0 ? void 0 : _e.script).toBe(BROWSE_PATH_SCRIPT);
    expect((_f = state.userModified) === null || _f === void 0 ? void 0 : _f.script).toBe(true);
    // this should not have changed since we already modified it
    expect((_g = state.launchSpec) === null || _g === void 0 ? void 0 : _g.name).toBe('Custom Job');
    const workDirBrowseEl = document.body.querySelector('#launcher-browse-workdir-path');
    onBrowse = new Promise(resolve => {
        window.addEventListener('message', message => {
            if (message.data.type !== 'browse_path') {
                return;
            }
            resolve();
        });
    });
    workDirBrowseEl.click();
    yield onBrowse;
    const workDirPathEl = document.body.querySelector('#launcher-workdir-path');
    expect(workDirPathEl.textContent).toBe(BROWSE_PATH_DIRECTORY);
    expect((_h = state.jobOptions) === null || _h === void 0 ? void 0 : _h.workDir).toBe(BROWSE_PATH_DIRECTORY);
    expect((_j = state.userModified) === null || _j === void 0 ? void 0 : _j.workDir).toBe(true);
}));
test('test that warning appears when non-local config is selected', () => __awaiter(void 0, void 0, void 0, function* () {
    var _a, _b, _c;
    const fileName = 'script_file';
    const validPath = `/path/to/${fileName}.R`;
    const { state, api } = createDefaultApi();
    const { window } = yield newDom(api, validPath);
    const { document } = window;
    yield Promise.all([
        new Promise(resolve => {
            window.addEventListener('message', message => {
                if (message.data.type !== 'get_info') {
                    return;
                }
                resolve();
            });
        }),
        new Promise(resolve => {
            window.addEventListener('message', message => {
                if (message.data.type !== 'get_info_for_script') {
                    return;
                }
                resolve();
            });
        })
    ]);
    // change the cluster to not "Local"
    const clusterEl = document.body.querySelector('#rstudio_label_cluster');
    clusterEl.value = 'Kubernetes';
    clusterEl.dispatchEvent(new window.Event('change'));
    expect((_a = state.launchSpec) === null || _a === void 0 ? void 0 : _a.cluster).toBe('Kubernetes');
    // verify that the warning is still not visible before changing the path
    const warningEl = document.body.querySelector('#version-path-warning');
    expect(warningEl.style.display).toBe('none');
    // change the version to a non-default value
    const versionSelectorEl = document.body.querySelector('#r-version-selector');
    versionSelectorEl.value = '4.2.1';
    versionSelectorEl.dispatchEvent(new window.Event('change'));
    // verify warning is visible and contains a mention of the cluster name and the version
    expect(warningEl.style.display).not.toBe('none');
    expect((_b = warningEl.textContent) === null || _b === void 0 ? void 0 : _b.includes('Kubernetes')).toBe(true);
    expect((_c = warningEl.textContent) === null || _c === void 0 ? void 0 : _c.includes('4.2.1')).toBe(true);
}));
describe('arguments', () => {
    test('modification hidden by default', () => __awaiter(void 0, void 0, void 0, function* () {
        const fileName = 'script_file';
        const validPath = `/path/to/${fileName}.R`;
        const { state, api } = createDefaultApi();
        const { window } = yield newDom(api, validPath);
        const { document } = window;
        state.jobOptions = {
            script: validPath
        };
        yield Promise.all([
            new Promise(resolve => {
                window.addEventListener('message', message => {
                    if (message.data.type !== 'get_info') {
                        return;
                    }
                    resolve();
                });
            }),
            new Promise(resolve => {
                window.addEventListener('message', message => {
                    if (message.data.type !== 'get_info_for_script') {
                        return;
                    }
                    resolve();
                });
            })
        ]);
        const formGridEl = document.body.querySelector('#form-grid');
        expect(Array.from(formGridEl.classList).includes('show-arguments')).toBe(false);
        const argumentEls = formGridEl.querySelectorAll('.argument');
        for (const argumentEl of argumentEls) {
            expect(window.getComputedStyle(argumentEl).display === 'none');
        }
    }));
    test('arguments show correctly', () => __awaiter(void 0, void 0, void 0, function* () {
        const fileName = 'script_file';
        const validPath = `/path/to/${fileName}.R`;
        const { state, api } = createDefaultApi();
        const { window } = yield newDom(api, validPath);
        const { document } = window;
        state.jobOptions = {
            script: validPath
        };
        yield Promise.all([
            new Promise(resolve => {
                window.addEventListener('message', message => {
                    if (message.data.type !== 'get_info') {
                        return;
                    }
                    resolve();
                });
            }),
            new Promise(resolve => {
                window.addEventListener('message', message => {
                    if (message.data.type !== 'get_info_for_script') {
                        return;
                    }
                    resolve();
                });
            })
        ]);
        const modifyArgumentsButtomEl = document.body.querySelector('div.argument-summary + button.browse-button');
        modifyArgumentsButtomEl.click();
        const formGridEl = document.body.querySelector('#form-grid');
        expect(Array.from(formGridEl.classList).includes('show-arguments')).toBe(true);
        const argumentEls = formGridEl.querySelectorAll('.argument');
        for (const argumentEl of argumentEls) {
            expect(window.getComputedStyle(argumentEl).display !== 'none');
        }
    }));
    test('modify multiple', () => __awaiter(void 0, void 0, void 0, function* () {
        var _a, _b;
        const fileName = 'script_file';
        const validPath = `/path/to/${fileName}.R`;
        const { state, api } = createDefaultApi();
        const { window } = yield newDom(api, validPath);
        const { document } = window;
        state.jobOptions = {
            script: validPath
        };
        yield Promise.all([
            new Promise(resolve => {
                window.addEventListener('message', message => {
                    if (message.data.type !== 'get_info') {
                        return;
                    }
                    resolve();
                });
            }),
            new Promise(resolve => {
                window.addEventListener('message', message => {
                    if (message.data.type !== 'get_info_for_script') {
                        return;
                    }
                    resolve();
                });
            })
        ]);
        const modifyArgumentsButtomEl = document.body.querySelector('div.argument-summary + button.browse-button');
        modifyArgumentsButtomEl.click();
        const argumentInputEls = Array.from(document.body.querySelectorAll('#form-grid input.form-control.argument'));
        expect(argumentInputEls.length).toBe(test_data_1.rScriptInfo.scriptArgs.length);
        for (let a = 0; a < argumentInputEls.length; a++) {
            const argumentInputEl = argumentInputEls[a];
            argumentInputEl.value = `--arg${a}`;
            argumentInputEl.dispatchEvent(new window.Event('change'));
        }
        const argSummaryEl = document.body.querySelector('div.argument-summary');
        expect(argSummaryEl.textContent).toBe(test_data_1.rScriptInfo.scriptArgs.map((_, index) => `--arg${index}`).join(' '));
        // outputArgs[1] is the array of arguments associated with the general-purpose list
        expect((_a = state.outputArgs) === null || _a === void 0 ? void 0 : _a[1]).toEqual(test_data_1.rScriptInfo.scriptArgs.map((_, index) => { return { name: undefined, value: `--arg${index}` }; }));
        // perform an arg reset and re-verify that all args are correct
        const resetButtonEl = document.body.querySelector('#form-grid button.reset-arguments');
        resetButtonEl.click();
        expect(argSummaryEl.textContent).toBe(test_data_1.rScriptInfo.scriptArgs.map(arg => arg.name).join(' '));
        // outputArgs[1] is the array of arguments associated with the general-purpose list
        expect((_b = state.outputArgs) === null || _b === void 0 ? void 0 : _b[1]).toEqual(test_data_1.rScriptInfo.scriptArgs.map(arg => { return { name: undefined, value: arg.name }; }));
    }));
    test('add new', () => __awaiter(void 0, void 0, void 0, function* () {
        var _a, _b;
        const fileName = 'script_file';
        const validPath = `/path/to/${fileName}.R`;
        const { state, api } = createDefaultApi();
        const { window } = yield newDom(api, validPath);
        const { document } = window;
        state.jobOptions = {
            script: validPath
        };
        yield Promise.all([
            new Promise(resolve => {
                window.addEventListener('message', message => {
                    if (message.data.type !== 'get_info') {
                        return;
                    }
                    resolve();
                });
            }),
            new Promise(resolve => {
                window.addEventListener('message', message => {
                    if (message.data.type !== 'get_info_for_script') {
                        return;
                    }
                    resolve();
                });
            })
        ]);
        const modifyArgumentsButtomEl = document.body.querySelector('div.argument-summary + button.browse-button');
        modifyArgumentsButtomEl.click();
        // verify lengths match before adding new
        expect(document.body.querySelectorAll('#form-grid input.form-control.argument').length).toBe(test_data_1.rScriptInfo.scriptArgs.length);
        const addArgumentButtonEl = document.body.querySelector('#form-grid button.add-argument');
        addArgumentButtonEl.click();
        // verify that there are now scriptArgs.length + 1 elements queried
        expect(document.body.querySelectorAll('#form-grid input.form-control.argument').length).toBe(test_data_1.rScriptInfo.scriptArgs.length + 1);
        // verify that the summary has not changed yet, adding a new blank argument does not affect summary
        const argSummaryEl = document.body.querySelector('div.argument-summary');
        expect(argSummaryEl.textContent).toBe(test_data_1.rScriptInfo.scriptArgs.map(arg => arg.name).join(' '));
        // modify the final element
        const newArg = { name: '--new-arg' };
        const lastArgumentEl = document.body.querySelector('#form-grid input.form-control.argument:last-of-type');
        lastArgumentEl.value = newArg.name;
        lastArgumentEl.dispatchEvent(new window.Event('change'));
        // verify summary and state
        expect(argSummaryEl.textContent).toBe([...test_data_1.rScriptInfo.scriptArgs, newArg].map(arg => arg.name).join(' '));
        // outputArgs[1] is the array of arguments associated with the general-purpose list
        expect((_a = state.outputArgs) === null || _a === void 0 ? void 0 : _a[1]).toEqual([...test_data_1.rScriptInfo.scriptArgs, newArg].map(arg => { return { name: undefined, value: arg.name }; }));
        // perform an arg reset and re-verify that all args are correct
        const resetButtonEl = document.body.querySelector('#form-grid button.reset-arguments');
        resetButtonEl.click();
        expect(argSummaryEl.textContent).toBe(test_data_1.rScriptInfo.scriptArgs.map(arg => arg.name).join(' '));
        // outputArgs[1] is the array of arguments associated with the general-purpose list
        expect((_b = state.outputArgs) === null || _b === void 0 ? void 0 : _b[1]).toEqual(test_data_1.rScriptInfo.scriptArgs.map(arg => { return { name: undefined, value: arg.name }; }));
    }));
    test('remove one', () => __awaiter(void 0, void 0, void 0, function* () {
        var _a, _b;
        const fileName = 'script_file';
        const validPath = `/path/to/${fileName}.R`;
        const { state, api } = createDefaultApi();
        const { window } = yield newDom(api, validPath);
        const { document } = window;
        state.jobOptions = {
            script: validPath
        };
        yield Promise.all([
            new Promise(resolve => {
                window.addEventListener('message', message => {
                    if (message.data.type !== 'get_info') {
                        return;
                    }
                    resolve();
                });
            }),
            new Promise(resolve => {
                window.addEventListener('message', message => {
                    if (message.data.type !== 'get_info_for_script') {
                        return;
                    }
                    resolve();
                });
            })
        ]);
        const modifyArgumentsButtomEl = document.body.querySelector('div.argument-summary + button.browse-button');
        modifyArgumentsButtomEl.click();
        // verify lengths match before adding new
        expect(document.body.querySelectorAll('#form-grid input.form-control.argument').length).toBe(test_data_1.rScriptInfo.scriptArgs.length);
        const deleteArgumentButtonEl = document.body.querySelector('#form-grid button.delete-argument');
        deleteArgumentButtonEl.click();
        // verify that there are now scriptArgs.length + 1 elements queried
        expect(document.body.querySelectorAll('#form-grid input.form-control.argument').length).toBe(test_data_1.rScriptInfo.scriptArgs.length - 1);
        // verify summary and state
        // use slice(1) because we've removed the first argument
        const argSummaryEl = document.body.querySelector('div.argument-summary');
        expect(argSummaryEl.textContent).toBe(test_data_1.rScriptInfo.scriptArgs.slice(1).map(arg => arg.name).join(' '));
        // outputArgs[1] is the array of arguments associated with the general-purpose list
        expect((_a = state.outputArgs) === null || _a === void 0 ? void 0 : _a[1]).toEqual(test_data_1.rScriptInfo.scriptArgs.slice(1).map(arg => { return { name: undefined, value: arg.name }; }));
        // perform an arg reset and re-verify that all args are correct
        const resetButtonEl = document.body.querySelector('#form-grid button.reset-arguments');
        resetButtonEl.click();
        expect(argSummaryEl.textContent).toBe(test_data_1.rScriptInfo.scriptArgs.map(arg => arg.name).join(' '));
        // outputArgs[1] is the array of arguments associated with the general-purpose list
        expect((_b = state.outputArgs) === null || _b === void 0 ? void 0 : _b[1]).toEqual(test_data_1.rScriptInfo.scriptArgs.map(arg => { return { name: undefined, value: arg.name }; }));
    }));
    test('remove all', () => __awaiter(void 0, void 0, void 0, function* () {
        var _a, _b;
        const fileName = 'script_file';
        const validPath = `/path/to/${fileName}.R`;
        const { state, api } = createDefaultApi();
        const { window } = yield newDom(api, validPath);
        const { document } = window;
        state.jobOptions = {
            script: validPath
        };
        yield Promise.all([
            new Promise(resolve => {
                window.addEventListener('message', message => {
                    if (message.data.type !== 'get_info') {
                        return;
                    }
                    resolve();
                });
            }),
            new Promise(resolve => {
                window.addEventListener('message', message => {
                    if (message.data.type !== 'get_info_for_script') {
                        return;
                    }
                    resolve();
                });
            })
        ]);
        const modifyArgumentsButtomEl = document.body.querySelector('div.argument-summary + button.browse-button');
        modifyArgumentsButtomEl.click();
        // verify lengths match before adding new
        expect(document.body.querySelectorAll('#form-grid input.form-control.argument').length).toBe(test_data_1.rScriptInfo.scriptArgs.length);
        // delete all arguments
        const deleteArgumentButtonEls = Array.from(document.body.querySelectorAll('#form-grid button.delete-argument'));
        for (const deleteArgumentButtonEl of deleteArgumentButtonEls) {
            deleteArgumentButtonEl.click();
        }
        // verify that there are now ZERO  elements queried
        expect(document.body.querySelectorAll('#form-grid input.form-control.argument').length).toBe(0);
        // verify summary and state
        const argSummaryEl = document.body.querySelector('div.argument-summary');
        expect(argSummaryEl.textContent).toBe('');
        // outputArgs[1] is the array of arguments associated with the general-purpose list
        expect((_a = state.outputArgs) === null || _a === void 0 ? void 0 : _a[1]).toEqual([]);
        // perform an arg reset and re-verify that all args are correct
        const resetButtonEl = document.body.querySelector('#form-grid button.reset-arguments');
        resetButtonEl.click();
        expect(argSummaryEl.textContent).toBe(test_data_1.rScriptInfo.scriptArgs.map(arg => arg.name).join(' '));
        // outputArgs[1] is the array of arguments associated with the general-purpose list
        expect((_b = state.outputArgs) === null || _b === void 0 ? void 0 : _b[1]).toEqual(test_data_1.rScriptInfo.scriptArgs.map(arg => { return { name: undefined, value: arg.name }; }));
    }));
});
describe('managed credentials', () => {
    test('checkbox section hidden by default', () => __awaiter(void 0, void 0, void 0, function* () {
        const { api } = createDefaultApi();
        const { window } = yield newDom(api);
        const { document } = window;
        yield new Promise(resolve => {
            window.addEventListener('message', message => {
                if (message.data.type !== 'get_info') {
                    return;
                }
                resolve();
            });
        });
        // Verify that the managed credentials section is hidden by default
        const managedCredsSection = document.body.querySelector('#managed-credentials');
        expect(managedCredsSection).not.toBe(null);
        expect(managedCredsSection.classList.contains('hidden')).toBe(true);
        // Verify the checkbox exists but is not visible
        const managedCredsCheckbox = document.body.querySelector('#managed-credentials-enabled');
        expect(managedCredsCheckbox).not.toBe(null);
        expect(managedCredsCheckbox.type).toBe('checkbox');
    }));
    test('checkbox shows when enabled via message', () => __awaiter(void 0, void 0, void 0, function* () {
        const { api } = createDefaultApi();
        const { window } = yield newDom(api);
        const { document } = window;
        yield new Promise(resolve => {
            window.addEventListener('message', message => {
                if (message.data.type !== 'get_info') {
                    return;
                }
                resolve();
            });
        });
        // Send managed_credentials enabled message
        window.postMessage({
            type: 'managed_credentials',
            enabled: true
        }, '*');
        // Wait a bit for the message to be processed
        yield new Promise(resolve => setTimeout(resolve, 10));
        // Verify that the managed credentials section is now visible
        const managedCredsSection = document.body.querySelector('#managed-credentials');
        expect(managedCredsSection.classList.contains('hidden')).toBe(false);
        // Verify the checkbox is checked by default when enabled
        const managedCredsCheckbox = document.body.querySelector('#managed-credentials-enabled');
        expect(managedCredsCheckbox.checked).toBe(true);
    }));
    test('checkbox hides when disabled via message', () => __awaiter(void 0, void 0, void 0, function* () {
        const { api } = createDefaultApi();
        const { window } = yield newDom(api);
        const { document } = window;
        yield new Promise(resolve => {
            window.addEventListener('message', message => {
                if (message.data.type !== 'get_info') {
                    return;
                }
                resolve();
            });
        });
        // First enable the checkbox
        window.postMessage({
            type: 'managed_credentials',
            enabled: true
        }, '*');
        yield new Promise(resolve => setTimeout(resolve, 10));
        // Verify it's visible
        const managedCredsSection = document.body.querySelector('#managed-credentials');
        expect(managedCredsSection.classList.contains('hidden')).toBe(false);
        // Now disable it
        window.postMessage({
            type: 'managed_credentials',
            enabled: false
        }, '*');
        yield new Promise(resolve => setTimeout(resolve, 10));
        // Verify it's hidden again
        expect(managedCredsSection.classList.contains('hidden')).toBe(true);
    }));
    test('checkbox user interaction updates state', () => __awaiter(void 0, void 0, void 0, function* () {
        var _a, _b, _c;
        const { api, state } = createDefaultApi();
        const { window } = yield newDom(api);
        const { document } = window;
        yield new Promise(resolve => {
            window.addEventListener('message', message => {
                if (message.data.type !== 'get_info') {
                    return;
                }
                resolve();
            });
        });
        // Enable managed credentials
        window.postMessage({
            type: 'managed_credentials',
            enabled: true
        }, '*');
        yield new Promise(resolve => setTimeout(resolve, 10));
        const managedCredsCheckbox = document.body.querySelector('#managed-credentials-enabled');
        // Initially should be checked
        expect(managedCredsCheckbox.checked).toBe(true);
        expect((_a = state.jobOptions) === null || _a === void 0 ? void 0 : _a.managedCredentialsEnabled).toBe(true);
        // Uncheck the checkbox
        managedCredsCheckbox.checked = false;
        managedCredsCheckbox.dispatchEvent(new window.Event('click'));
        // State should be updated
        expect((_b = state.jobOptions) === null || _b === void 0 ? void 0 : _b.managedCredentialsEnabled).toBe(false);
        // Check it again
        managedCredsCheckbox.checked = true;
        managedCredsCheckbox.dispatchEvent(new window.Event('click'));
        // State should be updated again
        expect((_c = state.jobOptions) === null || _c === void 0 ? void 0 : _c.managedCredentialsEnabled).toBe(true);
    }));
    test('user modification prevents automatic updates', () => __awaiter(void 0, void 0, void 0, function* () {
        const { api } = createDefaultApi();
        const { window } = yield newDom(api);
        const { document } = window;
        yield new Promise(resolve => {
            window.addEventListener('message', message => {
                if (message.data.type !== 'get_info') {
                    return;
                }
                resolve();
            });
        });
        // Enable managed credentials
        window.postMessage({
            type: 'managed_credentials',
            enabled: true
        }, '*');
        yield new Promise(resolve => setTimeout(resolve, 10));
        const managedCredsCheckbox = document.body.querySelector('#managed-credentials-enabled');
        // Initially should be checked
        expect(managedCredsCheckbox.checked).toBe(true);
        // User unchecks the checkbox
        managedCredsCheckbox.checked = false;
        managedCredsCheckbox.dispatchEvent(new window.Event('click'));
        // Now send another enable message
        window.postMessage({
            type: 'managed_credentials',
            enabled: true
        }, '*');
        yield new Promise(resolve => setTimeout(resolve, 10));
        // The checkbox should remain unchecked because user has modified it
        expect(managedCredsCheckbox.checked).toBe(false);
        // But the section should still be visible
        const managedCredsSection = document.body.querySelector('#managed-credentials');
        expect(managedCredsSection.classList.contains('hidden')).toBe(false);
    }));
    test('checkbox state persists in validation with enabled credentials', () => __awaiter(void 0, void 0, void 0, function* () {
        var _a, _b;
        const fileName = 'script_file';
        const validPath = `/path/to/${fileName}.R`;
        const { api, state } = createDefaultApi();
        const { window } = yield newDom(api, validPath);
        const { document } = window;
        yield Promise.all([
            new Promise(resolve => {
                window.addEventListener('message', message => {
                    if (message.data.type !== 'get_info') {
                        return;
                    }
                    resolve();
                });
            }),
            new Promise(resolve => {
                window.addEventListener('message', message => {
                    if (message.data.type !== 'get_info_for_script') {
                        return;
                    }
                    resolve();
                });
            })
        ]);
        // Enable managed credentials
        window.postMessage({
            type: 'managed_credentials',
            enabled: true
        }, '*');
        yield new Promise(resolve => setTimeout(resolve, 10));
        // Get initial state
        const managedCredsCheckbox = document.body.querySelector('#managed-credentials-enabled');
        expect(managedCredsCheckbox.checked).toBe(true);
        expect((_a = state.jobOptions) === null || _a === void 0 ? void 0 : _a.managedCredentialsEnabled).toBe(true);
        // Change job name to trigger validation
        const jobNameEl = document.body.querySelector('#rstudio_label_job_name');
        jobNameEl.value = 'Test Job';
        jobNameEl.dispatchEvent(new window.Event('change'));
        // Managed credentials should still be enabled
        expect((_b = state.jobOptions) === null || _b === void 0 ? void 0 : _b.managedCredentialsEnabled).toBe(true);
        // Start button should be enabled (valid config with managed credentials)
        const startButtonEl = document.body.querySelector('#submit');
        expect(startButtonEl === null || startButtonEl === void 0 ? void 0 : startButtonEl.classList.contains('disabled')).toBe(false);
    }));
    test('checkbox checked state reflects in job launch', () => __awaiter(void 0, void 0, void 0, function* () {
        const fileName = 'script_file';
        const validPath = `/path/to/${fileName}.R`;
        let capturedLaunchMessage = null;
        const { api } = createDefaultApi();
        // Override the postMessage to capture the launch message
        const originalPostMessage = api.postMessage;
        api.postMessage = (message) => {
            if (message.event === 'launch') {
                capturedLaunchMessage = message;
            }
            return originalPostMessage.call(api, message);
        };
        const { window } = yield newDom(api, validPath);
        const { document } = window;
        yield Promise.all([
            new Promise(resolve => {
                window.addEventListener('message', message => {
                    if (message.data.type !== 'get_info') {
                        return;
                    }
                    resolve();
                });
            }),
            new Promise(resolve => {
                window.addEventListener('message', message => {
                    if (message.data.type !== 'get_info_for_script') {
                        return;
                    }
                    resolve();
                });
            })
        ]);
        // Enable managed credentials and check the checkbox
        window.postMessage({
            type: 'managed_credentials',
            enabled: true
        }, '*');
        yield new Promise(resolve => setTimeout(resolve, 10));
        const managedCredsCheckbox = document.body.querySelector('#managed-credentials-enabled');
        expect(managedCredsCheckbox.checked).toBe(true);
        // Uncheck the checkbox manually
        managedCredsCheckbox.checked = false;
        managedCredsCheckbox.dispatchEvent(new window.Event('click'));
        // Click the start button
        const startButtonEl = document.body.querySelector('#submit');
        startButtonEl.click();
        // Verify that managedCredentialsEnabled is false in the job options
        expect(capturedLaunchMessage).not.toBe(null);
        expect(capturedLaunchMessage.jobOptions.managedCredentialsEnabled).toBe(false);
    }));
    test('setting change from enabled to disabled hides section', () => __awaiter(void 0, void 0, void 0, function* () {
        const { api } = createDefaultApi();
        const { window } = yield newDom(api);
        const { document } = window;
        yield new Promise(resolve => {
            window.addEventListener('message', message => {
                if (message.data.type !== 'get_info') {
                    return;
                }
                resolve();
            });
        });
        // First enable managed credentials
        window.postMessage({
            type: 'managed_credentials',
            enabled: true
        }, '*');
        yield new Promise(resolve => setTimeout(resolve, 10));
        // Verify it's visible and checked
        const managedCredsSection = document.body.querySelector('#managed-credentials');
        const managedCredsCheckbox = document.body.querySelector('#managed-credentials-enabled');
        expect(managedCredsSection.classList.contains('hidden')).toBe(false);
        expect(managedCredsCheckbox.checked).toBe(true);
        // Now disable via setting change
        window.postMessage({
            type: 'managed_credentials',
            enabled: false
        }, '*');
        yield new Promise(resolve => setTimeout(resolve, 10));
        // Verify it's hidden (this is what should happen)
        expect(managedCredsSection.classList.contains('hidden')).toBe(true);
        // The checkbox state becomes irrelevant when section is hidden
    }));
    test('setting change after user modification preserves user choice when re-enabled', () => __awaiter(void 0, void 0, void 0, function* () {
        const { api } = createDefaultApi();
        const { window } = yield newDom(api);
        const { document } = window;
        yield new Promise(resolve => {
            window.addEventListener('message', message => {
                if (message.data.type !== 'get_info') {
                    return;
                }
                resolve();
            });
        });
        // Enable managed credentials
        window.postMessage({
            type: 'managed_credentials',
            enabled: true
        }, '*');
        yield new Promise(resolve => setTimeout(resolve, 10));
        const managedCredsSection = document.body.querySelector('#managed-credentials');
        const managedCredsCheckbox = document.body.querySelector('#managed-credentials-enabled');
        // User unchecks the checkbox
        managedCredsCheckbox.checked = false;
        managedCredsCheckbox.dispatchEvent(new window.Event('click'));
        // Disable the setting
        window.postMessage({
            type: 'managed_credentials',
            enabled: false
        }, '*');
        yield new Promise(resolve => setTimeout(resolve, 10));
        expect(managedCredsSection.classList.contains('hidden')).toBe(true);
        // Re-enable the setting
        window.postMessage({
            type: 'managed_credentials',
            enabled: true
        }, '*');
        yield new Promise(resolve => setTimeout(resolve, 10));
        // Section should be visible again, but checkbox should remain unchecked (user's choice)
        expect(managedCredsSection.classList.contains('hidden')).toBe(false);
        expect(managedCredsCheckbox.checked).toBe(false);
    }));
});
describe('valid config', () => {
    // validateState logic:
    //   valid when state.launchSpec?.name is defined and state.jobOptions?.script is defined
    test('start button enabled on valid config', () => __awaiter(void 0, void 0, void 0, function* () {
        const { api } = createDefaultApi();
        const { window } = yield newDom(api);
        const { document } = window;
        yield new Promise(resolve => {
            window.addEventListener('message', message => {
                if (message.data.type !== 'get_info') {
                    return;
                }
                resolve();
            });
        });
        // verify that the start button is diabled 
        const startButtonEl = document.body.querySelector('#submit');
        expect(startButtonEl === null || startButtonEl === void 0 ? void 0 : startButtonEl.classList.contains('disabled')).toBe(true);
        // set the job name and script
        const jobNameEl = document.body.querySelector('#rstudio_label_job_name');
        jobNameEl.value = 'Custom Job';
        jobNameEl.dispatchEvent(new window.Event('change'));
        const scriptBrowseEl = document.body.querySelector('#launcher-browse-script-path');
        const onBrowse = new Promise(resolve => {
            window.addEventListener('message', message => {
                if (message.data.type !== 'browse_path') {
                    return;
                }
                resolve();
            });
        });
        scriptBrowseEl.click();
        yield onBrowse;
        yield new Promise(resolve => {
            window.addEventListener('message', message => {
                if (message.data.type !== 'get_info_for_script') {
                    return;
                }
                resolve();
            });
        });
        // it should now be possible to launch the job
        expect(startButtonEl === null || startButtonEl === void 0 ? void 0 : startButtonEl.classList.contains('disabled')).toBe(false);
    }));
    test('start button disabled on invalid config', () => __awaiter(void 0, void 0, void 0, function* () {
        const fileName = 'script_file';
        const validPath = `/path/to/${fileName}.R`;
        const { api, state } = createDefaultApi();
        const { window } = yield newDom(api, validPath);
        const { document } = window;
        state.launchSpec = {
            name: "Custom Job",
            cluster: "Local",
            placement_constraints: [],
            resource_limits: [],
            queues: [],
            container_image: '',
            default_image: '',
            container_images: ['']
        };
        yield Promise.all([
            new Promise(resolve => {
                window.addEventListener('message', message => {
                    if (message.data.type !== 'get_info') {
                        return;
                    }
                    resolve();
                });
            }),
            new Promise(resolve => {
                window.addEventListener('message', message => {
                    if (message.data.type !== 'get_info_for_script') {
                        return;
                    }
                    resolve();
                });
            })
        ]);
        // we're starting with a valid config
        // verify that the start button is enabled
        const startButtonEl = document.body.querySelector('#submit');
        expect(startButtonEl === null || startButtonEl === void 0 ? void 0 : startButtonEl.classList.contains('disabled')).toBe(false);
        // clear the job name
        const jobNameEl = document.body.querySelector('#rstudio_label_job_name');
        jobNameEl.value = '';
        jobNameEl.dispatchEvent(new window.Event('change'));
        // the state should now be invalid because the job has been cleared
        expect(startButtonEl === null || startButtonEl === void 0 ? void 0 : startButtonEl.classList.contains('disabled')).toBe(true);
    }));
    test('validate state funciton on invalid state', () => {
        expect((0, __1.validateState)(baseState())).toBe(false);
    });
    test('validate state funciton on valid state', () => {
        const validState = baseState();
        validState.jobOptions.script = BROWSE_PATH_SCRIPT;
        validState.launchSpec.name = 'a non empty value';
        expect((0, __1.validateState)(validState)).toBe(true);
    });
});
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaW5kZXgudGVzdC5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uLy4uLy4uL3NyYy90ZXN0cy9pbmRleC50ZXN0LnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7Ozs7Ozs7Ozs7Ozs7O0FBQUEsaUNBQThCO0FBQzlCLDRDQUFvQjtBQUNwQixnREFBd0I7QUFHeEIsNkNBQWdEO0FBQ2hELDJDQUFzRTtBQUN0RSwwQkFBbUM7QUFFbkMscUJBQXFCO0FBQ3JCLElBQUksUUFBUSxHQUFHLFlBQUUsQ0FBQyxZQUFZLENBQUMsY0FBSSxDQUFDLE9BQU8sQ0FBQyxjQUFjLENBQUMsRUFBRSxNQUFNLENBQUMsQ0FBQztBQUVyRSw4QkFBOEI7QUFDOUIsUUFBUSxHQUFHLFFBQVEsQ0FBQyxPQUFPLENBQUMsZUFBZSxFQUFFLGNBQUksQ0FBQyxPQUFPLENBQUMsc0JBQXNCLENBQUMsQ0FBQyxDQUFDO0FBQ25GLFFBQVEsR0FBRyxRQUFRLENBQUMsT0FBTyxDQUFDLFVBQVUsRUFBRSxjQUFJLENBQUMsT0FBTyxDQUFDLGVBQWUsQ0FBQyxDQUFDLENBQUM7QUFFdkUsd0JBQXdCO0FBQ3hCLE1BQU0sWUFBWSxHQUFHLElBQUksYUFBSyxDQUFDLFFBQVEsQ0FBQyxDQUFDO0FBRXpDLHNGQUFzRjtBQUN0RixLQUFLLE1BQU0sS0FBSyxJQUFJLFlBQVksQ0FBQyxNQUFNLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxRQUFRLEVBQUUsQ0FBQztJQUMvRCxJQUFJLEtBQUssQ0FBQyxTQUFTLENBQUMsV0FBVyxFQUFFLEtBQUssTUFBTSxJQUFJLEtBQUssQ0FBQyxZQUFZLENBQUMsTUFBTSxDQUFDLEtBQUssYUFBYSxFQUFFLENBQUM7UUFBQyxTQUFTO0lBQUMsQ0FBQztJQUMzRyxLQUFLLENBQUMsTUFBTSxFQUFFLENBQUM7QUFDakIsQ0FBQztBQUVELG1FQUFtRTtBQUNuRSxtREFBbUQ7QUFDbkQsUUFBUSxHQUFHLFlBQVksQ0FBQyxTQUFTLEVBQUUsQ0FBQztBQUlwQyxTQUFlLE1BQU07eURBQUMsR0FBd0IsRUFBRSxVQUFVLEdBQUcsRUFBRTtRQUU3RCxNQUFNLFdBQVcsR0FBRyxRQUFRLENBQUMsT0FBTyxDQUFDLGVBQWUsRUFBRSxVQUFVLENBQUMsQ0FBQztRQUVsRSxNQUFNLFlBQVksR0FBRyxJQUFJLGFBQUssQ0FBQyxXQUFXLEVBQUU7WUFDMUMsR0FBRyxFQUFFLFVBQVUsY0FBSSxDQUFDLE9BQU8sQ0FBQyxLQUFLLENBQUMsRUFBRTtZQUNwQyxVQUFVLEVBQUUsYUFBYTtZQUN6QixTQUFTLEVBQUUsUUFBUTtTQUNwQixDQUFDLENBQUM7UUFFSCxNQUFNLENBQUMsTUFBTSxDQUFDLFlBQVksQ0FBQyxNQUFNLEVBQUU7WUFDakMsZ0JBQWdCLEVBQUUsR0FBRyxFQUFFO2dCQUNyQixJQUFJLEdBQUcsRUFBRSxDQUFDO29CQUNSLEdBQUcsQ0FBQyxRQUFRLEdBQUcsT0FBTyxDQUFDLEVBQUUsR0FBRyxZQUFZLENBQUMsTUFBTSxDQUFDLFdBQVcsQ0FBQyxPQUFPLEVBQUUsR0FBRyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUM7Z0JBQy9FLENBQUM7Z0JBQ0QsT0FBTyxHQUFHLElBQUksSUFBQSw2QkFBZ0IsR0FBYSxDQUFDO1lBQzlDLENBQUM7U0FDRixDQUFDLENBQUM7UUFFSCxNQUFNLENBQUMsTUFBTSxDQUFDLFlBQVksQ0FBQyxNQUFNLENBQUMsS0FBSyxDQUFDLFNBQVMsRUFBRTtZQUNqRCxxQkFBcUIsRUFBRSxHQUFHLEVBQUU7Z0JBQzFCLE9BQU87b0JBQ0wsQ0FBQyxFQUFFLENBQUM7b0JBQ0osQ0FBQyxFQUFFLENBQUM7b0JBQ0osS0FBSyxFQUFFLENBQUM7b0JBQ1IsTUFBTSxFQUFFLENBQUM7b0JBQ1QsR0FBRyxFQUFFLENBQUM7b0JBQ04sS0FBSyxFQUFFLENBQUM7b0JBQ1IsTUFBTSxFQUFFLENBQUM7b0JBQ1QsSUFBSSxFQUFFLENBQUM7b0JBQ1AsTUFBTSxFQUFFLElBQUksQ0FBQyxFQUFFLEVBQUU7aUJBQ2xCLENBQUM7WUFDSixDQUFDO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsTUFBTSxJQUFJLE9BQU8sQ0FBTyxPQUFPLENBQUMsRUFBRTtZQUNoQyxZQUFZLENBQUMsTUFBTSxDQUFDLGdCQUFnQixDQUFDLE1BQU0sRUFBRSxHQUFHLEVBQUU7Z0JBQ2hELE9BQU8sRUFBRSxDQUFDO1lBQ1osQ0FBQyxDQUFDLENBQUM7UUFDTCxDQUFDLENBQUMsQ0FBQztRQUVILE9BQU8sWUFBWSxDQUFDO0lBQ3RCLENBQUM7Q0FBQTtBQUVELE1BQU0sa0JBQWtCLEdBQUcsaUJBQWlCLENBQUM7QUFDN0MsTUFBTSxxQkFBcUIsR0FBRyxXQUFXLENBQUM7QUFFMUMsU0FBUyxnQkFBZ0IsQ0FBRSxXQUFvQztJQUM3RCxNQUFNLEtBQUssR0FBYyxFQUFFLENBQUM7SUFDNUIsTUFBTSxHQUFHLEdBQUcsSUFBQSw2QkFBZ0IsRUFBQyxLQUFLLENBQXVCLENBQUM7SUFDMUQsR0FBRyxDQUFDLFFBQVEsR0FBRyxXQUFXLElBQUksQ0FBQyxHQUFHLEVBQUUsR0FBb0IsQ0FBQyxDQUFDLENBQUM7SUFDM0QsR0FBRyxDQUFDLFdBQVcsR0FBRyxDQUFPLE9BQVksRUFBRSxFQUFFOztRQUN2QyxRQUFRLE9BQU8sQ0FBQyxLQUFLLEVBQUUsQ0FBQztZQUN4QixLQUFLLFFBQVE7Z0JBQ1gsT0FBTyxDQUFDLEdBQUcsQ0FBQztvQkFDVixVQUFVLEVBQUUsT0FBTyxDQUFDLFVBQVU7b0JBQzlCLFVBQVUsRUFBRSxPQUFPLENBQUMsVUFBVTtpQkFDL0IsQ0FBQyxDQUFDO2dCQUNILE1BQU07WUFDUixLQUFLLFVBQVU7Z0JBQ2IsTUFBQSxHQUFHLENBQUMsUUFBUSxvREFBRztvQkFDYixJQUFJLEVBQUUsVUFBVTtvQkFDaEIsT0FBTyxFQUFFLElBQUksQ0FBQyxTQUFTLENBQUMsd0JBQVksRUFBRSxJQUFJLEVBQUUsQ0FBQyxDQUFDO2lCQUMvQyxDQUFDLENBQUM7Z0JBQ0gsTUFBTTtZQUNSLEtBQUssYUFBYTtnQkFDaEIsTUFBQSxHQUFHLENBQUMsUUFBUSxvREFBRztvQkFDYixJQUFJLEVBQUUsYUFBYTtvQkFDbkIsRUFBRSxFQUFFLE9BQU8sQ0FBQyxFQUFFO29CQUNkLE9BQU8sRUFBRSxPQUFPLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQyxxQkFBcUIsQ0FBQyxDQUFDLENBQUMsa0JBQWtCO29CQUN2RSxRQUFRLEVBQUUsS0FBSztpQkFDaEIsQ0FBQyxDQUFDO2dCQUNILE1BQU07WUFDUixLQUFLLHFCQUFxQixDQUFDLENBQUMsQ0FBQztnQkFDM0IsSUFBSSxRQUFRLEdBQUcsSUFBSSxDQUFDO2dCQUNwQixRQUFPLE9BQU8sQ0FBQyxHQUFHLEVBQUUsQ0FBQztvQkFDckIsS0FBSyxHQUFHO3dCQUNOLFFBQVEsR0FBRyxJQUFJLENBQUMsU0FBUyxDQUFDLHVCQUFXLEVBQUUsSUFBSSxFQUFFLENBQUMsQ0FBQyxDQUFDO3dCQUNoRCxNQUFNO29CQUNSLEtBQUssSUFBSTt3QkFDUCxRQUFRLEdBQUcsSUFBSSxDQUFDLFNBQVMsQ0FBQyx3QkFBWSxFQUFFLElBQUksRUFBRSxDQUFDLENBQUMsQ0FBQzt3QkFDakQsTUFBTTtnQkFDUixDQUFDO2dCQUNELE1BQUEsR0FBRyxDQUFDLFFBQVEsb0RBQUc7b0JBQ2IsSUFBSSxFQUFFLHFCQUFxQjtvQkFDM0IsRUFBRSxFQUFFLE9BQU8sQ0FBQyxFQUFFO29CQUNkLE9BQU8sRUFBRSxRQUFRO2lCQUNsQixDQUFDLENBQUM7Z0JBQ0gsTUFBTTtZQUNSLENBQUM7UUFDRCxDQUFDO0lBQ0gsQ0FBQyxDQUFBLENBQUM7SUFFRixPQUFPLEVBQUUsS0FBSyxFQUFFLEdBQUcsRUFBRSxDQUFDO0FBQ3hCLENBQUM7QUFFRCxNQUFNLFNBQVMsR0FBRyxHQUFHLEVBQUU7SUFDckIsT0FBTztRQUNMLEtBQUssRUFBRTtZQUNMLGFBQWEsRUFBRSxDQUFDLGFBQWEsQ0FBQztZQUM5QixhQUFhLEVBQUUsaUJBQWlCO1lBQ2hDLGdCQUFnQixFQUFFLEtBQUs7WUFDdkIsV0FBVyxFQUFFLENBQUMsU0FBUyxDQUFDO1lBQ3hCLFdBQVcsRUFBRSxJQUFJO1NBQ2xCO1FBQ0QsU0FBUyxFQUFFO1lBQ1QsY0FBYyxFQUFFLEVBQUU7WUFDbEIsZ0JBQWdCLEVBQUUsWUFBWTtZQUM5QixpQkFBaUIsRUFBRSxtQkFBbUI7WUFDdEMsZ0JBQWdCLEVBQUUsc0JBQXNCO1lBQ3hDLHNCQUFzQixFQUFFLGtDQUFrQztZQUMxRCxjQUFjLEVBQUUsVUFBVTtZQUMxQixvQkFBb0IsRUFBRSxTQUFTO1lBQy9CLFdBQVcsRUFBRSx3QkFBd0I7WUFDckMsZ0JBQWdCLEVBQUUsRUFBRTtZQUNwQixZQUFZLEVBQUUsY0FBYztZQUM1QixXQUFXLEVBQUUsSUFBSTtZQUNqQixZQUFZLEVBQUUsSUFBSTtZQUNsQixhQUFhLEVBQUUsY0FBYztZQUM3QixtQkFBbUIsRUFBRSwwQkFBMEI7U0FDaEQ7UUFDRCxVQUFVLEVBQUU7WUFDVixJQUFJLEVBQUUsYUFBYTtZQUNuQixPQUFPLEVBQUUsT0FBTztZQUNoQixxQkFBcUIsRUFBRSxFQUFFO1lBQ3pCLGVBQWUsRUFBRSxFQUFFO1lBQ25CLE1BQU0sRUFBRSxFQUFFO1lBQ1YsZUFBZSxFQUFFLEVBQUU7WUFDbkIsYUFBYSxFQUFFLEVBQUU7WUFDakIsZ0JBQWdCLEVBQUUsQ0FBQyxFQUFFLENBQUM7U0FDdkI7UUFDRCxVQUFVLEVBQUUsRUFBRSxNQUFNLEVBQUUsRUFBRSxFQUFFLE9BQU8sRUFBRSxFQUFFLEVBQUUsSUFBSSxFQUFFLEVBQUUsRUFBRTtRQUNqRCxZQUFZLEVBQUUsRUFBRTtRQUNoQixVQUFVLEVBQUUsRUFBRTtLQUNGLENBQUM7QUFDakIsQ0FBQyxDQUFDO0FBRUYsSUFBSSxDQUFDLGtDQUFrQyxFQUFFLEdBQVMsRUFBRTtJQUNsRCxNQUFNLEVBQUUsR0FBRyxFQUFFLEdBQUcsZ0JBQWdCLEVBQUUsQ0FBQztJQUNuQyxNQUFNLEdBQUcsR0FBRyxNQUFNLE1BQU0sQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUU5QixNQUFNLElBQUksT0FBTyxDQUFPLE9BQU8sQ0FBQyxFQUFFO1FBQ2hDLEdBQUcsQ0FBQyxNQUFNLENBQUMsZ0JBQWdCLENBQUMsU0FBUyxFQUFFLE9BQU8sQ0FBQyxFQUFFO1lBQy9DLElBQUcsT0FBTyxDQUFDLElBQUksQ0FBQyxJQUFJLEtBQUssVUFBVSxFQUFFLENBQUM7Z0JBQUMsT0FBTztZQUFDLENBQUM7WUFDaEQsT0FBTyxFQUFFLENBQUM7UUFDWixDQUFDLENBQUMsQ0FBQztJQUNMLENBQUMsQ0FBQyxDQUFDO0FBQ0wsQ0FBQyxDQUFBLENBQUMsQ0FBQztBQUVILFFBQVEsQ0FBQyxnQkFBZ0IsRUFBRSxHQUFHLEVBQUU7SUFFOUIsSUFBSSxDQUFDLFlBQVksRUFBRSxHQUFTLEVBQUU7UUFDNUIsTUFBTSxFQUFFLEtBQUssRUFBRSxHQUFHLEVBQUUsR0FBRyxnQkFBZ0IsRUFBRSxDQUFDO1FBQzFDLE1BQU0sRUFBRSxNQUFNLEVBQUUsR0FBRyxNQUFNLE1BQU0sQ0FBQyxHQUFHLENBQUMsQ0FBQztRQUNyQyxNQUFNLEVBQUUsUUFBUSxFQUFFLEdBQUcsTUFBTSxDQUFDO1FBRTVCLE1BQU0sSUFBSSxPQUFPLENBQU8sT0FBTyxDQUFDLEVBQUU7WUFDaEMsTUFBTSxDQUFDLGdCQUFnQixDQUFDLFNBQVMsRUFBRSxPQUFPLENBQUMsRUFBRTtnQkFDM0MsSUFBRyxPQUFPLENBQUMsSUFBSSxDQUFDLElBQUksS0FBSyxVQUFVLEVBQUUsQ0FBQztvQkFBQyxPQUFPO2dCQUFDLENBQUM7Z0JBQ2hELE9BQU8sRUFBRSxDQUFDO1lBQ1osQ0FBQyxDQUFDLENBQUM7UUFDTCxDQUFDLENBQUMsQ0FBQztRQUVILE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxPQUFPLENBQUMsU0FBUyxFQUFFLENBQUMsQ0FBQztRQUVuQyxNQUFNLFNBQVMsR0FBRyxRQUFRLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyx5QkFBeUIsQ0FBcUIsQ0FBQztRQUM3RixNQUFNLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsQ0FBQztJQUM5QyxDQUFDLENBQUEsQ0FBQyxDQUFDO0lBRUgsSUFBSSxDQUFDLHFCQUFxQixFQUFFLEdBQVMsRUFBRTtRQUVyQyxNQUFNLFFBQVEsR0FBRyxhQUFhLENBQUM7UUFDL0IsTUFBTSxTQUFTLEdBQUcsWUFBWSxRQUFRLElBQUksQ0FBQztRQUUzQyxNQUFNLEVBQUUsS0FBSyxFQUFFLEdBQUcsRUFBRSxHQUFHLGdCQUFnQixFQUFFLENBQUM7UUFDMUMsTUFBTSxFQUFFLE1BQU0sRUFBRSxHQUFHLE1BQU0sTUFBTSxDQUFDLEdBQUcsRUFBRSxTQUFTLENBQUMsQ0FBQztRQUNoRCxNQUFNLEVBQUUsUUFBUSxFQUFFLEdBQUcsTUFBTSxDQUFDO1FBRTVCLE1BQU0sT0FBTyxDQUFDLEdBQUcsQ0FBQztZQUNoQixJQUFJLE9BQU8sQ0FBTyxPQUFPLENBQUMsRUFBRTtnQkFDMUIsTUFBTSxDQUFDLGdCQUFnQixDQUFDLFNBQVMsRUFBRSxPQUFPLENBQUMsRUFBRTtvQkFDM0MsSUFBRyxPQUFPLENBQUMsSUFBSSxDQUFDLElBQUksS0FBSyxVQUFVLEVBQUUsQ0FBQzt3QkFBQyxPQUFPO29CQUFDLENBQUM7b0JBQ2hELE9BQU8sRUFBRSxDQUFDO2dCQUNaLENBQUMsQ0FBQyxDQUFDO1lBQ0wsQ0FBQyxDQUFDO1lBQ0YsSUFBSSxPQUFPLENBQU8sT0FBTyxDQUFDLEVBQUU7Z0JBQzFCLE1BQU0sQ0FBQyxnQkFBZ0IsQ0FBQyxTQUFTLEVBQUUsT0FBTyxDQUFDLEVBQUU7b0JBQzNDLElBQUcsT0FBTyxDQUFDLElBQUksQ0FBQyxJQUFJLEtBQUsscUJBQXFCLEVBQUUsQ0FBQzt3QkFBQyxPQUFPO29CQUFDLENBQUM7b0JBQzNELE9BQU8sRUFBRSxDQUFDO2dCQUNaLENBQUMsQ0FBQyxDQUFDO1lBQ0wsQ0FBQyxDQUFDO1NBQ0gsQ0FBQyxDQUFDO1FBRUgsTUFBTSxZQUFZLEdBQUc7WUFDbkI7Z0JBQ0UsSUFBSSxFQUFFLFdBQVc7Z0JBQ2pCLEtBQUssRUFBRSxrQkFBa0I7YUFDMUI7WUFDRDtnQkFDRSxJQUFJLEVBQUUsU0FBUztnQkFDZixLQUFLLEVBQUUsSUFBSTthQUNaO1lBQ0Q7Z0JBQ0UsSUFBSSxFQUFFLFNBQVM7Z0JBQ2YsS0FBSyxFQUFFLFdBQVc7YUFDbkI7WUFDRDtnQkFDRSxJQUFJLEVBQUUsU0FBUztnQkFDZixLQUFLLEVBQUUsY0FBYzthQUN0QjtTQUNGLENBQUM7UUFFRixvREFBb0Q7UUFDcEQsK0NBQStDO1FBQy9DLE1BQU0sSUFBSSxHQUFHLFNBQVMsRUFBRSxDQUFDO1FBQ3pCLE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxPQUFPLGlDQUNoQixJQUFJLEtBQ1AsVUFBVSxrQ0FDTCxJQUFJLENBQUMsVUFBVSxLQUNsQixJQUFJLEVBQUUsWUFBWSxFQUNsQixNQUFNLEVBQUUsU0FBUyxFQUNqQixPQUFPLEVBQUUsd0JBQVksQ0FBQyxnQkFBZ0IsS0FFeEMsVUFBVSxrQ0FDTCxJQUFJLENBQUMsVUFBVSxLQUNsQixJQUFJLEVBQUUsUUFBUSxLQUVoQixVQUFVLEVBQUU7Z0JBQ1YsQ0FBRSxZQUFZLENBQUMsQ0FBQyxDQUFDLENBQUUsRUFBRSxZQUFZLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQzthQUMzQyxJQUNELENBQUM7UUFFSCxrREFBa0Q7UUFDbEQsTUFBTSxTQUFTLEdBQUcsUUFBUSxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMseUJBQXlCLENBQXFCLENBQUM7UUFDN0YsTUFBTSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLENBQUM7UUFFNUMsTUFBTSxZQUFZLEdBQUcsUUFBUSxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsdUJBQXVCLENBQW1CLENBQUM7UUFDNUYsTUFBTSxDQUFDLFlBQVksQ0FBQyxXQUFXLENBQUMsQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLENBQUM7UUFFakQsTUFBTSxpQkFBaUIsR0FBRyxRQUFRLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxxQkFBcUIsQ0FBc0IsQ0FBQztRQUNsRyxNQUFNLENBQUMsaUJBQWlCLENBQUMsS0FBSyxDQUFDLENBQUMsSUFBSSxDQUFDLGtCQUFrQixDQUFDLENBQUM7SUFDM0QsQ0FBQyxDQUFBLENBQUMsQ0FBQztJQUVILElBQUksQ0FBQyxzQkFBc0IsRUFBRSxHQUFTLEVBQUU7UUFFdEMsTUFBTSxRQUFRLEdBQUcsYUFBYSxDQUFDO1FBQy9CLE1BQU0sU0FBUyxHQUFHLFlBQVksUUFBUSxLQUFLLENBQUM7UUFFNUMsTUFBTSxFQUFFLEtBQUssRUFBRSxHQUFHLEVBQUUsR0FBRyxnQkFBZ0IsRUFBRSxDQUFDO1FBQzFDLE1BQU0sRUFBRSxNQUFNLEVBQUUsR0FBRyxNQUFNLE1BQU0sQ0FBQyxHQUFHLEVBQUUsU0FBUyxDQUFDLENBQUM7UUFDaEQsTUFBTSxFQUFFLFFBQVEsRUFBRSxHQUFHLE1BQU0sQ0FBQztRQUU1QixNQUFNLE9BQU8sQ0FBQyxHQUFHLENBQUM7WUFDaEIsSUFBSSxPQUFPLENBQU8sT0FBTyxDQUFDLEVBQUU7Z0JBQzFCLE1BQU0sQ0FBQyxnQkFBZ0IsQ0FBQyxTQUFTLEVBQUUsT0FBTyxDQUFDLEVBQUU7b0JBQzNDLElBQUcsT0FBTyxDQUFDLElBQUksQ0FBQyxJQUFJLEtBQUssVUFBVSxFQUFFLENBQUM7d0JBQUMsT0FBTztvQkFBQyxDQUFDO29CQUNoRCxPQUFPLEVBQUUsQ0FBQztnQkFDWixDQUFDLENBQUMsQ0FBQztZQUNMLENBQUMsQ0FBQztZQUNGLElBQUksT0FBTyxDQUFPLE9BQU8sQ0FBQyxFQUFFO2dCQUMxQixNQUFNLENBQUMsZ0JBQWdCLENBQUMsU0FBUyxFQUFFLE9BQU8sQ0FBQyxFQUFFO29CQUMzQyxJQUFHLE9BQU8sQ0FBQyxJQUFJLENBQUMsSUFBSSxLQUFLLHFCQUFxQixFQUFFLENBQUM7d0JBQUMsT0FBTztvQkFBQyxDQUFDO29CQUMzRCxPQUFPLEVBQUUsQ0FBQztnQkFDWixDQUFDLENBQUMsQ0FBQztZQUNMLENBQUMsQ0FBQztTQUNILENBQUMsQ0FBQztRQUVILE1BQU0sWUFBWSxHQUFHO1lBQ25CO2dCQUNFLElBQUksRUFBRSxZQUFZO2dCQUNsQixLQUFLLEVBQUUsa0JBQWtCO2FBQzFCO1lBQ0Q7Z0JBQ0UsSUFBSSxFQUFFLFNBQVM7Z0JBQ2YsS0FBSyxFQUFFLElBQUk7YUFDWjtTQUNGLENBQUM7UUFFRixvREFBb0Q7UUFDcEQsK0NBQStDO1FBQy9DLE1BQU0sSUFBSSxHQUFHLFNBQVMsRUFBRSxDQUFDO1FBQ3pCLE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxPQUFPLGlDQUNoQixJQUFJLEtBQ1AsVUFBVSxrQ0FDTCxJQUFJLENBQUMsVUFBVSxLQUNsQixJQUFJLEVBQUUsWUFBWSxFQUNsQixNQUFNLEVBQUUsU0FBUyxFQUNqQixPQUFPLEVBQUUsd0JBQVksQ0FBQyxnQkFBZ0IsS0FFeEMsVUFBVSxrQ0FDTCxJQUFJLENBQUMsVUFBVSxLQUNsQixJQUFJLEVBQUUsUUFBUSxLQUVoQixVQUFVLEVBQUU7Z0JBQ1YsQ0FBRSxZQUFZLENBQUMsQ0FBQyxDQUFDLENBQUUsRUFBRSxZQUFZLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQzthQUMzQyxJQUNELENBQUM7UUFFSCxrREFBa0Q7UUFDbEQsTUFBTSxTQUFTLEdBQUcsUUFBUSxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMseUJBQXlCLENBQXFCLENBQUM7UUFDN0YsTUFBTSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLENBQUM7UUFFNUMsTUFBTSxZQUFZLEdBQUcsUUFBUSxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsdUJBQXVCLENBQW1CLENBQUM7UUFDNUYsTUFBTSxDQUFDLFlBQVksQ0FBQyxXQUFXLENBQUMsQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLENBQUM7UUFFakQsTUFBTSxpQkFBaUIsR0FBRyxRQUFRLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxzQkFBc0IsQ0FBc0IsQ0FBQztRQUNuRyxNQUFNLENBQUMsaUJBQWlCLENBQUMsS0FBSyxDQUFDLENBQUMsSUFBSSxDQUFDLGtCQUFrQixDQUFDLENBQUM7SUFDM0QsQ0FBQyxDQUFBLENBQUMsQ0FBQztJQUVIOzs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7TUF1TEU7SUFFRixJQUFJLENBQUMscUJBQXFCLEVBQUUsR0FBUyxFQUFFO1FBRXJDLE1BQU0sUUFBUSxHQUFHLHFDQUFxQyxDQUFDO1FBQ3ZELE1BQU0sU0FBUyxHQUFHLFFBQVEsQ0FBQztRQUUzQixNQUFNLEVBQUUsS0FBSyxFQUFFLEdBQUcsRUFBRSxHQUFHLGdCQUFnQixFQUFFLENBQUM7UUFDMUMsTUFBTSxFQUFFLE1BQU0sRUFBRSxHQUFHLE1BQU0sTUFBTSxDQUFDLEdBQUcsRUFBRSxTQUFTLENBQUMsQ0FBQztRQUNoRCxNQUFNLEVBQUUsUUFBUSxFQUFFLEdBQUcsTUFBTSxDQUFDO1FBRTVCLCtEQUErRDtRQUMvRCx3REFBd0Q7UUFDeEQsTUFBTSxJQUFJLE9BQU8sQ0FBTyxPQUFPLENBQUMsRUFBRTtZQUNoQyxNQUFNLENBQUMsZ0JBQWdCLENBQUMsU0FBUyxFQUFFLE9BQU8sQ0FBQyxFQUFFO2dCQUMzQyxJQUFHLE9BQU8sQ0FBQyxJQUFJLENBQUMsSUFBSSxLQUFLLFVBQVUsRUFBRSxDQUFDO29CQUFDLE9BQU87Z0JBQUMsQ0FBQztnQkFDaEQsT0FBTyxFQUFFLENBQUM7WUFDWixDQUFDLENBQUMsQ0FBQztRQUNMLENBQUMsQ0FBQyxDQUFDO1FBRUgsb0VBQW9FO1FBQ3BFLE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxPQUFPLENBQUMsU0FBUyxFQUFFLENBQUMsQ0FBQztRQUVuQyxrREFBa0Q7UUFDbEQsTUFBTSxTQUFTLEdBQUcsUUFBUSxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMseUJBQXlCLENBQXFCLENBQUM7UUFDN0YsTUFBTSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLENBQUM7UUFFNUMsTUFBTSxZQUFZLEdBQUcsUUFBUSxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsdUJBQXVCLENBQW1CLENBQUM7UUFDNUYsTUFBTSxDQUFDLFlBQVksQ0FBQyxXQUFXLENBQUMsQ0FBQyxJQUFJLENBQUMsRUFBRSxDQUFDLENBQUM7SUFDNUMsQ0FBQyxDQUFBLENBQUMsQ0FBQztJQUVILElBQUksQ0FBQyxxQkFBcUIsRUFBRSxHQUFTLEVBQUU7UUFFckMsTUFBTSxVQUFVLEdBQUcsYUFBYSxDQUFDO1FBQ2pDLE1BQU0sU0FBUyxHQUFHLFlBQVksVUFBVSxJQUFJLENBQUM7UUFFN0MsTUFBTSxFQUFFLEtBQUssRUFBRSxHQUFHLEVBQUUsR0FBRyxnQkFBZ0IsRUFBRSxDQUFDO1FBRTFDLEtBQUssQ0FBQyxVQUFVLEdBQUc7WUFDakIsTUFBTSxFQUFFLFNBQVM7U0FDbEIsQ0FBQztRQUVGLEtBQUssQ0FBQyxVQUFVLEdBQUc7WUFDakIsQ0FBQyxFQUFFLElBQUksRUFBRSxXQUFXLEVBQUUsS0FBSyxFQUFFLE9BQU8sRUFBRSxDQUFDO1lBQ3ZDLENBQUMsRUFBRSxLQUFLLEVBQUUsV0FBVyxFQUFFLEVBQUUsRUFBRSxLQUFLLEVBQUUsZUFBZSxFQUFFLENBQUM7U0FDckQsQ0FBQztRQUVGLEtBQUssQ0FBQyxVQUFVLEdBQUc7WUFDakIsSUFBSSxFQUFFLFlBQVk7WUFDbEIsT0FBTyxFQUFFLFlBQVk7WUFDckIscUJBQXFCLEVBQUUsRUFBRTtZQUN6QixlQUFlLEVBQUU7Z0JBQ2YsRUFBRSxJQUFJLEVBQUUsVUFBVSxFQUFFLEtBQUssRUFBRSxHQUFHLEVBQUU7Z0JBQ2hDLEVBQUUsSUFBSSxFQUFFLFFBQVEsRUFBRSxLQUFLLEVBQUUsS0FBSyxFQUFFO2FBQ2pDO1lBQ0QsTUFBTSxFQUFFLEVBQUU7WUFDVixlQUFlLEVBQUUseUJBQXlCO1lBQzFDLGFBQWEsRUFBRSxFQUFFO1lBQ2pCLGdCQUFnQixFQUFFLENBQUUsRUFBRSxDQUFFO1NBQ3pCLENBQUM7UUFFRixNQUFNLEVBQUUsTUFBTSxFQUFFLEdBQUcsTUFBTSxNQUFNLENBQUMsR0FBRyxDQUFDLENBQUM7UUFDckMsTUFBTSxFQUFFLFFBQVEsRUFBRSxHQUFHLE1BQU0sQ0FBQztRQUU1QixNQUFNLE9BQU8sQ0FBQyxHQUFHLENBQUM7WUFDaEIsSUFBSSxPQUFPLENBQU8sT0FBTyxDQUFDLEVBQUU7Z0JBQzFCLE1BQU0sQ0FBQyxnQkFBZ0IsQ0FBQyxTQUFTLEVBQUUsT0FBTyxDQUFDLEVBQUU7b0JBQzNDLElBQUcsT0FBTyxDQUFDLElBQUksQ0FBQyxJQUFJLEtBQUssVUFBVSxFQUFFLENBQUM7d0JBQUMsT0FBTztvQkFBQyxDQUFDO29CQUNoRCxPQUFPLEVBQUUsQ0FBQztnQkFDWixDQUFDLENBQUMsQ0FBQztZQUNMLENBQUMsQ0FBQztZQUNGLElBQUksT0FBTyxDQUFPLE9BQU8sQ0FBQyxFQUFFO2dCQUMxQixNQUFNLENBQUMsZ0JBQWdCLENBQUMsU0FBUyxFQUFFLE9BQU8sQ0FBQyxFQUFFO29CQUMzQyxJQUFHLE9BQU8sQ0FBQyxJQUFJLENBQUMsSUFBSSxLQUFLLHFCQUFxQixFQUFFLENBQUM7d0JBQUMsT0FBTztvQkFBQyxDQUFDO29CQUMzRCxPQUFPLEVBQUUsQ0FBQztnQkFDWixDQUFDLENBQUMsQ0FBQztZQUNMLENBQUMsQ0FBQztTQUNILENBQUMsQ0FBQztRQUVILGtEQUFrRDtRQUNsRCxNQUFNLFNBQVMsR0FBRyxRQUFRLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyx5QkFBeUIsQ0FBcUIsQ0FBQztRQUM3RixNQUFNLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxDQUFDLElBQUksQ0FBQyxZQUFZLENBQUMsQ0FBQztRQUUzQyxxRUFBcUU7UUFDckUsMkNBQTJDO1FBQzNDLE1BQU0sa0JBQWtCLEdBQUcsUUFBUSxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMscUJBQXFCLENBQXFCLENBQUM7UUFDbEcsTUFBTSxDQUFDLGtCQUFrQixDQUFDLEtBQUssQ0FBQyxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsQ0FBQztRQUUzQyxNQUFNLG9CQUFvQixHQUFHLFFBQVEsQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLHVCQUF1QixDQUFxQixDQUFDO1FBQ3RHLE1BQU0sQ0FBQyxvQkFBb0IsQ0FBQyxLQUFLLENBQUMsQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLENBQUM7UUFFaEQsTUFBTSxZQUFZLEdBQUcsUUFBUSxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsdUJBQXVCLENBQW1CLENBQUM7UUFDNUYsTUFBTSxDQUFDLFlBQVksQ0FBQyxXQUFXLENBQUMsQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLENBQUM7UUFFakQsTUFBTSxZQUFZLEdBQUcsUUFBUSxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsc0JBQXNCLENBQW1CLENBQUM7UUFDM0YsTUFBTSxDQUFDLFlBQVksQ0FBQyxXQUFXLENBQUMsQ0FBQyxJQUFJLENBQUMseUJBQXlCLENBQUMsQ0FBQztRQUVqRSxNQUFNLFdBQVcsR0FBRyxRQUFRLENBQUMsSUFBSSxDQUFDLGdCQUFnQixDQUFDLDZCQUE2QixDQUFDLENBQUM7UUFDbEYsTUFBTSxDQUFDLFdBQVcsQ0FBQyxNQUFNLENBQUMsQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLENBQUM7SUFFckMsQ0FBQyxDQUFBLENBQUMsQ0FBQztBQUNMLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLHlDQUF5QyxFQUFFLEdBQVMsRUFBRTs7SUFFekQsTUFBTSxRQUFRLEdBQUcsYUFBYSxDQUFDO0lBQy9CLE1BQU0sU0FBUyxHQUFHLFlBQVksUUFBUSxJQUFJLENBQUM7SUFFM0MsTUFBTSxFQUFFLEtBQUssRUFBRSxHQUFHLEVBQUUsR0FBRyxnQkFBZ0IsRUFBRSxDQUFDO0lBQzFDLE1BQU0sRUFBRSxNQUFNLEVBQUUsR0FBRyxNQUFNLE1BQU0sQ0FBQyxHQUFHLEVBQUUsU0FBUyxDQUFDLENBQUM7SUFDaEQsTUFBTSxFQUFFLFFBQVEsRUFBRSxHQUFHLE1BQU0sQ0FBQztJQUU1QixNQUFNLE9BQU8sQ0FBQyxHQUFHLENBQUM7UUFDaEIsSUFBSSxPQUFPLENBQU8sT0FBTyxDQUFDLEVBQUU7WUFDMUIsTUFBTSxDQUFDLGdCQUFnQixDQUFDLFNBQVMsRUFBRSxPQUFPLENBQUMsRUFBRTtnQkFDM0MsSUFBRyxPQUFPLENBQUMsSUFBSSxDQUFDLElBQUksS0FBSyxVQUFVLEVBQUUsQ0FBQztvQkFBQyxPQUFPO2dCQUFDLENBQUM7Z0JBQ2hELE9BQU8sRUFBRSxDQUFDO1lBQ1osQ0FBQyxDQUFDLENBQUM7UUFDTCxDQUFDLENBQUM7UUFDRixJQUFJLE9BQU8sQ0FBTyxPQUFPLENBQUMsRUFBRTtZQUMxQixNQUFNLENBQUMsZ0JBQWdCLENBQUMsU0FBUyxFQUFFLE9BQU8sQ0FBQyxFQUFFO2dCQUMzQyxJQUFHLE9BQU8sQ0FBQyxJQUFJLENBQUMsSUFBSSxLQUFLLHFCQUFxQixFQUFFLENBQUM7b0JBQUMsT0FBTztnQkFBQyxDQUFDO2dCQUMzRCxPQUFPLEVBQUUsQ0FBQztZQUNaLENBQUMsQ0FBQyxDQUFDO1FBQ0wsQ0FBQyxDQUFDO0tBQ0gsQ0FBQyxDQUFDO0lBRUgsdUVBQXVFO0lBQ3ZFLGlFQUFpRTtJQUNqRSxNQUFNLFNBQVMsR0FBRyxRQUFRLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyx5QkFBeUIsQ0FBcUIsQ0FBQztJQUU3RixTQUFTLENBQUMsS0FBSyxHQUFHLFlBQVksQ0FBQztJQUMvQixTQUFTLENBQUMsYUFBYSxDQUFDLElBQUksTUFBTSxDQUFDLEtBQUssQ0FBQyxRQUFRLENBQUMsQ0FBQyxDQUFDO0lBQ3BELE1BQU0sQ0FBQyxNQUFBLEtBQUssQ0FBQyxVQUFVLDBDQUFFLElBQUksQ0FBQyxDQUFDLElBQUksQ0FBQyxZQUFZLENBQUMsQ0FBQztJQUVsRCxNQUFNLFNBQVMsR0FBRyxRQUFRLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyx3QkFBd0IsQ0FBc0IsQ0FBQztJQUM3RixTQUFTLENBQUMsS0FBSyxHQUFHLFlBQVksQ0FBQztJQUMvQixTQUFTLENBQUMsYUFBYSxDQUFDLElBQUksTUFBTSxDQUFDLEtBQUssQ0FBQyxRQUFRLENBQUMsQ0FBQyxDQUFDO0lBQ3BELE1BQU0sQ0FBQyxNQUFBLEtBQUssQ0FBQyxVQUFVLDBDQUFFLE9BQU8sQ0FBQyxDQUFDLElBQUksQ0FBQyxZQUFZLENBQUMsQ0FBQztJQUVyRCwyREFBMkQ7SUFDM0QsTUFBTSxrQkFBa0IsR0FBRyxRQUFRLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxxQkFBcUIsQ0FBcUIsQ0FBQztJQUNsRyxrQkFBa0IsQ0FBQyxLQUFLLEdBQUcsR0FBRyxDQUFDO0lBQy9CLGtCQUFrQixDQUFDLGFBQWEsQ0FBQyxJQUFJLE1BQU0sQ0FBQyxLQUFLLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQztJQUMzRCxNQUFNLENBQUMsTUFBQSxLQUFLLENBQUMsVUFBVSwwQ0FBRSxlQUFlLENBQUMsQ0FBQyxFQUFFLEtBQUssQ0FBQyxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUU3RCxNQUFNLG9CQUFvQixHQUFHLFFBQVEsQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLHVCQUF1QixDQUFxQixDQUFDO0lBQ3RHLG9CQUFvQixDQUFDLEtBQUssR0FBRyxLQUFLLENBQUMsQ0FBQywwQ0FBMEM7SUFDOUUsb0JBQW9CLENBQUMsYUFBYSxDQUFDLElBQUksTUFBTSxDQUFDLEtBQUssQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDO0lBQzdELE1BQU0sQ0FBQyxNQUFBLEtBQUssQ0FBQyxVQUFVLDBDQUFFLGVBQWUsQ0FBQyxDQUFDLEVBQUUsS0FBSyxDQUFDLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxDQUFDO0lBRWpFLE1BQU0sY0FBYyxHQUFHLFFBQVEsQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLDhCQUE4QixDQUFzQixDQUFDO0lBRXhHLElBQUksUUFBUSxHQUFHLElBQUksT0FBTyxDQUFPLE9BQU8sQ0FBQyxFQUFFO1FBQ3pDLE1BQU0sQ0FBQyxnQkFBZ0IsQ0FBQyxTQUFTLEVBQUUsT0FBTyxDQUFDLEVBQUU7WUFDM0MsSUFBRyxPQUFPLENBQUMsSUFBSSxDQUFDLElBQUksS0FBSyxhQUFhLEVBQUUsQ0FBQztnQkFBQyxPQUFPO1lBQUMsQ0FBQztZQUNuRCxPQUFPLEVBQUUsQ0FBQztRQUNaLENBQUMsQ0FBQyxDQUFDO0lBQ0wsQ0FBQyxDQUFDLENBQUM7SUFFSCxjQUFjLENBQUMsS0FBSyxFQUFFLENBQUM7SUFDdkIsTUFBTSxRQUFRLENBQUM7SUFFZixNQUFNLFlBQVksR0FBRyxRQUFRLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyx1QkFBdUIsQ0FBbUIsQ0FBQztJQUM1RixNQUFNLENBQUMsWUFBWSxDQUFDLFdBQVcsQ0FBQyxDQUFDLElBQUksQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDO0lBQzFELE1BQU0sQ0FBQyxNQUFBLEtBQUssQ0FBQyxVQUFVLDBDQUFFLE1BQU0sQ0FBQyxDQUFDLElBQUksQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDO0lBQzFELE1BQU0sQ0FBQyxNQUFBLEtBQUssQ0FBQyxZQUFZLDBDQUFFLE1BQU0sQ0FBQyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQztJQUU5Qyw0REFBNEQ7SUFDNUQsTUFBTSxDQUFDLE1BQUEsS0FBSyxDQUFDLFVBQVUsMENBQUUsSUFBSSxDQUFDLENBQUMsSUFBSSxDQUFDLFlBQVksQ0FBQyxDQUFDO0lBRWxELE1BQU0sZUFBZSxHQUFHLFFBQVEsQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLCtCQUErQixDQUFzQixDQUFDO0lBRTFHLFFBQVEsR0FBRyxJQUFJLE9BQU8sQ0FBTyxPQUFPLENBQUMsRUFBRTtRQUNyQyxNQUFNLENBQUMsZ0JBQWdCLENBQUMsU0FBUyxFQUFFLE9BQU8sQ0FBQyxFQUFFO1lBQzNDLElBQUcsT0FBTyxDQUFDLElBQUksQ0FBQyxJQUFJLEtBQUssYUFBYSxFQUFFLENBQUM7Z0JBQUMsT0FBTztZQUFDLENBQUM7WUFDbkQsT0FBTyxFQUFFLENBQUM7UUFDWixDQUFDLENBQUMsQ0FBQztJQUNMLENBQUMsQ0FBQyxDQUFDO0lBRUgsZUFBZSxDQUFDLEtBQUssRUFBRSxDQUFDO0lBQ3hCLE1BQU0sUUFBUSxDQUFDO0lBRWYsTUFBTSxhQUFhLEdBQUcsUUFBUSxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsd0JBQXdCLENBQW1CLENBQUM7SUFDOUYsTUFBTSxDQUFDLGFBQWEsQ0FBQyxXQUFXLENBQUMsQ0FBQyxJQUFJLENBQUMscUJBQXFCLENBQUMsQ0FBQztJQUM5RCxNQUFNLENBQUMsTUFBQSxLQUFLLENBQUMsVUFBVSwwQ0FBRSxPQUFPLENBQUMsQ0FBQyxJQUFJLENBQUMscUJBQXFCLENBQUMsQ0FBQztJQUM5RCxNQUFNLENBQUMsTUFBQSxLQUFLLENBQUMsWUFBWSwwQ0FBRSxPQUFPLENBQUMsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUM7QUFFakQsQ0FBQyxDQUFBLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyw2REFBNkQsRUFBRSxHQUFTLEVBQUU7O0lBRTdFLE1BQU0sUUFBUSxHQUFHLGFBQWEsQ0FBQztJQUMvQixNQUFNLFNBQVMsR0FBRyxZQUFZLFFBQVEsSUFBSSxDQUFDO0lBRTNDLE1BQU0sRUFBRSxLQUFLLEVBQUUsR0FBRyxFQUFFLEdBQUcsZ0JBQWdCLEVBQUUsQ0FBQztJQUMxQyxNQUFNLEVBQUUsTUFBTSxFQUFFLEdBQUcsTUFBTSxNQUFNLENBQUMsR0FBRyxFQUFFLFNBQVMsQ0FBQyxDQUFDO0lBQ2hELE1BQU0sRUFBRSxRQUFRLEVBQUUsR0FBRyxNQUFNLENBQUM7SUFFNUIsTUFBTSxPQUFPLENBQUMsR0FBRyxDQUFDO1FBQ2hCLElBQUksT0FBTyxDQUFPLE9BQU8sQ0FBQyxFQUFFO1lBQzFCLE1BQU0sQ0FBQyxnQkFBZ0IsQ0FBQyxTQUFTLEVBQUUsT0FBTyxDQUFDLEVBQUU7Z0JBQzNDLElBQUcsT0FBTyxDQUFDLElBQUksQ0FBQyxJQUFJLEtBQUssVUFBVSxFQUFFLENBQUM7b0JBQUMsT0FBTztnQkFBQyxDQUFDO2dCQUNoRCxPQUFPLEVBQUUsQ0FBQztZQUNaLENBQUMsQ0FBQyxDQUFDO1FBQ0wsQ0FBQyxDQUFDO1FBQ0YsSUFBSSxPQUFPLENBQU8sT0FBTyxDQUFDLEVBQUU7WUFDMUIsTUFBTSxDQUFDLGdCQUFnQixDQUFDLFNBQVMsRUFBRSxPQUFPLENBQUMsRUFBRTtnQkFDM0MsSUFBRyxPQUFPLENBQUMsSUFBSSxDQUFDLElBQUksS0FBSyxxQkFBcUIsRUFBRSxDQUFDO29CQUFDLE9BQU87Z0JBQUMsQ0FBQztnQkFDM0QsT0FBTyxFQUFFLENBQUM7WUFDWixDQUFDLENBQUMsQ0FBQztRQUNMLENBQUMsQ0FBQztLQUNILENBQUMsQ0FBQztJQUVILG9DQUFvQztJQUNwQyxNQUFNLFNBQVMsR0FBRyxRQUFRLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyx3QkFBd0IsQ0FBc0IsQ0FBQztJQUM3RixTQUFTLENBQUMsS0FBSyxHQUFHLFlBQVksQ0FBQztJQUMvQixTQUFTLENBQUMsYUFBYSxDQUFDLElBQUksTUFBTSxDQUFDLEtBQUssQ0FBQyxRQUFRLENBQUMsQ0FBQyxDQUFDO0lBQ3BELE1BQU0sQ0FBQyxNQUFBLEtBQUssQ0FBQyxVQUFVLDBDQUFFLE9BQU8sQ0FBQyxDQUFDLElBQUksQ0FBQyxZQUFZLENBQUMsQ0FBQztJQUVyRCx3RUFBd0U7SUFDeEUsTUFBTSxTQUFTLEdBQUcsUUFBUSxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsdUJBQXVCLENBQW1CLENBQUM7SUFDekYsTUFBTSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLENBQUMsSUFBSSxDQUFDLE1BQU0sQ0FBQyxDQUFDO0lBRTdDLDRDQUE0QztJQUM1QyxNQUFNLGlCQUFpQixHQUFHLFFBQVEsQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLHFCQUFxQixDQUFzQixDQUFDO0lBQ2xHLGlCQUFpQixDQUFDLEtBQUssR0FBRyxPQUFPLENBQUM7SUFDbEMsaUJBQWlCLENBQUMsYUFBYSxDQUFDLElBQUksTUFBTSxDQUFDLEtBQUssQ0FBQyxRQUFRLENBQUMsQ0FBQyxDQUFDO0lBRTVELHVGQUF1RjtJQUN2RixNQUFNLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsQ0FBQyxHQUFHLENBQUMsSUFBSSxDQUFDLE1BQU0sQ0FBQyxDQUFDO0lBQ2pELE1BQU0sQ0FBQyxNQUFBLFNBQVMsQ0FBQyxXQUFXLDBDQUFFLFFBQVEsQ0FBQyxZQUFZLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQztJQUNqRSxNQUFNLENBQUMsTUFBQSxTQUFTLENBQUMsV0FBVywwQ0FBRSxRQUFRLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUM7QUFFOUQsQ0FBQyxDQUFBLENBQUMsQ0FBQztBQUVILFFBQVEsQ0FBQyxXQUFXLEVBQUUsR0FBRyxFQUFFO0lBQ3pCLElBQUksQ0FBQyxnQ0FBZ0MsRUFBRSxHQUFTLEVBQUU7UUFFaEQsTUFBTSxRQUFRLEdBQUcsYUFBYSxDQUFDO1FBQy9CLE1BQU0sU0FBUyxHQUFHLFlBQVksUUFBUSxJQUFJLENBQUM7UUFFM0MsTUFBTSxFQUFFLEtBQUssRUFBRSxHQUFHLEVBQUUsR0FBRyxnQkFBZ0IsRUFBRSxDQUFDO1FBQzFDLE1BQU0sRUFBRSxNQUFNLEVBQUUsR0FBRyxNQUFNLE1BQU0sQ0FBQyxHQUFHLEVBQUUsU0FBUyxDQUFDLENBQUM7UUFDaEQsTUFBTSxFQUFFLFFBQVEsRUFBRSxHQUFHLE1BQU0sQ0FBQztRQUU1QixLQUFLLENBQUMsVUFBVSxHQUFHO1lBQ2pCLE1BQU0sRUFBRSxTQUFTO1NBQ2xCLENBQUM7UUFFRixNQUFNLE9BQU8sQ0FBQyxHQUFHLENBQUM7WUFDaEIsSUFBSSxPQUFPLENBQU8sT0FBTyxDQUFDLEVBQUU7Z0JBQzFCLE1BQU0sQ0FBQyxnQkFBZ0IsQ0FBQyxTQUFTLEVBQUUsT0FBTyxDQUFDLEVBQUU7b0JBQzNDLElBQUksT0FBTyxDQUFDLElBQUksQ0FBQyxJQUFJLEtBQUssVUFBVSxFQUFFLENBQUM7d0JBQUMsT0FBTztvQkFBQyxDQUFDO29CQUNqRCxPQUFPLEVBQUUsQ0FBQztnQkFDWixDQUFDLENBQUMsQ0FBQztZQUNMLENBQUMsQ0FBQztZQUNGLElBQUksT0FBTyxDQUFPLE9BQU8sQ0FBQyxFQUFFO2dCQUMxQixNQUFNLENBQUMsZ0JBQWdCLENBQUMsU0FBUyxFQUFFLE9BQU8sQ0FBQyxFQUFFO29CQUMzQyxJQUFJLE9BQU8sQ0FBQyxJQUFJLENBQUMsSUFBSSxLQUFLLHFCQUFxQixFQUFFLENBQUM7d0JBQUMsT0FBTztvQkFBQyxDQUFDO29CQUM1RCxPQUFPLEVBQUUsQ0FBQztnQkFDWixDQUFDLENBQUMsQ0FBQztZQUNMLENBQUMsQ0FBQztTQUNILENBQUMsQ0FBQztRQUVILE1BQU0sVUFBVSxHQUFHLFFBQVEsQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLFlBQVksQ0FBbUIsQ0FBQztRQUMvRSxNQUFNLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsU0FBUyxDQUFDLENBQUMsUUFBUSxDQUFDLGdCQUFnQixDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLENBQUM7UUFFaEYsTUFBTSxXQUFXLEdBQUcsVUFBVSxDQUFDLGdCQUFnQixDQUFDLFdBQVcsQ0FBQyxDQUFDO1FBQzdELEtBQUssTUFBTSxVQUFVLElBQUksV0FBVyxFQUFFLENBQUM7WUFDckMsTUFBTSxDQUFDLE1BQU0sQ0FBQyxnQkFBZ0IsQ0FBQyxVQUFVLENBQUMsQ0FBQyxPQUFPLEtBQUssTUFBTSxDQUFDLENBQUM7UUFDakUsQ0FBQztJQUVILENBQUMsQ0FBQSxDQUFDLENBQUM7SUFFSCxJQUFJLENBQUMsMEJBQTBCLEVBQUUsR0FBUyxFQUFFO1FBRTFDLE1BQU0sUUFBUSxHQUFHLGFBQWEsQ0FBQztRQUMvQixNQUFNLFNBQVMsR0FBRyxZQUFZLFFBQVEsSUFBSSxDQUFDO1FBRTNDLE1BQU0sRUFBRSxLQUFLLEVBQUUsR0FBRyxFQUFFLEdBQUcsZ0JBQWdCLEVBQUUsQ0FBQztRQUMxQyxNQUFNLEVBQUUsTUFBTSxFQUFFLEdBQUcsTUFBTSxNQUFNLENBQUMsR0FBRyxFQUFFLFNBQVMsQ0FBQyxDQUFDO1FBQ2hELE1BQU0sRUFBRSxRQUFRLEVBQUUsR0FBRyxNQUFNLENBQUM7UUFFNUIsS0FBSyxDQUFDLFVBQVUsR0FBRztZQUNqQixNQUFNLEVBQUUsU0FBUztTQUNsQixDQUFDO1FBRUYsTUFBTSxPQUFPLENBQUMsR0FBRyxDQUFDO1lBQ2hCLElBQUksT0FBTyxDQUFPLE9BQU8sQ0FBQyxFQUFFO2dCQUMxQixNQUFNLENBQUMsZ0JBQWdCLENBQUMsU0FBUyxFQUFFLE9BQU8sQ0FBQyxFQUFFO29CQUMzQyxJQUFJLE9BQU8sQ0FBQyxJQUFJLENBQUMsSUFBSSxLQUFLLFVBQVUsRUFBRSxDQUFDO3dCQUFDLE9BQU87b0JBQUMsQ0FBQztvQkFDakQsT0FBTyxFQUFFLENBQUM7Z0JBQ1osQ0FBQyxDQUFDLENBQUM7WUFDTCxDQUFDLENBQUM7WUFDRixJQUFJLE9BQU8sQ0FBTyxPQUFPLENBQUMsRUFBRTtnQkFDMUIsTUFBTSxDQUFDLGdCQUFnQixDQUFDLFNBQVMsRUFBRSxPQUFPLENBQUMsRUFBRTtvQkFDM0MsSUFBSSxPQUFPLENBQUMsSUFBSSxDQUFDLElBQUksS0FBSyxxQkFBcUIsRUFBRSxDQUFDO3dCQUFDLE9BQU87b0JBQUMsQ0FBQztvQkFDNUQsT0FBTyxFQUFFLENBQUM7Z0JBQ1osQ0FBQyxDQUFDLENBQUM7WUFDTCxDQUFDLENBQUM7U0FDSCxDQUFDLENBQUM7UUFFSCxNQUFNLHVCQUF1QixHQUFHLFFBQVEsQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLDZDQUE2QyxDQUFzQixDQUFDO1FBQ2hJLHVCQUF1QixDQUFDLEtBQUssRUFBRSxDQUFDO1FBRWhDLE1BQU0sVUFBVSxHQUFHLFFBQVEsQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLFlBQVksQ0FBbUIsQ0FBQztRQUMvRSxNQUFNLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsU0FBUyxDQUFDLENBQUMsUUFBUSxDQUFDLGdCQUFnQixDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUM7UUFFL0UsTUFBTSxXQUFXLEdBQUcsVUFBVSxDQUFDLGdCQUFnQixDQUFDLFdBQVcsQ0FBQyxDQUFDO1FBQzdELEtBQUssTUFBTSxVQUFVLElBQUksV0FBVyxFQUFFLENBQUM7WUFDckMsTUFBTSxDQUFDLE1BQU0sQ0FBQyxnQkFBZ0IsQ0FBQyxVQUFVLENBQUMsQ0FBQyxPQUFPLEtBQUssTUFBTSxDQUFDLENBQUM7UUFDakUsQ0FBQztJQUVILENBQUMsQ0FBQSxDQUFDLENBQUM7SUFFSCxJQUFJLENBQUMsaUJBQWlCLEVBQUUsR0FBUyxFQUFFOztRQUVqQyxNQUFNLFFBQVEsR0FBRyxhQUFhLENBQUM7UUFDL0IsTUFBTSxTQUFTLEdBQUcsWUFBWSxRQUFRLElBQUksQ0FBQztRQUUzQyxNQUFNLEVBQUUsS0FBSyxFQUFFLEdBQUcsRUFBRSxHQUFHLGdCQUFnQixFQUFFLENBQUM7UUFDMUMsTUFBTSxFQUFFLE1BQU0sRUFBRSxHQUFHLE1BQU0sTUFBTSxDQUFDLEdBQUcsRUFBRSxTQUFTLENBQUMsQ0FBQztRQUNoRCxNQUFNLEVBQUUsUUFBUSxFQUFFLEdBQUcsTUFBTSxDQUFDO1FBRTVCLEtBQUssQ0FBQyxVQUFVLEdBQUc7WUFDakIsTUFBTSxFQUFFLFNBQVM7U0FDbEIsQ0FBQztRQUVGLE1BQU0sT0FBTyxDQUFDLEdBQUcsQ0FBQztZQUNoQixJQUFJLE9BQU8sQ0FBTyxPQUFPLENBQUMsRUFBRTtnQkFDMUIsTUFBTSxDQUFDLGdCQUFnQixDQUFDLFNBQVMsRUFBRSxPQUFPLENBQUMsRUFBRTtvQkFDM0MsSUFBSSxPQUFPLENBQUMsSUFBSSxDQUFDLElBQUksS0FBSyxVQUFVLEVBQUUsQ0FBQzt3QkFBQyxPQUFPO29CQUFDLENBQUM7b0JBQ2pELE9BQU8sRUFBRSxDQUFDO2dCQUNaLENBQUMsQ0FBQyxDQUFDO1lBQ0wsQ0FBQyxDQUFDO1lBQ0YsSUFBSSxPQUFPLENBQU8sT0FBTyxDQUFDLEVBQUU7Z0JBQzFCLE1BQU0sQ0FBQyxnQkFBZ0IsQ0FBQyxTQUFTLEVBQUUsT0FBTyxDQUFDLEVBQUU7b0JBQzNDLElBQUksT0FBTyxDQUFDLElBQUksQ0FBQyxJQUFJLEtBQUsscUJBQXFCLEVBQUUsQ0FBQzt3QkFBQyxPQUFPO29CQUFDLENBQUM7b0JBQzVELE9BQU8sRUFBRSxDQUFDO2dCQUNaLENBQUMsQ0FBQyxDQUFDO1lBQ0wsQ0FBQyxDQUFDO1NBQ0gsQ0FBQyxDQUFDO1FBRUgsTUFBTSx1QkFBdUIsR0FBRyxRQUFRLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyw2Q0FBNkMsQ0FBc0IsQ0FBQztRQUNoSSx1QkFBdUIsQ0FBQyxLQUFLLEVBQUUsQ0FBQztRQUVoQyxNQUFNLGdCQUFnQixHQUFHLEtBQUssQ0FBQyxJQUFJLENBQ2pDLFFBQVEsQ0FBQyxJQUFJLENBQUMsZ0JBQWdCLENBQUMsd0NBQXdDLENBQUMsQ0FDbkQsQ0FBQztRQUV4QixNQUFNLENBQUMsZ0JBQWdCLENBQUMsTUFBTSxDQUFDLENBQUMsSUFBSSxDQUFDLHVCQUFXLENBQUMsVUFBVSxDQUFDLE1BQU0sQ0FBQyxDQUFDO1FBQ3BFLEtBQUssSUFBSSxDQUFDLEdBQUcsQ0FBQyxFQUFFLENBQUMsR0FBRyxnQkFBZ0IsQ0FBQyxNQUFNLEVBQUUsQ0FBQyxFQUFFLEVBQUUsQ0FBQztZQUNqRCxNQUFNLGVBQWUsR0FBRyxnQkFBZ0IsQ0FBQyxDQUFDLENBQUMsQ0FBQztZQUM1QyxlQUFlLENBQUMsS0FBSyxHQUFHLFFBQVEsQ0FBQyxFQUFFLENBQUM7WUFDcEMsZUFBZSxDQUFDLGFBQWEsQ0FBQyxJQUFJLE1BQU0sQ0FBQyxLQUFLLENBQUMsUUFBUSxDQUFDLENBQUMsQ0FBQztRQUM1RCxDQUFDO1FBRUQsTUFBTSxZQUFZLEdBQUcsUUFBUSxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsc0JBQXNCLENBQW1CLENBQUM7UUFDM0YsTUFBTSxDQUFDLFlBQVksQ0FBQyxXQUFXLENBQUMsQ0FBQyxJQUFJLENBQUMsdUJBQVcsQ0FBQyxVQUFVLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxFQUFFLEtBQUssRUFBRSxFQUFFLENBQUMsUUFBUSxLQUFLLEVBQUUsQ0FBQyxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDO1FBRTNHLG1GQUFtRjtRQUNuRixNQUFNLENBQUMsTUFBQSxLQUFLLENBQUMsVUFBVSwwQ0FBRyxDQUFDLENBQUMsQ0FBQyxDQUFDLE9BQU8sQ0FDbkMsdUJBQVcsQ0FBQyxVQUFVLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxFQUFFLEtBQUssRUFBRSxFQUFFLEdBQUcsT0FBTyxFQUFFLElBQUksRUFBRSxTQUFTLEVBQUUsS0FBSyxFQUFFLFFBQVEsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUNsRyxDQUFDO1FBRUYsK0RBQStEO1FBQy9ELE1BQU0sYUFBYSxHQUFHLFFBQVEsQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLG1DQUFtQyxDQUFzQixDQUFDO1FBQzVHLGFBQWEsQ0FBQyxLQUFLLEVBQUUsQ0FBQztRQUV0QixNQUFNLENBQUMsWUFBWSxDQUFDLFdBQVcsQ0FBQyxDQUFDLElBQUksQ0FBQyx1QkFBVyxDQUFDLFVBQVUsQ0FBQyxHQUFHLENBQUMsR0FBRyxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsSUFBSSxDQUFDLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUM7UUFFN0YsbUZBQW1GO1FBQ25GLE1BQU0sQ0FBQyxNQUFBLEtBQUssQ0FBQyxVQUFVLDBDQUFHLENBQUMsQ0FBQyxDQUFDLENBQUMsT0FBTyxDQUNuQyx1QkFBVyxDQUFDLFVBQVUsQ0FBQyxHQUFHLENBQUMsR0FBRyxDQUFDLEVBQUUsR0FBRyxPQUFPLEVBQUUsSUFBSSxFQUFFLFNBQVMsRUFBRSxLQUFLLEVBQUUsR0FBRyxDQUFDLElBQUksRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQ3BGLENBQUM7SUFFSixDQUFDLENBQUEsQ0FBQyxDQUFDO0lBRUgsSUFBSSxDQUFDLFNBQVMsRUFBRSxHQUFTLEVBQUU7O1FBRXpCLE1BQU0sUUFBUSxHQUFHLGFBQWEsQ0FBQztRQUMvQixNQUFNLFNBQVMsR0FBRyxZQUFZLFFBQVEsSUFBSSxDQUFDO1FBRTNDLE1BQU0sRUFBRSxLQUFLLEVBQUUsR0FBRyxFQUFFLEdBQUcsZ0JBQWdCLEVBQUUsQ0FBQztRQUMxQyxNQUFNLEVBQUUsTUFBTSxFQUFFLEdBQUcsTUFBTSxNQUFNLENBQUMsR0FBRyxFQUFFLFNBQVMsQ0FBQyxDQUFDO1FBQ2hELE1BQU0sRUFBRSxRQUFRLEVBQUUsR0FBRyxNQUFNLENBQUM7UUFFNUIsS0FBSyxDQUFDLFVBQVUsR0FBRztZQUNqQixNQUFNLEVBQUUsU0FBUztTQUNsQixDQUFDO1FBRUYsTUFBTSxPQUFPLENBQUMsR0FBRyxDQUFDO1lBQ2hCLElBQUksT0FBTyxDQUFPLE9BQU8sQ0FBQyxFQUFFO2dCQUMxQixNQUFNLENBQUMsZ0JBQWdCLENBQUMsU0FBUyxFQUFFLE9BQU8sQ0FBQyxFQUFFO29CQUMzQyxJQUFJLE9BQU8sQ0FBQyxJQUFJLENBQUMsSUFBSSxLQUFLLFVBQVUsRUFBRSxDQUFDO3dCQUFDLE9BQU87b0JBQUMsQ0FBQztvQkFDakQsT0FBTyxFQUFFLENBQUM7Z0JBQ1osQ0FBQyxDQUFDLENBQUM7WUFDTCxDQUFDLENBQUM7WUFDRixJQUFJLE9BQU8sQ0FBTyxPQUFPLENBQUMsRUFBRTtnQkFDMUIsTUFBTSxDQUFDLGdCQUFnQixDQUFDLFNBQVMsRUFBRSxPQUFPLENBQUMsRUFBRTtvQkFDM0MsSUFBSSxPQUFPLENBQUMsSUFBSSxDQUFDLElBQUksS0FBSyxxQkFBcUIsRUFBRSxDQUFDO3dCQUFDLE9BQU87b0JBQUMsQ0FBQztvQkFDNUQsT0FBTyxFQUFFLENBQUM7Z0JBQ1osQ0FBQyxDQUFDLENBQUM7WUFDTCxDQUFDLENBQUM7U0FDSCxDQUFDLENBQUM7UUFFSCxNQUFNLHVCQUF1QixHQUFHLFFBQVEsQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLDZDQUE2QyxDQUFzQixDQUFDO1FBQ2hJLHVCQUF1QixDQUFDLEtBQUssRUFBRSxDQUFDO1FBRWhDLHlDQUF5QztRQUN6QyxNQUFNLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyx3Q0FBd0MsQ0FBQyxDQUFDLE1BQU0sQ0FBQyxDQUFDLElBQUksQ0FBQyx1QkFBVyxDQUFDLFVBQVUsQ0FBQyxNQUFNLENBQUMsQ0FBQztRQUU1SCxNQUFNLG1CQUFtQixHQUFHLFFBQVEsQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLGdDQUFnQyxDQUFzQixDQUFDO1FBQy9HLG1CQUFtQixDQUFDLEtBQUssRUFBRSxDQUFDO1FBRTVCLG1FQUFtRTtRQUNuRSxNQUFNLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyx3Q0FBd0MsQ0FBQyxDQUFDLE1BQU0sQ0FBQyxDQUFDLElBQUksQ0FBQyx1QkFBVyxDQUFDLFVBQVUsQ0FBQyxNQUFNLEdBQUcsQ0FBQyxDQUFDLENBQUM7UUFFaEksbUdBQW1HO1FBQ25HLE1BQU0sWUFBWSxHQUFHLFFBQVEsQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLHNCQUFzQixDQUFtQixDQUFDO1FBQzNGLE1BQU0sQ0FBQyxZQUFZLENBQUMsV0FBVyxDQUFDLENBQUMsSUFBSSxDQUFDLHVCQUFXLENBQUMsVUFBVSxDQUFDLEdBQUcsQ0FBQyxHQUFHLENBQUMsRUFBRSxDQUFDLEdBQUcsQ0FBQyxJQUFJLENBQUMsQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQztRQUU3RiwyQkFBMkI7UUFDM0IsTUFBTSxNQUFNLEdBQVEsRUFBRSxJQUFJLEVBQUUsV0FBVyxFQUFFLENBQUM7UUFDMUMsTUFBTSxjQUFjLEdBQUcsUUFBUSxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMscURBQXFELENBQXFCLENBQUM7UUFDOUgsY0FBYyxDQUFDLEtBQUssR0FBRyxNQUFNLENBQUMsSUFBYyxDQUFDO1FBQzdDLGNBQWMsQ0FBQyxhQUFhLENBQUMsSUFBSSxNQUFNLENBQUMsS0FBSyxDQUFDLFFBQVEsQ0FBQyxDQUFDLENBQUM7UUFFekQsMkJBQTJCO1FBQzNCLE1BQU0sQ0FBQyxZQUFZLENBQUMsV0FBVyxDQUFDLENBQUMsSUFBSSxDQUFDLENBQUUsR0FBRyx1QkFBVyxDQUFDLFVBQVUsRUFBRSxNQUFNLENBQUUsQ0FBQyxHQUFHLENBQUMsR0FBRyxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsSUFBSSxDQUFDLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUM7UUFFNUcsbUZBQW1GO1FBQ25GLE1BQU0sQ0FBQyxNQUFBLEtBQUssQ0FBQyxVQUFVLDBDQUFHLENBQUMsQ0FBQyxDQUFDLENBQUMsT0FBTyxDQUNuQyxDQUFFLEdBQUcsdUJBQVcsQ0FBQyxVQUFVLEVBQUUsTUFBTSxDQUFFLENBQUMsR0FBRyxDQUFDLEdBQUcsQ0FBQyxFQUFFLEdBQUcsT0FBTyxFQUFFLElBQUksRUFBRSxTQUFTLEVBQUUsS0FBSyxFQUFFLEdBQUcsQ0FBQyxJQUFJLEVBQUUsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUNuRyxDQUFDO1FBRUYsK0RBQStEO1FBQy9ELE1BQU0sYUFBYSxHQUFHLFFBQVEsQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLG1DQUFtQyxDQUFzQixDQUFDO1FBQzVHLGFBQWEsQ0FBQyxLQUFLLEVBQUUsQ0FBQztRQUV0QixNQUFNLENBQUMsWUFBWSxDQUFDLFdBQVcsQ0FBQyxDQUFDLElBQUksQ0FBQyx1QkFBVyxDQUFDLFVBQVUsQ0FBQyxHQUFHLENBQUMsR0FBRyxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsSUFBSSxDQUFDLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUM7UUFFN0YsbUZBQW1GO1FBQ25GLE1BQU0sQ0FBQyxNQUFBLEtBQUssQ0FBQyxVQUFVLDBDQUFHLENBQUMsQ0FBQyxDQUFDLENBQUMsT0FBTyxDQUNuQyx1QkFBVyxDQUFDLFVBQVUsQ0FBQyxHQUFHLENBQUMsR0FBRyxDQUFDLEVBQUUsR0FBRyxPQUFPLEVBQUUsSUFBSSxFQUFFLFNBQVMsRUFBRSxLQUFLLEVBQUUsR0FBRyxDQUFDLElBQUksRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQ3BGLENBQUM7SUFFSixDQUFDLENBQUEsQ0FBQyxDQUFDO0lBRUgsSUFBSSxDQUFDLFlBQVksRUFBRSxHQUFTLEVBQUU7O1FBRTVCLE1BQU0sUUFBUSxHQUFHLGFBQWEsQ0FBQztRQUMvQixNQUFNLFNBQVMsR0FBRyxZQUFZLFFBQVEsSUFBSSxDQUFDO1FBRTNDLE1BQU0sRUFBRSxLQUFLLEVBQUUsR0FBRyxFQUFFLEdBQUcsZ0JBQWdCLEVBQUUsQ0FBQztRQUMxQyxNQUFNLEVBQUUsTUFBTSxFQUFFLEdBQUcsTUFBTSxNQUFNLENBQUMsR0FBRyxFQUFFLFNBQVMsQ0FBQyxDQUFDO1FBQ2hELE1BQU0sRUFBRSxRQUFRLEVBQUUsR0FBRyxNQUFNLENBQUM7UUFFNUIsS0FBSyxDQUFDLFVBQVUsR0FBRztZQUNqQixNQUFNLEVBQUUsU0FBUztTQUNsQixDQUFDO1FBRUYsTUFBTSxPQUFPLENBQUMsR0FBRyxDQUFDO1lBQ2hCLElBQUksT0FBTyxDQUFPLE9BQU8sQ0FBQyxFQUFFO2dCQUMxQixNQUFNLENBQUMsZ0JBQWdCLENBQUMsU0FBUyxFQUFFLE9BQU8sQ0FBQyxFQUFFO29CQUMzQyxJQUFJLE9BQU8sQ0FBQyxJQUFJLENBQUMsSUFBSSxLQUFLLFVBQVUsRUFBRSxDQUFDO3dCQUFDLE9BQU87b0JBQUMsQ0FBQztvQkFDakQsT0FBTyxFQUFFLENBQUM7Z0JBQ1osQ0FBQyxDQUFDLENBQUM7WUFDTCxDQUFDLENBQUM7WUFDRixJQUFJLE9BQU8sQ0FBTyxPQUFPLENBQUMsRUFBRTtnQkFDMUIsTUFBTSxDQUFDLGdCQUFnQixDQUFDLFNBQVMsRUFBRSxPQUFPLENBQUMsRUFBRTtvQkFDM0MsSUFBSSxPQUFPLENBQUMsSUFBSSxDQUFDLElBQUksS0FBSyxxQkFBcUIsRUFBRSxDQUFDO3dCQUFDLE9BQU87b0JBQUMsQ0FBQztvQkFDNUQsT0FBTyxFQUFFLENBQUM7Z0JBQ1osQ0FBQyxDQUFDLENBQUM7WUFDTCxDQUFDLENBQUM7U0FDSCxDQUFDLENBQUM7UUFFSCxNQUFNLHVCQUF1QixHQUFHLFFBQVEsQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLDZDQUE2QyxDQUFzQixDQUFDO1FBQ2hJLHVCQUF1QixDQUFDLEtBQUssRUFBRSxDQUFDO1FBRWhDLHlDQUF5QztRQUN6QyxNQUFNLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyx3Q0FBd0MsQ0FBQyxDQUFDLE1BQU0sQ0FBQyxDQUFDLElBQUksQ0FBQyx1QkFBVyxDQUFDLFVBQVUsQ0FBQyxNQUFNLENBQUMsQ0FBQztRQUU1SCxNQUFNLHNCQUFzQixHQUFHLFFBQVEsQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLG1DQUFtQyxDQUFzQixDQUFDO1FBQ3JILHNCQUFzQixDQUFDLEtBQUssRUFBRSxDQUFDO1FBRS9CLG1FQUFtRTtRQUNuRSxNQUFNLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyx3Q0FBd0MsQ0FBQyxDQUFDLE1BQU0sQ0FBQyxDQUFDLElBQUksQ0FBQyx1QkFBVyxDQUFDLFVBQVUsQ0FBQyxNQUFNLEdBQUcsQ0FBQyxDQUFDLENBQUM7UUFFaEksMkJBQTJCO1FBQzNCLHdEQUF3RDtRQUN4RCxNQUFNLFlBQVksR0FBRyxRQUFRLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxzQkFBc0IsQ0FBbUIsQ0FBQztRQUMzRixNQUFNLENBQUMsWUFBWSxDQUFDLFdBQVcsQ0FBQyxDQUFDLElBQUksQ0FBQyx1QkFBVyxDQUFDLFVBQVUsQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLENBQUMsR0FBRyxDQUFDLEdBQUcsQ0FBQyxFQUFFLENBQUMsR0FBRyxDQUFDLElBQUksQ0FBQyxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDO1FBRXRHLG1GQUFtRjtRQUNuRixNQUFNLENBQUMsTUFBQSxLQUFLLENBQUMsVUFBVSwwQ0FBRyxDQUFDLENBQUMsQ0FBQyxDQUFDLE9BQU8sQ0FDbkMsdUJBQVcsQ0FBQyxVQUFVLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQyxDQUFDLEdBQUcsQ0FBQyxHQUFHLENBQUMsRUFBRSxHQUFHLE9BQU8sRUFBRSxJQUFJLEVBQUUsU0FBUyxFQUFFLEtBQUssRUFBRSxHQUFHLENBQUMsSUFBSSxFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FDN0YsQ0FBQztRQUVGLCtEQUErRDtRQUMvRCxNQUFNLGFBQWEsR0FBRyxRQUFRLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxtQ0FBbUMsQ0FBc0IsQ0FBQztRQUM1RyxhQUFhLENBQUMsS0FBSyxFQUFFLENBQUM7UUFFdEIsTUFBTSxDQUFDLFlBQVksQ0FBQyxXQUFXLENBQUMsQ0FBQyxJQUFJLENBQUMsdUJBQVcsQ0FBQyxVQUFVLENBQUMsR0FBRyxDQUFDLEdBQUcsQ0FBQyxFQUFFLENBQUMsR0FBRyxDQUFDLElBQUksQ0FBQyxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDO1FBRTdGLG1GQUFtRjtRQUNuRixNQUFNLENBQUMsTUFBQSxLQUFLLENBQUMsVUFBVSwwQ0FBRyxDQUFDLENBQUMsQ0FBQyxDQUFDLE9BQU8sQ0FDbkMsdUJBQVcsQ0FBQyxVQUFVLENBQUMsR0FBRyxDQUFDLEdBQUcsQ0FBQyxFQUFFLEdBQUcsT0FBTyxFQUFFLElBQUksRUFBRSxTQUFTLEVBQUUsS0FBSyxFQUFFLEdBQUcsQ0FBQyxJQUFJLEVBQUUsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUNwRixDQUFDO0lBRUosQ0FBQyxDQUFBLENBQUMsQ0FBQztJQUVILElBQUksQ0FBQyxZQUFZLEVBQUUsR0FBUyxFQUFFOztRQUU1QixNQUFNLFFBQVEsR0FBRyxhQUFhLENBQUM7UUFDL0IsTUFBTSxTQUFTLEdBQUcsWUFBWSxRQUFRLElBQUksQ0FBQztRQUUzQyxNQUFNLEVBQUUsS0FBSyxFQUFFLEdBQUcsRUFBRSxHQUFHLGdCQUFnQixFQUFFLENBQUM7UUFDMUMsTUFBTSxFQUFFLE1BQU0sRUFBRSxHQUFHLE1BQU0sTUFBTSxDQUFDLEdBQUcsRUFBRSxTQUFTLENBQUMsQ0FBQztRQUNoRCxNQUFNLEVBQUUsUUFBUSxFQUFFLEdBQUcsTUFBTSxDQUFDO1FBRTVCLEtBQUssQ0FBQyxVQUFVLEdBQUc7WUFDakIsTUFBTSxFQUFFLFNBQVM7U0FDbEIsQ0FBQztRQUVGLE1BQU0sT0FBTyxDQUFDLEdBQUcsQ0FBQztZQUNoQixJQUFJLE9BQU8sQ0FBTyxPQUFPLENBQUMsRUFBRTtnQkFDMUIsTUFBTSxDQUFDLGdCQUFnQixDQUFDLFNBQVMsRUFBRSxPQUFPLENBQUMsRUFBRTtvQkFDM0MsSUFBSSxPQUFPLENBQUMsSUFBSSxDQUFDLElBQUksS0FBSyxVQUFVLEVBQUUsQ0FBQzt3QkFBQyxPQUFPO29CQUFDLENBQUM7b0JBQ2pELE9BQU8sRUFBRSxDQUFDO2dCQUNaLENBQUMsQ0FBQyxDQUFDO1lBQ0wsQ0FBQyxDQUFDO1lBQ0YsSUFBSSxPQUFPLENBQU8sT0FBTyxDQUFDLEVBQUU7Z0JBQzFCLE1BQU0sQ0FBQyxnQkFBZ0IsQ0FBQyxTQUFTLEVBQUUsT0FBTyxDQUFDLEVBQUU7b0JBQzNDLElBQUksT0FBTyxDQUFDLElBQUksQ0FBQyxJQUFJLEtBQUsscUJBQXFCLEVBQUUsQ0FBQzt3QkFBQyxPQUFPO29CQUFDLENBQUM7b0JBQzVELE9BQU8sRUFBRSxDQUFDO2dCQUNaLENBQUMsQ0FBQyxDQUFDO1lBQ0wsQ0FBQyxDQUFDO1NBQ0gsQ0FBQyxDQUFDO1FBRUgsTUFBTSx1QkFBdUIsR0FBRyxRQUFRLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyw2Q0FBNkMsQ0FBc0IsQ0FBQztRQUNoSSx1QkFBdUIsQ0FBQyxLQUFLLEVBQUUsQ0FBQztRQUVoQyx5Q0FBeUM7UUFDekMsTUFBTSxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsZ0JBQWdCLENBQUMsd0NBQXdDLENBQUMsQ0FBQyxNQUFNLENBQUMsQ0FBQyxJQUFJLENBQUMsdUJBQVcsQ0FBQyxVQUFVLENBQUMsTUFBTSxDQUFDLENBQUM7UUFFNUgsdUJBQXVCO1FBQ3ZCLE1BQU0sdUJBQXVCLEdBQUcsS0FBSyxDQUFDLElBQUksQ0FDeEMsUUFBUSxDQUFDLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxtQ0FBbUMsQ0FBQyxDQUM3QyxDQUFDO1FBQ3pCLEtBQUssTUFBTSxzQkFBc0IsSUFBSSx1QkFBdUIsRUFBRSxDQUFDO1lBQzdELHNCQUFzQixDQUFDLEtBQUssRUFBRSxDQUFDO1FBQ2pDLENBQUM7UUFFRCxtREFBbUQ7UUFDbkQsTUFBTSxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsZ0JBQWdCLENBQUMsd0NBQXdDLENBQUMsQ0FBQyxNQUFNLENBQUMsQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLENBQUM7UUFFaEcsMkJBQTJCO1FBQzNCLE1BQU0sWUFBWSxHQUFHLFFBQVEsQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLHNCQUFzQixDQUFtQixDQUFDO1FBQzNGLE1BQU0sQ0FBQyxZQUFZLENBQUMsV0FBVyxDQUFDLENBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxDQUFDO1FBRTFDLG1GQUFtRjtRQUNuRixNQUFNLENBQUMsTUFBQSxLQUFLLENBQUMsVUFBVSwwQ0FBRyxDQUFDLENBQUMsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxFQUFFLENBQUMsQ0FBQztRQUUxQywrREFBK0Q7UUFDL0QsTUFBTSxhQUFhLEdBQUcsUUFBUSxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsbUNBQW1DLENBQXNCLENBQUM7UUFDNUcsYUFBYSxDQUFDLEtBQUssRUFBRSxDQUFDO1FBRXRCLE1BQU0sQ0FBQyxZQUFZLENBQUMsV0FBVyxDQUFDLENBQUMsSUFBSSxDQUFDLHVCQUFXLENBQUMsVUFBVSxDQUFDLEdBQUcsQ0FBQyxHQUFHLENBQUMsRUFBRSxDQUFDLEdBQUcsQ0FBQyxJQUFJLENBQUMsQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQztRQUU3RixtRkFBbUY7UUFDbkYsTUFBTSxDQUFDLE1BQUEsS0FBSyxDQUFDLFVBQVUsMENBQUcsQ0FBQyxDQUFDLENBQUMsQ0FBQyxPQUFPLENBQ25DLHVCQUFXLENBQUMsVUFBVSxDQUFDLEdBQUcsQ0FBQyxHQUFHLENBQUMsRUFBRSxHQUFHLE9BQU8sRUFBRSxJQUFJLEVBQUUsU0FBUyxFQUFFLEtBQUssRUFBRSxHQUFHLENBQUMsSUFBSSxFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FDcEYsQ0FBQztJQUVKLENBQUMsQ0FBQSxDQUFDLENBQUM7QUFFTCxDQUFDLENBQUMsQ0FBQztBQUVILFFBQVEsQ0FBQyxxQkFBcUIsRUFBRSxHQUFHLEVBQUU7SUFDbkMsSUFBSSxDQUFDLG9DQUFvQyxFQUFFLEdBQVMsRUFBRTtRQUNwRCxNQUFNLEVBQUUsR0FBRyxFQUFFLEdBQUcsZ0JBQWdCLEVBQUUsQ0FBQztRQUNuQyxNQUFNLEVBQUUsTUFBTSxFQUFFLEdBQUcsTUFBTSxNQUFNLENBQUMsR0FBRyxDQUFDLENBQUM7UUFDckMsTUFBTSxFQUFFLFFBQVEsRUFBRSxHQUFHLE1BQU0sQ0FBQztRQUU1QixNQUFNLElBQUksT0FBTyxDQUFPLE9BQU8sQ0FBQyxFQUFFO1lBQ2hDLE1BQU0sQ0FBQyxnQkFBZ0IsQ0FBQyxTQUFTLEVBQUUsT0FBTyxDQUFDLEVBQUU7Z0JBQzNDLElBQUksT0FBTyxDQUFDLElBQUksQ0FBQyxJQUFJLEtBQUssVUFBVSxFQUFFLENBQUM7b0JBQUMsT0FBTztnQkFBQyxDQUFDO2dCQUNqRCxPQUFPLEVBQUUsQ0FBQztZQUNaLENBQUMsQ0FBQyxDQUFDO1FBQ0wsQ0FBQyxDQUFDLENBQUM7UUFFSCxtRUFBbUU7UUFDbkUsTUFBTSxtQkFBbUIsR0FBRyxRQUFRLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxzQkFBc0IsQ0FBd0IsQ0FBQztRQUN2RyxNQUFNLENBQUMsbUJBQW1CLENBQUMsQ0FBQyxHQUFHLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDO1FBQzNDLE1BQU0sQ0FBQyxtQkFBbUIsQ0FBQyxTQUFTLENBQUMsUUFBUSxDQUFDLFFBQVEsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDO1FBRXBFLGdEQUFnRDtRQUNoRCxNQUFNLG9CQUFvQixHQUFHLFFBQVEsQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLDhCQUE4QixDQUFxQixDQUFDO1FBQzdHLE1BQU0sQ0FBQyxvQkFBb0IsQ0FBQyxDQUFDLEdBQUcsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUM7UUFDNUMsTUFBTSxDQUFDLG9CQUFvQixDQUFDLElBQUksQ0FBQyxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsQ0FBQztJQUNyRCxDQUFDLENBQUEsQ0FBQyxDQUFDO0lBRUgsSUFBSSxDQUFDLHlDQUF5QyxFQUFFLEdBQVMsRUFBRTtRQUN6RCxNQUFNLEVBQUUsR0FBRyxFQUFFLEdBQUcsZ0JBQWdCLEVBQUUsQ0FBQztRQUNuQyxNQUFNLEVBQUUsTUFBTSxFQUFFLEdBQUcsTUFBTSxNQUFNLENBQUMsR0FBRyxDQUFDLENBQUM7UUFDckMsTUFBTSxFQUFFLFFBQVEsRUFBRSxHQUFHLE1BQU0sQ0FBQztRQUU1QixNQUFNLElBQUksT0FBTyxDQUFPLE9BQU8sQ0FBQyxFQUFFO1lBQ2hDLE1BQU0sQ0FBQyxnQkFBZ0IsQ0FBQyxTQUFTLEVBQUUsT0FBTyxDQUFDLEVBQUU7Z0JBQzNDLElBQUksT0FBTyxDQUFDLElBQUksQ0FBQyxJQUFJLEtBQUssVUFBVSxFQUFFLENBQUM7b0JBQUMsT0FBTztnQkFBQyxDQUFDO2dCQUNqRCxPQUFPLEVBQUUsQ0FBQztZQUNaLENBQUMsQ0FBQyxDQUFDO1FBQ0wsQ0FBQyxDQUFDLENBQUM7UUFFSCwyQ0FBMkM7UUFDM0MsTUFBTSxDQUFDLFdBQVcsQ0FBQztZQUNqQixJQUFJLEVBQUUscUJBQXFCO1lBQzNCLE9BQU8sRUFBRSxJQUFJO1NBQ2QsRUFBRSxHQUFHLENBQUMsQ0FBQztRQUVSLDZDQUE2QztRQUM3QyxNQUFNLElBQUksT0FBTyxDQUFDLE9BQU8sQ0FBQyxFQUFFLENBQUMsVUFBVSxDQUFDLE9BQU8sRUFBRSxFQUFFLENBQUMsQ0FBQyxDQUFDO1FBRXRELDZEQUE2RDtRQUM3RCxNQUFNLG1CQUFtQixHQUFHLFFBQVEsQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLHNCQUFzQixDQUF3QixDQUFDO1FBQ3ZHLE1BQU0sQ0FBQyxtQkFBbUIsQ0FBQyxTQUFTLENBQUMsUUFBUSxDQUFDLFFBQVEsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxDQUFDO1FBRXJFLHlEQUF5RDtRQUN6RCxNQUFNLG9CQUFvQixHQUFHLFFBQVEsQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLDhCQUE4QixDQUFxQixDQUFDO1FBQzdHLE1BQU0sQ0FBQyxvQkFBb0IsQ0FBQyxPQUFPLENBQUMsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUM7SUFDbEQsQ0FBQyxDQUFBLENBQUMsQ0FBQztJQUVILElBQUksQ0FBQywwQ0FBMEMsRUFBRSxHQUFTLEVBQUU7UUFDMUQsTUFBTSxFQUFFLEdBQUcsRUFBRSxHQUFHLGdCQUFnQixFQUFFLENBQUM7UUFDbkMsTUFBTSxFQUFFLE1BQU0sRUFBRSxHQUFHLE1BQU0sTUFBTSxDQUFDLEdBQUcsQ0FBQyxDQUFDO1FBQ3JDLE1BQU0sRUFBRSxRQUFRLEVBQUUsR0FBRyxNQUFNLENBQUM7UUFFNUIsTUFBTSxJQUFJLE9BQU8sQ0FBTyxPQUFPLENBQUMsRUFBRTtZQUNoQyxNQUFNLENBQUMsZ0JBQWdCLENBQUMsU0FBUyxFQUFFLE9BQU8sQ0FBQyxFQUFFO2dCQUMzQyxJQUFJLE9BQU8sQ0FBQyxJQUFJLENBQUMsSUFBSSxLQUFLLFVBQVUsRUFBRSxDQUFDO29CQUFDLE9BQU87Z0JBQUMsQ0FBQztnQkFDakQsT0FBTyxFQUFFLENBQUM7WUFDWixDQUFDLENBQUMsQ0FBQztRQUNMLENBQUMsQ0FBQyxDQUFDO1FBRUgsNEJBQTRCO1FBQzVCLE1BQU0sQ0FBQyxXQUFXLENBQUM7WUFDakIsSUFBSSxFQUFFLHFCQUFxQjtZQUMzQixPQUFPLEVBQUUsSUFBSTtTQUNkLEVBQUUsR0FBRyxDQUFDLENBQUM7UUFFUixNQUFNLElBQUksT0FBTyxDQUFDLE9BQU8sQ0FBQyxFQUFFLENBQUMsVUFBVSxDQUFDLE9BQU8sRUFBRSxFQUFFLENBQUMsQ0FBQyxDQUFDO1FBRXRELHNCQUFzQjtRQUN0QixNQUFNLG1CQUFtQixHQUFHLFFBQVEsQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLHNCQUFzQixDQUF3QixDQUFDO1FBQ3ZHLE1BQU0sQ0FBQyxtQkFBbUIsQ0FBQyxTQUFTLENBQUMsUUFBUSxDQUFDLFFBQVEsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxDQUFDO1FBRXJFLGlCQUFpQjtRQUNqQixNQUFNLENBQUMsV0FBVyxDQUFDO1lBQ2pCLElBQUksRUFBRSxxQkFBcUI7WUFDM0IsT0FBTyxFQUFFLEtBQUs7U0FDZixFQUFFLEdBQUcsQ0FBQyxDQUFDO1FBRVIsTUFBTSxJQUFJLE9BQU8sQ0FBQyxPQUFPLENBQUMsRUFBRSxDQUFDLFVBQVUsQ0FBQyxPQUFPLEVBQUUsRUFBRSxDQUFDLENBQUMsQ0FBQztRQUV0RCwyQkFBMkI7UUFDM0IsTUFBTSxDQUFDLG1CQUFtQixDQUFDLFNBQVMsQ0FBQyxRQUFRLENBQUMsUUFBUSxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUM7SUFDdEUsQ0FBQyxDQUFBLENBQUMsQ0FBQztJQUVILElBQUksQ0FBQyx5Q0FBeUMsRUFBRSxHQUFTLEVBQUU7O1FBQ3pELE1BQU0sRUFBRSxHQUFHLEVBQUUsS0FBSyxFQUFFLEdBQUcsZ0JBQWdCLEVBQUUsQ0FBQztRQUMxQyxNQUFNLEVBQUUsTUFBTSxFQUFFLEdBQUcsTUFBTSxNQUFNLENBQUMsR0FBRyxDQUFDLENBQUM7UUFDckMsTUFBTSxFQUFFLFFBQVEsRUFBRSxHQUFHLE1BQU0sQ0FBQztRQUU1QixNQUFNLElBQUksT0FBTyxDQUFPLE9BQU8sQ0FBQyxFQUFFO1lBQ2hDLE1BQU0sQ0FBQyxnQkFBZ0IsQ0FBQyxTQUFTLEVBQUUsT0FBTyxDQUFDLEVBQUU7Z0JBQzNDLElBQUksT0FBTyxDQUFDLElBQUksQ0FBQyxJQUFJLEtBQUssVUFBVSxFQUFFLENBQUM7b0JBQUMsT0FBTztnQkFBQyxDQUFDO2dCQUNqRCxPQUFPLEVBQUUsQ0FBQztZQUNaLENBQUMsQ0FBQyxDQUFDO1FBQ0wsQ0FBQyxDQUFDLENBQUM7UUFFSCw2QkFBNkI7UUFDN0IsTUFBTSxDQUFDLFdBQVcsQ0FBQztZQUNqQixJQUFJLEVBQUUscUJBQXFCO1lBQzNCLE9BQU8sRUFBRSxJQUFJO1NBQ2QsRUFBRSxHQUFHLENBQUMsQ0FBQztRQUVSLE1BQU0sSUFBSSxPQUFPLENBQUMsT0FBTyxDQUFDLEVBQUUsQ0FBQyxVQUFVLENBQUMsT0FBTyxFQUFFLEVBQUUsQ0FBQyxDQUFDLENBQUM7UUFFdEQsTUFBTSxvQkFBb0IsR0FBRyxRQUFRLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyw4QkFBOEIsQ0FBcUIsQ0FBQztRQUU3Ryw4QkFBOEI7UUFDOUIsTUFBTSxDQUFDLG9CQUFvQixDQUFDLE9BQU8sQ0FBQyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUNoRCxNQUFNLENBQUMsTUFBQSxLQUFLLENBQUMsVUFBVSwwQ0FBRSx5QkFBeUIsQ0FBQyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUUvRCx1QkFBdUI7UUFDdkIsb0JBQW9CLENBQUMsT0FBTyxHQUFHLEtBQUssQ0FBQztRQUNyQyxvQkFBb0IsQ0FBQyxhQUFhLENBQUMsSUFBSSxNQUFNLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUM7UUFFOUQsMEJBQTBCO1FBQzFCLE1BQU0sQ0FBQyxNQUFBLEtBQUssQ0FBQyxVQUFVLDBDQUFFLHlCQUF5QixDQUFDLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxDQUFDO1FBRWhFLGlCQUFpQjtRQUNqQixvQkFBb0IsQ0FBQyxPQUFPLEdBQUcsSUFBSSxDQUFDO1FBQ3BDLG9CQUFvQixDQUFDLGFBQWEsQ0FBQyxJQUFJLE1BQU0sQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQztRQUU5RCxnQ0FBZ0M7UUFDaEMsTUFBTSxDQUFDLE1BQUEsS0FBSyxDQUFDLFVBQVUsMENBQUUseUJBQXlCLENBQUMsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUM7SUFDakUsQ0FBQyxDQUFBLENBQUMsQ0FBQztJQUVILElBQUksQ0FBQyw4Q0FBOEMsRUFBRSxHQUFTLEVBQUU7UUFDOUQsTUFBTSxFQUFFLEdBQUcsRUFBRSxHQUFHLGdCQUFnQixFQUFFLENBQUM7UUFDbkMsTUFBTSxFQUFFLE1BQU0sRUFBRSxHQUFHLE1BQU0sTUFBTSxDQUFDLEdBQUcsQ0FBQyxDQUFDO1FBQ3JDLE1BQU0sRUFBRSxRQUFRLEVBQUUsR0FBRyxNQUFNLENBQUM7UUFFNUIsTUFBTSxJQUFJLE9BQU8sQ0FBTyxPQUFPLENBQUMsRUFBRTtZQUNoQyxNQUFNLENBQUMsZ0JBQWdCLENBQUMsU0FBUyxFQUFFLE9BQU8sQ0FBQyxFQUFFO2dCQUMzQyxJQUFJLE9BQU8sQ0FBQyxJQUFJLENBQUMsSUFBSSxLQUFLLFVBQVUsRUFBRSxDQUFDO29CQUFDLE9BQU87Z0JBQUMsQ0FBQztnQkFDakQsT0FBTyxFQUFFLENBQUM7WUFDWixDQUFDLENBQUMsQ0FBQztRQUNMLENBQUMsQ0FBQyxDQUFDO1FBRUgsNkJBQTZCO1FBQzdCLE1BQU0sQ0FBQyxXQUFXLENBQUM7WUFDakIsSUFBSSxFQUFFLHFCQUFxQjtZQUMzQixPQUFPLEVBQUUsSUFBSTtTQUNkLEVBQUUsR0FBRyxDQUFDLENBQUM7UUFFUixNQUFNLElBQUksT0FBTyxDQUFDLE9BQU8sQ0FBQyxFQUFFLENBQUMsVUFBVSxDQUFDLE9BQU8sRUFBRSxFQUFFLENBQUMsQ0FBQyxDQUFDO1FBRXRELE1BQU0sb0JBQW9CLEdBQUcsUUFBUSxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsOEJBQThCLENBQXFCLENBQUM7UUFFN0csOEJBQThCO1FBQzlCLE1BQU0sQ0FBQyxvQkFBb0IsQ0FBQyxPQUFPLENBQUMsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUM7UUFFaEQsNkJBQTZCO1FBQzdCLG9CQUFvQixDQUFDLE9BQU8sR0FBRyxLQUFLLENBQUM7UUFDckMsb0JBQW9CLENBQUMsYUFBYSxDQUFDLElBQUksTUFBTSxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDO1FBRTlELGtDQUFrQztRQUNsQyxNQUFNLENBQUMsV0FBVyxDQUFDO1lBQ2pCLElBQUksRUFBRSxxQkFBcUI7WUFDM0IsT0FBTyxFQUFFLElBQUk7U0FDZCxFQUFFLEdBQUcsQ0FBQyxDQUFDO1FBRVIsTUFBTSxJQUFJLE9BQU8sQ0FBQyxPQUFPLENBQUMsRUFBRSxDQUFDLFVBQVUsQ0FBQyxPQUFPLEVBQUUsRUFBRSxDQUFDLENBQUMsQ0FBQztRQUV0RCxvRUFBb0U7UUFDcEUsTUFBTSxDQUFDLG9CQUFvQixDQUFDLE9BQU8sQ0FBQyxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsQ0FBQztRQUVqRCwwQ0FBMEM7UUFDMUMsTUFBTSxtQkFBbUIsR0FBRyxRQUFRLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxzQkFBc0IsQ0FBd0IsQ0FBQztRQUN2RyxNQUFNLENBQUMsbUJBQW1CLENBQUMsU0FBUyxDQUFDLFFBQVEsQ0FBQyxRQUFRLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsQ0FBQztJQUN2RSxDQUFDLENBQUEsQ0FBQyxDQUFDO0lBRUgsSUFBSSxDQUFDLGdFQUFnRSxFQUFFLEdBQVMsRUFBRTs7UUFDaEYsTUFBTSxRQUFRLEdBQUcsYUFBYSxDQUFDO1FBQy9CLE1BQU0sU0FBUyxHQUFHLFlBQVksUUFBUSxJQUFJLENBQUM7UUFFM0MsTUFBTSxFQUFFLEdBQUcsRUFBRSxLQUFLLEVBQUUsR0FBRyxnQkFBZ0IsRUFBRSxDQUFDO1FBQzFDLE1BQU0sRUFBRSxNQUFNLEVBQUUsR0FBRyxNQUFNLE1BQU0sQ0FBQyxHQUFHLEVBQUUsU0FBUyxDQUFDLENBQUM7UUFDaEQsTUFBTSxFQUFFLFFBQVEsRUFBRSxHQUFHLE1BQU0sQ0FBQztRQUU1QixNQUFNLE9BQU8sQ0FBQyxHQUFHLENBQUM7WUFDaEIsSUFBSSxPQUFPLENBQU8sT0FBTyxDQUFDLEVBQUU7Z0JBQzFCLE1BQU0sQ0FBQyxnQkFBZ0IsQ0FBQyxTQUFTLEVBQUUsT0FBTyxDQUFDLEVBQUU7b0JBQzNDLElBQUksT0FBTyxDQUFDLElBQUksQ0FBQyxJQUFJLEtBQUssVUFBVSxFQUFFLENBQUM7d0JBQUMsT0FBTztvQkFBQyxDQUFDO29CQUNqRCxPQUFPLEVBQUUsQ0FBQztnQkFDWixDQUFDLENBQUMsQ0FBQztZQUNMLENBQUMsQ0FBQztZQUNGLElBQUksT0FBTyxDQUFPLE9BQU8sQ0FBQyxFQUFFO2dCQUMxQixNQUFNLENBQUMsZ0JBQWdCLENBQUMsU0FBUyxFQUFFLE9BQU8sQ0FBQyxFQUFFO29CQUMzQyxJQUFJLE9BQU8sQ0FBQyxJQUFJLENBQUMsSUFBSSxLQUFLLHFCQUFxQixFQUFFLENBQUM7d0JBQUMsT0FBTztvQkFBQyxDQUFDO29CQUM1RCxPQUFPLEVBQUUsQ0FBQztnQkFDWixDQUFDLENBQUMsQ0FBQztZQUNMLENBQUMsQ0FBQztTQUNILENBQUMsQ0FBQztRQUVILDZCQUE2QjtRQUM3QixNQUFNLENBQUMsV0FBVyxDQUFDO1lBQ2pCLElBQUksRUFBRSxxQkFBcUI7WUFDM0IsT0FBTyxFQUFFLElBQUk7U0FDZCxFQUFFLEdBQUcsQ0FBQyxDQUFDO1FBRVIsTUFBTSxJQUFJLE9BQU8sQ0FBQyxPQUFPLENBQUMsRUFBRSxDQUFDLFVBQVUsQ0FBQyxPQUFPLEVBQUUsRUFBRSxDQUFDLENBQUMsQ0FBQztRQUV0RCxvQkFBb0I7UUFDcEIsTUFBTSxvQkFBb0IsR0FBRyxRQUFRLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyw4QkFBOEIsQ0FBcUIsQ0FBQztRQUM3RyxNQUFNLENBQUMsb0JBQW9CLENBQUMsT0FBTyxDQUFDLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDO1FBQ2hELE1BQU0sQ0FBQyxNQUFBLEtBQUssQ0FBQyxVQUFVLDBDQUFFLHlCQUF5QixDQUFDLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDO1FBRS9ELHdDQUF3QztRQUN4QyxNQUFNLFNBQVMsR0FBRyxRQUFRLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyx5QkFBeUIsQ0FBcUIsQ0FBQztRQUM3RixTQUFTLENBQUMsS0FBSyxHQUFHLFVBQVUsQ0FBQztRQUM3QixTQUFTLENBQUMsYUFBYSxDQUFDLElBQUksTUFBTSxDQUFDLEtBQUssQ0FBQyxRQUFRLENBQUMsQ0FBQyxDQUFDO1FBRXBELDhDQUE4QztRQUM5QyxNQUFNLENBQUMsTUFBQSxLQUFLLENBQUMsVUFBVSwwQ0FBRSx5QkFBeUIsQ0FBQyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUUvRCx5RUFBeUU7UUFDekUsTUFBTSxhQUFhLEdBQUcsUUFBUSxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsU0FBUyxDQUFDLENBQUM7UUFDN0QsTUFBTSxDQUFDLGFBQWEsYUFBYixhQUFhLHVCQUFiLGFBQWEsQ0FBRSxTQUFTLENBQUMsUUFBUSxDQUFDLFVBQVUsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxDQUFDO0lBQ3BFLENBQUMsQ0FBQSxDQUFDLENBQUM7SUFFSCxJQUFJLENBQUMsK0NBQStDLEVBQUUsR0FBUyxFQUFFO1FBQy9ELE1BQU0sUUFBUSxHQUFHLGFBQWEsQ0FBQztRQUMvQixNQUFNLFNBQVMsR0FBRyxZQUFZLFFBQVEsSUFBSSxDQUFDO1FBRTNDLElBQUkscUJBQXFCLEdBQVEsSUFBSSxDQUFDO1FBQ3RDLE1BQU0sRUFBRSxHQUFHLEVBQUUsR0FBRyxnQkFBZ0IsRUFBRSxDQUFDO1FBRW5DLHlEQUF5RDtRQUN6RCxNQUFNLG1CQUFtQixHQUFHLEdBQUcsQ0FBQyxXQUFXLENBQUM7UUFDNUMsR0FBRyxDQUFDLFdBQVcsR0FBRyxDQUFDLE9BQVksRUFBRSxFQUFFO1lBQ2pDLElBQUksT0FBTyxDQUFDLEtBQUssS0FBSyxRQUFRLEVBQUUsQ0FBQztnQkFDL0IscUJBQXFCLEdBQUcsT0FBTyxDQUFDO1lBQ2xDLENBQUM7WUFDRCxPQUFPLG1CQUFtQixDQUFDLElBQUksQ0FBQyxHQUFHLEVBQUUsT0FBTyxDQUFDLENBQUM7UUFDaEQsQ0FBQyxDQUFDO1FBRUYsTUFBTSxFQUFFLE1BQU0sRUFBRSxHQUFHLE1BQU0sTUFBTSxDQUFDLEdBQUcsRUFBRSxTQUFTLENBQUMsQ0FBQztRQUNoRCxNQUFNLEVBQUUsUUFBUSxFQUFFLEdBQUcsTUFBTSxDQUFDO1FBRTVCLE1BQU0sT0FBTyxDQUFDLEdBQUcsQ0FBQztZQUNoQixJQUFJLE9BQU8sQ0FBTyxPQUFPLENBQUMsRUFBRTtnQkFDMUIsTUFBTSxDQUFDLGdCQUFnQixDQUFDLFNBQVMsRUFBRSxPQUFPLENBQUMsRUFBRTtvQkFDM0MsSUFBSSxPQUFPLENBQUMsSUFBSSxDQUFDLElBQUksS0FBSyxVQUFVLEVBQUUsQ0FBQzt3QkFBQyxPQUFPO29CQUFDLENBQUM7b0JBQ2pELE9BQU8sRUFBRSxDQUFDO2dCQUNaLENBQUMsQ0FBQyxDQUFDO1lBQ0wsQ0FBQyxDQUFDO1lBQ0YsSUFBSSxPQUFPLENBQU8sT0FBTyxDQUFDLEVBQUU7Z0JBQzFCLE1BQU0sQ0FBQyxnQkFBZ0IsQ0FBQyxTQUFTLEVBQUUsT0FBTyxDQUFDLEVBQUU7b0JBQzNDLElBQUksT0FBTyxDQUFDLElBQUksQ0FBQyxJQUFJLEtBQUsscUJBQXFCLEVBQUUsQ0FBQzt3QkFBQyxPQUFPO29CQUFDLENBQUM7b0JBQzVELE9BQU8sRUFBRSxDQUFDO2dCQUNaLENBQUMsQ0FBQyxDQUFDO1lBQ0wsQ0FBQyxDQUFDO1NBQ0gsQ0FBQyxDQUFDO1FBRUgsb0RBQW9EO1FBQ3BELE1BQU0sQ0FBQyxXQUFXLENBQUM7WUFDakIsSUFBSSxFQUFFLHFCQUFxQjtZQUMzQixPQUFPLEVBQUUsSUFBSTtTQUNkLEVBQUUsR0FBRyxDQUFDLENBQUM7UUFFUixNQUFNLElBQUksT0FBTyxDQUFDLE9BQU8sQ0FBQyxFQUFFLENBQUMsVUFBVSxDQUFDLE9BQU8sRUFBRSxFQUFFLENBQUMsQ0FBQyxDQUFDO1FBRXRELE1BQU0sb0JBQW9CLEdBQUcsUUFBUSxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsOEJBQThCLENBQXFCLENBQUM7UUFDN0csTUFBTSxDQUFDLG9CQUFvQixDQUFDLE9BQU8sQ0FBQyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUVoRCxnQ0FBZ0M7UUFDaEMsb0JBQW9CLENBQUMsT0FBTyxHQUFHLEtBQUssQ0FBQztRQUNyQyxvQkFBb0IsQ0FBQyxhQUFhLENBQUMsSUFBSSxNQUFNLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUM7UUFFOUQseUJBQXlCO1FBQ3pCLE1BQU0sYUFBYSxHQUFHLFFBQVEsQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLFNBQVMsQ0FBc0IsQ0FBQztRQUNsRixhQUFhLENBQUMsS0FBSyxFQUFFLENBQUM7UUFFdEIsb0VBQW9FO1FBQ3BFLE1BQU0sQ0FBQyxxQkFBcUIsQ0FBQyxDQUFDLEdBQUcsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUM7UUFDN0MsTUFBTSxDQUFDLHFCQUFxQixDQUFDLFVBQVUsQ0FBQyx5QkFBeUIsQ0FBQyxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsQ0FBQztJQUNqRixDQUFDLENBQUEsQ0FBQyxDQUFDO0lBRUgsSUFBSSxDQUFDLHVEQUF1RCxFQUFFLEdBQVMsRUFBRTtRQUN2RSxNQUFNLEVBQUUsR0FBRyxFQUFFLEdBQUcsZ0JBQWdCLEVBQUUsQ0FBQztRQUNuQyxNQUFNLEVBQUUsTUFBTSxFQUFFLEdBQUcsTUFBTSxNQUFNLENBQUMsR0FBRyxDQUFDLENBQUM7UUFDckMsTUFBTSxFQUFFLFFBQVEsRUFBRSxHQUFHLE1BQU0sQ0FBQztRQUU1QixNQUFNLElBQUksT0FBTyxDQUFPLE9BQU8sQ0FBQyxFQUFFO1lBQ2hDLE1BQU0sQ0FBQyxnQkFBZ0IsQ0FBQyxTQUFTLEVBQUUsT0FBTyxDQUFDLEVBQUU7Z0JBQzNDLElBQUksT0FBTyxDQUFDLElBQUksQ0FBQyxJQUFJLEtBQUssVUFBVSxFQUFFLENBQUM7b0JBQUMsT0FBTztnQkFBQyxDQUFDO2dCQUNqRCxPQUFPLEVBQUUsQ0FBQztZQUNaLENBQUMsQ0FBQyxDQUFDO1FBQ0wsQ0FBQyxDQUFDLENBQUM7UUFFSCxtQ0FBbUM7UUFDbkMsTUFBTSxDQUFDLFdBQVcsQ0FBQztZQUNqQixJQUFJLEVBQUUscUJBQXFCO1lBQzNCLE9BQU8sRUFBRSxJQUFJO1NBQ2QsRUFBRSxHQUFHLENBQUMsQ0FBQztRQUVSLE1BQU0sSUFBSSxPQUFPLENBQUMsT0FBTyxDQUFDLEVBQUUsQ0FBQyxVQUFVLENBQUMsT0FBTyxFQUFFLEVBQUUsQ0FBQyxDQUFDLENBQUM7UUFFdEQsa0NBQWtDO1FBQ2xDLE1BQU0sbUJBQW1CLEdBQUcsUUFBUSxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsc0JBQXNCLENBQXdCLENBQUM7UUFDdkcsTUFBTSxvQkFBb0IsR0FBRyxRQUFRLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyw4QkFBOEIsQ0FBcUIsQ0FBQztRQUU3RyxNQUFNLENBQUMsbUJBQW1CLENBQUMsU0FBUyxDQUFDLFFBQVEsQ0FBQyxRQUFRLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsQ0FBQztRQUNyRSxNQUFNLENBQUMsb0JBQW9CLENBQUMsT0FBTyxDQUFDLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDO1FBRWhELGlDQUFpQztRQUNqQyxNQUFNLENBQUMsV0FBVyxDQUFDO1lBQ2pCLElBQUksRUFBRSxxQkFBcUI7WUFDM0IsT0FBTyxFQUFFLEtBQUs7U0FDZixFQUFFLEdBQUcsQ0FBQyxDQUFDO1FBRVIsTUFBTSxJQUFJLE9BQU8sQ0FBQyxPQUFPLENBQUMsRUFBRSxDQUFDLFVBQVUsQ0FBQyxPQUFPLEVBQUUsRUFBRSxDQUFDLENBQUMsQ0FBQztRQUV0RCxrREFBa0Q7UUFDbEQsTUFBTSxDQUFDLG1CQUFtQixDQUFDLFNBQVMsQ0FBQyxRQUFRLENBQUMsUUFBUSxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUM7UUFDcEUsK0RBQStEO0lBQ2pFLENBQUMsQ0FBQSxDQUFDLENBQUM7SUFFSCxJQUFJLENBQUMsOEVBQThFLEVBQUUsR0FBUyxFQUFFO1FBQzlGLE1BQU0sRUFBRSxHQUFHLEVBQUUsR0FBRyxnQkFBZ0IsRUFBRSxDQUFDO1FBQ25DLE1BQU0sRUFBRSxNQUFNLEVBQUUsR0FBRyxNQUFNLE1BQU0sQ0FBQyxHQUFHLENBQUMsQ0FBQztRQUNyQyxNQUFNLEVBQUUsUUFBUSxFQUFFLEdBQUcsTUFBTSxDQUFDO1FBRTVCLE1BQU0sSUFBSSxPQUFPLENBQU8sT0FBTyxDQUFDLEVBQUU7WUFDaEMsTUFBTSxDQUFDLGdCQUFnQixDQUFDLFNBQVMsRUFBRSxPQUFPLENBQUMsRUFBRTtnQkFDM0MsSUFBSSxPQUFPLENBQUMsSUFBSSxDQUFDLElBQUksS0FBSyxVQUFVLEVBQUUsQ0FBQztvQkFBQyxPQUFPO2dCQUFDLENBQUM7Z0JBQ2pELE9BQU8sRUFBRSxDQUFDO1lBQ1osQ0FBQyxDQUFDLENBQUM7UUFDTCxDQUFDLENBQUMsQ0FBQztRQUVILDZCQUE2QjtRQUM3QixNQUFNLENBQUMsV0FBVyxDQUFDO1lBQ2pCLElBQUksRUFBRSxxQkFBcUI7WUFDM0IsT0FBTyxFQUFFLElBQUk7U0FDZCxFQUFFLEdBQUcsQ0FBQyxDQUFDO1FBRVIsTUFBTSxJQUFJLE9BQU8sQ0FBQyxPQUFPLENBQUMsRUFBRSxDQUFDLFVBQVUsQ0FBQyxPQUFPLEVBQUUsRUFBRSxDQUFDLENBQUMsQ0FBQztRQUV0RCxNQUFNLG1CQUFtQixHQUFHLFFBQVEsQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLHNCQUFzQixDQUF3QixDQUFDO1FBQ3ZHLE1BQU0sb0JBQW9CLEdBQUcsUUFBUSxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsOEJBQThCLENBQXFCLENBQUM7UUFFN0csNkJBQTZCO1FBQzdCLG9CQUFvQixDQUFDLE9BQU8sR0FBRyxLQUFLLENBQUM7UUFDckMsb0JBQW9CLENBQUMsYUFBYSxDQUFDLElBQUksTUFBTSxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDO1FBRTlELHNCQUFzQjtRQUN0QixNQUFNLENBQUMsV0FBVyxDQUFDO1lBQ2pCLElBQUksRUFBRSxxQkFBcUI7WUFDM0IsT0FBTyxFQUFFLEtBQUs7U0FDZixFQUFFLEdBQUcsQ0FBQyxDQUFDO1FBRVIsTUFBTSxJQUFJLE9BQU8sQ0FBQyxPQUFPLENBQUMsRUFBRSxDQUFDLFVBQVUsQ0FBQyxPQUFPLEVBQUUsRUFBRSxDQUFDLENBQUMsQ0FBQztRQUN0RCxNQUFNLENBQUMsbUJBQW1CLENBQUMsU0FBUyxDQUFDLFFBQVEsQ0FBQyxRQUFRLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUVwRSx3QkFBd0I7UUFDeEIsTUFBTSxDQUFDLFdBQVcsQ0FBQztZQUNqQixJQUFJLEVBQUUscUJBQXFCO1lBQzNCLE9BQU8sRUFBRSxJQUFJO1NBQ2QsRUFBRSxHQUFHLENBQUMsQ0FBQztRQUVSLE1BQU0sSUFBSSxPQUFPLENBQUMsT0FBTyxDQUFDLEVBQUUsQ0FBQyxVQUFVLENBQUMsT0FBTyxFQUFFLEVBQUUsQ0FBQyxDQUFDLENBQUM7UUFFdEQsd0ZBQXdGO1FBQ3hGLE1BQU0sQ0FBQyxtQkFBbUIsQ0FBQyxTQUFTLENBQUMsUUFBUSxDQUFDLFFBQVEsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxDQUFDO1FBQ3JFLE1BQU0sQ0FBQyxvQkFBb0IsQ0FBQyxPQUFPLENBQUMsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLENBQUM7SUFDbkQsQ0FBQyxDQUFBLENBQUMsQ0FBQztBQUNMLENBQUMsQ0FBQyxDQUFDO0FBRUgsUUFBUSxDQUFDLGNBQWMsRUFBRSxHQUFHLEVBQUU7SUFFNUIsdUJBQXVCO0lBQ3ZCLHlGQUF5RjtJQUV6RixJQUFJLENBQUMsc0NBQXNDLEVBQUUsR0FBUyxFQUFFO1FBRXRELE1BQU0sRUFBRSxHQUFHLEVBQUUsR0FBRyxnQkFBZ0IsRUFBRSxDQUFDO1FBQ25DLE1BQU0sRUFBRSxNQUFNLEVBQUUsR0FBRyxNQUFNLE1BQU0sQ0FBQyxHQUFHLENBQUMsQ0FBQztRQUNyQyxNQUFNLEVBQUUsUUFBUSxFQUFFLEdBQUcsTUFBTSxDQUFDO1FBRTVCLE1BQU0sSUFBSSxPQUFPLENBQU8sT0FBTyxDQUFDLEVBQUU7WUFDaEMsTUFBTSxDQUFDLGdCQUFnQixDQUFDLFNBQVMsRUFBRSxPQUFPLENBQUMsRUFBRTtnQkFDM0MsSUFBRyxPQUFPLENBQUMsSUFBSSxDQUFDLElBQUksS0FBSyxVQUFVLEVBQUUsQ0FBQztvQkFBQyxPQUFPO2dCQUFDLENBQUM7Z0JBQ2hELE9BQU8sRUFBRSxDQUFDO1lBQ1osQ0FBQyxDQUFDLENBQUM7UUFDTCxDQUFDLENBQUMsQ0FBQztRQUVILDJDQUEyQztRQUMzQyxNQUFNLGFBQWEsR0FBRyxRQUFRLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxTQUFTLENBQUMsQ0FBQztRQUM3RCxNQUFNLENBQUMsYUFBYSxhQUFiLGFBQWEsdUJBQWIsYUFBYSxDQUFFLFNBQVMsQ0FBQyxRQUFRLENBQUMsVUFBVSxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUM7UUFFakUsOEJBQThCO1FBQzlCLE1BQU0sU0FBUyxHQUFHLFFBQVEsQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLHlCQUF5QixDQUFxQixDQUFDO1FBQzdGLFNBQVMsQ0FBQyxLQUFLLEdBQUcsWUFBWSxDQUFDO1FBQy9CLFNBQVMsQ0FBQyxhQUFhLENBQUMsSUFBSSxNQUFNLENBQUMsS0FBSyxDQUFDLFFBQVEsQ0FBQyxDQUFDLENBQUM7UUFFcEQsTUFBTSxjQUFjLEdBQUcsUUFBUSxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsOEJBQThCLENBQXNCLENBQUM7UUFFeEcsTUFBTSxRQUFRLEdBQUcsSUFBSSxPQUFPLENBQU8sT0FBTyxDQUFDLEVBQUU7WUFDM0MsTUFBTSxDQUFDLGdCQUFnQixDQUFDLFNBQVMsRUFBRSxPQUFPLENBQUMsRUFBRTtnQkFDM0MsSUFBRyxPQUFPLENBQUMsSUFBSSxDQUFDLElBQUksS0FBSyxhQUFhLEVBQUUsQ0FBQztvQkFBQyxPQUFPO2dCQUFDLENBQUM7Z0JBQ25ELE9BQU8sRUFBRSxDQUFDO1lBQ1osQ0FBQyxDQUFDLENBQUM7UUFDTCxDQUFDLENBQUMsQ0FBQztRQUVILGNBQWMsQ0FBQyxLQUFLLEVBQUUsQ0FBQztRQUN2QixNQUFNLFFBQVEsQ0FBQztRQUVmLE1BQU0sSUFBSSxPQUFPLENBQU8sT0FBTyxDQUFDLEVBQUU7WUFDaEMsTUFBTSxDQUFDLGdCQUFnQixDQUFDLFNBQVMsRUFBRSxPQUFPLENBQUMsRUFBRTtnQkFDM0MsSUFBRyxPQUFPLENBQUMsSUFBSSxDQUFDLElBQUksS0FBSyxxQkFBcUIsRUFBRSxDQUFDO29CQUFDLE9BQU87Z0JBQUMsQ0FBQztnQkFDM0QsT0FBTyxFQUFFLENBQUM7WUFDWixDQUFDLENBQUMsQ0FBQztRQUNMLENBQUMsQ0FBQyxDQUFDO1FBRUgsOENBQThDO1FBQzlDLE1BQU0sQ0FBQyxhQUFhLGFBQWIsYUFBYSx1QkFBYixhQUFhLENBQUUsU0FBUyxDQUFDLFFBQVEsQ0FBQyxVQUFVLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsQ0FBQztJQUVwRSxDQUFDLENBQUEsQ0FBQyxDQUFDO0lBRUgsSUFBSSxDQUFDLHlDQUF5QyxFQUFFLEdBQVMsRUFBRTtRQUV6RCxNQUFNLFFBQVEsR0FBRyxhQUFhLENBQUM7UUFDL0IsTUFBTSxTQUFTLEdBQUcsWUFBWSxRQUFRLElBQUksQ0FBQztRQUUzQyxNQUFNLEVBQUUsR0FBRyxFQUFFLEtBQUssRUFBRSxHQUFHLGdCQUFnQixFQUFFLENBQUM7UUFDMUMsTUFBTSxFQUFFLE1BQU0sRUFBRSxHQUFHLE1BQU0sTUFBTSxDQUFDLEdBQUcsRUFBRSxTQUFTLENBQUMsQ0FBQztRQUNoRCxNQUFNLEVBQUUsUUFBUSxFQUFFLEdBQUcsTUFBTSxDQUFDO1FBRTVCLEtBQUssQ0FBQyxVQUFVLEdBQUc7WUFDakIsSUFBSSxFQUFFLFlBQVk7WUFDbEIsT0FBTyxFQUFFLE9BQU87WUFDaEIscUJBQXFCLEVBQUUsRUFBRTtZQUN6QixlQUFlLEVBQUUsRUFBRTtZQUNuQixNQUFNLEVBQUUsRUFBRTtZQUNWLGVBQWUsRUFBRSxFQUFFO1lBQ25CLGFBQWEsRUFBRSxFQUFFO1lBQ2pCLGdCQUFnQixFQUFFLENBQUUsRUFBRSxDQUFFO1NBQ3pCLENBQUM7UUFFRixNQUFNLE9BQU8sQ0FBQyxHQUFHLENBQUM7WUFDaEIsSUFBSSxPQUFPLENBQU8sT0FBTyxDQUFDLEVBQUU7Z0JBQzFCLE1BQU0sQ0FBQyxnQkFBZ0IsQ0FBQyxTQUFTLEVBQUUsT0FBTyxDQUFDLEVBQUU7b0JBQzNDLElBQUcsT0FBTyxDQUFDLElBQUksQ0FBQyxJQUFJLEtBQUssVUFBVSxFQUFFLENBQUM7d0JBQUMsT0FBTztvQkFBQyxDQUFDO29CQUNoRCxPQUFPLEVBQUUsQ0FBQztnQkFDWixDQUFDLENBQUMsQ0FBQztZQUNMLENBQUMsQ0FBQztZQUNGLElBQUksT0FBTyxDQUFPLE9BQU8sQ0FBQyxFQUFFO2dCQUMxQixNQUFNLENBQUMsZ0JBQWdCLENBQUMsU0FBUyxFQUFFLE9BQU8sQ0FBQyxFQUFFO29CQUMzQyxJQUFHLE9BQU8sQ0FBQyxJQUFJLENBQUMsSUFBSSxLQUFLLHFCQUFxQixFQUFFLENBQUM7d0JBQUMsT0FBTztvQkFBQyxDQUFDO29CQUMzRCxPQUFPLEVBQUUsQ0FBQztnQkFDWixDQUFDLENBQUMsQ0FBQztZQUNMLENBQUMsQ0FBQztTQUNILENBQUMsQ0FBQztRQUVILHFDQUFxQztRQUNyQywwQ0FBMEM7UUFDMUMsTUFBTSxhQUFhLEdBQUcsUUFBUSxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsU0FBUyxDQUFDLENBQUM7UUFDN0QsTUFBTSxDQUFDLGFBQWEsYUFBYixhQUFhLHVCQUFiLGFBQWEsQ0FBRSxTQUFTLENBQUMsUUFBUSxDQUFDLFVBQVUsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxDQUFDO1FBRWxFLHFCQUFxQjtRQUNyQixNQUFNLFNBQVMsR0FBRyxRQUFRLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyx5QkFBeUIsQ0FBcUIsQ0FBQztRQUM3RixTQUFTLENBQUMsS0FBSyxHQUFHLEVBQUUsQ0FBQztRQUNyQixTQUFTLENBQUMsYUFBYSxDQUFDLElBQUksTUFBTSxDQUFDLEtBQUssQ0FBQyxRQUFRLENBQUMsQ0FBQyxDQUFDO1FBRXBELG1FQUFtRTtRQUNuRSxNQUFNLENBQUMsYUFBYSxhQUFiLGFBQWEsdUJBQWIsYUFBYSxDQUFFLFNBQVMsQ0FBQyxRQUFRLENBQUMsVUFBVSxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUM7SUFFbkUsQ0FBQyxDQUFBLENBQUMsQ0FBQztJQUVILElBQUksQ0FBQywwQ0FBMEMsRUFBRSxHQUFHLEVBQUU7UUFDcEQsTUFBTSxDQUFDLElBQUEsaUJBQWEsRUFBQyxTQUFTLEVBQUUsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxDQUFDO0lBQ2pELENBQUMsQ0FBQyxDQUFDO0lBRUgsSUFBSSxDQUFDLHdDQUF3QyxFQUFFLEdBQUcsRUFBRTtRQUNsRCxNQUFNLFVBQVUsR0FBRyxTQUFTLEVBQXlCLENBQUM7UUFDdEQsVUFBVSxDQUFDLFVBQVUsQ0FBQyxNQUFNLEdBQUcsa0JBQWtCLENBQUM7UUFDbEQsVUFBVSxDQUFDLFVBQVUsQ0FBQyxJQUFJLEdBQUcsbUJBQW1CLENBQUM7UUFDakQsTUFBTSxDQUFDLElBQUEsaUJBQWEsRUFBQyxVQUFVLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQztJQUMvQyxDQUFDLENBQUMsQ0FBQztBQUdMLENBQUMsQ0FBQyxDQUFDIn0=