"use strict";
/*
 * languages.ts
 *
 * Copyright (C) 2022 by Posit Software, PBC
 *
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.LANGUAGES = exports.R_MARKDOWN = exports.QUARTO = exports.JUPYTER = exports.PYTHON = exports.R = exports.commandLineArgType = void 0;
exports.clearAfter = clearAfter;
exports.extension = extension;
exports.detectFileType = detectFileType;
exports.getScriptName = getScriptName;
exports.validScriptExtension = validScriptExtension;
const index_1 = require("./index");
function clearAfter(parentEl, childEl) {
    const children = Array.from(parentEl.children).reverse();
    for (const child of children) {
        if (child === childEl) {
            break;
        }
        parentEl.removeChild(child);
    }
}
function createVersionSelectorBuilder({ id, valueName, valueListName, labelText }) {
    return ({ currents = [], defaults = [], set = () => { } }) => {
        var _a;
        // always source the versions list from the defaults -- it should never change
        const listValue = defaults.find((dflt) => dflt.name === valueListName);
        const list = ((_a = listValue === null || listValue === void 0 ? void 0 : listValue.value) !== null && _a !== void 0 ? _a : []);
        // pick the value from the currents, because the user can modify it
        const value = currents.find((dflt) => dflt.name === valueName);
        const label = document.createElement("label");
        label.classList.add("col-sm-3", "control-label");
        label.textContent = labelText;
        label.setAttribute("for", id);
        const select = document.createElement("select");
        select.classList.add("form-control");
        select.id = id;
        for (const value of list) {
            const option = document.createElement("option");
            option.value = value.id;
            option.textContent = value.name;
            option.dataset.language = value.language;
            if (value.path) {
                option.dataset.path = value.path;
            }
            if (value.venv) {
                option.dataset.venv = value.venv;
            }
            select.appendChild(option);
        }
        const option = document.createElement("option");
        const OTHER_TEXT = "Other...";
        option.value = OTHER_TEXT;
        option.textContent = OTHER_TEXT;
        select.appendChild(option);
        const pathSelectionWrapper = document.createElement("div");
        pathSelectionWrapper.classList.add("form-grid");
        pathSelectionWrapper.style.gridColumn = "2 / -1";
        pathSelectionWrapper.style.display = "none";
        const pathInputEl = document.createElement("input");
        pathInputEl.classList.add("form-control");
        pathInputEl.type = "text";
        pathInputEl.style.width = "100%";
        pathInputEl.placeholder = "Path to executable";
        pathInputEl.id = `${id}-path`;
        pathInputEl.style.gridColumn = "1 / 3";
        const pathBrowseEl = document.createElement("button");
        pathBrowseEl.classList.add("secondary", "browse-button");
        pathBrowseEl.textContent = "Browse";
        pathSelectionWrapper.append(pathInputEl);
        pathSelectionWrapper.append(pathBrowseEl);
        pathBrowseEl.addEventListener("click", () => {
            (0, index_1.handleOnBrowse)(false, (event) => {
                const { message } = event.data;
                pathInputEl.value = message;
                pathInputEl.dispatchEvent(new Event("change"));
            });
        });
        pathInputEl.addEventListener("change", () => {
            set([
                {
                    name: `${valueName}-other`,
                    value: pathInputEl.value
                },
                {
                    name: valueName,
                    value: OTHER_TEXT
                }
            ]);
        });
        select.addEventListener("change", () => {
            pathSelectionWrapper.style.display = select.value !== OTHER_TEXT ? "none" : "";
            set([
                {
                    name: valueName,
                    value: select.value
                }
            ]);
        });
        // Populate other version path if it was specified
        const otherVersionIndex = currents.findIndex((dflt) => dflt.name === `${valueName}-other`);
        if (otherVersionIndex !== -1) {
            const otherVersionPath = currents[otherVersionIndex].value;
            if (otherVersionPath) {
                pathInputEl.value = otherVersionPath;
                select.value = OTHER_TEXT;
                pathInputEl.dispatchEvent(new Event("change"));
            }
        }
        else if (value === null || value === void 0 ? void 0 : value.value) {
            select.value = value.value;
        }
        // Trigger a change to select whatever version is currently displayed in the selector
        select.dispatchEvent(new Event("change"));
        // include a spacer div for the unused grid column
        // include a warning placeholder
        const warningEl = document.createElement("div");
        warningEl.id = "version-path-warning";
        warningEl.style.gridColumn = "2 / -1";
        warningEl.style.color = "var(--vscode-editorWarning-foreground)";
        warningEl.style.display = "none";
        return [label, select, document.createElement("div"), pathSelectionWrapper, warningEl];
    };
}
// create element builders/renderers for r versions and python verisons
const buildRVersionSelector = createVersionSelectorBuilder({
    id: "r-version-selector",
    valueListName: "r-versions",
    valueName: "r-version",
    labelText: "R version"
});
const buildPyVersionSelector = createVersionSelectorBuilder({
    id: "py-version-selector",
    valueListName: "py-versions",
    valueName: "py-version",
    labelText: "Python version"
});
function buildArgumentListEditor({ currents = [], defaults = [], set = () => { } }) {
    const result = [];
    let first;
    let last;
    // construct the linked list from defaults array
    const createLinkedList = (defaults) => {
        first = undefined;
        last = undefined;
        if (defaults.length) {
            first = { name: defaults[0].name, value: defaults[0].value };
            last = first;
            for (const dflt of defaults.slice(1)) {
                const current = Object.assign(Object.assign({}, dflt), { last });
                last.next = current;
                last = current;
            }
        }
    };
    createLinkedList(currents);
    // flatten the linked list to an array.
    // called when passing args array to `set` callback
    const argList = () => {
        const result = [];
        if (!last) {
            return result;
        }
        let current = last;
        while (current) {
            if (current.name || current.value) {
                // only include arguments that are not completely empty
                // we're traversing backwards, so we're using `unshift` to
                // preserve visual order
                result.unshift({ name: current.name, value: current.value });
            }
            current = current.last;
        }
        return result;
    };
    // create a mechanism for showing/hiding arguments UI
    const argLabel = document.createElement("label");
    argLabel.classList.add("col-sm-3", "control-label");
    argLabel.style.width = "unset";
    argLabel.textContent = "Arguments";
    const argSummary = document.createElement("div");
    argSummary.classList.add("form-control", "argument-summary");
    argSummary.style.width = "unset";
    argSummary.style.opacity = "60%";
    const updateArgPreview = () => {
        argSummary.textContent = argList()
            .map((arg) => { var _a; return (_a = arg.value) === null || _a === void 0 ? void 0 : _a.trim(); })
            .join(" ");
    };
    updateArgPreview();
    // hijack the set callback to trigger updates to the args preview
    // whenever the callback is called -- easier than remembering to call both
    const oldSet = set;
    set = (args) => {
        updateArgPreview();
        oldSet(args);
    };
    let showingArgs = false;
    const argShowHideButton = document.createElement("button");
    argShowHideButton.classList.add("browse-button", "secondary");
    argShowHideButton.addEventListener("click", () => {
        toggleArgs();
    });
    const toggleArgs = (override) => {
        var _a;
        showingArgs = override !== null && override !== void 0 ? override : !showingArgs;
        argShowHideButton.textContent = showingArgs ? "Hide" : "Modify";
        (_a = argShowHideButton.parentElement) === null || _a === void 0 ? void 0 : _a.classList.toggle("show-arguments", showingArgs);
        set(argList());
    };
    toggleArgs(false);
    result.push(argLabel, argSummary, argShowHideButton);
    const removers = new Set();
    // create the UI elements which manage a particular argument
    const createListItem = (argument) => {
        const valueInputEl = document.createElement("input");
        valueInputEl.type = "text";
        valueInputEl.placeholder = "Flag / Argument";
        valueInputEl.value = argument.value || "";
        valueInputEl.classList.add("form-control", "argument");
        valueInputEl.style.width = "unset";
        valueInputEl.style.gridColumn = "1 / 3";
        valueInputEl.addEventListener("change", () => {
            argument.value = valueInputEl.value;
            set(argList());
        });
        const deleteArgEl = document.createElement("button");
        deleteArgEl.classList.add("delete-button", "argument", "delete-argument");
        deleteArgEl.innerHTML = "&#x2715;";
        const remover = () => {
            if (argument.last) {
                argument.last.next = argument.next;
            }
            if (argument.next) {
                argument.next.last = argument.last;
            }
            // ensure that the last argument is always kept updated when removing
            // this is important so we can append new arguments correctly
            if (argument === last) {
                last = argument.last;
            }
            valueInputEl.remove();
            deleteArgEl.remove();
            removers.delete(remover);
            set(argList());
        };
        deleteArgEl.addEventListener("click", remover);
        removers.add(remover);
        return [valueInputEl, deleteArgEl];
    };
    const drawList = () => {
        let current = first;
        const result = [];
        while (current) {
            result.push(...createListItem(current));
            last = current;
            current = current.next;
        }
        return result;
    };
    result.push(...drawList());
    const addButton = document.createElement("button");
    addButton.classList.add("secondary", "argument", "add-argument");
    addButton.style.gridColumn = "1 / 3";
    addButton.textContent = "+ Add argument";
    addButton.addEventListener("click", () => {
        const argument = { last };
        if (last) {
            last.next = argument;
        }
        createListItem(argument).forEach((el) => addButton.insertAdjacentElement("beforebegin", el));
        last = argument;
        set(argList());
    });
    result.push(addButton);
    const resetButton = document.createElement("button");
    resetButton.classList.add("delete-button", "argument", "reset-arguments");
    resetButton.textContent = "Reset";
    resetButton.addEventListener("click", () => {
        // remove everything in the list first, then clear the removers cache
        // so they aren't called again another another reset
        removers.forEach((remover) => remover());
        removers.clear();
        // reset the linked list based off the defaults, not the currents
        createLinkedList(defaults);
        drawList().forEach((el) => addButton.insertAdjacentElement("beforebegin", el));
        // re-update the currents and redraw the preview
        set(argList());
    });
    result.push(resetButton);
    return result;
}
exports.commandLineArgType = [
    {
        name: "Command Line",
        type: "list",
        render: buildArgumentListEditor
    }
];
exports.R = "r";
exports.PYTHON = "py";
exports.JUPYTER = "ipynb";
exports.QUARTO = "qmd";
exports.R_MARKDOWN = "rmd";
exports.LANGUAGES = new Map()
    .set(exports.R, {
    name: "R",
    ext: exports.R,
    controls: [
        {
            name: "R Version",
            type: "select",
            render: buildRVersionSelector
        },
        ...exports.commandLineArgType
    ]
})
    .set(exports.PYTHON, {
    name: "Python",
    ext: exports.PYTHON,
    controls: [
        {
            name: "Python Version",
            type: "select",
            render: buildPyVersionSelector
        },
        ...exports.commandLineArgType
    ]
});
/*
  .set(JUPYTER, {
    name: 'Jupyter Notebook',
    ext: JUPYTER,
    controls: commandLineArgType
  })
  .set(QUARTO, {
    name: 'Quarto Markdown',
    ext: QUARTO,
    controls: commandLineArgType
  })
  .set(R_MARKDOWN, {
    name: 'R Markdown',
    ext: R_MARKDOWN,
    controls: commandLineArgType
  });
  */
function extension(path) {
    if (!path) {
        return "";
    }
    const ext = path.toLowerCase().split(".").at(-1);
    if (!ext) {
        return "";
    }
    return exports.LANGUAGES.has(ext) ? ext : "";
}
function detectFileType(path) {
    return exports.LANGUAGES.get(extension(path));
}
function getScriptName(path) {
    var _a;
    return ((_a = path.split("/").at(-1)) === null || _a === void 0 ? void 0 : _a.split(".").slice(0, -1).join("")) || "New Workbench Job";
}
function validScriptExtension(path) {
    var _a, _b;
    // output streams in VS Code count as files, and we don't want to
    // trigger changes when the user selects one of these.
    // Only continue if the active file has one of the extension types
    // we care about, defined in LANGAUGES.keys
    const langs = Array.from(exports.LANGUAGES.keys());
    // at(-1) accesses the element at the end of the array, which is the
    // extension in this case. LowerCase guarantees we can find it even if
    // the user has decided to change its capitalization (for whatever reason)
    return !path.includes("extension-output") && langs.includes((_b = (_a = path.split(".").at(-1)) === null || _a === void 0 ? void 0 : _a.toLowerCase()) !== null && _b !== void 0 ? _b : "");
}
//# sourceMappingURL=data:application/json;base64,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