// Copyright 2018-2024 the Deno authors. All rights reserved. MIT license.
// This module is browser compatible.
/**
 * Returns a tuple of two arrays with the first one containing all elements in
 * the given array that match the given predicate and the second one containing
 * all that do not.
 *
 * @template T The type of the elements in the array.
 *
 * @param array The array to partition.
 * @param predicate The predicate function to determine which array an element
 * belongs to.
 *
 * @returns A tuple of two arrays. The first array contains all elements that
 * match the predicate, the second contains all elements that do not.
 *
 * @example Basic usage
 * ```ts
 * import { partition } from "@std/collections/partition";
 * import { assertEquals } from "@std/assert/assert-equals";
 *
 * const numbers = [5, 6, 7, 8, 9];
 * const [even, odd] = partition(numbers, (it) => it % 2 === 0);
 *
 * assertEquals(even, [6, 8]);
 * assertEquals(odd, [5, 7, 9]);
 * ```
 */ export function partition(array, predicate) {
  const matches = [];
  const rest = [];
  for (const element of array){
    if (predicate(element)) {
      matches.push(element);
    } else {
      rest.push(element);
    }
  }
  return [
    matches,
    rest
  ];
}
//# sourceMappingURL=data:application/json;base64,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