// Copyright 2018-2024 the Deno authors. All rights reserved. MIT license.
// This module is browser compatible.
import { getLevelByName, getLevelName, LogLevels } from "./levels.ts";
export class LoggerConfig {
  level;
  handlers;
}
/**
 * An object that encapsulates provided message and arguments as well some
 * metadata that can be later used when formatting a message.
 */ export class LogRecord {
  msg;
  #args;
  #datetime;
  level;
  levelName;
  loggerName;
  constructor(options){
    this.msg = options.msg;
    this.#args = [
      ...options.args
    ];
    this.level = options.level;
    this.loggerName = options.loggerName;
    this.#datetime = new Date();
    this.levelName = getLevelName(options.level);
  }
  get args() {
    return [
      ...this.#args
    ];
  }
  get datetime() {
    return new Date(this.#datetime.getTime());
  }
}
export class Logger {
  #level;
  handlers;
  #loggerName;
  constructor(loggerName, levelName, options = {}){
    this.#loggerName = loggerName;
    this.#level = getLevelByName(levelName);
    this.handlers = options.handlers ?? [];
  }
  /** Use this to retrieve the current numeric log level. */ get level() {
    return this.#level;
  }
  /** Use this to set the numeric log level. */ set level(level) {
    try {
      this.#level = getLevelByName(getLevelName(level));
    } catch (_) {
      throw new TypeError(`Invalid log level: ${level}`);
    }
  }
  get levelName() {
    return getLevelName(this.#level);
  }
  set levelName(levelName) {
    this.#level = getLevelByName(levelName);
  }
  get loggerName() {
    return this.#loggerName;
  }
  /**
   * If the level of the logger is greater than the level to log, then nothing
   * is logged, otherwise a log record is passed to each log handler.  `msg` data
   * passed in is returned.  If a function is passed in, it is only evaluated
   * if the msg will be logged and the return value will be the result of the
   * function, not the function itself, unless the function isn't called, in which
   * case undefined is returned.  All types are coerced to strings for logging.
   */ #log(level, msg, ...args) {
    if (this.level > level) {
      return msg instanceof Function ? undefined : msg;
    }
    let fnResult;
    let logMessage;
    if (msg instanceof Function) {
      fnResult = msg();
      logMessage = this.asString(fnResult);
    } else {
      logMessage = this.asString(msg);
    }
    const record = new LogRecord({
      msg: logMessage,
      args: args,
      level: level,
      loggerName: this.loggerName
    });
    this.handlers.forEach((handler)=>{
      handler.handle(record);
    });
    return msg instanceof Function ? fnResult : msg;
  }
  asString(data, isProperty = false) {
    if (typeof data === "string") {
      if (isProperty) return `"${data}"`;
      return data;
    } else if (data === null || typeof data === "number" || typeof data === "bigint" || typeof data === "boolean" || typeof data === "undefined" || typeof data === "symbol") {
      return String(data);
    } else if (data instanceof Error) {
      return data.stack;
    } else if (typeof data === "object") {
      return `{${Object.entries(data).map(([k, v])=>`"${k}":${this.asString(v, true)}`).join(",")}}`;
    }
    return "undefined";
  }
  debug(msg, ...args) {
    return this.#log(LogLevels.DEBUG, msg, ...args);
  }
  info(msg, ...args) {
    return this.#log(LogLevels.INFO, msg, ...args);
  }
  warn(msg, ...args) {
    return this.#log(LogLevels.WARN, msg, ...args);
  }
  error(msg, ...args) {
    return this.#log(LogLevels.ERROR, msg, ...args);
  }
  critical(msg, ...args) {
    return this.#log(LogLevels.CRITICAL, msg, ...args);
  }
}
//# sourceMappingURL=data:application/json;base64,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