'use strict';

var nodeConfigProvider = require('@smithy/node-config-provider');
var nodeHttpHandler = require('@smithy/node-http-handler');
var protocolHttp = require('@smithy/protocol-http');
var utilStream = require('@smithy/util-stream');

var Endpoint;
(function (Endpoint) {
    Endpoint["IPv4"] = "http://169.254.169.254";
    Endpoint["IPv6"] = "http://[fd00:ec2::254]";
})(Endpoint || (Endpoint = {}));

var EndpointMode;
(function (EndpointMode) {
    EndpointMode["IPv4"] = "IPv4";
    EndpointMode["IPv6"] = "IPv6";
})(EndpointMode || (EndpointMode = {}));

const ENV_ENDPOINT_NAME = "AWS_EC2_METADATA_SERVICE_ENDPOINT";
const CONFIG_ENDPOINT_NAME = "ec2_metadata_service_endpoint";
const ENDPOINT_SELECTORS = {
    environmentVariableSelector: (env) => env[ENV_ENDPOINT_NAME],
    configFileSelector: (profile) => profile[CONFIG_ENDPOINT_NAME],
    default: Endpoint.IPv4,
};
({
    default: EndpointMode.IPv4,
});
const AWS_EC2_METADATA_V1_DISABLED = "AWS_EC2_METADATA_V1_DISABLED";
const PROFILE_AWS_EC2_METADATA_V1_DISABLED = "ec2_metadata_v1_disabled";
const IMDSv1_DISABLED_SELECTORS = {
    environmentVariableSelector: (env) => env[AWS_EC2_METADATA_V1_DISABLED] ? env[AWS_EC2_METADATA_V1_DISABLED] !== "false" : undefined,
    configFileSelector: (profile) => profile[PROFILE_AWS_EC2_METADATA_V1_DISABLED]
        ? profile[PROFILE_AWS_EC2_METADATA_V1_DISABLED] !== "false"
        : undefined,
    default: false,
};

class MetadataService {
    disableFetchToken;
    config;
    constructor(options = {}) {
        this.config = (async () => {
            const profile = options?.profile || process.env.AWS_PROFILE;
            return {
                endpoint: options.endpoint ?? (await nodeConfigProvider.loadConfig(ENDPOINT_SELECTORS, { profile })()),
                httpOptions: {
                    timeout: options?.httpOptions?.timeout || 0,
                },
                ec2MetadataV1Disabled: options?.ec2MetadataV1Disabled ?? (await nodeConfigProvider.loadConfig(IMDSv1_DISABLED_SELECTORS, { profile })()),
            };
        })();
        this.disableFetchToken = options?.disableFetchToken || false;
    }
    async request(path, options) {
        const { endpoint, ec2MetadataV1Disabled, httpOptions } = await this.config;
        const handler = new nodeHttpHandler.NodeHttpHandler({
            requestTimeout: httpOptions?.timeout,
            connectionTimeout: httpOptions?.timeout,
        });
        const endpointUrl = new URL(endpoint);
        const headers = options.headers || {};
        if (this.disableFetchToken && ec2MetadataV1Disabled) {
            throw new Error("IMDSv1 is disabled and fetching token is disabled, cannot make the request.");
        }
        if (!this.disableFetchToken) {
            try {
                headers["x-aws-ec2-metadata-token"] = await this.fetchMetadataToken();
            }
            catch (err) {
                if (ec2MetadataV1Disabled) {
                    throw err;
                }
            }
        }
        const request = new protocolHttp.HttpRequest({
            method: options.method || "GET",
            headers: headers,
            hostname: endpointUrl.hostname,
            path: endpointUrl.pathname + path,
            protocol: endpointUrl.protocol,
        });
        try {
            const { response } = await handler.handle(request, {});
            if (response.statusCode === 200 && response.body) {
                return utilStream.sdkStreamMixin(response.body).transformToString();
            }
            else {
                throw new Error(`Request failed with status code ${response.statusCode}`);
            }
        }
        catch (error) {
            throw new Error(`Error making request to the metadata service: ${error}`);
        }
    }
    async fetchMetadataToken() {
        const { endpoint, httpOptions } = await this.config;
        const handler = new nodeHttpHandler.NodeHttpHandler({
            requestTimeout: httpOptions?.timeout,
            connectionTimeout: httpOptions?.timeout,
        });
        const endpointUrl = new URL(endpoint);
        const tokenRequest = new protocolHttp.HttpRequest({
            method: "PUT",
            headers: {
                "x-aws-ec2-metadata-token-ttl-seconds": "21600",
            },
            hostname: endpointUrl.hostname,
            path: "/latest/api/token",
            protocol: endpointUrl.protocol,
        });
        try {
            const { response } = await handler.handle(tokenRequest, {});
            if (response.statusCode === 200 && response.body) {
                return utilStream.sdkStreamMixin(response.body).transformToString();
            }
            else {
                throw new Error(`Failed to fetch metadata token with status code ${response.statusCode}`);
            }
        }
        catch (error) {
            if (error?.statusCode === 400) {
                throw new Error(`Error fetching metadata token: ${error}`);
            }
            else if (error.message === "TimeoutError" || [403, 404, 405].includes(error.statusCode)) {
                this.disableFetchToken = true;
                throw new Error(`Error fetching metadata token: ${error}. disableFetchToken is enabled`);
            }
            throw new Error(`Error fetching metadata token: ${error}`);
        }
    }
}

exports.MetadataService = MetadataService;
