// Copyright 2018-2024 the Deno authors. All rights reserved. MIT license.
// Copyright (c) 2014 Jameson Little. MIT License.
// This module is browser compatible.
/**
 * Utilities for
 * {@link https://datatracker.ietf.org/doc/html/rfc4648#section-6 | base32}
 * encoding and decoding.
 *
 * Modified from {@link https://github.com/beatgammit/base64-js}.
 *
 * This module is browser compatible.
 *
 * ```ts
 * import { encodeBase32, decodeBase32 } from "https://deno.land/std@$STD_VERSION/encoding/base32.ts";
 *
 * const encoded = encodeBase32("foobar"); // "MZXW6YTBOI======"
 *
 * decodeBase32(encoded); // Uint8Array(6) [ 102, 111, 111, 98, 97, 114 ]
 * ```
 *
 * @module
 */ import { validateBinaryLike } from "./_util.ts";
const lookup = "ABCDEFGHIJKLMNOPQRSTUVWXYZ234567".split("");
const revLookup = [];
lookup.forEach((c, i)=>revLookup[c.charCodeAt(0)] = i);
const placeHolderPadLookup = [
  0,
  1,
  ,
  2,
  3,
  ,
  4
];
function _getPadLen(placeHoldersLen) {
  const maybeLen = placeHolderPadLookup[placeHoldersLen];
  if (typeof maybeLen !== "number") {
    throw new Error("Invalid pad length");
  }
  return maybeLen;
}
function getLens(b32) {
  const len = b32.length;
  if (len % 8 > 0) {
    throw new Error("Invalid string. Length must be a multiple of 8");
  }
  let validLen = b32.indexOf("=");
  if (validLen === -1) validLen = len;
  const placeHoldersLen = validLen === len ? 0 : 8 - validLen % 8;
  return [
    validLen,
    placeHoldersLen
  ];
}
function _byteLength(validLen, placeHoldersLen) {
  return (validLen + placeHoldersLen) * 5 / 8 - _getPadLen(placeHoldersLen);
}
/**
 * Decodes a base32-encoded string.
 *
 * @see {@link https://datatracker.ietf.org/doc/html/rfc4648#section-6}
 *
 * @param b32 The base32-encoded string to decode.
 * @returns The decoded data.
 *
 * @example
 * ```ts
 * import { decodeBase32 } from "https://deno.land/std@$STD_VERSION/encoding/base32.ts";
 *
 * decodeBase32("NRQMA===");
 * // Uint8Array(3) [ 108, 96, 192 ]
 * ```
 */ export function decodeBase32(b32) {
  let tmp;
  const [validLen, placeHoldersLen] = getLens(b32);
  const arr = new Uint8Array(_byteLength(validLen, placeHoldersLen));
  let curByte = 0;
  // if there are placeholders, only get up to the last complete 8 chars
  const len = placeHoldersLen > 0 ? validLen - 8 : validLen;
  let i;
  for(i = 0; i < len; i += 8){
    tmp = revLookup[b32.charCodeAt(i)] << 20 | revLookup[b32.charCodeAt(i + 1)] << 15 | revLookup[b32.charCodeAt(i + 2)] << 10 | revLookup[b32.charCodeAt(i + 3)] << 5 | revLookup[b32.charCodeAt(i + 4)];
    arr[curByte++] = tmp >> 17 & 0xff;
    arr[curByte++] = tmp >> 9 & 0xff;
    arr[curByte++] = tmp >> 1 & 0xff;
    tmp = (tmp & 1) << 15 | revLookup[b32.charCodeAt(i + 5)] << 10 | revLookup[b32.charCodeAt(i + 6)] << 5 | revLookup[b32.charCodeAt(i + 7)];
    arr[curByte++] = tmp >> 8 & 0xff;
    arr[curByte++] = tmp & 0xff;
  }
  if (placeHoldersLen === 1) {
    tmp = revLookup[b32.charCodeAt(i)] << 20 | revLookup[b32.charCodeAt(i + 1)] << 15 | revLookup[b32.charCodeAt(i + 2)] << 10 | revLookup[b32.charCodeAt(i + 3)] << 5 | revLookup[b32.charCodeAt(i + 4)];
    arr[curByte++] = tmp >> 17 & 0xff;
    arr[curByte++] = tmp >> 9 & 0xff;
    arr[curByte++] = tmp >> 1 & 0xff;
    tmp = (tmp & 1) << 7 | revLookup[b32.charCodeAt(i + 5)] << 2 | revLookup[b32.charCodeAt(i + 6)] >> 3;
    arr[curByte++] = tmp & 0xff;
  } else if (placeHoldersLen === 3) {
    tmp = revLookup[b32.charCodeAt(i)] << 19 | revLookup[b32.charCodeAt(i + 1)] << 14 | revLookup[b32.charCodeAt(i + 2)] << 9 | revLookup[b32.charCodeAt(i + 3)] << 4 | revLookup[b32.charCodeAt(i + 4)] >> 1;
    arr[curByte++] = tmp >> 16 & 0xff;
    arr[curByte++] = tmp >> 8 & 0xff;
    arr[curByte++] = tmp & 0xff;
  } else if (placeHoldersLen === 4) {
    tmp = revLookup[b32.charCodeAt(i)] << 11 | revLookup[b32.charCodeAt(i + 1)] << 6 | revLookup[b32.charCodeAt(i + 2)] << 1 | revLookup[b32.charCodeAt(i + 3)] >> 4;
    arr[curByte++] = tmp >> 8 & 0xff;
    arr[curByte++] = tmp & 0xff;
  } else if (placeHoldersLen === 6) {
    tmp = revLookup[b32.charCodeAt(i)] << 3 | revLookup[b32.charCodeAt(i + 1)] >> 2;
    arr[curByte++] = tmp & 0xff;
  }
  return arr;
}
function encodeChunk(uint8, start, end) {
  let tmp;
  const output = [];
  for(let i = start; i < end; i += 5){
    tmp = uint8[i] << 16 & 0xff0000 | uint8[i + 1] << 8 & 0xff00 | uint8[i + 2] & 0xff;
    output.push(lookup[tmp >> 19 & 0x1f]);
    output.push(lookup[tmp >> 14 & 0x1f]);
    output.push(lookup[tmp >> 9 & 0x1f]);
    output.push(lookup[tmp >> 4 & 0x1f]);
    tmp = (tmp & 0xf) << 16 | uint8[i + 3] << 8 & 0xff00 | uint8[i + 4] & 0xff;
    output.push(lookup[tmp >> 15 & 0x1f]);
    output.push(lookup[tmp >> 10 & 0x1f]);
    output.push(lookup[tmp >> 5 & 0x1f]);
    output.push(lookup[tmp & 0x1f]);
  }
  return output.join("");
}
/**
 * Converts data into a base32-encoded string.
 *
 * @see {@link https://datatracker.ietf.org/doc/html/rfc4648#section-6}
 *
 * @param data The data to encode.
 * @returns The base32-encoded string.
 *
 * @example
 * ```ts
 * import { encodeBase32 } from "https://deno.land/std@$STD_VERSION/encoding/base32.ts";
 *
 * encodeBase32("6c60c0"); // "NRQMA==="
 * ```
 */ export function encodeBase32(data) {
  const uint8 = validateBinaryLike(data);
  let tmp;
  const len = uint8.length;
  const extraBytes = len % 5;
  const parts = [];
  const maxChunkLength = 16385; // must be multiple of 5
  const len2 = len - extraBytes;
  // go through the array every 5 bytes, we'll deal with trailing stuff later
  for(let i = 0; i < len2; i += maxChunkLength){
    parts.push(encodeChunk(uint8, i, i + maxChunkLength > len2 ? len2 : i + maxChunkLength));
  }
  // pad the end with zeros, but make sure to not forget the extra bytes
  if (extraBytes === 4) {
    tmp = (uint8[len2] & 0xff) << 16 | (uint8[len2 + 1] & 0xff) << 8 | uint8[len2 + 2] & 0xff;
    parts.push(lookup[tmp >> 19 & 0x1f]);
    parts.push(lookup[tmp >> 14 & 0x1f]);
    parts.push(lookup[tmp >> 9 & 0x1f]);
    parts.push(lookup[tmp >> 4 & 0x1f]);
    tmp = (tmp & 0xf) << 11 | uint8[len2 + 3] << 3;
    parts.push(lookup[tmp >> 10 & 0x1f]);
    parts.push(lookup[tmp >> 5 & 0x1f]);
    parts.push(lookup[tmp & 0x1f]);
    parts.push("=");
  } else if (extraBytes === 3) {
    tmp = (uint8[len2] & 0xff) << 17 | (uint8[len2 + 1] & 0xff) << 9 | (uint8[len2 + 2] & 0xff) << 1;
    parts.push(lookup[tmp >> 20 & 0x1f]);
    parts.push(lookup[tmp >> 15 & 0x1f]);
    parts.push(lookup[tmp >> 10 & 0x1f]);
    parts.push(lookup[tmp >> 5 & 0x1f]);
    parts.push(lookup[tmp & 0x1f]);
    parts.push("===");
  } else if (extraBytes === 2) {
    tmp = (uint8[len2] & 0xff) << 12 | (uint8[len2 + 1] & 0xff) << 4;
    parts.push(lookup[tmp >> 15 & 0x1f]);
    parts.push(lookup[tmp >> 10 & 0x1f]);
    parts.push(lookup[tmp >> 5 & 0x1f]);
    parts.push(lookup[tmp & 0x1f]);
    parts.push("====");
  } else if (extraBytes === 1) {
    tmp = (uint8[len2] & 0xff) << 2;
    parts.push(lookup[tmp >> 5 & 0x1f]);
    parts.push(lookup[tmp & 0x1f]);
    parts.push("======");
  }
  return parts.join("");
}
//# sourceMappingURL=data:application/json;base64,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