#!/usr/bin/env bash
# This script will install extensions configured in positron.extensions.conf

set -e

server_bin_dir=/usr/lib/rstudio-server/bin

# eval POSITRON_ARGS to handle ~ or environment variables that might be in --user-data-dir
evaled_positron_args=$(eval "echo $POSITRON_ARGS")

function help() {
	cat <<EOF
usage: install-positron-ext.sh [options]

This script install extensions configured in positron.extensions.conf.
It expects the enviornment variable POSITRON_EXEC_COMMAND to be set to the path of the Positron binary.
If this is not set, it will look for the path in the positron.conf file.

Examples
   install-positron-ext.sh

Options
   -h
         Show this help.
EOF
}


while getopts h: flag
do
    case "${flag}" in
        h)
           help
           exit 0
           ;;
        *)
           echo "Invalid flag: ${flag}"
           exit 1
           ;;
    esac
done

# load and parse positron config file
configDirs="$XDG_CONFIG_DIRS:/etc"
origIFS=$IFS
IFS=':'; read -ra configDirList <<< "$configDirs"
for configDir in "${configDirList[@]}"
do
   if test -e "$configDir/rstudio/positron.conf"
   then
      configFile="$configDir/rstudio/positron.conf"
      break
   fi
done
IFS=$origIFS

# Default to use the environment variable set by Workbench
# If it's not been set, use the values in positron.conf 
# Otherwise exit with an error
if [[ -n "$POSITRON_EXEC_COMMAND" ]]
then
   cs_exe=$POSITRON_EXEC_COMMAND
elif [[ -f "$configFile" ]]
then
   cs_exe=$(grep "^[ \t]*exe" $configFile | sed 's/.*\(exe=\)\(.[^ ]*\).*/\2/')
   if [ -z "$cs_exe" ]; then
      # use the default
      cs_exe="$server_bin_dir/positron-server/bin/positron-server"
   fi
   ext_dir=$(grep 'extensions-dir[= ]' $configFile | sed 's/.*\(extensions-dir[= ]\)\(.[^ ]*\).*/\2/')
   if [ ! -z "$ext_dir" ] && [ ! -d "$ext_dir" ]
   then
      echo "$ext_dir does not exist - creating..."
      mkdir $ext_dir
   fi

   if [ -z "$ext_dir" ]; then
      echo "Warning: A global extensions directory is not configured. The extensions will only be installed for the user running this command. Set extensions-dir in positron.conf to install extensions globally."
   fi

   if [ -z "$evaluated_positron_args" ]; then
      evaled_positron_args=$(grep "^[ \t]*args" $configFile | sed 's/.*\(args=\)\(.*\).*/\2/')
   fi
else
   if [ -z "$POST_INSTALL_SCRIPT" ]; then
      echo "ERROR: Could not install extension. The Positron executable must be set by POSITRON_EXEC_COMMAND or as exe in positron.conf."
   fi
   exit 1
fi

# read the extension config file to see if we should install any extensions
extFile="$configDir/rstudio/positron.extensions.conf"
if test -e "$extFile"; then
   echo "Installing extensions from $extFile..."
   while read line
   do
      ext_arg=$(echo "$line" | sed 's/ *$//g')
      if [[ -n "$ext_arg" && ${ext_arg:0:1} != "#" ]]; then
         echo -e '\nRunning command "'$cs_exe' '$evaled_positron_args' --install-extension '$ext_arg'"...\n'
         echo $ext_arg | xargs $cs_exe $evaled_positron_args --install-extension
      fi
   done < "$extFile"
else
   echo "Did not locate $extFile, no extensions to install."
fi
