import { BytesList } from "../bytes/bytes_list.ts";
const CR = "\r".charCodeAt(0);
const LF = "\n".charCodeAt(0);
export class LineStream extends TransformStream {
    #bufs = new BytesList();
    #prevHadCR = false;
    constructor() {
        super({
            transform: (chunk, controller) => {
                this.#handle(chunk, controller);
            },
            flush: (controller) => {
                controller.enqueue(this.#mergeBufs(false));
            },
        });
    }
    #handle(chunk, controller) {
        const lfIndex = chunk.indexOf(LF);
        if (this.#prevHadCR) {
            this.#prevHadCR = false;
            if (lfIndex === 0) {
                controller.enqueue(this.#mergeBufs(true));
                this.#handle(chunk.subarray(1), controller);
                return;
            }
        }
        if (lfIndex === -1) {
            if (chunk.at(-1) === CR) {
                this.#prevHadCR = true;
            }
            this.#bufs.add(chunk);
        }
        else {
            let crOrLfIndex = lfIndex;
            if (chunk[lfIndex - 1] === CR) {
                crOrLfIndex--;
            }
            this.#bufs.add(chunk.subarray(0, crOrLfIndex));
            controller.enqueue(this.#mergeBufs(false));
            this.#handle(chunk.subarray(lfIndex + 1), controller);
        }
    }
    #mergeBufs(prevHadCR) {
        const mergeBuf = this.#bufs.concat();
        this.#bufs = new BytesList();
        if (prevHadCR) {
            return mergeBuf.subarray(0, -1);
        }
        else {
            return mergeBuf;
        }
    }
}
export class TextLineStream extends TransformStream {
    #buf = "";
    #prevHadCR = false;
    constructor() {
        super({
            transform: (chunk, controller) => {
                this.#handle(chunk, controller);
            },
            flush: (controller) => {
                controller.enqueue(this.#getBuf(false));
            },
        });
    }
    #handle(chunk, controller) {
        const lfIndex = chunk.indexOf("\n");
        if (this.#prevHadCR) {
            this.#prevHadCR = false;
            if (lfIndex === 0) {
                controller.enqueue(this.#getBuf(true));
                this.#handle(chunk.slice(1), controller);
                return;
            }
        }
        if (lfIndex === -1) {
            if (chunk.at(-1) === "\r") {
                this.#prevHadCR = true;
            }
            this.#buf += chunk;
        }
        else {
            let crOrLfIndex = lfIndex;
            if (chunk[lfIndex - 1] === "\r") {
                crOrLfIndex--;
            }
            this.#buf += chunk.slice(0, crOrLfIndex);
            controller.enqueue(this.#getBuf(false));
            this.#handle(chunk.slice(lfIndex + 1), controller);
        }
    }
    #getBuf(prevHadCR) {
        const buf = this.#buf;
        this.#buf = "";
        if (prevHadCR) {
            return buf.slice(0, -1);
        }
        else {
            return buf;
        }
    }
}
export class DelimiterStream extends TransformStream {
    #bufs = new BytesList();
    #delimiter;
    #inspectIndex = 0;
    #matchIndex = 0;
    #delimLen;
    #delimLPS;
    constructor(delimiter) {
        super({
            transform: (chunk, controller) => {
                this.#handle(chunk, controller);
            },
            flush: (controller) => {
                controller.enqueue(this.#bufs.concat());
            },
        });
        this.#delimiter = delimiter;
        this.#delimLen = delimiter.length;
        this.#delimLPS = createLPS(delimiter);
    }
    #handle(chunk, controller) {
        this.#bufs.add(chunk);
        let localIndex = 0;
        while (this.#inspectIndex < this.#bufs.size()) {
            if (chunk[localIndex] === this.#delimiter[this.#matchIndex]) {
                this.#inspectIndex++;
                localIndex++;
                this.#matchIndex++;
                if (this.#matchIndex === this.#delimLen) {
                    const matchEnd = this.#inspectIndex - this.#delimLen;
                    const readyBytes = this.#bufs.slice(0, matchEnd);
                    controller.enqueue(readyBytes);
                    this.#bufs.shift(this.#inspectIndex);
                    this.#inspectIndex = 0;
                    this.#matchIndex = 0;
                }
            }
            else {
                if (this.#matchIndex === 0) {
                    this.#inspectIndex++;
                    localIndex++;
                }
                else {
                    this.#matchIndex = this.#delimLPS[this.#matchIndex - 1];
                }
            }
        }
    }
}
export class TextDelimiterStream extends TransformStream {
    #buf = "";
    #delimiter;
    #inspectIndex = 0;
    #matchIndex = 0;
    #delimLPS;
    constructor(delimiter) {
        super({
            transform: (chunk, controller) => {
                this.#handle(chunk, controller);
            },
            flush: (controller) => {
                controller.enqueue(this.#buf);
            },
        });
        this.#delimiter = delimiter;
        this.#delimLPS = createLPS(new TextEncoder().encode(delimiter));
    }
    #handle(chunk, controller) {
        this.#buf += chunk;
        let localIndex = 0;
        while (this.#inspectIndex < this.#buf.length) {
            if (chunk[localIndex] === this.#delimiter[this.#matchIndex]) {
                this.#inspectIndex++;
                localIndex++;
                this.#matchIndex++;
                if (this.#matchIndex === this.#delimiter.length) {
                    const matchEnd = this.#inspectIndex - this.#delimiter.length;
                    const readyString = this.#buf.slice(0, matchEnd);
                    controller.enqueue(readyString);
                    this.#buf = this.#buf.slice(this.#inspectIndex);
                    this.#inspectIndex = 0;
                    this.#matchIndex = 0;
                }
            }
            else {
                if (this.#matchIndex === 0) {
                    this.#inspectIndex++;
                    localIndex++;
                }
                else {
                    this.#matchIndex = this.#delimLPS[this.#matchIndex - 1];
                }
            }
        }
    }
}
function createLPS(pat) {
    const lps = new Uint8Array(pat.length);
    lps[0] = 0;
    let prefixEnd = 0;
    let i = 1;
    while (i < lps.length) {
        if (pat[i] == pat[prefixEnd]) {
            prefixEnd++;
            lps[i] = prefixEnd;
            i++;
        }
        else if (prefixEnd === 0) {
            lps[i] = 0;
            i++;
        }
        else {
            prefixEnd = lps[prefixEnd - 1];
        }
    }
    return lps;
}
//# sourceMappingURL=data:application/json;base64,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