import { DateTimeFormatter } from "./formatter.ts";
export const SECOND = 1e3;
export const MINUTE = SECOND * 60;
export const HOUR = MINUTE * 60;
export const DAY = HOUR * 24;
export const WEEK = DAY * 7;
const DAYS_PER_WEEK = 7;
var Day;
(function (Day) {
    Day[Day["Sun"] = 0] = "Sun";
    Day[Day["Mon"] = 1] = "Mon";
    Day[Day["Tue"] = 2] = "Tue";
    Day[Day["Wed"] = 3] = "Wed";
    Day[Day["Thu"] = 4] = "Thu";
    Day[Day["Fri"] = 5] = "Fri";
    Day[Day["Sat"] = 6] = "Sat";
})(Day || (Day = {}));
export function parse(dateString, formatString) {
    const formatter = new DateTimeFormatter(formatString);
    const parts = formatter.parseToParts(dateString);
    const sortParts = formatter.sortDateTimeFormatPart(parts);
    return formatter.partsToDate(sortParts);
}
export function format(date, formatString) {
    const formatter = new DateTimeFormatter(formatString);
    return formatter.format(date);
}
export function dayOfYear(date) {
    const yearStart = new Date(date);
    yearStart.setUTCFullYear(date.getUTCFullYear(), 0, 0);
    const diff = date.getTime() -
        yearStart.getTime();
    return Math.floor(diff / DAY);
}
export function weekOfYear(date) {
    const workingDate = new Date(Date.UTC(date.getFullYear(), date.getMonth(), date.getDate()));
    const day = workingDate.getUTCDay();
    const nearestThursday = workingDate.getUTCDate() +
        Day.Thu -
        (day === Day.Sun ? DAYS_PER_WEEK : day);
    workingDate.setUTCDate(nearestThursday);
    const yearStart = new Date(Date.UTC(workingDate.getUTCFullYear(), 0, 1));
    return Math.ceil((workingDate.getTime() - yearStart.getTime() + DAY) / WEEK);
}
export function toIMF(date) {
    function dtPad(v, lPad = 2) {
        return v.padStart(lPad, "0");
    }
    const d = dtPad(date.getUTCDate().toString());
    const h = dtPad(date.getUTCHours().toString());
    const min = dtPad(date.getUTCMinutes().toString());
    const s = dtPad(date.getUTCSeconds().toString());
    const y = date.getUTCFullYear();
    const days = ["Sun", "Mon", "Tue", "Wed", "Thu", "Fri", "Sat"];
    const months = [
        "Jan",
        "Feb",
        "Mar",
        "Apr",
        "May",
        "Jun",
        "Jul",
        "Aug",
        "Sep",
        "Oct",
        "Nov",
        "Dec",
    ];
    return `${days[date.getUTCDay()]}, ${d} ${months[date.getUTCMonth()]} ${y} ${h}:${min}:${s} GMT`;
}
export function isLeap(year) {
    const yearNumber = year instanceof Date ? year.getFullYear() : year;
    return ((yearNumber % 4 === 0 && yearNumber % 100 !== 0) || yearNumber % 400 === 0);
}
export function difference(from, to, options) {
    const uniqueUnits = options?.units ? [...new Set(options?.units)] : [
        "milliseconds",
        "seconds",
        "minutes",
        "hours",
        "days",
        "weeks",
        "months",
        "quarters",
        "years",
    ];
    const bigger = Math.max(from.getTime(), to.getTime());
    const smaller = Math.min(from.getTime(), to.getTime());
    const differenceInMs = bigger - smaller;
    const differences = {};
    for (const uniqueUnit of uniqueUnits) {
        switch (uniqueUnit) {
            case "milliseconds":
                differences.milliseconds = differenceInMs;
                break;
            case "seconds":
                differences.seconds = Math.floor(differenceInMs / SECOND);
                break;
            case "minutes":
                differences.minutes = Math.floor(differenceInMs / MINUTE);
                break;
            case "hours":
                differences.hours = Math.floor(differenceInMs / HOUR);
                break;
            case "days":
                differences.days = Math.floor(differenceInMs / DAY);
                break;
            case "weeks":
                differences.weeks = Math.floor(differenceInMs / WEEK);
                break;
            case "months":
                differences.months = calculateMonthsDifference(bigger, smaller);
                break;
            case "quarters":
                differences.quarters = Math.floor((typeof differences.months !== "undefined" &&
                    differences.months / 4) ||
                    calculateMonthsDifference(bigger, smaller) / 4);
                break;
            case "years":
                differences.years = Math.floor((typeof differences.months !== "undefined" &&
                    differences.months / 12) ||
                    calculateMonthsDifference(bigger, smaller) / 12);
                break;
        }
    }
    return differences;
}
function calculateMonthsDifference(bigger, smaller) {
    const biggerDate = new Date(bigger);
    const smallerDate = new Date(smaller);
    const yearsDiff = biggerDate.getFullYear() - smallerDate.getFullYear();
    const monthsDiff = biggerDate.getMonth() - smallerDate.getMonth();
    const calendarDifferences = Math.abs(yearsDiff * 12 + monthsDiff);
    const compareResult = biggerDate > smallerDate ? 1 : -1;
    biggerDate.setMonth(biggerDate.getMonth() - compareResult * calendarDifferences);
    const isLastMonthNotFull = biggerDate > smallerDate
        ? 1
        : -1 === -compareResult
            ? 1
            : 0;
    const months = compareResult * (calendarDifferences - isLastMonthNotFull);
    return months === 0 ? 0 : months;
}
//# sourceMappingURL=data:application/json;base64,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