// Ported from js-yaml v3.13.1:
// https://github.com/nodeca/js-yaml/commit/665aadda42349dcae869f12040d9b10ef18d12da
// Copyright 2011-2015 by Vitaly Puzrin. All rights reserved. MIT license.
// Copyright 2018-2024 the Deno authors. All rights reserved. MIT license.
import { Type } from "../type.ts";
import { isNegativeZero } from "../_utils.ts";
const YAML_FLOAT_PATTERN = new RegExp(// 2.5e4, 2.5 and integers
"^(?:[-+]?(?:0|[1-9][0-9_]*)(?:\\.[0-9_]*)?(?:[eE][-+]?[0-9]+)?" + // .2e4, .2
// special case, seems not from spec
"|\\.[0-9_]+(?:[eE][-+]?[0-9]+)?" + // 20:59
"|[-+]?[0-9][0-9_]*(?::[0-5]?[0-9])+\\.[0-9_]*" + // .inf
"|[-+]?\\.(?:inf|Inf|INF)" + // .nan
"|\\.(?:nan|NaN|NAN))$");
function resolveYamlFloat(data) {
  if (!YAML_FLOAT_PATTERN.test(data) || // Quick hack to not allow integers end with `_`
  // Probably should update regexp & check speed
  data[data.length - 1] === "_") {
    return false;
  }
  return true;
}
function constructYamlFloat(data) {
  let value = data.replace(/_/g, "").toLowerCase();
  const sign = value[0] === "-" ? -1 : 1;
  const digits = [];
  if (value[0] && "+-".indexOf(value[0]) >= 0) {
    value = value.slice(1);
  }
  if (value === ".inf") {
    return sign === 1 ? Number.POSITIVE_INFINITY : Number.NEGATIVE_INFINITY;
  }
  if (value === ".nan") {
    return NaN;
  }
  if (value.indexOf(":") >= 0) {
    value.split(":").forEach((v)=>{
      digits.unshift(parseFloat(v));
    });
    let valueNb = 0.0;
    let base = 1;
    digits.forEach((d)=>{
      valueNb += d * base;
      base *= 60;
    });
    return sign * valueNb;
  }
  return sign * parseFloat(value);
}
const SCIENTIFIC_WITHOUT_DOT = /^[-+]?[0-9]+e/;
function representYamlFloat(object, style) {
  if (isNaN(object)) {
    switch(style){
      case "lowercase":
        return ".nan";
      case "uppercase":
        return ".NAN";
      case "camelcase":
        return ".NaN";
    }
  } else if (Number.POSITIVE_INFINITY === object) {
    switch(style){
      case "lowercase":
        return ".inf";
      case "uppercase":
        return ".INF";
      case "camelcase":
        return ".Inf";
    }
  } else if (Number.NEGATIVE_INFINITY === object) {
    switch(style){
      case "lowercase":
        return "-.inf";
      case "uppercase":
        return "-.INF";
      case "camelcase":
        return "-.Inf";
    }
  } else if (isNegativeZero(object)) {
    return "-0.0";
  }
  const res = object.toString(10);
  // JS stringifier can build scientific format without dots: 5e-100,
  // while YAML requires dot: 5.e-100. Fix it with simple hack
  return SCIENTIFIC_WITHOUT_DOT.test(res) ? res.replace("e", ".e") : res;
}
function isFloat(object) {
  return Object.prototype.toString.call(object) === "[object Number]" && (object % 1 !== 0 || isNegativeZero(object));
}
export const float = new Type("tag:yaml.org,2002:float", {
  construct: constructYamlFloat,
  defaultStyle: "lowercase",
  kind: "scalar",
  predicate: isFloat,
  represent: representYamlFloat,
  resolve: resolveYamlFloat
});
//# sourceMappingURL=data:application/json;base64,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