// Copyright 2018-2022 the Deno authors. All rights reserved. MIT license.
// This module is browser compatible.
import { BytesList } from "../bytes/bytes_list.ts";
const CR = "\r".charCodeAt(0);
const LF = "\n".charCodeAt(0);
/** @deprecated Use TextLineStream instead, as it can handle empty lines.
 *
 * Transform a stream into a stream where each chunk is divided by a newline,
 * be it `\n` or `\r\n`.
 *
 * ```ts
 * import { LineStream } from "./delimiter.ts";
 * const res = await fetch("https://example.com");
 * const lines = res.body!.pipeThrough(new LineStream());
 * ```
 */ export class LineStream extends TransformStream {
    #bufs = new BytesList();
    #prevHadCR = false;
    constructor(){
        super({
            transform: (chunk, controller)=>{
                this.#handle(chunk, controller);
            },
            flush: (controller)=>{
                controller.enqueue(this.#mergeBufs(false));
            }
        });
    }
    #handle(chunk, controller) {
        const lfIndex = chunk.indexOf(LF);
        if (this.#prevHadCR) {
            this.#prevHadCR = false;
            if (lfIndex === 0) {
                controller.enqueue(this.#mergeBufs(true));
                this.#handle(chunk.subarray(1), controller);
                return;
            }
        }
        if (lfIndex === -1) {
            if (chunk.at(-1) === CR) {
                this.#prevHadCR = true;
            }
            this.#bufs.add(chunk);
        } else {
            let crOrLfIndex = lfIndex;
            if (chunk[lfIndex - 1] === CR) {
                crOrLfIndex--;
            }
            this.#bufs.add(chunk.subarray(0, crOrLfIndex));
            controller.enqueue(this.#mergeBufs(false));
            this.#handle(chunk.subarray(lfIndex + 1), controller);
        }
    }
    #mergeBufs(prevHadCR) {
        const mergeBuf = this.#bufs.concat();
        this.#bufs = new BytesList();
        if (prevHadCR) {
            return mergeBuf.subarray(0, -1);
        } else {
            return mergeBuf;
        }
    }
}
/** Transform a stream into a stream where each chunk is divided by a newline,
 * be it `\n` or `\r\n`. `\r` can be enabled via the `allowCR` option.
 *
 * ```ts
 * import { TextLineStream } from "./delimiter.ts";
 * const res = await fetch("https://example.com");
 * const lines = res.body!
 *   .pipeThrough(new TextDecoderStream())
 *   .pipeThrough(new TextLineStream());
 * ```
 */ export class TextLineStream extends TransformStream {
    #allowCR;
    #buf = "";
    constructor(options){
        super({
            transform: (chunk, controller)=>this.#handle(chunk, controller),
            flush: (controller)=>this.#handle("\r\n", controller)
        });
        this.#allowCR = options?.allowCR ?? false;
    }
    #handle(chunk1, controller1) {
        chunk1 = this.#buf + chunk1;
        for(;;){
            const lfIndex1 = chunk1.indexOf("\n");
            if (this.#allowCR) {
                const crIndex = chunk1.indexOf("\r");
                if (crIndex !== -1 && crIndex !== chunk1.length - 1 && (lfIndex1 === -1 || lfIndex1 - 1 > crIndex)) {
                    controller1.enqueue(chunk1.slice(0, crIndex));
                    chunk1 = chunk1.slice(crIndex + 1);
                    continue;
                }
            }
            if (lfIndex1 !== -1) {
                let crOrLfIndex1 = lfIndex1;
                if (chunk1[lfIndex1 - 1] === "\r") {
                    crOrLfIndex1--;
                }
                controller1.enqueue(chunk1.slice(0, crOrLfIndex1));
                chunk1 = chunk1.slice(lfIndex1 + 1);
                continue;
            }
            break;
        }
        this.#buf = chunk1;
    }
}
/** Transform a stream into a stream where each chunk is divided by a given delimiter.
 *
 * ```ts
 * import { DelimiterStream } from "./delimiter.ts";
 * const res = await fetch("https://example.com");
 * const parts = res.body!
 *   .pipeThrough(new DelimiterStream(new TextEncoder().encode("foo")))
 *   .pipeThrough(new TextDecoderStream());
 * ```
 */ export class DelimiterStream extends TransformStream {
    #bufs = new BytesList();
    #delimiter;
    #inspectIndex = 0;
    #matchIndex = 0;
    #delimLen;
    #delimLPS;
    constructor(delimiter){
        super({
            transform: (chunk, controller)=>{
                this.#handle(chunk, controller);
            },
            flush: (controller)=>{
                controller.enqueue(this.#bufs.concat());
            }
        });
        this.#delimiter = delimiter;
        this.#delimLen = delimiter.length;
        this.#delimLPS = createLPS(delimiter);
    }
    #handle(chunk2, controller2) {
        this.#bufs.add(chunk2);
        let localIndex = 0;
        while(this.#inspectIndex < this.#bufs.size()){
            if (chunk2[localIndex] === this.#delimiter[this.#matchIndex]) {
                this.#inspectIndex++;
                localIndex++;
                this.#matchIndex++;
                if (this.#matchIndex === this.#delimLen) {
                    // Full match
                    const matchEnd = this.#inspectIndex - this.#delimLen;
                    const readyBytes = this.#bufs.slice(0, matchEnd);
                    controller2.enqueue(readyBytes);
                    // Reset match, different from KMP.
                    this.#bufs.shift(this.#inspectIndex);
                    this.#inspectIndex = 0;
                    this.#matchIndex = 0;
                }
            } else {
                if (this.#matchIndex === 0) {
                    this.#inspectIndex++;
                    localIndex++;
                } else {
                    this.#matchIndex = this.#delimLPS[this.#matchIndex - 1];
                }
            }
        }
    }
}
/** Transform a stream into a stream where each chunk is divided by a given delimiter.
 *
 * ```ts
 * import { TextDelimiterStream } from "./delimiter.ts";
 * const res = await fetch("https://example.com");
 * const parts = res.body!
 *   .pipeThrough(new TextDecoderStream())
 *   .pipeThrough(new TextDelimiterStream("foo"));
 * ```
 */ export class TextDelimiterStream extends TransformStream {
    #buf = "";
    #delimiter;
    #inspectIndex = 0;
    #matchIndex = 0;
    #delimLPS;
    constructor(delimiter){
        super({
            transform: (chunk, controller)=>{
                this.#handle(chunk, controller);
            },
            flush: (controller)=>{
                controller.enqueue(this.#buf);
            }
        });
        this.#delimiter = delimiter;
        this.#delimLPS = createLPS(new TextEncoder().encode(delimiter));
    }
    #handle(chunk3, controller3) {
        this.#buf += chunk3;
        let localIndex1 = 0;
        while(this.#inspectIndex < this.#buf.length){
            if (chunk3[localIndex1] === this.#delimiter[this.#matchIndex]) {
                this.#inspectIndex++;
                localIndex1++;
                this.#matchIndex++;
                if (this.#matchIndex === this.#delimiter.length) {
                    // Full match
                    const matchEnd1 = this.#inspectIndex - this.#delimiter.length;
                    const readyString = this.#buf.slice(0, matchEnd1);
                    controller3.enqueue(readyString);
                    // Reset match, different from KMP.
                    this.#buf = this.#buf.slice(this.#inspectIndex);
                    this.#inspectIndex = 0;
                    this.#matchIndex = 0;
                }
            } else {
                if (this.#matchIndex === 0) {
                    this.#inspectIndex++;
                    localIndex1++;
                } else {
                    this.#matchIndex = this.#delimLPS[this.#matchIndex - 1];
                }
            }
        }
    }
}
/** Generate longest proper prefix which is also suffix array. */ function createLPS(pat) {
    const lps = new Uint8Array(pat.length);
    lps[0] = 0;
    let prefixEnd = 0;
    let i = 1;
    while(i < lps.length){
        if (pat[i] == pat[prefixEnd]) {
            prefixEnd++;
            lps[i] = prefixEnd;
            i++;
        } else if (prefixEnd === 0) {
            lps[i] = 0;
            i++;
        } else {
            prefixEnd = lps[prefixEnd - 1];
        }
    }
    return lps;
}
//# sourceMappingURL=data:application/json;base64,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