// Copyright 2018-2022 the Deno authors. All rights reserved. MIT license.
/**
 * Functions for encoding binary data in array buffers.
 *
 * @module
 */ const rawTypeSizes = {
    int8: 1,
    uint8: 1,
    int16: 2,
    uint16: 2,
    int32: 4,
    uint32: 4,
    int64: 8,
    uint64: 8,
    float32: 4,
    float64: 8
};
/** Number of bytes required to store `dataType`. */ export function sizeof(dataType) {
    return rawTypeSizes[dataType];
}
/** Reads the exact number of bytes from `r` required to fill `b`.
 *
 * Throws `Deno.errors.UnexpectedEof` if `n` bytes cannot be read. */ export async function readExact(r, b) {
    let totalRead = 0;
    do {
        const tmp = new Uint8Array(b.length - totalRead);
        const nRead = await r.read(tmp);
        if (nRead === null) throw new Deno.errors.UnexpectedEof();
        b.set(tmp, totalRead);
        totalRead += nRead;
    }while (totalRead < b.length)
}
/** Reads exactly `n` bytes from `r`.
 *
 * Resolves it in a `Uint8Array`, or throws `Deno.errors.UnexpectedEof` if `n` bytes cannot be read. */ export async function getNBytes(r, n) {
    const scratch = new Uint8Array(n);
    await readExact(r, scratch);
    return scratch;
}
/** Decodes a number from `b`. If `o.bytes` is shorter than `sizeof(o.dataType)`, returns `null`.
 *
 * `o.dataType` defaults to `"int32"`. */ export function varnum(b, o = {}) {
    o.dataType = o.dataType ?? "int32";
    const littleEndian = (o.endian ?? "big") === "little" ? true : false;
    if (b.length < sizeof(o.dataType)) return null;
    const view = new DataView(b.buffer);
    switch(o.dataType){
        case "int8":
            return view.getInt8(b.byteOffset);
        case "uint8":
            return view.getUint8(b.byteOffset);
        case "int16":
            return view.getInt16(b.byteOffset, littleEndian);
        case "uint16":
            return view.getUint16(b.byteOffset, littleEndian);
        case "int32":
            return view.getInt32(b.byteOffset, littleEndian);
        case "uint32":
            return view.getUint32(b.byteOffset, littleEndian);
        case "float32":
            return view.getFloat32(b.byteOffset, littleEndian);
        case "float64":
            return view.getFloat64(b.byteOffset, littleEndian);
    }
}
/** Decodes a bigint from `b`. If `o.bytes` is shorter than `sizeof(o.dataType)`, returns `null`.
 *
 * `o.dataType` defaults to `"int64"`. */ export function varbig(b, o = {}) {
    o.dataType = o.dataType ?? "int64";
    const littleEndian = (o.endian ?? "big") === "little" ? true : false;
    if (b.length < sizeof(o.dataType)) return null;
    const view = new DataView(b.buffer);
    switch(o.dataType){
        case "int8":
            return BigInt(view.getInt8(b.byteOffset));
        case "uint8":
            return BigInt(view.getUint8(b.byteOffset));
        case "int16":
            return BigInt(view.getInt16(b.byteOffset, littleEndian));
        case "uint16":
            return BigInt(view.getUint16(b.byteOffset, littleEndian));
        case "int32":
            return BigInt(view.getInt32(b.byteOffset, littleEndian));
        case "uint32":
            return BigInt(view.getUint32(b.byteOffset, littleEndian));
        case "int64":
            return view.getBigInt64(b.byteOffset, littleEndian);
        case "uint64":
            return view.getBigUint64(b.byteOffset, littleEndian);
    }
}
/** Encodes number `x` into `b`. Returns the number of bytes used, or `0` if `b` is shorter than `sizeof(o.dataType)`.
 *
 * `o.dataType` defaults to `"int32"`. */ export function putVarnum(b, x, o = {}) {
    o.dataType = o.dataType ?? "int32";
    const littleEndian = (o.endian ?? "big") === "little" ? true : false;
    if (b.length < sizeof(o.dataType)) return 0;
    const view = new DataView(b.buffer);
    switch(o.dataType){
        case "int8":
            view.setInt8(b.byteOffset, x);
            break;
        case "uint8":
            view.setUint8(b.byteOffset, x);
            break;
        case "int16":
            view.setInt16(b.byteOffset, x, littleEndian);
            break;
        case "uint16":
            view.setUint16(b.byteOffset, x, littleEndian);
            break;
        case "int32":
            view.setInt32(b.byteOffset, x, littleEndian);
            break;
        case "uint32":
            view.setUint32(b.byteOffset, x, littleEndian);
            break;
        case "float32":
            view.setFloat32(b.byteOffset, x, littleEndian);
            break;
        case "float64":
            view.setFloat64(b.byteOffset, x, littleEndian);
            break;
    }
    return sizeof(o.dataType);
}
/** Encodes bigint `x` into `b`. Returns the number of bytes used, or `0` if `b` is shorter than `sizeof(o.dataType)`.
 *
 * `o.dataType` defaults to `"int64"`. */ export function putVarbig(b, x, o = {}) {
    o.dataType = o.dataType ?? "int64";
    const littleEndian = (o.endian ?? "big") === "little" ? true : false;
    if (b.length < sizeof(o.dataType)) return 0;
    const view = new DataView(b.buffer);
    switch(o.dataType){
        case "int8":
            view.setInt8(b.byteOffset, Number(x));
            break;
        case "uint8":
            view.setUint8(b.byteOffset, Number(x));
            break;
        case "int16":
            view.setInt16(b.byteOffset, Number(x), littleEndian);
            break;
        case "uint16":
            view.setUint16(b.byteOffset, Number(x), littleEndian);
            break;
        case "int32":
            view.setInt32(b.byteOffset, Number(x), littleEndian);
            break;
        case "uint32":
            view.setUint32(b.byteOffset, Number(x), littleEndian);
            break;
        case "int64":
            view.setBigInt64(b.byteOffset, x, littleEndian);
            break;
        case "uint64":
            view.setBigUint64(b.byteOffset, x, littleEndian);
            break;
    }
    return sizeof(o.dataType);
}
/** Decodes a number from `r`, consuming `sizeof(o.dataType)` bytes. If less than `sizeof(o.dataType)` bytes were read, throws `Deno.errors.unexpectedEof`.
 *
 * `o.dataType` defaults to `"int32"`. */ export async function readVarnum(r, o = {}) {
    o.dataType = o.dataType ?? "int32";
    const scratch = await getNBytes(r, sizeof(o.dataType));
    return varnum(scratch, o);
}
/** Decodes a bigint from `r`, consuming `sizeof(o.dataType)` bytes. If less than `sizeof(o.dataType)` bytes were read, throws `Deno.errors.unexpectedEof`.
 *
 * `o.dataType` defaults to `"int64"`. */ export async function readVarbig(r, o = {}) {
    o.dataType = o.dataType ?? "int64";
    const scratch = await getNBytes(r, sizeof(o.dataType));
    return varbig(scratch, o);
}
/** Encodes and writes `x` to `w`. Resolves to the number of bytes written.
 *
 * `o.dataType` defaults to `"int32"`. */ export function writeVarnum(w, x, o = {}) {
    o.dataType = o.dataType ?? "int32";
    const scratch = new Uint8Array(sizeof(o.dataType));
    putVarnum(scratch, x, o);
    return w.write(scratch);
}
/** Encodes and writes `x` to `w`. Resolves to the number of bytes written.
 *
 * `o.dataType` defaults to `"int64"`. */ export function writeVarbig(w, x, o = {}) {
    o.dataType = o.dataType ?? "int64";
    const scratch = new Uint8Array(sizeof(o.dataType));
    putVarbig(scratch, x, o);
    return w.write(scratch);
}
/** Encodes `x` into a new `Uint8Array`.
 *
 * `o.dataType` defaults to `"int32"` */ export function varnumBytes(x, o = {}) {
    o.dataType = o.dataType ?? "int32";
    const b = new Uint8Array(sizeof(o.dataType));
    putVarnum(b, x, o);
    return b;
}
/** Encodes `x` into a new `Uint8Array`.
 *
 * `o.dataType` defaults to `"int64"` */ export function varbigBytes(x, o = {}) {
    o.dataType = o.dataType ?? "int64";
    const b = new Uint8Array(sizeof(o.dataType));
    putVarbig(b, x, o);
    return b;
}
//# sourceMappingURL=data:application/json;base64,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